﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Include/CommandLineParserWrapper.h"
#include <Windows.h>
#include <util/UtilError.h>
#include <util/UtilErrorCode.h>
#include <cctype>

namespace nn {
namespace g3dTool {
namespace {

template< typename TType >
TType GetProcessAddr( const HMODULE& hModule, const char* pStr )
{
    TType returnValue = reinterpret_cast< TType >( GetProcAddress( hModule, pStr ) );
    if( returnValue == NULL )
    {
        THROW_ERROR(ERRCODE_INTERNAL, "Failed to get the DLL function pointer.");
    }

    return returnValue;
}

inline void FormatToNwOption( std::string* pOutStr, const char* pKey, const char* pReplaceTo )
{
    std::string::size_type pos = 0;
    pos = pOutStr->find( pKey, pos, 1 );
    while( pos != std::string::npos )
    {
        if( pos == std::string::npos )
        {
            break;	// 一文字も見つからない場合はアーリーアウト
        }
        pOutStr->replace( pos, 1, pReplaceTo );
        ++pos;
        pos = pOutStr->find( pKey, pos, 1 );
    }

    pOutStr->insert( 0, "--" );	// gfxTool のパーサではオプション名には "--" は含まれないため付加する。
}

inline void FormatToNwOption( std::string* pOutStr, const std::string& suffics, const char* pTarget, const char* pReplaceTo )
{
    FormatToNwOption( pOutStr, pTarget, pReplaceTo );
    *pOutStr += "=";
    *pOutStr += suffics;
}

//!< @brief 旧 nw が内部で扱っているフォーマットに成形し、pOutStrArray に保存します。"フルオプション名=値" に整形します。
template< typename TType >
inline void ConvertToNwOptionNameFmt( std::vector< std::wstring >* pOutStrArray, const TType& option )
{
    if( !option.IsExisting() )
    {
        return;
    }

    if( option.GetOptionValueType() == nn::gfxTool::OptionValueType::None )
    {
        std::string oldFormat( option.GetDefinition().pLongName );
        FormatToNwOption( &oldFormat, "-", "-" );
        pOutStrArray->push_back( std::wstring( oldFormat.begin(), oldFormat.end() ) );
    }
    else if( option.GetOptionValueType() == nn::gfxTool::OptionValueType::One )
    {
        std::string oldFormat( option.GetDefinition().pLongName );

        // option の値が文字列の時に'\"'が入る場合がある（ args-file に output="hoge.bfres" と記述した場合等）
        // なので、文字列の先頭と最後の'\"'を除去しておく。
        std::string optionValue( option.GetValue() );
        if( *optionValue.begin() == '\"' && *( optionValue.end() - 1 ) == '\"' )
        {
            optionValue.erase( optionValue.begin() );
            optionValue.erase( optionValue.end() - 1 );
        }

        FormatToNwOption( &oldFormat, optionValue, "-", "-" );

        pOutStrArray->push_back( std::wstring( oldFormat.begin(), oldFormat.end() ) );
    }
}

template<>
inline void ConvertToNwOptionNameFmt
    <nn::gfxTool::CommandLineOption<nn::gfxTool::Custom<std::string>::Type[]>>
    ( std::vector< std::wstring >* pOutStrArray,
    const nn::gfxTool::CommandLineOption<nn::gfxTool::Custom<std::string>::Type[]>& option )
{
    // ここでは "option name = val0" "option name = val1" ... という風に文字列バッファに入れている。
    auto& optValues = option.GetValue();
    for( int optValueIdx = 0; optValueIdx < optValues.size(); ++optValueIdx )
    {
        std::string oldFormat( option.GetDefinition().pLongName );

        // option の値が文字列の時に'\"'が入る場合がある（ args-file に output="hoge.bfres" と記述した場合等）
        // なので、文字列の先頭と最後の'\"'を除去しておく。
        std::string optionValue( optValues[ optValueIdx ].c_str() );
        if( *optionValue.begin() == '\"' && *( optionValue.end() - 1 ) == '\"' )
        {
            optionValue.erase( optionValue.begin() );
            optionValue.erase( optionValue.end() - 1 );
        }

        FormatToNwOption( &oldFormat, optionValue, "-", "-" );

        pOutStrArray->push_back( std::wstring( oldFormat.begin(), oldFormat.end() ) );
    }
}

template<>
inline void ConvertToNwOptionNameFmt< nn::gfxTool::CommandLineOption<> >(
    std::vector< std::wstring >* pOutStrArray, const nn::gfxTool::CommandLineOption<>& option )
{
    if( !option.IsExisting() )
    {
        return;
    }

    if( option.GetOptionValueType() == nn::gfxTool::OptionValueType::None )
    {
        std::string oldFormat( option.GetDefinition().pLongName );
        FormatToNwOption( &oldFormat, "-", "-" );
        pOutStrArray->push_back( std::wstring( oldFormat.begin(), oldFormat.end() ) );
    }
}

}

CommandLineParserWrapper::CommandLineParserWrapper( int argc, const wchar_t* argv[] )
{
    m_pGlobalOptionParser.reset( new GlobalOptionParser() );

    // wchar から char に変換します。
    for( int idx = 0; idx < argc; ++idx )
    {
        std::wstring str( argv[ idx ] );
        m_CharBuff.push_back( std::string( str.begin(), str.end() ) );
    }

    m_pGlobalOptionParser->Parse( m_CharBuff );

    // 必須オプションのチェックを行います。
    {
        auto inputArgs	= m_pGlobalOptionParser->GetInputArgs();
        auto options	= m_pGlobalOptionParser->GetCommandLineOptionDefinitions();
        bool checkRequiredOption = true;
        if( options.help.IsExisting() ||
            options.version.IsExisting() ||
            options.verify.IsExisting()
            )
        {
            checkRequiredOption = false;
        }
        else
        {
            for( auto inputFile = inputArgs.begin(); inputFile != inputArgs.end(); ++inputFile )
            {
                // 入力ファイルに .fsca|b が含まれるか調べる。
                size_t index = inputFile->find( ".fsc", 0 );
                if( index != std::string::npos )
                {
                    checkRequiredOption = false;
                    continue;
                }

            }
        }
        if( checkRequiredOption )
        {
            m_pGlobalOptionParser->CheckRequiredOption();
        }
    }

    // Siglo コマンドラインオプションを旧オプションに変換します。
    // value を持つ option は = で接続します。（旧オプションの仕様）
    {
        // 旧コンバータの仕様上、以下の順番で文字列ポインタを並べないといけない。
        // 1. 実行ファイル名　2. option　3. input_file_name

        // 1. 実行ファイル名を追加します。
        auto inputArgs	= m_pGlobalOptionParser->GetInputArgs();
        auto addStr = [ this ]( std::string str )
        {
            this->m_WideCharBuff.push_back( std::wstring( str.begin(), str.end() ) );
        };
        std::for_each( inputArgs.begin(), inputArgs.begin() + 1, addStr );

        // 2. option 名を追加します。
        const GlobalOptionParserBase::CommandLineOptionDefinitions& options	= m_pGlobalOptionParser->GetCommandLineOptionDefinitions();
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.dumpShaderSource );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.dumpStatisticsInformation);
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.forceVariation );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.help );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.output );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.preProcess );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.rootConfig );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.rootDefinition );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.silent );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.unifiedAnnotation );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.verify );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.version );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.glslVersion );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.defineMacro );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.autoExtract );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.apiType );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.codeType );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.debugInfoLevel );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.debugInfoDirectory );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.shaderCacheDirectory );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.shaderCacheWriteDirectory);
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.shaderCacheReadDirectory);
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.glslExtension );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.usePreprocessorVariation );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.decomposeBinary );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.jobs );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.glslcOptionFlags );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.gfxShaderConverterOptions );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.variationBeginRatio );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.variationEndRatio );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.skipConvert );
        ConvertToNwOptionNameFmt( &m_WideCharBuff, options.mergeShaderArchiveFile);

        // 3. input_file_name を追加します。
        if( inputArgs.begin() + 1 != inputArgs.end() )
        {
            std::for_each( inputArgs.begin() + 1, inputArgs.end(), addStr );
        }
    }

    {
        const wchar_t**	wideCharStrPtrArray = new( const wchar_t* [ m_WideCharBuff.size() ] );
        for( int idx = 0; idx < static_cast< int >( m_WideCharBuff.size() ); ++idx )
        {
            wideCharStrPtrArray[ idx ] = m_WideCharBuff[ idx ].c_str();
        }

        m_pCommandArgs.reset( new nw::g3d::tool::CmdArgs( static_cast< int >( m_WideCharBuff.size() ), wideCharStrPtrArray ) );
        delete[] wideCharStrPtrArray;
    }

}

void CommandLineParserWrapper::LoadDllFuncs( const HMODULE& hModule )
{
    NN_UNUSED( hModule );
    THROW_ERROR(ERRCODE_INTERNAL, "Internal error");	// TODO: 翻訳: dll ロードは未実装です。
}

}
}
