﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using nw.g3d.nw4f_3dif;
using nw4f.meshlib;

namespace nw.g3d.iflib.src.Optimize.Model.PlygonReductionOptimize
{
    internal class ReductionUtility
    {
        internal class MeshPair
        {
            public nw4f.meshlib.Mesh OriginalMesh { get; set; }
            public nw4f.meshlib.Mesh ExecuteMesh { get; set; }
        }

        internal static readonly Dictionary<SourceType, string> SourceTypeNames = new Dictionary<SourceType, string>
        {
            { SourceType.Position, "position" },
            { SourceType.Normal, "normal" },
            { SourceType.Color, "color" },
            { SourceType.Texture, "uv" },
            { SourceType.Binormal, "binormal" },
            { SourceType.Tangent, "tangent" },
            { SourceType.BlendWeight, "blendweight" },
            { SourceType.EtceteraDouble, "etceteraDouble" },
            { SourceType.EtceteraInt, "etceteraInt" },
            { SourceType.NONE, "none" }
        };

        internal static QemSimp.QemSimpParam CreateParams(IfModelPolygonReductionOptimizer.Options options, float targetRate)
        {
            var qemSimpParam = new QemSimp.QemSimpParam
            {
                TargetRate = targetRate,
                TargetRatePerMaterialShape = options.TargetRatePerShapes,
                DistOffs = options.DistOffset,
                Smoothing = options.Smoothing,
                LockUvHardEdgeAngle = options.LockUvHardEdgeAngle,
                PropUseflag = options.PropFlag,
                Policy = options.Policy,
                Quality = options.Quality,
                OptPosPolicy = options.OptPosPolicy,
                RemoveUnusedVtx = true,
                OldLockOpenEdgeAngle = options.OldLockOpenEdgeAngle,
                OldLockUvHardEdgeAngle = options.OldLockUvHardEdgeAngle,
                OldFeatureFaceAngle = options.OldFeatureFaceAngle,
                LockOpenEdgeAngle = options.LockOpenEdgeAngle,
                FeatureFaceAngle = options.FeatureFaceAngle,
                ExcludeMeshNames = options.ExcludeMeshNames
            };
            nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(qemSimpParam.ToString());
            qemSimpParam.PropWeights[(int)SourceType.Texture] = options.UvWeight;
            qemSimpParam.PropWeights[(int)SourceType.Normal] = options.NormalWeight;
            qemSimpParam.PropWeights[(int)SourceType.BlendWeight] = options.AnimBlendWeight;

            return qemSimpParam;
        }

        internal static Dictionary<Mesh, Dictionary<int, ReductionUtility.MeshPair>> ConvertToExecList(Model meshModel, IfModelPolygonReductionOptimizer.Options options)
        {
            var meshes = new Dictionary<Mesh, Dictionary<int, ReductionUtility.MeshPair>>();
            {// データ入力

                // 入れ物作成
                foreach (var shape in meshModel.Shapes)
                {
                    foreach (var mesh in shape.Meshes)
                    {
                        lock (meshes)
                        {
                            Dictionary<int, ReductionUtility.MeshPair> meshesByTargetRate;
                            if (meshes.TryGetValue(mesh, out meshesByTargetRate) == false)
                            {
                                meshesByTargetRate = new Dictionary<int, ReductionUtility.MeshPair>();
                                meshes.Add(mesh, meshesByTargetRate);
                            }
                        }
                    }
                }

                G3dParallel.ForEach(meshModel.Shapes, delegate (Shape shape)
                {
                    G3dParallel.ForEach(shape.Meshes, delegate (Mesh mesh)
                    {
                        var sourcesDictionary = new Dictionary<SourceType, IList<Tuple<G3dStream, vtx_attribType>>>();
                        foreach (var sourcesItem in shape.SourcesDictionary)
                        {
                            var sources = sourcesItem.Value.Select(source => Tuple.Create(source.OriginalStream, source.OriginalVtxAttribType)).ToList();
                            sourcesDictionary.Add(sourcesItem.Key, sources);
                        }

                        var reductionMesh = ReductionUtility.LoadMesh(mesh, sourcesDictionary);
                        if (reductionMesh == null)
                        {
                            throw ExcepHandle.CreateException("LoadMesh Failed");
                        }

                        reductionMesh.MeshName = shape.OrignalShapeType.name + QemSimp.MeshNameSymbol + mesh.OriginalMeshType.index;

                        reductionMesh.AddMaterialShapeInfo(shape.OrignalShapeType.index, shape.OrignalShapeType.name, reductionMesh.Sources);
                        reductionMesh.CoveredMeshbyMaterialShape(0);

                        G3dParallel.ForEach(options.TargetRates, delegate (KeyValuePair<int, float> targetRate)
                        {
                            var executeMesh = reductionMesh.Clone();

                            // (SIGLO-28482) 2016/06/08 位置座標を事前に変換する
                            var posStream = executeMesh.GetSource(SourceType.Position, null);
                            if (posStream != null && options.IsMargeShape)
                            {
                                var posCount = posStream.Count;
                                nw4f.tinymathlib.Vector3 tmp = new nw4f.tinymathlib.Vector3();
                                Vector4 tmpV4 = new Vector4();
                                for (int pi = 0; pi < posCount; pi++)
                                {
                                    MeshSourceDblCtrler.GetAsVec3(posStream, pi, ref tmp);
                                    tmpV4[0] = (float)tmp[0];
                                    tmpV4[1] = (float)tmp[1];
                                    tmpV4[2] = (float)tmp[2];
                                    tmpV4[3] = 1.0f;

                                    var transPos = shape.AffineMatrx * tmpV4;
                                    tmp[0] = Convert.ToDouble(transPos[0]);
                                    tmp[1] = Convert.ToDouble(transPos[1]);
                                    tmp[2] = Convert.ToDouble(transPos[2]);

                                    // 上書き
                                    MeshSourceDblCtrler.SetAsVec3(posStream, pi, tmp);
                                }
                            }

                            lock (meshes)
                            {
                                Dictionary<int, ReductionUtility.MeshPair> meshesByTargetRate;
                                if (meshes.TryGetValue(mesh, out meshesByTargetRate) == false)
                                {
                                    throw ExcepHandle.CreateException("Execute Failed");
                                }
                                meshesByTargetRate.Add(targetRate.Key,
                                    new ReductionUtility.MeshPair { OriginalMesh = reductionMesh, ExecuteMesh = executeMesh });
                            }
                        });
                    });
                });
            }
            return meshes;
        }

        internal static nw4f.meshlib.Mesh LoadMesh(Mesh mesh, IEnumerable<KeyValuePair<SourceType, IList<Tuple<G3dStream, vtx_attribType>>>> sourcesDictionary)
        {
            var reductionMesh = new nw4f.meshlib.Mesh();

            if (ImportSources(reductionMesh, sourcesDictionary) == false)
            {
                throw ExcepHandle.CreateException("ImportSources Failed");
            }

            {
                var sourcesCount = sourcesDictionary.Sum(valuePair => valuePair.Value.Count);
                if (ImportFaces(reductionMesh, mesh, sourcesCount) == false)
                {
                    throw ExcepHandle.CreateException("ImportFace Failed");
                }
            }

            if (ImportSubMeshInfos(reductionMesh, mesh.SubMeshes) == false)
            {
                throw ExcepHandle.CreateException("ImportSubMeshInfos Failed");
            }

            return reductionMesh;
        }

        private static MeshSourceBase CreateSource(nw4f.meshlib.Mesh outMesh, SourceType type, vtx_attribType attribute, G3dStream stream)
        {
            var stride = stream.column;
            var count = outMesh.GetSources(type).Count;
            var meshSource = outMesh.AddSource(ReductionUtility.SourceTypeNames[type] + count, type, stride);
            if (meshSource == null)
            {
                throw ExcepHandle.CreateException("outMesh.AddSource Failed");
            }
            // 量子化型をチェックする
            meshSource.QuantizeType = attribute.quantize_type;
            return meshSource;
        }

        private static bool ImportSources(nw4f.meshlib.Mesh outMesh, IEnumerable<KeyValuePair<SourceType, IList<Tuple<G3dStream, vtx_attribType>>>> sourcesDictionary)
        {
            foreach (var sourcesItem in sourcesDictionary)
            {
                var sourceType = sourcesItem.Key;
                var sources = sourcesItem.Value;

                foreach (var source in sources)
                {
                    var attribute = source.Item2;
                    var stream = source.Item1;
                    var stride = stream.column;

                    switch (stream.type)
                    {
                        case stream_typeType.@float:
                            {
                                var meshSource = CreateSource(outMesh, sourceType, attribute, stream) as MeshSource<double>;
                                var streamDatas = stream.FloatData;
                                meshSource.SetLength(streamDatas.Count, true);
                                for (var dataIndex = 0; dataIndex < streamDatas.Count; dataIndex += stride)
                                {
                                    for (var strideIndex = 0; strideIndex < stride; ++strideIndex)
                                    {
                                        meshSource.SetRowValue(dataIndex + strideIndex, (double)streamDatas[dataIndex + strideIndex]);
                                    }
                                }
                            }
                        break;
                        case stream_typeType.@int:
                            {
                                var meshSource = CreateSource(outMesh, sourceType, attribute, stream) as MeshSource<int>;
                                var streamDatas = stream.IntData;
                                meshSource.SetLength(streamDatas.Count, true);
                                for (var dataIndex = 0; dataIndex < streamDatas.Count; dataIndex += stride)
                                {
                                    for (var strideIndex = 0; strideIndex < stride; ++strideIndex)
                                    {
                                        meshSource.SetRowValue(dataIndex + strideIndex, (int)streamDatas[dataIndex + strideIndex]);
                                    }
                                }
                            }
                            break;
                        case stream_typeType.@byte:
                            {
                                var meshSource = CreateSource(outMesh, sourceType, attribute, stream) as MeshSource<int>;
                                var streamDatas = stream.ByteData;
                                // 量子化型をチェックする
                                meshSource.SetLength(streamDatas.Count, true);
                                for (var dataIndex = 0; dataIndex < streamDatas.Count; dataIndex += stride)
                                {
                                    for (var strideIndex = 0; strideIndex < stride; ++strideIndex)
                                    {
                                        meshSource.SetRowValue(dataIndex + strideIndex, (int)streamDatas[dataIndex + strideIndex]);
                                    }
                                }
                            }
                            break;
                        case stream_typeType.@string:
                            throw new ArgumentOutOfRangeException();
                        case stream_typeType.wstring:
                            throw new ArgumentOutOfRangeException();
                        default:
                            throw new ArgumentOutOfRangeException();
                    }
                }
            }

            return true;
        }

        private static bool ImportFaces(nw4f.meshlib.Mesh outMesh, Mesh mesh, int sourcesCount)
        {
            var indicesStream = mesh.IndexSource.OriginalStream.IntData;

            if (mesh.OriginalMeshType.mode == mesh_modeType.triangles)
            {
                var faceCount = indicesStream.Count / 3;

                outMesh.ResizeFaces(faceCount, faceCount * 3 * sourcesCount);

                for (int indicesStreamIndex = 0, submeshSize = indicesStream.Count, faceIndex = 0; indicesStreamIndex < submeshSize; indicesStreamIndex += 3, ++faceIndex)
                {
                    var offset = faceIndex * 3;
                    outMesh.VNums[faceIndex] = 3;
                    outMesh.VOffset[faceIndex] = offset;

                    for (var vertexIndex = 0; vertexIndex < 3; ++vertexIndex)
                    {
                        for (var streamIndex = 0; streamIndex < sourcesCount; ++streamIndex)
                        {
                            outMesh.VBuffer[(offset + vertexIndex) * sourcesCount + streamIndex] = indicesStream[indicesStreamIndex + vertexIndex];
                        }
                    }
                }
            }
            else if (mesh.OriginalMeshType.mode == mesh_modeType.triangle_fan)
            {
                var faceCount = indicesStream.Count - 2;

                outMesh.ResizeFaces(faceCount, faceCount * 3 * sourcesCount);

                for (int indicesStreamIndex = 1, submeshSize = indicesStream.Count, faceIndex = 0; indicesStreamIndex < submeshSize; ++indicesStreamIndex, ++faceIndex)
                {
                    var offset = faceIndex * 3;
                    outMesh.VNums[faceIndex] = 3;
                    outMesh.VOffset[faceIndex] = offset;

                    // 基点
                    for (var streamIndex = 0; streamIndex < sourcesCount; ++streamIndex)
                    {
                        outMesh.VBuffer[offset * sourcesCount + streamIndex] = indicesStream[0];
                    }

                    // その他
                    for (var vertexIndex = 1; vertexIndex < 3; ++vertexIndex)
                    {
                        for (var streamIndex = 0; streamIndex < sourcesCount; ++streamIndex)
                        {
                            outMesh.VBuffer[(offset + vertexIndex) * sourcesCount + streamIndex] = indicesStream[indicesStreamIndex + vertexIndex];
                        }
                    }
                }
            }
            else if (mesh.OriginalMeshType.mode == mesh_modeType.triangle_strip)
            {
                var faceCount = indicesStream.Count - 2;

                outMesh.ResizeFaces(faceCount, faceCount * 3 * sourcesCount);

                for (int indicesStreamIndex = 0, submeshSize = indicesStream.Count, faceIndex = 0; indicesStreamIndex < submeshSize; ++indicesStreamIndex, ++faceIndex)
                {
                    var offset = faceIndex * 3;
                    outMesh.VNums[faceIndex] = 3;
                    outMesh.VOffset[faceIndex] = offset;

                    for (var vertexIndex = 0; vertexIndex < 3; ++vertexIndex)
                    {
                        for (var streamIndex = 0; streamIndex < sourcesCount; ++streamIndex)
                        {
                            outMesh.VBuffer[(offset + vertexIndex) * sourcesCount + streamIndex] = indicesStream[indicesStreamIndex + vertexIndex];
                        }
                    }
                }
            }

            return true;
        }

        private static bool ImportSubMeshInfos(nw4f.meshlib.Mesh outMesh, IEnumerable<SubMesh> subMeshes)
        {
            var index = 0;
            foreach (var subMesh in subMeshes)
            {
                outMesh.SetSubMeshInfo(index++, subMesh.IndexOffset == 0 ? 0 : subMesh.IndexOffset / 3, subMesh.IndexCount == 0 ? 0 : subMesh.IndexCount / 3);
            }
            return true;
        }

        internal static bool SaveMesh(out Mesh outMesh, out IDictionary<SourceType, IList<Stream<object>>> outSourcesDictionary, nw4f.meshlib.Mesh reductionMesh, Mesh originalMesh, bool IsShaerSources)
        {
            if (IsShaerSources == false)
            {
                if (ExportSources(out outSourcesDictionary, reductionMesh.Sources) == false)
                {
                    throw ExcepHandle.CreateException("ExportSources Failed");
                }
            }
            else
            {
                outSourcesDictionary = null;
            }

            if (ExportFaces(out outMesh, reductionMesh, originalMesh) == false)
            {
                throw ExcepHandle.CreateException("ExportSources Failed");
            }

            IList<SubMesh> submeshes = new List<SubMesh>();
            if (ExportSubMeshInfos(out submeshes, reductionMesh, originalMesh.SubMeshes) == false)
            {
                throw ExcepHandle.CreateException("ExportSubMeshInfos Failed");
            }

            outMesh.SubMeshes = submeshes;

            return true;
        }

        private static bool ExportSources(out IDictionary<SourceType, IList<Stream<object>>> outSourcesDictionary,
            IEnumerable<MeshSourceBase> meshSourceBases)
        {
            outSourcesDictionary = new Dictionary<SourceType, IList<Stream<object>>>();

            foreach (var meshSourceBase in meshSourceBases)
            {
                if (outSourcesDictionary.ContainsKey(meshSourceBase.Type) == false)
                {
                    outSourcesDictionary.Add(meshSourceBase.Type, new List<Stream<object>>());
                }
                var outSources = outSourcesDictionary[meshSourceBase.Type];

                var source = new Stream<object> { Column = meshSourceBase.Stride };
                foreach (var data in meshSourceBase.Data)
                {
                    source.Datas.Add(data);
                }

                outSources.Add(source);
            }

            return true;
        }

        private static bool ExportFaces(out Mesh outMesh, nw4f.meshlib.Mesh reductionMesh, Mesh originalMesh)
        {
            outMesh = new Mesh();
            outMesh.IndexSource.AddStream.Column = 3;
            outMesh.OriginalMeshType = originalMesh.OriginalMeshType;
            outMesh.IndexSource.OriginalStream = originalMesh.IndexSource.OriginalStream;

            var sourcesCount = reductionMesh.Sources.Count;
            for (var index = 0; index < reductionMesh.VBuffer.Count; index += sourcesCount)
            {
                var data = reductionMesh.VBuffer[index];
                outMesh.IndexSource.AddStream.Datas.Add(data);
            }

            return true;
        }

        private static bool ExportSubMeshInfos(out IList<SubMesh> subMeshes, nw4f.meshlib.Mesh reductionMesh, IList<SubMesh> originalSubmeshes)
        {
            subMeshes = new List<SubMesh>();

            var submeshIndex = 0;
            foreach (var subMeshInfo in reductionMesh.SubMeshInformationList)
            {
                subMeshes.Add(new SubMesh { IndexCount = subMeshInfo.Value.subMeshCount * 3, IndexOffset = subMeshInfo.Value.subMeshOffset * 3, OriginalSubmeshType = originalSubmeshes[subMeshInfo.Value.subMeshindex].OriginalSubmeshType });
                ++submeshIndex;
            }

            return true;
        }

        internal static bool UpdateShape(Shape outShape, IEnumerable<KeyValuePair<int, Mesh>> meshes,
            IDictionary<int, IDictionary<SourceType, IList<Stream<object>>>> sourcesDictionaries)
        {
            foreach (var newMesh in meshes)
            {
                outShape.Meshes.Add(newMesh.Value);

                var newSourcesDictionary = sourcesDictionaries[newMesh.Key];
                if (newSourcesDictionary == null)
                {
                    continue;
                }
                foreach (var newSourceItem in newSourcesDictionary)
                {
                    if (outShape.SourcesDictionary.ContainsKey(newSourceItem.Key) == false)
                    {
                        throw ExcepHandle.CreateException("shape.SourcesDictionary.ContainsKey Failed");
                    }
                    var sources = outShape.SourcesDictionary[newSourceItem.Key];

                    for (var index = 0; index < newSourceItem.Value.Count; ++index)
                    {
                        var addStream = sources[index].AddStream;
                        if (newMesh.Value.SourceOffset == 0)
                        {
                            var originalVtxAttribType = sources[index].OriginalVtxAttribType;
                            var baseCount = originalVtxAttribType.count;
                            var addCount = addStream.Datas.Count == 0 ? 0 : addStream.Datas.Count / addStream.Column;
                            newMesh.Value.SourceOffset = baseCount + addCount;
                        }

                        var newStream = newSourceItem.Value[index];
                        if (addStream.Column == 0)
                        {
                            addStream.Column = newStream.Column;
                        }
                        else
                        {
                            if (addStream.Column != newStream.Column)
                            {
                                throw ExcepHandle.CreateException("addStream.Column != newStream.Column Failed");
                            }
                        }
                        foreach (var data in newStream.Datas)
                        {
                            addStream.Datas.Add(data);
                        }
                    }
                }
            }

            return true;
        }

        [Conditional("DEBUG")]
        internal static void CheckPolygonCount(SortedDictionary<int, float> targetRates, SortedDictionary<int, Mesh> meshes)
        {
            var sortedList = new SortedList<float, List<int>>();
            foreach (var targetRate in targetRates)
            {
                if (sortedList.ContainsKey(targetRate.Value) == false)
                {
                    sortedList.Add(targetRate.Value, new List<int>());
                }
                sortedList[targetRate.Value].Add(meshes[targetRate.Key].IndexSource.AddStream.Datas.Count);
            }

            for (var index = 1; index < sortedList.Count; ++index)
            {
                var before = sortedList.ElementAt(index - 1).Value;
                var current = sortedList.ElementAt(index).Value;
                foreach (var currentCount in current)
                {
                    foreach (var beforeCount in before)
                    {
                        if (currentCount < beforeCount)
                        {
                            throw new Exception("check polygon count error");
                        }
                    }
                }
            }
        }
    }

    public class Reduction
    {
        public static bool Execute(Model model, IfModelPolygonReductionOptimizer.Options options)
        {
            var meshes = ReductionUtility.ConvertToExecList(model, options);
            {// リダクション
                G3dParallel.ForEach(options.TargetRates, delegate (KeyValuePair<int, float> targetRate)
                {
                    var meshPairMapByName = new Dictionary<string, ReductionUtility.MeshPair>();
                    var executeMeshes = new List<nw4f.meshlib.Mesh>();
                    foreach (var meshItem in meshes)
                    {
                        var meshPair = meshItem.Value[targetRate.Key];
                        var executeMesh = meshPair.ExecuteMesh;
                        executeMeshes.Add(executeMesh);
                        meshPairMapByName.Add(executeMesh.MeshName, meshPair);
                    }

                    if (Execute(executeMeshes, options, targetRate.Value) == false)
                    {
                        throw ExcepHandle.CreateException("Execute Failed");
                    }

                    foreach (var executeMesh in executeMeshes)
                    {
                        var totalArea = executeMesh.mFaceArea.Sum();
                        // 1,合計面積がゼロであるならば、メッシュが完全に消えてしまう
                        // そうするとfmdbで整合性に問題がでるので触らない事で解決する
                        // 2,除外メッシュリストに含まれている場合は、簡略化は行ってないものとして元メッシュのクローンを取得する
                        if (totalArea < 1e-5 || options.ExcludeMeshNames.Any(pattern => Regex.IsMatch(executeMesh.MeshName, pattern + QemSimp.MeshNameSymbol + "[0-9]*")))
                        {
                            meshPairMapByName[executeMesh.MeshName].ExecuteMesh = meshPairMapByName[executeMesh.MeshName].OriginalMesh.Clone();
                        }
                        else
                        {
                            meshPairMapByName[executeMesh.MeshName].ExecuteMesh = executeMesh;
                        }
                    }
                });

                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Complete_All_Notify"));
            }

            {
                // 後処理
                // データ出力
                var isShareSources = options.OptPosPolicy == QemOptPos.ZeroOne && options.IsMargeShape == false;

                G3dParallel.ForEach(model.Shapes, delegate (Shape shape)
                {
                    var newMeshes = new SortedDictionary<int, Mesh>();
                    var newSourcesDictionaries = new Dictionary<int, IDictionary<SourceType, IList<Stream<object>>>>();

                    G3dParallel.ForEach(shape.Meshes, delegate (Mesh mesh)
                    {
                        var meshesByTargetRate = meshes[mesh];
                        G3dParallel.ForEach(options.TargetRates, delegate (KeyValuePair<int, float> targetRate)
                        {
                            var meshPair = meshesByTargetRate[targetRate.Key];
                            var reductionMesh = meshPair.OriginalMesh;
                            var executeMesh = meshPair.ExecuteMesh;

                            // シェイプ結合していない場合はボーンローカルからモデルローカル座標へ変換する行列は
                            //適用されないので、再計算不要。
                            if (options.IsMargeShape)
                            {
                                // (SIGLO-28482) 2016/06/08 位置座標を事前に変換する
                                // shape.AffineMatrx は ボーンローカルからモデルローカルへ変換する行列
                                var modelToBoneMatrix = new Matrix44(shape.AffineMatrx);
                                modelToBoneMatrix.Invert();

                                // 位置
                                {
                                    var posStream = executeMesh.GetSource(SourceType.Position, null);
                                    if(posStream != null)
                                    {
                                        var posCount = posStream.Count;
                                        nw4f.tinymathlib.Vector3 tmp = new nw4f.tinymathlib.Vector3();
                                        Vector4 tmpV4 = new Vector4();

                                        for (int pi = 0; pi < posCount; pi++)
                                        {
                                            MeshSourceDblCtrler.GetAsVec3(posStream, pi, ref tmp);
                                            Vector4 position = new Vector4((float)tmp.x, (float)tmp.y, (float)tmp.z, 1.0f);

                                            position = modelToBoneMatrix * position;
                                            tmp.x = (double)(position.X);
                                            tmp.y = (double)(position.Y);
                                            tmp.z = (double)(position.Z);

                                            // 上書き
                                            MeshSourceDblCtrler.SetAsVec3(posStream, pi, tmp);
                                        }
                                    }
                                }

                                // 法線 / 接線 / 従法線
                                {
                                    // モデルローカル -> ボーンローカル行列の逆転置行列を取得
                                    var directionTransformMatrix = ModelCompressUtility.GetNormalTransform(modelToBoneMatrix);

                                    var sourceTypes = new SourceType[] { SourceType.Normal, SourceType.Tangent, SourceType.Binormal };
                                    foreach (var source in sourceTypes)
                                    {
                                        var sourceStream = executeMesh.GetSource(source, null);
                                        if (sourceStream == null)
                                        {
                                            continue;
                                        }

                                        var tempSource = new nw4f.tinymathlib.Vector3();
                                        for (int sourceIndex = 0, sourceCount = sourceStream.Count; sourceIndex < sourceCount; ++sourceIndex)
                                        {
                                            // 接線、従法線は vec4 型だが、 w に格納されている符号情報は変化しないため vec3 で取得
                                            MeshSourceDblCtrler.GetAsVec3(sourceStream, sourceIndex, ref tempSource);

                                            Vector4 direction = new Vector4((float)tempSource.x, (float)tempSource.y, (float)tempSource.z, 1.0f);
                                            direction = directionTransformMatrix * direction;
                                            tempSource.x = (double)(direction.X);
                                            tempSource.y = (double)(direction.Y);
                                            tempSource.z = (double)(direction.Z);
                                            tempSource.Normalize();

                                            MeshSourceDblCtrler.SetAsVec3(sourceStream, sourceIndex, tempSource);
                                        }
                                    }
                                }
                            }

                            if (isShareSources)
                            {
                                var indexTrack = new MeshIndexTrack { SourceMesh = reductionMesh, DestMesh = executeMesh };
                                indexTrack.Percentage = ShapeMerger.Percentage;
                                // インデックス置換
                                indexTrack.Run();

                                // その後に、ソースをごっそり入れ替える
                                for (int si = 0; si < executeMesh.SourceN; si++)
                                {
                                    executeMesh.Sources[si] = reductionMesh.Sources[si].Clone();
                                }
                            }

                            Mesh newMesh;
                            IDictionary<SourceType, IList<Stream<object>>> newSourcesDictionary;
                            if (ReductionUtility.SaveMesh(out newMesh, out newSourcesDictionary, executeMesh, mesh, isShareSources) == false)
                            {
                                throw ExcepHandle.CreateException("SaveMesh Failed");
                            }
                            lock (newMeshes)
                            {
                                newMeshes.Add(targetRate.Key, newMesh);
                            }
                            lock (newSourcesDictionaries)
                            {
                                newSourcesDictionaries.Add(targetRate.Key, newSourcesDictionary);
                            }
                        });
                    });
                    if (options.IsMargeShape == false)
                    {
                        ReductionUtility.CheckPolygonCount(options.TargetRates, newMeshes);
                    }

                    if (ReductionUtility.UpdateShape(shape, newMeshes, newSourcesDictionaries) == false)
                    {
                        throw ExcepHandle.CreateException("UpdateShape Failed");
                    }
                });
            }

            return true;
        }

#if false
        public static bool Execute(Shape shape, IfModelPolygonReductionOptimizer.Options options)
        {
            var newMeshes = new SortedDictionary<int, Mesh>();
            var newSourcesDictionaries = new Dictionary<int, IDictionary<SourceType, IList<Stream<object>>>>();

            G3dParallel.ForEach(shape.Meshes, delegate(Mesh mesh)
            {
                var sourcesDictionary = new Dictionary<SourceType, IList<G3dStream>>();
                foreach (var sourcesItem in shape.SourcesDictionary)
                {
                    var sources = sourcesItem.Value.Select(source => source.OriginalStream).ToList();
                    sourcesDictionary.Add(sourcesItem.Key, sources);
                }
                var reductionMesh = LoadMesh(mesh, sourcesDictionary);
                if (reductionMesh == null)
                {
                    throw new Exception("LoadMesh Failed");
                }

                G3dParallel.ForEach(options.TargetRates, delegate(KeyValuePair<int, float> targetRate)
                {
                    var isShareSources = options.OptPosPolicy == QemOptPos.ZeroOne;
                    var executeMesh = reductionMesh.Clone();

                    if (Execute(executeMesh, options, targetRate.Value) == false)
                    {
                        throw new Exception("Execute Failed");
                    }

                    if (isShareSources)
                    {
                        var indexTrack = new MeshIndexTrack {SourceMesh = reductionMesh, DestMesh = executeMesh};
                        // インデックス置換
                        indexTrack.Run();
                    }

                    Mesh newMesh;
                    IDictionary<SourceType, IList<Stream<object>>> newSourcesDictionary;
                    if (SaveMesh(out newMesh, out newSourcesDictionary, executeMesh, mesh, isShareSources) == false)
                    {
                        throw new Exception("SaveMesh Failed");
                    }

                    lock (newMeshes)
                    {
                        newMeshes.Add(targetRate.Key, newMesh);
                    }
                    lock (newSourcesDictionaries)
                    {
                        newSourcesDictionaries.Add(targetRate.Key, newSourcesDictionary);
                    }
                });

                CheckPolygonCount(options.TargetRates, newMeshes);
            });

            if (UpdateShape(shape, newMeshes, newSourcesDictionaries) == false)
            {
                throw new Exception("UpdateShape Failed");
            }

            return true;
        }
#endif

        private static bool Execute(List<nw4f.meshlib.Mesh> meshes, IfModelPolygonReductionOptimizer.Options options, float targetRate)
        {
            var qemSimpParam = ReductionUtility.CreateParams(options, targetRate);
            ShapeMerger.Percentage = options.ClosedVertexThreshold;

            //　指定メッシュが存在しない場合には、エラーとする
            foreach (var keyValuePair in qemSimpParam.TargetRatePerMaterialShape)
            {
                var find = false;
                for (var i = 0; i < meshes.Count; i++)
                {
                    var mesh = meshes[i];
                    // 除外メッシュリストに含まれている場合にはエラーメッセージを表示
                    if (Regex.IsMatch(mesh.MeshName, keyValuePair.Key + QemSimp.MeshNameSymbol + "[0-9]*"))
                    {
                        find = true;
                        break;
                    }
                }
                if (!find)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Find_NoShape", keyValuePair.Key));
                    _3dIntermediateFilePlygonReduction.IfStrings.Throw("PolygonReduction_Find_NoShape", keyValuePair.Key);
                }
            }
            // 結合を実行しなかった場合には、通常のリダクション処理が走る様にしました
            // 結合が実行できた場合のみ、通常処理がスキップされます
            bool bMergedMesh = false;
            if (options.IsMargeShape)
            {
                var multiQemSimp = new MultiQemSimp(meshes, qemSimpParam);
                var executedMeshes = multiQemSimp.Execute();
                if (executedMeshes != null)
                {
                    bMergedMesh = true;
                    meshes.Clear();
                    foreach (var executedMesh in executedMeshes)
                    {
                        meshes.Add(executedMesh);
                    }
                }
            }

            // 結合を実行しなかった場合
            if (!bMergedMesh)
            {
                for (var i = 0; i < meshes.Count; i++)
                {
                    nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLog("\n-----------------\n");
                    var mesh = meshes[i];

                    // 面積を事前に計算していない場合
                    if (!mesh.mFaceArea.Any())
                    {
                        mesh.RecomputeFaceNormalsAndArea();
                    }
                    // 合計面積がゼロであるならば、メッシュが完全に消えてしまう
                    // そうするとfmdbで整合性に問題がでるので触らない事で解決する
                    var totalArea = mesh.mFaceArea.Sum();
                    if (totalArea < 1e-5)
                    {
                        continue;
                    }

                    // 除外メッシュリストに含まれている場合は、簡略化は行わない
                    if (qemSimpParam.ExcludeMeshNames.Any(pattern => Regex.IsMatch(mesh.MeshName, pattern + QemSimp.MeshNameSymbol + "[0-9]*")))
                    {
                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Exclude_Message", mesh.MeshName));
                        continue;
                    }

                    var original = mesh.Clone();
                    var cvm = new ClosedVtxMarger(mesh);
                    cvm.SetSpaceInfo(10, 10, 10);
                    var margedVtxCount = cvm.Run();

                    // メッシュの共有化時に行った頂点法線の共有は解除されているのでやり直し
                    mesh.ShrinkVertexNormals(null);
                    // 簡略化の前にブレンドウェイトを統合する
                    // 元々同じ値だったものが別々の値になることを回避したい
                    if ((qemSimpParam.PropUseflag & (int)QemUseProp.BWT) != 0)
                    {
                        var blendWeightList = mesh.GetSources(SourceType.BlendWeight);
                        foreach (var source in blendWeightList)
                        {
                            ClosedPropManager pmap = new ClosedPropManager();
                            pmap.Init(mesh, source.Type, source.Name);
                            pmap.Run();
                        }
                    }

                    if (margedVtxCount < 0)
                    {
                        meshes[i] = original;
                    }
                    else
                    {
                        var qemSimp = new QemSimp { Params = qemSimpParam.Clone() };

                        original.ComputeBounds();
                        if (qemSimp.Initialize(mesh) == false)
                        {
                            throw ExcepHandle.CreateException("qemSimp.Initialize Failed");
                        }

                        bool[] moved;
                        bool[] deleted;
                        qemSimp.Execute(out moved, out deleted);
                        // UVの強制丸め
                        //mesh.ComputeBounds();
                        mesh.ForceRoundingUvInBSquares(original.UVBSquares);
                        // 後処理、fmdb出力の為に、各ソースの数を揃える為の再配置処理を行う
                        var rearranger = new FaceAttributeRearrange(mesh);
                        rearranger.Run();
                    }
                }
            }
            return true;
        }
    }
}
