﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.iflib;
using nw4f.tinymathlib;
using Vector3 = nw4f.tinymathlib.Vector3;

namespace nw4f.meshlib
{
    using WERsList = ResizableList<WingedEdge>;
    using VtxWeList = ResizableList<VtxWnEdge>;

    /// <summary>
    /// 各頂点にどのエッジがぶら下がっているかを
    /// </summary>
    public class VtxWnEdge : ICloneable
    {
        internal ResizableList<WingedEdge> EdgeList { get; set; } = new ResizableList<WingedEdge>();
        public ResizableList<WingedEdge> Edges
        {
            get { return EdgeList; }
        }

        public WingedEdge Remove(int vid)
        {
            WingedEdge result = null;
            int e = 0;
            for (e = 0; e < EdgeList.Count; e++)
            {
                if (EdgeList[e].Vertex0 == vid || EdgeList[e].Vertex1 == vid)
                {
                    result = EdgeList[e];
                    EdgeList.RemoveAt(e);
                    break;
                }
            }
            return result;
        }

        public object Clone()
        {
            VtxWnEdge ret = new VtxWnEdge();
            ret.EdgeList = ResizableList<WingedEdge>.Clone(EdgeList);
            return ret;
        }
    }

    public partial class WingedEdgeMesh
    {
        //private VtxWeList mVtxEdgeList = new VtxWeList();

        private Dictionary<SourceKey, VtxWeList> mVtxEdgeList = new Dictionary<SourceKey, VtxWeList>(new SourceKeyComparer());

        public VtxWeList RepVEL
        {
            get { return mVtxEdgeList[mRepresentativeKey]; }
        }

        public VtxWeList GetVtxWEList(SourceType type, string name)
        {
            SourceKey key = new SourceKey(type, name);

            if (mVtxEdgeList.ContainsKey(key))
            {
                return mVtxEdgeList[key];
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        public void CreateVEL(SourceType type = SourceType.Position, string name = null)
        {
            try
            {
                WERsList wingedEdges = GetWingedEdges(type, name);
                VtxWeList vtxEdgeList = GetVtxWEList(type, name);
                if (vtxEdgeList == null)
                {
                    vtxEdgeList = new VtxWeList();
                    mVtxEdgeList.Add(new SourceKey(type, name), vtxEdgeList);
                }
                MeshSourceBase source = mMesh.GetSource(type, name);

                vtxEdgeList.Resize(source.Count, null);

                for (int v = 0; v < source.Count; v++)
                {
                    vtxEdgeList[v] = new VtxWnEdge();
                }
                WingedEdge edge = null;
                for (int e = 0; e < wingedEdges.Count; e++)
                {
                    edge = wingedEdges[e];
                    if (edge.Vertex0 == edge.Vertex1)
                    {
                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Degenerated_WingedEdge", edge.Vertex0, edge.Vertex1));
                    }
                    else if (edge.Vertex0 > edge.Vertex1)
                    {
                        throw ExcepHandle.CreateException("Index error is found.");
                    }

                    vtxEdgeList[edge.Vertex0].EdgeList.Add(edge);
                    vtxEdgeList[edge.Vertex1].EdgeList.Add(edge);
                }
            }
            catch (Exception ex)
            {
                ExcepHandle.CreateException(ex);
                Destroy();
                throw;
            }
        }

        /// <summary>
        /// 指定のインデックスを両端に持つエッジを探索
        /// </summary>
        /// <param name="vid0"></param>
        /// <param name="vid1"></param>
        /// <returns></returns>
        public List<WingedEdge> FindRepEdge(int vid0, int vid1)
        {
            return FindEdge(vid0, vid1, mRepresentativeKey.type, mRepresentativeKey.name);
        }

        /// <summary>
        /// 指定のインデックスを両端に持つエッジを探索
        /// </summary>
        /// <param name="vid0"></param>
        /// <param name="vid1"></param>
        /// <param name="type"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        public List<WingedEdge> FindEdge(int vid0, int vid1, SourceType type, string name)
        {
            VtxWeList targetVEL = GetVtxWEList(type, name);
            if (targetVEL == null)
            { return null; }
            List<WingedEdge> result = new List<WingedEdge>();
            if (vid0 > vid1)
            {
                int tmp = vid0;
                vid0 = vid1;
                vid1 = tmp;
            }
            foreach (var wedge in targetVEL[vid0].EdgeList)
            {
                if (wedge.Vertex0 == vid0 && wedge.Vertex1 == vid1)
                {
                    result.Add(wedge);
                }
            }
            return result;
        }

        /// <summary>
        /// エッジの方向を取得します
        /// </summary>
        /// <param name="edge"></param>
        /// <returns></returns>
        public Vector3 GetEdgeDir(WingedEdge edge)
        {
            Vector3 v0 = new Vector3();
            Vector3 v1 = new Vector3();
            mMesh.GetVertexPos(edge.Vertex0, ref v0);
            mMesh.GetVertexPos(edge.Vertex1, ref v1);
            return v1 - v0;
        }

        /// <summary>
        /// 重複した面を削除した後に、不正エッジを削除する
        /// 両翼の面のIDが一致している場合,両翼の面IDが不正の場合
        /// </summary>
        /// <param name="keep"></param>
        internal void DeleteDuplicateEdges(int vid, SourceType type = SourceType.Position, string name = null)
        {
            VtxWeList vtxEdgeList = GetVtxWEList(type, name);

            vtxEdgeList[vid].EdgeList.Sort();
            WingedEdge prev = null;
            int edgeN = 0;
            bool[] erasable = new bool[vtxEdgeList[vid].EdgeList.Count];
            foreach (WingedEdge curr in vtxEdgeList[vid].EdgeList)
            {
                erasable[edgeN] = false;
                if (prev.IsVerticesEqual(curr))
                {
                    erasable[edgeN] = true;
                }
                edgeN++;
                prev = curr;
            }

            edgeN = 0;
            for (int ei = 0; ei < vtxEdgeList[vid].EdgeList.Count; ei++)
            {
                if (!erasable[ei])
                {
                    vtxEdgeList[vid].EdgeList[edgeN] = vtxEdgeList[vid].EdgeList[ei];
                    edgeN++;
                }
            }
            vtxEdgeList[vid].EdgeList.Resize(edgeN, null);
        }

        /// <summary>
        /// 代表エッジのみを縮退させる（旧バージョン）: Opaque な関数
        /// </summary>
        /// <param name="keep"></param>
        /// <param name="kill"></param>
        public bool TestShrinkRepEdge(int keep, int kill)
        {
            return TestShrink(keep, kill, mRepresentativeKey);
        }

        /// <summary>
        /// Repエッジとそれに連なるエッジを削除します
        /// </summary>
        /// <param name="keep"></param>
        /// <param name="kill"></param>
        public void TestShrinkAll(int keep, int kill)
        {
            List<WingedEdge> edges = FindRepEdge(keep, kill);

            if (edges.Count > 1) { return; }

            WingedEdge targetRepEdge = edges[0];
            foreach (var pair in targetRepEdge.ChildEdges)
            {
                SourceKey key = pair.Key;
                foreach (var refEdge in pair.Value)
                {
                    TestShrink(refEdge.Vertex0, refEdge.Vertex1, key);
                }
            }
            TestShrink(keep, kill, mRepresentativeKey);
        }

        /// <summary>
        /// 非多様体になってしまう縮退かどうか？をチェックします
        /// </summary>
        /// <param name="keep"></param>
        /// <param name="kill"></param>
        /// <param name="type"></param>
        /// <param name="name"></param>
        public bool TestShrink(int keep, int kill, SourceType type, string name)
        {
            return TestShrink(keep, kill, new SourceKey(type, name));
        }

        /// <summary>
        /// 非多様体になってしまう縮退かどうか？をチェックします
        /// </summary>
        /// <param name="keep"></param>
        /// <param name="kill"></param>
        /// <returns></returns>
        public bool TestShrink(int keep, int kill, SourceKey key)
        {
            if (keep == kill && keep == int.MaxValue) { return false; }

            int edgeN = GetWingedEdgeN(key.type, key.name);
            if (keep > edgeN || kill > edgeN)
            {
                throw ExcepHandle.CreateException("Vertex indices are over bounds.");
            }

            VtxWeList vtxEdgeList = GetVtxWEList(key.type, key.name);
            // 消去予定のフェース
            int killFace0 = int.MaxValue;
            int killFace1 = int.MaxValue;

            //クローん
            VtxWnEdge keepEdges = (VtxWnEdge)vtxEdgeList[keep].Clone();
            VtxWnEdge killEdges = (VtxWnEdge)vtxEdgeList[kill].Clone();

            // keep-kill のエッジを消す
            WingedEdge r = null;
            do
            {
                r = keepEdges.Remove(kill);
                if (r != null)
                {
                    // 消去予定のフェース
                    killFace0 = r.Face0;
                    killFace1 = r.Face1;
                }
            }
            while (r != null);

            // keep-kill のエッジを消す
            r = null;
            do
            {
                r = killEdges.Remove(keep);
                if (r != null)
                {
                    // 消去予定のフェース
                    if (killFace0 != r.Face0 || killFace1 != r.Face1)
                    {
                        throw ExcepHandle.CreateException("Face index error is found.");
                    }
                }
            }
            while (r != null);

            if (killFace0 == int.MaxValue && killFace1 == int.MaxValue)
            {
                throw ExcepHandle.CreateException("ShrinkEdge() :: try to Shrink illegal edge");
            }

            // インデックスの書き換え
            keepEdges.EdgeList.AddRange(killEdges.EdgeList);
            foreach (var edge in keepEdges.EdgeList)
            {
                if (edge.Vertex0 == kill)
                {
                    edge.Vertex0 = keep;
                }
                else if (edge.Vertex1 == kill)
                {
                    edge.Vertex1 = keep;
                }
                // 入れ替え
                int minID = Math.Min(edge.Vertex0, edge.Vertex1);
                int maxID = Math.Max(edge.Vertex0, edge.Vertex1);

                edge.Vertex0 = minID;
                edge.Vertex1 = maxID;
            }
            //mVtxEdgeList[kill].edges.Clear();

            // 重複したエッジがあるので、それを消す
            keepEdges.EdgeList.Sort();

            edgeN = 0;
            WingedEdge prev = null;
            bool[] erasable = new bool[keepEdges.EdgeList.Count];

            int[] nface0 = new int[2] { int.MaxValue, int.MaxValue };
            int[] nface1 = new int[2] { int.MaxValue, int.MaxValue };

            int[] wing_faces0 = new int[4] { int.MaxValue, int.MaxValue, int.MaxValue, int.MaxValue };
            int[] wing_faces1 = new int[4] { int.MaxValue, int.MaxValue, int.MaxValue, int.MaxValue };

            foreach (WingedEdge curr in keepEdges.EdgeList)
            {
                if (curr.Vertex0 == kill || curr.Vertex1 == kill)
                {
                    throw ExcepHandle.CreateException("ShrinkEdge : Fail to assemple edge info");
                }

                erasable[edgeN] = false;
                if (prev != null)
                {
                    if (prev.IsVerticesEqual(curr))
                    {
                        erasable[edgeN] = true;
                        if ((curr.Face0 == killFace0 || curr.Face1 == killFace0) && (killFace0 != int.MaxValue))
                        {
                            wing_faces0[0] = prev.Face0;
                            wing_faces0[1] = prev.Face1;
                            wing_faces0[2] = curr.Face0;
                            wing_faces0[3] = curr.Face1;
                        }
                        else
                        {
                            wing_faces1[0] = prev.Face0;
                            wing_faces1[1] = prev.Face1;
                            wing_faces1[2] = curr.Face0;
                            wing_faces1[3] = curr.Face1;
                        }
                    }
                }
                prev = curr;
                edgeN++;
            }
            Array.Sort(wing_faces0);
            Array.Sort(wing_faces1);
            for (int i = 1; i < 4; i++)
            {
                if (wing_faces0[i] == wing_faces0[i - 1])
                {
                    wing_faces0[i] = wing_faces0[i - 1] = int.MaxValue;
                }
            }
            for (int i = 1; i < 4; i++)
            {
                if (wing_faces1[i] == wing_faces1[i - 1])
                {
                    wing_faces1[i] = wing_faces1[i - 1] = int.MaxValue;
                }
            }
            Array.Sort(wing_faces0);
            Array.Sort(wing_faces1);

            nface0[0] = wing_faces0[0];
            nface0[1] = wing_faces0[1];
            nface1[0] = wing_faces1[0];
            nface1[1] = wing_faces1[1];

            int enableN = 0;
            // 重複したエッジを削除して綺麗に並べる
            enableN = 0;
            int keN = keepEdges.EdgeList.Count;
            bool result = true;
            for (int t = 0; t < keN; t++)
            {
                if (!erasable[t])
                {
                    if (t + 1 < keN && erasable[t + 1])
                    {
                        WingedEdge curr = keepEdges.EdgeList[t];
                        WingedEdge next = keepEdges.EdgeList[t + 1];

                        if ((curr.Face0 != next.Face0 && curr.Face1 != next.Face1) &&
                            (curr.Face0 != next.Face1 && curr.Face1 != next.Face0))
                        {
                            result = false;
                        }

                        if (curr.Face0 == next.Face0)
                        {
                            curr.Face0 = next.Face1;
                        }
                        else if (curr.Face0 == next.Face1)
                        {
                            curr.Face0 = next.Face0;
                        }
                        else if (curr.Face1 == next.Face0)
                        {
                            curr.Face1 = next.Face1;
                        }
                        else if (curr.Face1 == next.Face1)
                        {
                            curr.Face1 = next.Face0;
                        }

                        int tmp0 = curr.Face0;
                        int tmp1 = curr.Face1;
                        curr.Face0 = Math.Min(tmp0, tmp1);
                        curr.Face1 = Math.Max(tmp0, tmp1);
                    }
                    keepEdges.EdgeList[enableN] = keepEdges.EdgeList[t];
                    enableN++;
                }
            }

            keepEdges.EdgeList.Resize(enableN, null);
            keN = keepEdges.EdgeList.Count;
            for (int t = 1; t < keN; t++)
            {
                if (keepEdges.EdgeList[t].IsVerticesEqual(keepEdges.EdgeList[t - 1]))
                {
                    mMesh.DebugOutput("error.obj");
                    throw new Exception();
                }
            }
            return result;
        }

        /// <summary>
        /// 代表エッジのみを縮退させる（旧バージョン）: Opaque な関数
        /// </summary>
        /// <param name="keep"></param>
        /// <param name="kill"></param>
        public void ShrinkRepEdge(int keep, int kill)
        {
            ShrinkEdge(keep, kill, mRepresentativeKey);
        }

        /// <summary>
        /// エッジ縮退をした場合に、各プロパティがどちら側に落ちるか？を取得する
        /// </summary>
        /// <param name="targetRepEdge"></param>
        /// <param name="keep"></param>
        /// <param name="kill"></param>
        /// <param name="keeps"></param>
        /// <param name="kills"></param>
        private void GetShrinkInfo(WingedEdge targetRepEdge, int keep, int kill, Dictionary<SourceKey, int2> keeps, Dictionary<SourceKey, int2> kills)
        {
            int fid = targetRepEdge.Face0;
            int vo = mMesh.mVoffs[fid];
            int vn = mMesh.mVnums[fid];
            int sn = mMesh.SourceN;
            int po = mMesh.GetSourceOfset(targetRepEdge.SpaceKey.type, targetRepEdge.SpaceKey.name);
            for (int vi = 0; vi < vn; vi++)
            {
                int vid = mMesh.mVbuffs[(vo + vi) * sn + po];

                if (vid == keep || vid == kill)
                {
                    for (int si = 0; si < sn; si++)
                    {
                        if (si == po) { continue; }
                        int sid = mMesh.mVbuffs[(vo + vi) * sn + si];
                        SourceKey key = mMesh.Sources[si].Key;
                        if (vid == kill)
                        {
                            kills[key] = new int2(sid, int.MaxValue);
                        }
                        else
                        {
                            keeps[key] = new int2(sid, int.MaxValue);
                        }
                    }
                }
            }

            if (targetRepEdge.Face1 != int.MaxValue)
            {
                fid = targetRepEdge.Face1;
                vo = mMesh.mVoffs[fid];
                vn = mMesh.mVnums[fid];
                for (int vi = 0; vi < vn; vi++)
                {
                    int vid = mMesh.mVbuffs[(vo + vi) * sn + po];
                    if (vid == keep || vid == kill)
                    {
                        for (int si = 0; si < sn; si++)
                        {
                            if (si == po) { continue; }
                            int sid = mMesh.mVbuffs[(vo + vi) * sn + si];
                            SourceKey key = mMesh.Sources[si].Key;

                            if (vid == kill)
                            {
                                kills[key][1] = sid;
                            }
                            if (vid == keep)
                            {
                                keeps[key][1] = sid;
                            }
                        }
                    }
                }
            }
        }

        [Conditional("DEBUG")]
        public void CheckChidRef(int vtx)
        {
            foreach (var edge in RepVEL[vtx].EdgeList)
            {
                CheckChidRef(edge);
            }
        }

        [Conditional("DEBUG")]
        private void CheckChidRef(WingedEdge targetRepEdge)
        {
            var keeps = new Dictionary<SourceKey, int2>();
            var kills = new Dictionary<SourceKey, int2>();
            int keep = targetRepEdge.Vertex0;
            int kill = targetRepEdge.Vertex1;
            GetShrinkInfo(targetRepEdge, keep, kill, keeps, kills);
            foreach (var pair in targetRepEdge.ChildEdges)
            {
                SourceKey key = pair.Key;
                int2 tmp;
                if (keeps.TryGetValue(key, out tmp))
                {
                    foreach (var refEdge in pair.Value)
                    {
                        // これは、最終的には外すべきもの
                        bool vi = (tmp[0] == refEdge.Vertex0 || tmp[1] == refEdge.Vertex0) && refEdge.Vertex0 != int.MaxValue;
                        bool vj = (tmp[0] == refEdge.Vertex1 || tmp[1] == refEdge.Vertex1) && refEdge.Vertex1 != int.MaxValue;
                        if (!vi && !vj)
                        {
                            throw new Exception("Winged edge is broken");
                        }
                    }
                }
                if (kills.TryGetValue(key, out tmp))
                {
                    foreach (var refEdge in pair.Value)
                    {
                        // これは、最終的には外すべきもの
                        bool vi = (tmp[0] == refEdge.Vertex0 || tmp[1] == refEdge.Vertex0) && refEdge.Vertex0 != int.MaxValue;
                        bool vj = (tmp[0] == refEdge.Vertex1 || tmp[1] == refEdge.Vertex1) && refEdge.Vertex1 != int.MaxValue;
                        if (!vi && !vj)
                        {
                            throw new Exception("Winged edge is broken");
                        }
                    }
                }
            }
        }
        /// <summary>
        /// Repエッジとそれに連なるエッジを削除します
        /// </summary>
        /// <param name="keep"></param>
        /// <param name="kill"></param>
        public bool ShrinkEdgeAll(int keep, int kill, out Dictionary<SourceKey, int2> keeps, out Dictionary<SourceKey, int2> kills)
        {
            List<WingedEdge> edges = FindRepEdge(keep, kill);
            if (edges.Count > 1)
            {
                throw ExcepHandle.CreateException(string.Format("Doubly edges are set for edge[{0},{1}]", keep, kill));
            }

            WingedEdge targetRepEdge = edges[0];

            //----------------------------------------------
            // どちら側に縮退すべきであるか？を検索する
            if (targetRepEdge.Face0 == int.MaxValue)
            {
                throw ExcepHandle.CreateException("Winged edge is broken");
            }
            // 返り値にも使う、縮退する方向の頂点ＩＤの集合
            // ハードエッジの場合は二つのＩＤが格納される
            keeps = new Dictionary<SourceKey, int2>();
            kills = new Dictionary<SourceKey, int2>();

            GetShrinkInfo(targetRepEdge, keep, kill, keeps, kills);
            foreach (var pair in targetRepEdge.ChildEdges)
            {
                if (pair.Value.Count > 1)
                {
                    bool b0 = false;
                    bool b1 = false;
                    SourceKey key = pair.Key;
                    int2 tmp;
                    // 両端での枝割れをチェック
                    if (keeps.TryGetValue(key, out tmp))
                    {
                        if (tmp[0] != tmp[1])
                        {
                            b0 = true;
                        }
                    }
                    if (kills.TryGetValue(key, out tmp))
                    {
                        if (tmp[0] != tmp[1])
                        {
                            b1 = true;
                        }
                    }
                    if (b0 && !b1)
                    {
                        return false;
                    }
                }
            }

            //----------------------------------------------
            foreach (var pair in targetRepEdge.ChildEdges)
            {
                SourceKey key = pair.Key;
                int2 tmp;
                if (keeps.TryGetValue(key, out tmp))
                {
                    foreach (var refEdge in pair.Value)
                    {
                        // これは、最終的には外すべきもの
                        int vi = 0;
                        int vj = 0;
                        if ((tmp[0] == refEdge.Vertex0 || tmp[1] == refEdge.Vertex0) &&
                            refEdge.Vertex0 != int.MaxValue)
                        {
                            vi = refEdge.Vertex0;
                            vj = refEdge.Vertex1;
                        }
                        else if ((tmp[0] == refEdge.Vertex1 || tmp[1] == refEdge.Vertex1) &&
                                 refEdge.Vertex1 != int.MaxValue)
                        {
                            vi = refEdge.Vertex1;
                            vj = refEdge.Vertex0;
                        }
                        else
                        {
                            throw ExcepHandle.CreateException("Winged edge is broken");
                        }

                        ShrinkEdge(vi, vj, key);
#if DEBUG
                        refEdge.Test = targetRepEdge;
#endif
                    }
                }
            }
            keeps.Add(targetRepEdge.SpaceKey, new int2(keep, int.MaxValue));
            kills.Add(targetRepEdge.SpaceKey, new int2(kill, int.MaxValue));
            ShrinkEdge(keep, kill, mRepresentativeKey);
            UpdateNeighborEdgeReference(edges[0]);
            return true;
        }

        /// <summary>
        /// エッジの縮退を指定のエッジ種別で行う
        /// </summary>
        /// <param name="keep"></param>
        /// <param name="kill"></param>
        /// <param name="type"></param>
        /// <param name="name"></param>
        public void ShrinkEdge(int keep, int kill, SourceType type, string name)
        {
            ShrinkEdge(keep, kill, new SourceKey(type, name));
        }

        /// <summary>
        /// エッジの縮退を行い、リストの整理をする
        /// </summary>
        /// <param name="keep">保持頂点</param>
        /// <param name="kill">削除頂点</param>
        public void ShrinkEdge(int keep, int kill, SourceKey key)
        {
            try
            {
                int edgeN = GetWingedEdgeN(key.type, key.name);
                VtxWeList vtxEdgeList = GetVtxWEList(key.type, key.name);

                if (keep > edgeN || kill > edgeN)
                {
                    throw ExcepHandle.CreateException("Vertex indices are over bounds.");
                }

                // 消去予定のフェース
                int killFace0 = int.MaxValue;
                int killFace1 = int.MaxValue;

                // keep-kill のエッジを消す
                WingedEdge r = null;
                do
                {
                    r = vtxEdgeList[keep].Remove(kill);
                    if (r != null)
                    {
                        // 消去予定のフェース
                        killFace0 = r.Face0;
                        killFace1 = r.Face1;
                        r.UnMovable = true;
                        //r.Vertex0 = int.MaxValue;
                        //r.Vertex1 = int.MaxValue;
                    }
                }
                while (r != null);

                // keep-kill のエッジを消す
                r = null;
                do
                {
                    r = vtxEdgeList[kill].Remove(keep);
                    if (r != null)
                    {
                        r.UnMovable = true;
                        // 消去予定のフェース
                        if (killFace0 != r.Face0 || killFace1 != r.Face1)
                        {
                            throw ExcepHandle.CreateException("Error");
                        }
                    }
                }
                while (r != null);

                if (killFace0 == int.MaxValue && killFace1 == int.MaxValue)
                {
                    throw ExcepHandle.CreateException("ShrinkEdge() :: try to Shrink illegal winged edge");
                }

                // インデックスの書き換え
                vtxEdgeList[keep].EdgeList.AddRange(vtxEdgeList[kill].EdgeList);
                foreach (var edge in vtxEdgeList[keep].EdgeList)
                {
                    if (edge.Vertex0 == kill)
                    {
                        edge.Vertex0 = keep;
                    }
                    else if (edge.Vertex1 == kill)
                    {
                        edge.Vertex1 = keep;
                    }
                    // 入れ替え
                    int minID = Math.Min(edge.Vertex0, edge.Vertex1);
                    int maxID = Math.Max(edge.Vertex0, edge.Vertex1);
                    edge.Vertex0 = minID;
                    edge.Vertex1 = maxID;
                }
                vtxEdgeList[kill].EdgeList.Clear();

                // 重複したエッジがあるので、それを消す
                vtxEdgeList[keep].EdgeList.Sort();

                edgeN = 0;
                WingedEdge prev = null;
                bool[] erasable = new bool[vtxEdgeList[keep].EdgeList.Count];

                int[] nface0 = new int[2] { int.MaxValue, int.MaxValue };
                int[] nface1 = new int[2] { int.MaxValue, int.MaxValue };

                int[] wing_faces0 = new int[4] { int.MaxValue, int.MaxValue, int.MaxValue, int.MaxValue };
                int[] wing_faces1 = new int[4] { int.MaxValue, int.MaxValue, int.MaxValue, int.MaxValue };

                foreach (WingedEdge curr in vtxEdgeList[keep].EdgeList)
                {
                    if (curr.Vertex0 == kill || curr.Vertex1 == kill)
                    {
                        throw ExcepHandle.CreateException("ShrinkEdge : Fail to assemple edge information.");
                    }

                    erasable[edgeN] = false;
                    if (prev != null)
                    {
                        if (prev.IsVerticesEqual(curr))
                        {
                            erasable[edgeN] = true;
                            if ((curr.Face0 == killFace0 || curr.Face1 == killFace0) && (killFace0 != int.MaxValue))
                            {
                                wing_faces0[0] = prev.Face0;
                                wing_faces0[1] = prev.Face1;
                                wing_faces0[2] = curr.Face0;
                                wing_faces0[3] = curr.Face1;
                            }
                            else
                            {
                                wing_faces1[0] = prev.Face0;
                                wing_faces1[1] = prev.Face1;
                                wing_faces1[2] = curr.Face0;
                                wing_faces1[3] = curr.Face1;
                            }
                        }
                    }
                    prev = curr;
                    edgeN++;
                }
                Array.Sort(wing_faces0);
                Array.Sort(wing_faces1);
                for (int i = 1; i < 4; i++)
                {
                    if (wing_faces0[i] == wing_faces0[i - 1])
                    {
                        wing_faces0[i] = wing_faces0[i - 1] = int.MaxValue;
                    }
                }

                for (int i = 1; i < 4; i++)
                {
                    if (wing_faces1[i] == wing_faces1[i - 1])
                    {
                        wing_faces1[i] = wing_faces1[i - 1] = int.MaxValue;
                    }
                }

                Array.Sort(wing_faces0);
                Array.Sort(wing_faces1);

                nface0[0] = wing_faces0[0];
                nface0[1] = wing_faces0[1];
                nface1[0] = wing_faces1[0];
                nface1[1] = wing_faces1[1];

                // 重複したエッジを削除して綺麗に並べる
                int enableN = 0;
                enableN = 0;
                int keN = vtxEdgeList[keep].EdgeList.Count;
                for (int t = 0; t < keN; t++)
                {
                    if (!erasable[t])
                    {
                        // 重複したエッジの一方のウイング情報を書き換えて
                        // もう一方は消し去る
                        if (t + 1 < keN && erasable[t + 1])
                        {
                            WingedEdge curr = vtxEdgeList[keep].EdgeList[t];
                            WingedEdge next = vtxEdgeList[keep].EdgeList[t + 1];

                            if (curr.Face0 == next.Face0)
                            {
                                curr.Face0 = next.Face1;
                            }
                            else if (curr.Face0 == next.Face1)
                            {
                                curr.Face0 = next.Face0;
                            }
                            else if (curr.Face1 == next.Face0)
                            {
                                curr.Face1 = next.Face1;
                            }
                            else if (curr.Face1 == next.Face1)
                            {
                                curr.Face1 = next.Face0;
                            }

                            int tmp0 = curr.Face0;
                            int tmp1 = curr.Face1;

                            curr.Face0 = Math.Min(tmp0, tmp1);
                            curr.Face1 = Math.Max(tmp0, tmp1);
                            // 消し去る方の参照エッジ情報を、もう一方に追加する
                            curr.AddChildEdgesFrom(next);

                            next.ChildEdges.Clear();
                        }
                        vtxEdgeList[keep].EdgeList[enableN] = vtxEdgeList[keep].EdgeList[t];
                        enableN++;
                    }
                    else
                    {
                        var edge = vtxEdgeList[keep].EdgeList[t];
                        int other = edge.Vertex0 == keep ? edge.Vertex1 : edge.Vertex0;

                        edge.Vertex0 = int.MaxValue;
                        edge.Vertex1 = int.MaxValue;
                        edge.UnMovable = true;
                        int otherEN = vtxEdgeList[other].EdgeList.Count;
                        for (int oe = 0; oe < otherEN; oe++)
                        {
                            var otedge = vtxEdgeList[other].EdgeList[oe];
                            if (otedge.Vertex0 == int.MaxValue && otedge.Vertex1 == int.MaxValue)
                            {
                                vtxEdgeList[other].EdgeList.RemoveAt(oe);
                                oe = 0;
                                otherEN = vtxEdgeList[other].EdgeList.Count;
                            }
                        }
                    }
                }
                vtxEdgeList[keep].EdgeList.Resize(enableN, null);

                // フェースインデックスを書き換える
                ResizableList<WingedEdge> tmpList = new ResizableList<WingedEdge>();
                foreach (var e in vtxEdgeList[keep].EdgeList)
                {
                    WingedEdge edge = new WingedEdge(e);
                    bool update = false;
                    if ((edge.Face0 == killFace0 || edge.Face1 == killFace0) && (killFace0 != int.MaxValue))
                    {
                        e.Face0 = nface0[0];
                        e.Face1 = nface0[1];
                        update = true;
                    }
                    else if ((edge.Face0 == killFace1 || edge.Face1 == killFace1) && (killFace1 != int.MaxValue))
                    {
                        e.Face0 = nface1[0];
                        e.Face1 = nface1[1];
                        update = true;
                    }

                    if (update)
                    {
                        if (e.Face0 == e.Face1)
                        {
                            e.Face1 = int.MaxValue;
                        }
                        if (nface1[0] == int.MaxValue || nface1[1] == int.MaxValue)
                        {
                            e.Boder = true;
                        }
                        if (e.Face0 == killFace0 || e.Face0 == killFace1)
                        {
                            e.Face0 = int.MaxValue;
                        }
                        if (e.Face1 == killFace0 || e.Face1 == killFace1)
                        {
                            e.Face1 = int.MaxValue;
                        }

                        int tmp0 = e.Face0;
                        int tmp1 = e.Face1;

                        e.Face0 = Math.Min(tmp0, tmp1);
                        e.Face1 = Math.Max(tmp0, tmp1);
                    }

                    // 両ウイングが開放端のエッジ( これはゴミ
                    if (e.Face0 == int.MaxValue && e.Face1 == int.MaxValue)
                    {
                        // ゴミを消す
                        int other = edge.Vertex0 == keep ? edge.Vertex1 : edge.Vertex0;
                        int eid = edge.ID;

                        int otherEN = vtxEdgeList[other].EdgeList.Count;
                        for (int oe = 0; oe < otherEN; oe++)
                        {
                            var otedge = vtxEdgeList[other].EdgeList[oe];
                            if (otedge.Face0 == int.MaxValue && otedge.Face1 == int.MaxValue)
                            {
                                otedge.UnMovable = true;
                                vtxEdgeList[other].EdgeList.RemoveAt(oe);
                                oe = 0;
                                otherEN = vtxEdgeList[other].EdgeList.Count;
                            }
                        }
                    }
                    else
                    {
                        tmpList.Add(e);
                    }
                }

                if (vtxEdgeList[keep].EdgeList.Count != tmpList.Count)
                {
                    vtxEdgeList[keep].EdgeList = tmpList;
                }
            }
            catch (Exception ex)
            {
                string message = string.Format("Fail to Shrink winged edge [{0},{1}]", keep, kill);
                throw ExcepHandle.CreateException(message, ex);
            }
        }

        /// <summary>
        /// 最大のウイング角を取得する
        /// </summary>
        /// <param name="vid"></param>
        /// <returns></returns>
        public double GetMaxWingAngle(int vid)
        {
            double ret = 0;
            foreach (var edge in RepVEL[vid].EdgeList)
            {
                ret = Math.Max(ret, GetWingAngle(edge));
            }
            return ret;
        }

        /// <summary>
        /// エッジの縮退後に指定のエッジの参照エッジを更新する処理を行う
        /// 削除済みエッジの相互参照を除去する
        /// </summary>
        /// <param name="edge"></param>
        private void UpdateEdgeReference(WingedEdge edge)
        {
            foreach (var refEdgeContainer in edge.ChildEdges)
            {
                SourceKey key = refEdgeContainer.Key;
                WERsList childs = refEdgeContainer.Value;
                int sizeN = refEdgeContainer.Value.Count;
                int i = 0;
                while (i < sizeN)
                {
                    WingedEdge childEdge = childs[i];
                    if (!childEdge.ParentEdges.Contains(edge))
                    {
                        throw ExcepHandle.CreateException("error : illegal winged edge reference");
                    }

                    if (childEdge.UnMovable)
                    {
                        childEdge.ParentEdges.Remove(edge);
                        childs.RemoveAt(i);
                        sizeN--;
                    }
                    else
                    {
                        i++;
                    }
                }
            }
        }

        /// <summary>
        /// 自分自身と近接エッジの参照関係を更新する更新
        /// </summary>
        /// <param name="edge"></param>
        private void UpdateNeighborEdgeReference(WingedEdge edge)
        {
            WERsList edges0 = mVtxEdgeList[edge.SpaceKey][edge.Vertex0].Edges;
            WERsList edges1 = mVtxEdgeList[edge.SpaceKey][edge.Vertex1].Edges;

            foreach (var wingedEdge in edges0)
            {
                UpdateEdgeReference(wingedEdge);
            }
            foreach (var wingedEdge in edges1)
            {
                UpdateEdgeReference(wingedEdge);
            }
        }
    }
}
