﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public class VtxTri
    {
        public ResizableList<int> TriList { get; set; } = new ResizableList<int>();
    }

    public class VtxTriMesh
    {
        private Mesh mMesh = null;
        private SourceKey mRepresentedSourceType = new SourceKey(SourceType.Position, null);
        private ResizableList<VtxTri> mVTList = new ResizableList<VtxTri>();
        private ResizableList<int> mNonMfdVList = new ResizableList<int>();

        internal Mesh Mesh
        {
            get { return mMesh; }
        }

        public ResizableList<VtxTri> VTL
        {
            get { return mVTList; }
        }

        public SourceKey RepKey
        {
            get { return mRepresentedSourceType; }
        }

        public ResizableList<int> NonMfdVertices
        {
            get { return mNonMfdVList; }
        }

        public static int GetRequiredMemSize(Mesh mesh, SourceKey sourceKey)
        {
            try
            {
                var adjNum = new ResizableList<int>();
                int faceN = mesh.FaceN;
                int vtxN = mesh.VertexN;
                int posOfs = mesh.GetSourceOfset(sourceKey.type, sourceKey.name);
                int sourceN = mesh.Sources.Count;

                adjNum.Resize(vtxN, 0);
                // 初期カウントアップ
                for (int f = 0; f < faceN; f++)
                {
                    int vnum = mesh.mVnums[f];
                    int vofs = mesh.mVoffs[f];
                    for (int v = 0; v < vnum; v++)
                    {
                        int vid = mesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                        adjNum[vid]++;
                    }
                }

                int sum = 0;
                // オフセット設定,バッファ確保
                for (int v = 0; v < vtxN; v++)
                {
                    sum += adjNum[v];
                }
                return sum * sizeof(int);
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(ex.Message);
            }
        }

        /// <summary>
        /// 初期化・データ生成処理
        /// </summary>
        /// <param name="mesh"></param>
        public void Create(Mesh mesh)
        {
            try
            {
                Destroy();
                mMesh = mesh;

                ResizableList<int> adjNum = new ResizableList<int>();

                int faceN = mMesh.FaceN;
                int vtxN = mMesh.VertexN;
                int posOfs = mMesh.GetSourceOfset(RepKey.type, RepKey.name);
                int sourceN = mMesh.Sources.Count;

                mVTList.Resize(vtxN, null);
                adjNum.Resize(vtxN, 0);

                // 初期カウントアップ
                for (int f = 0; f < faceN; f++)
                {
                    int vnum = mMesh.mVnums[f];
                    int vofs = mMesh.mVoffs[f];
                    for (int v = 0; v < vnum; v++)
                    {
                        int vid = mMesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                        adjNum[vid]++;
                    }
                }

                // オフセット設定,バッファ確保
                for (int v = 0; v < vtxN; v++)
                {
                    mVTList[v] = new VtxTri();
                    mVTList[v].TriList.Resize(adjNum[v], 0);
                }

                // 実際にバッファを設定する
                for (int v = 0; v < vtxN; v++)
                {
                    adjNum[v] = 0;
                }

                // リスト設定
                for (int f = 0; f < faceN; f++)
                {
                    int vnum = mMesh.mVnums[f];
                    int vofs = mMesh.mVoffs[f];

                    for (int v = 0; v < vnum; v++)
                    {
                        int v0 = mMesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                        mVTList[v0].TriList[adjNum[v0]] = f;
                        adjNum[v0]++;
                    }
                }
            }
            catch (Exception ex)
            {
                mVTList.Clear();
                mNonMfdVList.Clear();
                ExcepHandle.CreateException(ex);
                throw;
            }
        }

        /// <summary>
        /// 非多様体頂点の検出を行います。
        /// 今のところ、まだ空
        /// </summary>
        /// <returns></returns>
        private int PickNonManifoldVertices()
        {
            foreach (var vtxTri in mVTList)
            {
                foreach (var fi in vtxTri.TriList)
                {
                }
            }
            return 0;
        }

        /// <summary>
        /// 面積ゼロのフェースを削除します
        /// </summary>
        /// <param name="keepID"></param>
        /// <param name="newpos"></param>
        /// <returns></returns>
        internal int CleanDegenerateFace(int keepID)
        {
            int result = 0;

            int num = 0;
            int fid = 0;

            try
            {
                num = mVTList[keepID].TriList.Count;
                if (num == 0)
                {
                    return 0;
                }

                for (int f = 0; f < num; f++)
                {
                    fid = mVTList[keepID].TriList[f];
                    if (mMesh.IsFaceDegenerate(fid))
                    {
                        mMesh.InvalidateFace(fid);
                        result++;
                    }
                }
                fid = int.MaxValue;

                // 入れ物の内容物を入れ替え
                if (result > 0)
                {
                    ResizableList<int> tmp = new ResizableList<int>();
                    for (int f = 0; f < num; f++)
                    {
                        if (!mMesh.IsInvalidateFace(mVTList[keepID].TriList[f]))
                        {
                            tmp.Add(mVTList[keepID].TriList[f]);
                        }
                    }
                    mVTList[keepID].TriList.Clear();
                    mVTList[keepID].TriList = tmp;
                }
                return result;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(string.Format("Fail to degenere faces : fid {0}.", fid), ex);
            }
        }

        /// <summary>
        /// 指定頂点の周りの面の法線を再計算します
        /// </summary>
        /// <param name="vtxWedge">再計算したい頂点</param>
        internal void RecomputeFaceNormalsAndArea(int vtxId)
        {
            try
            {
                if (RepKey.type != SourceType.Position) { return; }
                var anm = mVTList[vtxId].TriList.Count;
                for (var i = 0; i < anm; i++)
                {
                    var afid = mVTList[vtxId].TriList[i];
                    mMesh.CalculateTriangleNormal(afid);
                    mMesh.ComputeFaceArea(afid);
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(string.Format("Fail to recompute vertex normal : vid {0}.", vtxId), ex);
            }
        }

        /// <summary>
        /// 重複したフェースが存在しないか？を確認する。
        /// </summary>
        /// <param name="keepVtx"></param>
        /// <param name="killVtx"></param>
        /// <param name="chkList"></param>
        /// <returns></returns>
        internal bool TestDuplicateFace(int keepVtx, int killVtx, ResizableList<int> chkList)
        {
            bool result = false;
            int keepID = keepVtx;
            int killID = killVtx;
            int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
            int sourceN = mMesh.Sources.Count;

            ResizableList<ResizableList<int>> vid = new ResizableList<ResizableList<int>>();

            vid.Resize(chkList.Count, null);
            for (int i = 0; i < chkList.Count; i++)
            {
                int afid = chkList[i];
                int vnum = mMesh.mVnums[afid];
                int vofs = mMesh.mVoffs[afid];

                vid[i] = new ResizableList<int>();
                vid[i].Resize(3, int.MaxValue);

                for (int v = 0; v < vnum; v++)
                {
                    int pid = mMesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                    vid[i][v] = pid;

                    // 削除頂点IDに一致する場合
                    if (pid == killID || pid == keepID)
                    {
                        vid[i][v] = keepID;
                    }
                }
            }

            // 完全に重複したフェースの存在を確認
            int faceN = chkList.Count;
            for (int fi = 0; fi < faceN - 1; fi++)
            {
                int afid = chkList[fi];
                if (mMesh.IsInvalidateFace(afid)) { continue; }

                int[] p = new int[3]
                    {
                        vid[fi][0],
                        vid[fi][1],
                        vid[fi][2]
                    };
                Array.Sort(p);

                for (int fj = fi + 1; fj < faceN; fj++)
                {
                    int afid2 = chkList[fj];
                    if (afid != afid2 && !mMesh.IsInvalidateFace(afid2))
                    {
                        int[] q = new int[3]
                            {
                                vid[fj][0],
                                vid[fj][1],
                                vid[fj][2]
                            };
                        Array.Sort(q);

                        if (p[0] == q[0] && p[1] == q[1] && p[2] == q[2])
                        {
                            result = true;
                        }
                    }
                }
            }
            return result;
        }

        /// <summary>
        /// エッジ縮退によって、フリップするフェースが出来ないか？をテストする
        /// </summary>
        /// <param name="keepVtx"></param>
        /// <param name="killVtx"></param>
        /// <param name="newPos"></param>
        /// <param name="chkList"></param>
        /// <returns></returns>
        internal bool TestFlip(int keepVtx, int killVtx, VectorN newPos, ResizableList<int> chkList)
        {
            int keepID = keepVtx;
            int killID = killVtx;
            int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
            int sourceN = mMesh.Sources.Count;
            MeshSourceBase PosSource = mMesh.GetSource(SourceType.Position, null);

            // 古い法線をコピーして保存しておきます
            Dictionary<int, Vector3> oldNml = new Dictionary<int, Vector3>();
            Dictionary<int, Vector3> newNml = new Dictionary<int, Vector3>();
            for (int i = 0; i < chkList.Count; i++)
            {
                int afid = chkList[i];
                oldNml[afid] = mMesh.mFaceNormal[afid];
            }

            // 新しい法線を計算します
            Vector3[] pos = new Vector3[3];
            int[] vid = new int[3];

            pos[0] = new Vector3();
            pos[1] = new Vector3();
            pos[2] = new Vector3();
            for (int a = 0; a < chkList.Count; a++)
            {
                int afid = chkList[a];
                int vnum = mMesh.mVnums[afid];
                int vofs = mMesh.mVoffs[afid];

                for (int v = 0; v < vnum; v++)
                {
                    int pid = mMesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                    vid[v] = pid;
                    MeshSourceDblCtrler.GetAsVec3(PosSource, pid, ref pos[v]);
                    // 削除頂点IDに一致する場合
                    if (pid == killID || pid == keepID)
                    {
                        vid[v] = keepID;
                        pos[v].x = newPos[0];
                        pos[v].y = newPos[1];
                        pos[v].z = newPos[2];
                    }
                }

                if (vid[0] == vid[1] || vid[0] == vid[2] || vid[1] == vid[2])
                {
                    newNml[afid] = null;
                }
                else
                {
                    var crs = Mesh.CalculateTriangleNormal(pos[0], pos[1], pos[2]);
                    newNml[afid] = crs;
                }
            }

            bool flipFace = false;
            // フリップチェック
            for (int a = 0; a < chkList.Count; a++)
            {
                int afid = chkList[a];
                if (newNml[afid] != null)
                {
                    if (newNml[afid].Norm() < double.Epsilon)
                    {
                        flipFace = true;
                    }

                    double dot = newNml[afid].Dot(oldNml[afid]);
                    // 裏返った場合
                    if (dot < Math.Cos(170.0 * 3.14 / 180.0))
                    {
                        flipFace = true;
                    }
                }
            }
            return flipFace;
        }

        /// <summary>
        /// エッジ縮退によってフリップが起こるか？をチェックします
        /// </summary>
        /// <param name="keepVtx">維持する頂点</param>
        /// <param name="killVtx">削除する頂点</param>
        /// <param name="newPos">新しい位置</param>
        /// <returns>true : フリップが起こる場合 false: フリップしない </returns>
        internal bool TestShrinkEdge(WedgeVertex keepVtx, WedgeVertex killVtx, VectorN newPos)
        {
            if (keepVtx == null || killVtx == null)
            {
                throw new ArgumentException();
            }
            int keepID = keepVtx.Vertex;
            int killID = killVtx.Vertex;
            return TestShrinkEdge(keepID, killID, newPos);
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="keepVtx"></param>
        /// <param name="killVtx"></param>
        /// <param name="newPos"></param>
        /// <returns></returns>
        public bool TestShrinkEdge(int keepVtx, int killVtx, VectorN newPos)
        {
            try
            {
                int keepID = keepVtx;
                int killID = killVtx;
                // 新しい法線を計算します
                ResizableList<int> chkList = new ResizableList<int>();
                chkList.AddRange(mVTList[keepID].TriList);
                chkList.AddRange(mVTList[killID].TriList);

                if (TestFlip(keepVtx, killVtx, newPos, chkList)) { return false; }

                if (TestDuplicateFace(keepVtx, killVtx, chkList)) { return false; }
                return true;
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException(string.Format("Fail to test hhrinkable edge  : [{0},{1}].", keepVtx, killVtx), ex);
            }
        }

        /// <summary>
        /// 指定の頂点の隣接フェースリストの中から、指定プロパティを使用している
        /// フェースを探してきて、そのリストを返します。
        /// </summary>
        /// <param name="vid"> 頂点ID</param>
        /// <param name="type"> プロパティの種類（頂点以外）</param>
        /// <param name="name"> そのプロパティの名前</param>
        /// <param name="pid">　そのプロパティのインデックス</param>
        /// <returns>同じプロパティを使用しているフェースリスト</returns>
        public List<int> GetPindexSharedFaceIndices(int vid, SourceType type, string name, int pid)
        {
            // インデックスオーバー
            if (vid >= mVTList.Count)
            { return null; }

            // 結果のフェースインデックスリスト
            List<int> result = new List<int>();
            int sn = mMesh.SourceN;
            int so = mMesh.GetSourceOfset(type, name);
            int po = mMesh.GetSourceOfset(SourceType.Position, null);

            // 隣接フェースリストを検索し
            // プロパティのインデックスがヒットするフェースを取得します
            foreach (int fid in mVTList[vid].TriList)
            {
                int vn = mMesh.mVnums[fid];
                int vo = mMesh.mVoffs[fid];
                for (int vi = 0; vi < vn; vi++)
                {
                    int sid = mMesh.VBuffer[(vo + vi) * sn + so];
                    int vtxId = mMesh.VBuffer[(vo + vi) * sn + po];
                    if (sid == pid && vtxId == vid)
                    {
                        if (!result.Contains(fid))
                        {
                            result.Add(fid);
                        }
                        //break;
                    }
                }
            }
            return result;
        }

        /// <summary>
        /// 指定のプロパティインデックスと近い（ほぼ一致している）プロパティを持っている
        /// フェースを取得して、そのリストを返します。
        /// </summary>
        /// <param name="vid"> 頂点ID　</param>
        /// <param name="type">プロパティ種別</param>
        /// <param name="name">プロパティ名</param>
        /// <param name="pid">プロパティID</param>
        /// <returns>フェースリスト</returns>
        public List<int> GetPdistSharedFaceIndices(int vid, SourceType type, string name, int pid)
        {
            // インデックスオーバー
            if (vid >= mVTList.Count)
            { return null; }

            // 結果のフェースインデックスリスト
            List<int> result = new List<int>();

            int sn = mMesh.SourceN;
            int so = mMesh.GetSourceOfset(type, name);
            int po = mMesh.GetSourceOfset(SourceType.Position, null);
            MeshSourceBase source = mMesh.GetSource(type, name);

            // 隣接フェースリストを検索し
            // プロパティのインデックスがヒットするフェースを取得します
            foreach (int fid in mVTList[vid].TriList)
            {
                int vn = mMesh.mVnums[fid];
                int vo = mMesh.mVoffs[fid];
                for (int vi = 0; vi < vn; vi++)
                {
                    int sid = mMesh.VBuffer[(vo + vi) * sn + so];
                    int vtxId = mMesh.VBuffer[(vo + vi) * sn + po];

                    if (vtxId == vid)
                    {
                        if (source.Stride == 2)
                        {
                            Vector2 s = new Vector2(), d = new Vector2();
                            MeshSourceDblCtrler.GetAsVec2(source, pid, ref s);
                            MeshSourceDblCtrler.GetAsVec2(source, sid, ref d);
                            Vector2 diff = s - d;
                            if (diff.Norm() < Mesh.minimalDistForProp)
                            {
                                if (!result.Contains(fid))
                                {
                                    result.Add(fid);
                                }
                            }
                        }
                        else if (source.Stride == 3)
                        {
                            Vector3 s = new Vector3(), d = new Vector3();
                            MeshSourceDblCtrler.GetAsVec3(source, pid, ref s);
                            MeshSourceDblCtrler.GetAsVec3(source, sid, ref d);
                            Vector3 diff = s - d;
                            if (diff.Norm() < Mesh.minimalDistForProp)
                            {
                                if (!result.Contains(fid))
                                {
                                    result.Add(fid);
                                }
                            }
                        }
                        else if (source.Stride == 4)
                        {
                            Vector4 s = new Vector4(), d = new Vector4();
                            MeshSourceDblCtrler.GetAsVec4(source, pid, ref s);
                            MeshSourceDblCtrler.GetAsVec4(source, sid, ref d);
                            Vector4 diff = s - d;
                            if (diff.Norm() < Mesh.minimalDistForProp)
                            {
                                if (!result.Contains(fid))
                                {
                                    result.Add(fid);
                                }
                            }
                        }
                    }
                }
            }
            return result;
        }

        /// <summary>
        /// 頂点バッファを取得します
        /// </summary>
        /// <param name="vbuff">格納先</param>
        /// <param name="faceId">フェースID</param>
        /// <param name="vtxId">頂点ID</param>
        internal void GetVBuff(ref int[] vbuff, int faceId, int vtxId)
        {
            try
            {
                int vo = mMesh.mVoffs[faceId];
                int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
                int sourceN = mMesh.Sources.Count;

                for (int vi = 0; vi < 3; vi++)
                {
                    int posid = mMesh.mVbuffs[(vo + vi) * sourceN + posOfs];
                    if (posid == vtxId)
                    {
                        for (int si = 0; si < sourceN; si++)
                        {
                            vbuff[si] = mMesh.mVbuffs[(vo + vi) * sourceN + si];
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                string message = string.Format("Fail to get vertex buffer: f{0},v{1}.", faceId, vtxId);
                throw ExcepHandle.CreateException(message, ex);
            }
        }

        /// <summary>
        /// プロパティインデックスの上書き
        /// </summary>
        /// <param name="vtxId"></param>
        /// <param name="killBuff"></param>
        /// <param name="keepBuff"></param>
        private void OverWritePropindex(int vtxId, int[] killBuff, int[] keepBuff, bool chkByIndex = true)
        {
            try
            {
                int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
                int sourceN = mMesh.Sources.Count;
                // face0 側のプロパティ隣接フェースのインデックス書き換えを行う
                for (int si = 0; si < sourceN; si++)
                {
                    if (si == posOfs) { continue; }
                    if (killBuff[si] == int.MaxValue) { continue; }
                    MeshSourceBase source = mMesh.Sources[si];
                    if (source == null) { continue; }

                    List<int> flist = null;
                    // 各プロパティについて、インデックスが一致するフェースを近傍から集めてFANを作る
                    if (chkByIndex)
                    {
                        flist = GetPindexSharedFaceIndices(vtxId, source.Type, source.Name, killBuff[si]);
                    }
                    else
                    {
                        flist = GetPdistSharedFaceIndices(vtxId, source.Type, source.Name, killBuff[si]);
                    }
                    foreach (var afid in flist)
                    {
                        int vnum = mMesh.mVnums[afid];
                        int vofs = mMesh.mVoffs[afid];
                        for (int v = 0; v < vnum; v++)
                        {
                            int pid = mMesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                            // 削除頂点IDに一致する場合
                            if (pid == vtxId)
                            {
                                // プロパティのインデックスを書き換える
                                mMesh.mVbuffs[(vofs + v) * sourceN + si] = keepBuff[si];
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                string message = string.Format("Fail to overwrite property index around vertex id {0}.", vtxId);
                throw ExcepHandle.CreateException(message, ex);
            }
        }

        /// <summary>
        /// プロパティを上書きします
        /// </summary>
        /// <param name="killVtx"></param>
        /// <param name="keepBuff"></param>
        private void OverWritePropindex(WedgeVertex killVtx, int[] keepBuff, bool chkByIndex = true)
        {
            try
            {
                int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
                int sourceN = mMesh.Sources.Count;
                int cntN = killVtx.Keys.Count;
                int ntnum = mVTList[killVtx.Vertex].TriList.Count;

                for (int i = 0; i < cntN; i++)
                {
                    string name = killVtx.Keys[i].name;
                    SourceType type = killVtx.Keys[i].type;
                    int si = mMesh.GetSourceOfset(type, name);
                    foreach (var propIndex in killVtx.Props[i])
                    {
                        List<int> flist = null;
                        if (chkByIndex)
                        {
                            flist = GetPindexSharedFaceIndices(killVtx.Vertex, type, name, propIndex);
                        }
                        else
                        {
                            flist = GetPdistSharedFaceIndices(killVtx.Vertex, type, name, propIndex);
                        }
                        foreach (var afid in flist)
                        {
                            int vnum = mMesh.mVnums[afid];
                            int vofs = mMesh.mVoffs[afid];
                            for (int v = 0; v < vnum; v++)
                            {
                                int pid = mMesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                                // 削除頂点IDに一致する場合
                                if (pid == killVtx.Vertex)
                                {
                                    mMesh.mVbuffs[(vofs + v) * sourceN + si] = keepBuff[si];
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                string message = string.Format("Fail to overwrite property index around veretex id {0}.", killVtx);
                throw ExcepHandle.CreateException(message, ex);
            }
        }
        /// <summary>
        /// 頂点インデックスの上書き
        /// </summary>
        /// <param name="vtxId"></param>
        /// <param name="killBuff"></param>
        /// <param name="keepBuff"></param>
        private void OverWriteVertexIndex(int vtxId, int keepID, bool chkByIndex = true)
        {
            try
            {
                int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
                int sourceN = mMesh.Sources.Count;
                // face0 側のプロパティ隣接フェースのインデックス書き換えを行う
                {
                    if (vtxId == int.MaxValue) { return; }
                    MeshSourceBase source = mMesh.Sources[posOfs];
                    if (source == null) { return; }

                    List<int> flist = null;
                    if (chkByIndex)
                    {
                        flist = GetPindexSharedFaceIndices(vtxId, source.Type, source.Name, vtxId);
                    }
                    else
                    {
                        flist = GetPdistSharedFaceIndices(vtxId, source.Type, source.Name, vtxId);
                    }

                    for (int a = 0; a < flist.Count; a++)
                    {
                        int afid = flist[a];
                        int vnum = mMesh.mVnums[afid];
                        int vofs = mMesh.mVoffs[afid];

                        for (int v = 0; v < vnum; v++)
                        {
                            int pid = mMesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                            // 削除頂点IDに一致する場合
                            if (pid == vtxId)
                            {
                                // プロパティのインデックスを書き換える
                                mMesh.mVbuffs[(vofs + v) * sourceN + posOfs] = keepID;
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                string message = string.Format("Fail to overwrite vertex index around vertex id {0}.", vtxId);
                throw ExcepHandle.CreateException(message, ex);
            }
        }

        [Conditional("DEBUG")]
        private void IsEqualIndices(int[] vi0, int[] vi1)
        {
            if (vi0.Length != vi1.Length)
            {
                throw new Exception("IsEqualIndex:error");
            }

            for (int i = 0; i < vi0.Length; i++)
            {
                if (vi0[i] != vi1[i])
                {
                    throw new Exception("IsEqualIndex:error");
                }
            }
        }

        /// <summary>
        /// 重複した面を除去して、エッジを更新
        /// </summary>
        /// <param name="wemesh"></param>
        /// <param name="keepVtx"></param>
        /// <returns></returns>
        public void DeleteDuplicateFaceAndEdges(WingedEdgeMesh wemesh, int keepVtx)
        {
            int keepID = keepVtx;
            int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
            int sourceN = mMesh.Sources.Count;

            // 完全に重複したフェースの存在を確認
            int faceN = mVTList[keepID].TriList.Count;
            for (int fi = 0; fi < faceN - 1; fi++)
            {
                int afid = mVTList[keepID].TriList[fi];
                if (mMesh.IsInvalidateFace(afid)) { continue; }

                int vofs = mMesh.mVoffs[afid];
                int[] p = new int[3]
                    {
                        mMesh.mVbuffs[(vofs + 0) * sourceN + posOfs],
                        mMesh.mVbuffs[(vofs + 1) * sourceN + posOfs],
                        mMesh.mVbuffs[(vofs + 2) * sourceN + posOfs]
                    };
                Array.Sort(p);

                bool duplicate = false;
                for (int fj = fi + 1; fj < faceN; fj++)
                {
                    int afid2 = mVTList[keepID].TriList[fj];
                    if (afid != afid2 && !mMesh.IsInvalidateFace(afid2))
                    {
                        int vofs2 = mMesh.mVoffs[afid2];
                        int[] q = new int[3]
                            {
                                mMesh.mVbuffs[(vofs2 + 0) * sourceN + posOfs],
                                mMesh.mVbuffs[(vofs2 + 1) * sourceN + posOfs],
                                mMesh.mVbuffs[(vofs2 + 2) * sourceN + posOfs]
                            };
                        Array.Sort(q);
                        // 重複するフェースを消して、情報を整理する
                        if (p[0] == q[0] && p[1] == q[1] && p[2] == q[2])
                        {
                            //throw new Exception(string.Empty);

                            duplicate = true;
                            mMesh.InvalidateFace(afid2);

                            if (wemesh != null)
                            {
                                for (int qi = 0; qi < 3; qi++)
                                {
                                    foreach (var edge in wemesh.RepVEL[q[qi]].EdgeList)
                                    {
                                        if (edge.Face0 == afid2) { edge.Face0 = afid; }
                                        if (edge.Face1 == afid2) { edge.Face1 = afid; }

                                        if (edge.Face0 == edge.Face1)
                                        {
                                            edge.Face1 = int.MaxValue;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                if (duplicate && wemesh != null)
                {
                    wemesh.DeleteDuplicateEdges(p[0]);
                    wemesh.DeleteDuplicateEdges(p[1]);
                    wemesh.DeleteDuplicateEdges(p[2]);
                }
            }
        }

        private int ShrinkNeighborTriangles(int keepID, int killID)
        {
            int result = 0;
            int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
            int sourceN = mMesh.Sources.Count;
            int anm = mVTList[killID].TriList.Count;

            // 縮退するフェースを洗い出して消す
            for (int a = 0; a < anm; a++)
            {
                int afid = mVTList[killID].TriList[a];
                int vofs = mMesh.mVoffs[afid];

                int p0 = mMesh.mVbuffs[(vofs + 0) * sourceN + posOfs];
                int p1 = mMesh.mVbuffs[(vofs + 1) * sourceN + posOfs];
                int p2 = mMesh.mVbuffs[(vofs + 2) * sourceN + posOfs];

                // 縮退エッジがあったので、消す
                if (p0 == p1 || p1 == p2 || p2 == p0)
                {
                    if (!mMesh.IsInvalidateFace(afid))
                    {
                        mMesh.InvalidateFace(afid);
                        result++;
                    }
                }
            }

            // 削除側の隣接頂点リストを追加, 削除されたフェースを消す
            mVTList[keepID].TriList.AddRange(mVTList[killID].TriList);

            // リストの再構成
            var tmp = new ResizableList<int>();
            foreach (var vtxTri in mVTList[keepID].TriList)
            {
                if (!mMesh.IsInvalidateFace(vtxTri))
                {
                    tmp.Add(vtxTri);
                }
            }

            mVTList[keepID].TriList = tmp;
            mVTList[killID].TriList.Clear();

            return result;
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="keeps"></param>
        /// <param name="kills"></param>
        /// <returns></returns>
        public int ShrinkEdge(Dictionary<SourceKey, int2> keeps, Dictionary<SourceKey, int2> kills)
        {
            SourceKey key = mMesh.GetSource(SourceType.Position, null).Key;

            if (!keeps.ContainsKey(key) || !kills.ContainsKey(key))
            {
                throw ExcepHandle.CreateException("Invalid edge shrink occur");
            }

            int result = 0;
            int keepID = keeps[key][0];
            int killID = kills[key][0];
            int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
            int sourceN = mMesh.Sources.Count;

            int[] vKeepBuff0 = new int[sourceN];
            int[] vKillBuff0 = new int[sourceN];
            int[] vKeepBuff1 = new int[sourceN];
            int[] vKillBuff1 = new int[sourceN];

            for (int si = 0; si < sourceN; si++)
            {
                SourceKey skey = mMesh.Sources[si].Key;
                vKeepBuff0[si] = keeps[skey][0];
                vKillBuff0[si] = kills[skey][0];
                vKeepBuff1[si] = keeps[skey][1];
                vKillBuff1[si] = kills[skey][1];
            }
            //インデックス書き換え
            OverWritePropindex(killID, vKillBuff0, vKeepBuff0);
            OverWritePropindex(killID, vKillBuff1, vKeepBuff1);
            OverWriteVertexIndex(vKillBuff0[posOfs], vKeepBuff0[posOfs]);
            OverWriteVertexIndex(vKillBuff1[posOfs], vKeepBuff1[posOfs]);

            result = ShrinkNeighborTriangles(keepID, killID);
            return result;
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="keepVtx"></param>
        /// <param name="killVtx"></param>
        /// <param name="face0"></param>
        /// <param name="face1"></param>
        /// <param name="branched"></param>
        /// <returns></returns>
        public int ShrinkEdge(WedgeVertex keepVtx, WedgeVertex killVtx, int face0, int face1, bool branched)
        {
            try
            {
                int keepID = keepVtx.Vertex;
                int killID = killVtx.Vertex;
                int result = 0;
                int posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
                int sourceN = mMesh.Sources.Count;
                int anm = mVTList[killID].TriList.Count;

                //Int32 keepPropN = keepVtx.GetMaxPropCount();
                //Int32 killPropN = killVtx.GetMaxPropCount();
                if (!branched)
                {
                    //if( keepPropN != killPropN )
                    //    throw new Exception("ShrinkEdge:prop count is different.");

                    int[] vKeepBuff0 = new int[sourceN];
                    int[] vKillBuff0 = new int[sourceN];
                    int[] vKeepBuff1 = new int[sourceN];
                    int[] vKillBuff1 = new int[sourceN];
                    GetVBuff(ref vKeepBuff0, face0, keepID);
                    GetVBuff(ref vKillBuff0, face0, killID);

                    if (face1 != int.MaxValue)
                    {
                        GetVBuff(ref vKeepBuff1, face1, keepID);
                        GetVBuff(ref vKillBuff1, face1, killID);
                        // 重複チェッカ
                        // 重複してたら、探索しなくても良いようにする
                        for (int si = 0; si < sourceN; si++)
                        {
                            if (vKillBuff0[si] == vKillBuff1[si])
                            {
                                vKillBuff1[si] = int.MaxValue;
                            }
                        }
                    }
                    else
                    {
                        for (int si = 0; si < sourceN; si++)
                        {
                            vKeepBuff1[si] = int.MaxValue;
                            vKillBuff1[si] = int.MaxValue;
                        }
                    }

                    //インデックス書き換え
                    OverWritePropindex(killID, vKillBuff0, vKeepBuff0);
                    OverWritePropindex(killID, vKillBuff1, vKeepBuff1);
                    OverWriteVertexIndex(vKillBuff0[posOfs], vKeepBuff0[posOfs]);
                    OverWriteVertexIndex(vKillBuff1[posOfs], vKeepBuff1[posOfs]);
                }
                else
                {
                    int[] vKeepBuff = new int[sourceN];
                    int[] vKeepBuff1 = new int[sourceN];

                    GetVBuff(ref vKeepBuff, face0, keepID);
                    GetVBuff(ref vKeepBuff1, face1, keepID);

                    OverWritePropindex(killVtx, vKeepBuff);
                    OverWriteVertexIndex(killID, vKeepBuff[posOfs]);
                }

                result = ShrinkNeighborTriangles(keepID, killID);
                return result;
            }
            catch (Exception ex)
            {
                string message = string.Format("Fail to Shrink edge [{0},{1}] l{2} r{3}", keepVtx, killVtx, face0, face1);
                throw ExcepHandle.CreateException(message, ex);
            }
        }

        public List<int> GetNeighborTrigs(int tid, int vid, int nvid)
        {
            List<int> result = new List<int>();
            foreach (var fi in mVTList[vid].TriList)
            {
                if (fi == tid) { continue; }
                int sn = mMesh.SourceN;
                int po = mMesh.GetSourceOfset(SourceType.Position, null);

                int vn = mMesh.VNums[fi];
                int vo = mMesh.VOffset[fi];
                int cnt = 0;
                for (int vj = 0; vj < vn; vj++)
                {
                    int mvid = mMesh.mVbuffs[(vo + vj) * sn + po];
                    if (mvid == vid) { cnt++; }
                    if (mvid == nvid) { cnt++; }
                }

                if (cnt == 2)
                {
                    result.Add(fi);
                }
            }

            return result;
        }

        public List<int> Get2RingIncidentTrianglesForProperty(WingedEdge edge, WingedEdgeMesh wem)
        {
            var result = new List<int>();
            var vtxEdgeList = wem.GetVtxWEList(edge.SpaceKey.type, edge.SpaceKey.name);

            foreach (var uvedge in vtxEdgeList[edge.Vertex0].Edges)
            {
                var pvIndex0 = uvedge.ParentEdges[0].Vertex0;
                var pvIndex1 = uvedge.ParentEdges[0].Vertex1;
                result.AddRange(GetIncidentTrianglesForProperty(uvedge, pvIndex0));
                result.AddRange(GetIncidentTrianglesForProperty(uvedge, pvIndex1));
            }
            foreach (var uvedge in vtxEdgeList[edge.Vertex1].Edges)
            {
                var pvIndex0 = uvedge.ParentEdges[0].Vertex0;
                var pvIndex1 = uvedge.ParentEdges[0].Vertex1;
                result.AddRange(GetIncidentTrianglesForProperty(uvedge, pvIndex0));
                result.AddRange(GetIncidentTrianglesForProperty(uvedge, pvIndex1));
            }
            result.Sort();
            return result.Distinct().ToList();
        }

        /// <summary>
        /// 近傍から、同一のプロパティIDをもつ面のみをピックアップする
        /// </summary>
        /// <param name="edge"></param>
        /// <param name="posId"></param>
        /// <returns></returns>
        public List<int> GetIncidentTrianglesForProperty(WingedEdge edge, int posId)
        {
            return GetIncidentTrianglesForProperty(edge.SpaceKey.type,
                edge.SpaceKey.name,
                edge.Vertex0,
                edge.Vertex1,
                posId);
        }

        /// <summary>
        /// 指定のプロパティIDを持つ、posIdの近接面を取得します
        /// </summary>
        /// <param name="type"></param>
        /// <param name="name"></param>
        /// <param name="propId0"></param>
        /// <param name="propId1"></param>
        /// <param name="posId"></param>
        /// <returns></returns>
        public List<int> GetIncidentTrianglesForProperty(SourceType type, string name, int propId0, int propId1, int posId)
        {
            var result = new List<int>();
            var triList = VTL[posId].TriList;

            var po = mMesh.GetSourceOfset(SourceType.Position, null);
            var offset = mMesh.GetSourceOfset(type, name);
            var sourceN = mMesh.SourceN;
            foreach (var ti in triList.Distinct())
            {
                var vofs = mMesh.VOffset[ti];
                for (var vi = 0; vi < 3; vi++)
                {
                    var localVId = mMesh.mVbuffs[(vofs + vi) * sourceN + po];

                    if (localVId == posId)
                    {
                        var localPId = mMesh.mVbuffs[(vofs + vi) * sourceN + offset];
                        if ((localPId == propId0 && propId0 != int.MaxValue) || (localPId == propId1 && propId1 != int.MaxValue))
                        {
                            if (!result.Contains(ti))
                            {
                                result.Add(ti);
                            }
                            break;
                        }
                    }
                }
            }
            return result;
        }

        /// <summary>
        /// 法線が一致するものを法線近傍で検索する
        /// </summary>
        /// <param name="posId"></param>
        /// <returns></returns>
        public Dictionary<int, int> ComouteNormalCoincedentMap(int posId)
        {
            int sn = mMesh.Sources.Count;
            int po = mMesh.GetSourceOfset(SourceType.Position, null);
            int no = mMesh.GetSourceOfset(SourceType.Normal, null);
            MeshSourceBase posSource = mMesh.GetSource(SourceType.Position, null);
            MeshSourceBase nmlSource = mMesh.GetSource(SourceType.Normal, null);
            if (nmlSource == null || posSource == null) { return null; }

            var coincidentNmlIndex = new Dictionary<int, int>();

            var hashMap = new Dictionary<long, List<int>>();

            foreach (var faceId in VTL[posId].TriList)
            {
                int vn = mMesh.mVnums[faceId];
                int vo = mMesh.mVoffs[faceId];

                for (int triVi = 0; triVi < vn; triVi++)
                {
                    var localPosId = mMesh.VBuffer[(sn * (vo + triVi)) + po];
                    if (localPosId == posId)
                    {
                        var nmlId = mMesh.VBuffer[(sn * (vo + triVi)) + no];
                        var tmpNml = new Vector3();
                        MeshSourceDblCtrler.GetAsVec3(nmlSource, nmlId, ref tmpNml);
                        var hash = SpacialHash.hash_func(tmpNml);

                        if (!hashMap.ContainsKey(hash))
                        {
                            hashMap.Add(hash, new List<int>());
                        }
                        hashMap[hash].Add(nmlId);

                        if (!coincidentNmlIndex.ContainsKey(nmlId))
                        {
                            coincidentNmlIndex.Add(nmlId, nmlId);
                        }
                    }
                }
            }

            // 重複がある場合は、一つにまとめる
            foreach (var dictValue in hashMap)
            {
                var normalIndices = dictValue.Value;
                if (normalIndices.Count() > 1)
                {
                    normalIndices.Sort();
                    var replaceIndex = normalIndices.First();
                    foreach (var normalIndex in normalIndices)
                    {
                        coincidentNmlIndex[normalIndex] = replaceIndex;
                    }
                }
            }

            return coincidentNmlIndex;
        }

        /// <summary>
        /// 指定の頂点周辺の法線を、除外形状を除いて再計算する
        /// </summary>
        /// <param name="posId"></param>
        /// <param name="nmlName"></param>
        public void ComputeVtxNormals(int posId, string nmlName)
        {
            int sn = mMesh.Sources.Count;
            int po = mMesh.GetSourceOfset(SourceType.Position, null);
            int no = mMesh.GetSourceOfset(SourceType.Normal, null);
            MeshSourceBase posSource = mMesh.GetSource(SourceType.Position, null);
            MeshSourceBase nmlSource = mMesh.GetSource(SourceType.Normal, null);
            if (nmlSource == null || posSource == null) { return; }

            var nmlVtxMap = new Dictionary<int, Vector3>();
            // 頂点ごとの法線をスムーシングされてる範囲で計算する
            foreach (var faceId in VTL[posId].TriList)
            {
                int vn = mMesh.mVnums[faceId];
                int vo = mMesh.mVoffs[faceId];
                var msid = mMesh.FaceMaterialShapeId[faceId];
                var shapeInfo = mMesh.MaterialShapeInformationList.Find(o => o.Index == msid);

                // 除外対象に含まれる場合は、再計算を行わないで、元々の法線を取得する
                if (shapeInfo != null && shapeInfo.IsExcludeForSimplification)
                {
                    for (int vi = 0; vi < vn; vi++)
                    {
                        var ni = mMesh.VBuffer[(sn * (vo + vi)) + no];
                        if (!nmlVtxMap.ContainsKey(ni))
                        {
                            nmlVtxMap.Add(ni, new Vector3());
                        }
                        var tmp = new Vector3();
                        MeshSourceDblCtrler.GetAsVec3(nmlSource, ni, ref tmp);
                        nmlVtxMap[ni] = tmp;
                    }
                }
                else
                {
                    var hitIndex = -1;
                    Vector3[] pos = new Vector3[vn];
                    for (int triVi = 0; triVi < vn; triVi++)
                    {
                        var localVtxIndex = mMesh.VBuffer[(sn * (vo + triVi)) + po];
                        pos[triVi] = new Vector3();
                        MeshSourceDblCtrler.GetAsVec3(posSource, localVtxIndex, ref pos[triVi]);

                        if (localVtxIndex == posId)
                        {
                            hitIndex = triVi;
                        }
                    }

                    if (hitIndex != -1)
                    {
                        var ni = mMesh.VBuffer[(sn * (vo + hitIndex)) + no];
                        var vi0 = hitIndex;
                        var vi1 = (hitIndex + 1) % vn;
                        var vi2 = (hitIndex + 2) % vn;

                        var e1 = pos[vi1] - pos[vi0];
                        var e2 = pos[vi2] - pos[vi0];
                        var cross = e1.Cross(e2);
                        var angle = e1.Dot(e2) / (e1.Norm() * e2.Norm());

                        // 計算誤差により後段の acos で NaN になってしまう可能性があるためクランプする
                        angle = Math.Min(1.0, Math.Max(-1.0, angle));
                        angle = Math.Acos(angle);

                        if (!nmlVtxMap.ContainsKey(ni))
                        {
                            nmlVtxMap.Add(ni, new Vector3());
                        }
                        nmlVtxMap[ni] += cross * angle;
                    }
                }
            }

            for (int i = 0; i < nmlVtxMap.Count - 1; i++)
            {
                var kvpI = nmlVtxMap.ElementAt(i);
                var nmlI = kvpI.Value;
                for (int j = i + 1; j < nmlVtxMap.Count; j++)
                {
                    var kvpJ = nmlVtxMap.ElementAt(j);
                    var nmlJ = kvpJ.Value;
                    var d = nmlI.GetAngle(nmlJ, false);
                    // 1度未満は同じにする
                    if (d < 1.0)
                    {
                        nmlVtxMap[kvpI.Key] = kvpJ.Value;
                    }
                }
            }

            // 計算したものを頂点ごとに格納する
            foreach (var faceId in VTL[posId].TriList)
            {
                int vn = mMesh.mVnums[faceId];
                int vo = mMesh.mVoffs[faceId];
                for (int triVi = 0; triVi < vn; triVi++)
                {
                    var localVtxIndex = mMesh.VBuffer[(sn * (vo + triVi)) + po];
                    if (localVtxIndex == posId)
                    {
                        // ハードエッジ解除を行う為に、インデックス統合を行う
                        var ni = mMesh.VBuffer[(sn * (vo + triVi)) + no];
                        // 新規に格納する場合は、頂点ごとにバラバラにする
                        if (!nmlVtxMap.ContainsKey(ni))
                        {
                            throw new Exception();
                        }
                        var nml = nmlVtxMap[ni];
                        nml.Normalize();
                        mMesh.VBuffer[(sn * (vo + triVi)) + no] = ni;
                        MeshSourceDblCtrler.SetAsVec3(nmlSource, mMesh.VBuffer[(sn * (vo + triVi)) + no], nml);
                    }
                }
            }
        }

        /// <summary>
        /// 指定頂点に紐づけられている法線インデックスを全て取得する
        /// </summary>
        /// <param name="posId"></param>
        /// <param name="nmlName"></param>
        /// <param name="nmlIndices"></param>
        public void GetNormalIndices(int posId, string nmlName, List<int> nmlIndices)
        {
            var triList0 = VTL[posId].TriList;
            var sourceN = mMesh.SourceN;
            var posOfs = mMesh.GetSourceOfset(SourceType.Position, null);
            var nmlOfs = mMesh.GetSourceOfset(SourceType.Normal, nmlName);
            foreach (var ti in triList0)
            {
                var voffset = mMesh.mVoffs[ti];

                for (var vi = 0; vi < 3; vi++)
                {
                    var vid = mMesh.VBuffer[(voffset + vi) * sourceN + posOfs];
                    if (vid == posId)
                    {
                        var nid = mMesh.VBuffer[(voffset + vi) * sourceN + nmlOfs];
                        if (!nmlIndices.Contains(nid))
                        {
                            nmlIndices.Add(nid);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 開放
        /// </summary>
        internal void Destroy()
        {
            mVTList.Clear();
        }
#if ENABLE_ANCIENT
        /// <summary>
        /// エッジを縮退させて、フェースを削除します。
        /// </summary>
        /// <param name="keepID">保持する頂点ID</param>
        /// <param name="killID">削除する頂点ID</param>
        /// <param name="newpos">新しい位置</param>
        /// <returns></returns>
        internal int ShrinkEdgeOld(WedgeVertex keepVtx, WedgeVertex killVtx, int face0, int face1)
        {
            try
            {
                int keepID  = keepVtx.vtx;
                int killID  = killVtx.vtx;
                int result  = 0;
                int posOfs  = mMesh.GetSourceOfset(SourceType.Position, null);
                int sourceN = mMesh.Sources.Count;
                int anm = mVTList[killID].triList.Count;

                //WedgeVertex edgeVKeep = null;
                //WedgeVertex edgeVKill = null;
                //UpdateFaceVtxWedge(ref edgeVKeep, face0, keepID);
                //UpdateFaceVtxWedge(ref edgeVKill, face0, killID);
                //UpdateFaceVtxWedge(ref edgeVKeep, face1, keepID);
                //UpdateFaceVtxWedge(ref edgeVKill, face1, killID);

                for (int a = 0; a < anm; a++)
                {
                    int afid = mVTList[killID].triList[a];
                    int vnum = mMesh.mVnums[afid];
                    int vofs = mMesh.mVoffs[afid];

                    for (int v = 0; v < vnum; v++)
                    {
                        int pid = mMesh.mVbuffs[(vofs + v) * sourceN + posOfs];
                        // 削除頂点IDに一致する場合
                        if (pid == killID)
                        {
                            // プロパティのインデックスを書き換える
                            for (int s = 0; s < sourceN; s++)
                            {
                                //Int32 killProp = mMesh.mVbuffs[(vofs + v) * sourceN + s];
                                MeshSourceBase meshsrc = mMesh.Sources[s];
                                if (meshsrc != null)
                                {
                                    int keepPnum = 0;
                                    int killPnum = 0;
                                    int[] keeparray = keepVtx.GetPropIndex(out keepPnum, meshsrc.Type, meshsrc.Name);
                                    int[] killarray = killVtx.GetPropIndex(out killPnum, meshsrc.Type, meshsrc.Name);
                                    //Todo 現状は１対１の対応で
                                    if (keeparray != null && keepPnum >= killPnum)
                                    {
                                        mMesh.mVbuffs[(vofs + v) * sourceN + s] = keeparray[0];

                                    }
                                 }
                            }
                            //mMesh.mVbuffs[(vofs + v) * sourceN + posOfs] = keepID;
                        }
                    }
                }

                // 縮退するフェースを洗い出して消す
                for (int a = 0; a < anm; a++)
                {
                    int afid = mVTList[killID].triList[a];
                    int vofs = mMesh.mVoffs[afid];

                    int p0 = mMesh.mVbuffs[(vofs + 0) * sourceN + posOfs];
                    int p1 = mMesh.mVbuffs[(vofs + 1) * sourceN + posOfs];
                    int p2 = mMesh.mVbuffs[(vofs + 2) * sourceN + posOfs];

                    // 縮退エッジがあったので、消す
                    if (p0 == p1 || p1 == p2 || p2 == p0)
                    {
                        if (!mMesh.IsInvalidateFace(afid) )
                        {
                            mMesh.InvalidateFace(afid);
                            result++;
                        }
                    }
                }

                // 削除側の隣接頂点リストを追加, 削除されたフェースを消す
                var tmp = new ResizableList<int>();
                mVTList[keepID].triList.AddRange( mVTList[killID].triList );
                foreach (var vtxTri in mVTList[keepID].triList)
                {
                    if (!mMesh.IsInvalidateFace(vtxTri))
                    {
                        tmp.Add(vtxTri);
                    }
                }
                mVTList[keepID].triList = tmp;
                mVTList[killID].triList.Clear();
                return result;
            }
            catch (Exception ex)
            {
               throw;
            }
        }
#endif
    }
}
