﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace nw4f.tinymathlib
{
    public class VectorN
    {
        private uint mDim = 0;
        private double[] mData = null;

        /// <summary>
        /// 次元
        /// </summary>
        public uint Dim
        {
            get { return mDim; }
        }

        /// <summary>
        /// 数値列
        /// </summary>
        public double[] Values
        {
            get { return mData; }
        }

        public double this[int index]
        {
            get { return mData[index]; }
            set { mData[index] = value; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="d">ベクタ次元数</param>
        public VectorN(uint d)
        {
            mDim = d;
            mData = new double[d];
            SetZero();
        }

        public VectorN(VectorN v)
        {
            mDim = v.Dim;
            mData = new double[v.Dim];
            int i = 0;
            foreach (var data in v.mData)
            {
                mData[i++] = data;
            }
        }

        /// <summary>
        /// Zeroで初期化
        /// </summary>
        public void SetZero()
        {
            Array.Clear(mData, 0, (int)mDim);
        }

        private const int BlockSize = sizeof(double);
        /// <summary>
        /// 値をコピー
        /// </summary>
        public void Set(VectorN rhs)
        {
            if (rhs.Dim <= Dim)
            {
                Buffer.BlockCopy(rhs.mData, 0, mData, 0, (int)rhs.Dim * BlockSize);
            }
        }

        /// <summary>
        /// 非数が含まれるかどうか？をチェック
        /// </summary>
        /// <param name="v"></param>
        /// <returns></returns>
        public bool IsNaN()
        {
            foreach (var data in mData)
            {
                if (double.IsNaN(data)) { return true; }
            }
            return false;
        }

        /// <summary>
        /// ほぼ一緒
        /// </summary>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        /// <returns></returns>
        public static bool IsNearEqual(VectorN lhs, VectorN rhs, double threshold = 1e-5)
        {
            VectorN diff = (lhs - rhs);
            double v = diff.Norm();
            if (Math.Sqrt(v) < threshold)
            {
                return true;
            }
            return false;
        }

        /// <summary>
        /// 内積を取得 ( 異なるサイズのベクタ同士の演算は不可
        /// </summary>
        public double Dot(VectorN rhs)
        {
            if (mDim != rhs.mDim)
            {
                throw new Exception("異なるサイズのベクトルの内積を取ろうとしてます");
            }

            double result = 0;
            for (int i = 0; i < rhs.mDim; i++)
            {
                result += mData[i] * rhs.mData[i];
            }
            return result;
        }

        /// <summary>
        /// 正規化
        /// </summary>
        /// <returns></returns>
        public VectorN Normalize()
        {
            double Len = Math.Sqrt(Dot(this));
            if (Len == 0)
            {
                throw new Exception("ゼロベクトルの正規化はできません");
            }
            VectorN result = new VectorN(Dim);
            result = this * (1 / Len);
            return result;
        }

        /// <summary>
        /// ノルムを求める
        /// </summary>
        /// <returns></returns>
        public double Norm()
        {
            double Len = Math.Sqrt(Dot(this));
            return Len;
        }

        /// <summary>
        /// 加算 ( 異なるサイズのベクタ同士の演算は不可
        /// </summary>
        public static VectorN operator +(VectorN lhs, VectorN rhs)
        {
            if (lhs.Dim != rhs.Dim)
            {
                throw new Exception("サイズが異るベクトルを足しています。");
            }

            VectorN result = new VectorN(lhs.Dim);
            for (uint i = 0; i < lhs.Dim; i++)
            {
                result.mData[i] = lhs.mData[i] + rhs.mData[i];
            }
            return result;
        }

        /// <summary>
        /// 減算 ( 異なるサイズのベクタ同士の演算は不可
        /// </summary>
        public static VectorN operator -(VectorN lhs, VectorN rhs)
        {
            if (lhs.Dim != rhs.Dim)
            {
                throw new Exception("サイズが異なるベクトルを減算しています。");
            }
            VectorN result = new VectorN(lhs.Dim);
            for (int i = 0; i < lhs.mDim; i++)
            {
                result.mData[i] = lhs.mData[i] - rhs.mData[i];
            }
            return result;
        }

        /// <summary>
        /// スカラーとの乗算
        /// </summary>
        public static VectorN operator *(VectorN lhs, double rhs)
        {
            VectorN result = new VectorN(lhs.Dim);
            for (int i = 0; i < lhs.mDim; i++)
            {
                result.mData[i] = lhs.mData[i] * rhs;
            }
            return result;
        }

        /// <summary>
        /// スカラーとの除算
        /// </summary>
        public static VectorN operator /(VectorN lhs, double rhs)
        {
            if (rhs == 0.0f)
            {
                throw new Exception("Zeroで除算しようとしてます。");
            }
            VectorN result = new VectorN(lhs.Dim);
            for (int i = 0; i < lhs.mDim; i++)
            {
                result.mData[i] = lhs.mData[i] / rhs;
            }
            return result;
        }

        /// <summary>
        /// 単項マイナス演算子
        /// </summary>
        public static VectorN operator -(VectorN lhs)
        {
            VectorN result = new VectorN(lhs.mDim);
            for (int i = 0; i < lhs.mDim; i++)
            {
                result[i] = -lhs[i];
            }
            return result;
        }
    }
}
