﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

// ReSharper disable TooWideLocalVariableScope
// ReSharper disable CompareOfFloatsByEqualityOperator
// ReSharper disable SuggestUseVarKeywordEvident

namespace nw4f.tinymathlib
{
    /// <summary>
    /// フリーサイズ行列
    /// </summary>
    public class MatrixMN
    {
        private double[,] data = null;
        private int col = 0;
        private int row = 0;

        /// <summary>
        /// 行のサイズ
        /// </summary>
        public int Column
        {
            get { return col; }
        }

        /// <summary>
        /// 列のサイズ
        /// </summary>
        public int Row
        {
            get { return row; }
        }

        /// <summary>
        /// インデクサ
        /// </summary>
        public double this[int c, int r]
        {
            get { return data[c, r]; }
            set { data[c, r] = value; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="c">行</param>
        /// <param name="r">列</param>
        public MatrixMN(int c, int r)
        {
            col = c;
            row = r;
            data = new double[c, r];
            SetZero();
        }

        /// <summary>
        /// 値をコピーします
        /// </summary>
        /// <param name="rhs"></param>
        public void Set(MatrixMN rhs)
        {
            Array.Copy(rhs.data, data, (col * row));
        }

        /// <summary>
        /// Zeroで初期化
        /// </summary>
        public void SetZero()
        {
            Array.Clear(data, 0, (col * row));
        }

        /// <summary>
        /// 単位行列を取得する
        /// </summary>
        public void Identity()
        {
            SetZero();
            for (int c = 0; c < col; c++)
            {
                data[c, c] = 1.0f;
            }
        }

        /// <summary>
        /// 行列中に非数が含まれているかどうか？をチェックします
        /// </summary>
        public bool IsNaN()
        {
            for (int c = 0; c < col; c++)
            {
                for (int r = 0; r < row; r++)
                {
                    if (double.IsNaN(data[c, r]))
                    {
                        return true;
                    }
                }
            }
            return false;
        }

#if GAUSS
       /// <summary>
        /// 逆行列を求める
        /// </summary>
        /// <param name="invMat">逆行列</param>
        /// <returns>Determinant</returns>
        public double GetInverse(out MatrixMN invMat)
        {
            double        dInvPiv=0.0;
            double        dPiv=0.0;
            double        dTemp;
            double        dMax;
            double        dDet = 1.0f;

            int            nPivRow = 0;
            //逆行列を初期化
            invMat = new MatrixMN(col,row);
            invMat.Identity();

            MatrixMN tmp = new MatrixMN(col, row);
            for (int c = 0; c < col; c++)
                for (int r = 0; r < row; r++)
                        tmp.data[c, r] = data[c, r];

            if (col != row)
                return 0.0;

            int dim = col;
            // 前進消去法
            for( int iPiv = 0; iPiv < dim; iPiv++ )
            {
                // 列での最大値を検索する
                dMax = 0;
                for( int iRow = iPiv; iRow < dim; iRow++ )
                {
                    if (Math.Abs(tmp.data[iRow, iPiv]) > dMax)
                    {
                        dMax = Math.Abs(tmp.data[iRow, iPiv]);
                        nPivRow = iRow;
                    }
                }
                // 失敗するので終了
                if( dMax == 0.0 )
                {
                    return 0.0f;
                }

                // ピボット選択
                if( iPiv != (int)nPivRow )
                {
                    for(int iCol = iPiv; iCol < dim; iCol++ )
                    {
                        dTemp = tmp.data[iPiv, iCol];
                        tmp.data[iPiv, iCol] = tmp.data[nPivRow, iCol];
                        tmp.data[nPivRow, iCol] = dTemp;
                    }

                    for( int iCol = 0; iCol < dim; iCol++ )
                    {
                        dTemp = invMat.data[iPiv,iCol];
                        invMat.data[iPiv,iCol]     = invMat.data[nPivRow,iCol];
                        invMat.data[nPivRow,iCol] = dTemp;
                    }
                    dDet = -dDet;
                }
                dPiv = tmp.data[iPiv, iPiv];
                dInvPiv = 1.0f/dPiv;
                dDet    *= dPiv;

                for( int iCol = iPiv+1; iCol < dim; iCol++ )
                    tmp.data[iPiv, iCol] *= dInvPiv;

                for(int iCol = 0; iCol < dim; iCol++ )
                    invMat.data[iPiv,iCol] *= dInvPiv;

                for(int iRow = iPiv+1; iRow < dim; iRow++ )
                {
                    dTemp = tmp.data[iRow, iPiv];
                    for(int iCol = iPiv+1; iCol < dim; iCol++ )
                        tmp.data[iRow, iCol] -= dTemp * tmp.data[iPiv, iCol];
                    for(int iCol = 0; iCol < dim; iCol++ )
                        invMat.data[iRow,iCol] -= dTemp*(invMat.data[iPiv,iCol]);
                }
            }

            // 後退代入法
            for( int iRow = dim-1; iRow>0; iRow--)
            {
                for(int iCol=0; iCol<iRow; iCol++)
                {
                    dTemp = tmp.data[iCol, iRow];
                    for (int k=0; k<dim; k++)
                    {
                        invMat.data[iCol, k] -= dTemp * (invMat.data[iRow,k]);
                    }
                }
            }
            return dDet;
        }
#else
        /// <summary>
        /// solve luMatrix * x = b
        /// </summary>
        /// <param name="luMatrix"></param>
        /// <param name="b"></param>
        /// <returns></returns>
        private double[] HelperSolve(MatrixMN luMatrix, double[] b)
        {
            int n = luMatrix.col;
            double[] x = new double[n];
            b.CopyTo(x, 0);

            for (int i = 1; i < n; ++i)
            {
                double sum = x[i];
                for (int j = 0; j < i; ++j)
                {
                    sum -= luMatrix.data[i, j] * x[j];
                }
                x[i] = sum;
            }

            x[n - 1] /= luMatrix.data[n - 1, n - 1];
            for (int i = n - 2; i >= 0; --i)
            {
                double sum = x[i];
                for (int j = i + 1; j < n; ++j)
                {
                    sum -= luMatrix.data[i, j] * x[j];
                }
                x[i] = sum / luMatrix.data[i, i];
            }
            return x;
        }

        /// <summary>
        /// LU分解を行う
        /// </summary>
        public void LUDecompose(out MatrixMN lu, out int[] order, out int toggle)
        {
            int i = 0, j = 0, k = 0;
            int width = 0;
            int height = 0;
            int d = 0;
            double max = 0.0;
            int temp = 0;

            order = new int[col];
            width = col;
            height = row;
            //逆行列を初期化
            lu = new MatrixMN(col, row);
            lu.Set(this);
            toggle = 1;

            for (i = 0; i < height; i++)
            {
                order[i] = i;
            }

            for (i = 0; i < height - 1; i++)
            {
                d = i;
                max = Math.Abs(lu.data[i, i]);
                for (j = i + 1; j < col; j++)
                {
                    if (max < Math.Abs(lu.data[j, i]))
                    {
                        d = j;
                        max = Math.Abs(lu.data[j, i]);
                    }
                }

                if (d != i)
                {
                    temp = order[i];
                    order[i] = order[d];
                    order[d] = temp;

                    ExchangeRowvector(ref lu, i, d);
                    toggle = -toggle;
                }

                if (Math.Abs(lu.data[i, i]) < double.Epsilon)
                {
                    lu = null;
                    return;
                }

                for (j = i + 1; j < height; j++)
                {
                    lu.data[j, i] /= lu.data[i, i];
                    for (k = i + 1; k < width; k++)
                    {
                        lu.data[j, k] -= (lu.data[j, i] * lu.data[i, k]);
                    }
                }
            }
        }

        /// <summary>
        /// 行列の指定列ベクトルを入れ替える
        /// </summary>
        /// <param name="m"></param>
        /// <param name="a"></param>
        /// <param name="b"></param>
        private static void ExchangeRowvector(ref MatrixMN m, int a, int b)
        {
            int i = 0;
            double temp = 0.0;
            for (i = 0; i < m.col; i++)
            {
                temp = m.data[a, i];
                m.data[a, i] = m.data[b, i];
                m.data[b, i] = temp;
            }
        }

        /// <summary>
        /// 逆行列を求める
        /// </summary>
        /// <param name="result">行列式</param>
        public double GetInverse(out MatrixMN result)
        {
            int n = col;
            result = new MatrixMN(col, row);
            MatrixMN lum;
            int[] perm;
            int toggle;

            LUDecompose(out lum, out perm, out toggle);
            if (lum == null)
            {
                return 0.0;
            }

            double det = toggle;
            double[] b = new double[n];
            for (int i = 0; i < n; ++i)
            {
                for (int j = 0; j < n; ++j)
                {
                    if (i == perm[j])
                    {
                        b[j] = 1.0;
                    }
                    else
                    {
                        b[j] = 0.0;
                    }
                }
                double[] x = HelperSolve(lum, b);
                for (int j = 0; j < n; ++j)
                {
                    result.data[j, i] = x[j];
                }
            }

            for (int i = 0; i < col; ++i)
            {
                det *= lum.data[i, i];
            }
            return det;
        }
#endif

        /// <summary>
        ///
        /// </summary>
        /// <returns></returns>
        public MatrixMN GetTranspose()
        {
            MatrixMN result = new MatrixMN(row, col);
            for (int i = 0; i < col; i++)
            {
                for (int j = 0; j < row; j++)
                {
                    result.data[i, j] = data[j, i];
                }
            }
            return result;
        }

        /// <summary>
        /// 行列の加算
        /// </summary>
        public static MatrixMN operator +(MatrixMN lhs, MatrixMN rhs)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(lhs.col == rhs.col, "サイズが異る行列を足しています。");
            Nintendo.Foundation.Contracts.Assertion.Operation.True(lhs.row == rhs.row, "サイズが異る行列を足しています。");

            MatrixMN result = new MatrixMN(lhs.col, lhs.row);
            for (int c = 0; c < lhs.col; c++)
            {
                for (int r = 0; r < lhs.row; r++)
                {
                    result.data[r, c] = lhs.data[r, c] + rhs.data[r, c];
                }
            }

            return result;
        }

        /// <summary>
        /// 行列の加算
        /// </summary>
        public static MatrixMN operator *(MatrixMN lhs, MatrixMN rhs)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(lhs.col == rhs.col, "サイズが異る行列を足しています。");
            Nintendo.Foundation.Contracts.Assertion.Operation.True(lhs.row == rhs.row, "サイズが異る行列を足しています。");

            MatrixMN result = new MatrixMN(lhs.col, lhs.row);
            result.SetZero();
            for (int i = 0; i < lhs.row; i++)
            {
                for (int j = 0; j < lhs.col; j++)
                {
                    for (int k = 0; k < lhs.col; k++)
                    {
                        result.data[i, j] += lhs.data[i, k] * rhs.data[k, j];
                    }
                }
            }
            return result;
        }

        /// <summary>
        /// 行列とベクタの乗算（左から掛ける）
        /// </summary>
        public static VectorN operator *(MatrixMN lhs, VectorN rhs)
        {
            if (lhs.col != rhs.Dim)
            {
                throw new Exception("サイズが異なる行列とベクトルを掛けています。");
            }
            if (lhs.row != rhs.Dim)
            {
                throw new Exception("サイズが異なる行列とベクトルを掛けています。");
            }

            VectorN result = new VectorN(rhs.Dim);
            result.SetZero();
            for (int r = 0; r < rhs.Dim; r++)
            {
                for (int c = 0; c < rhs.Dim; c++)
                {
                    result[r] += lhs.data[r, c] * rhs[c];
                }
            }

            return result;
        }

        /// <summary>
        /// 行列のスカラ倍
        /// </summary>
        public static MatrixMN operator *(MatrixMN lhs, double rhs)
        {
            MatrixMN result = new MatrixMN(lhs.col, lhs.row);
            for (int i = 0; i < lhs.col; i++)
            {
                for (int j = 0; j < lhs.row; j++)
                {
                    result.data[i, j] += lhs.data[i, j] * rhs;
                }
            }

            return result;
        }
    }
}
// ReSharper restore SuggestUseVarKeywordEvident
// ReSharper restore CompareOfFloatsByEqualityOperator
// ReSharper restore TooWideLocalVariableScope
