﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.meshlib;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
#if OLDQUAD
   public class Quadric
    {
        int munDim = 0;    // 次元

        MatrixMN mValueA;    // テンソル　：　２次係数候
        VectorN mValueB;    // ベクトル     :   １次係数
        double mValueC;    // 定数
        double mdArea;
        double mDistOfs = 0.0;
        private double[] mRamda = null;

        private void symmetric_subfrom(MatrixMN A, VectorN a, VectorN b)
        {
            for (int i = 0; i < A.Row; i++)
            {
                for (int j = 0; j < A.Column; j++)
                {
                    double av = a[i];
                    A[i, j] -= (av * b[j]);
                }
            }
        }

        public Quadric(int aunDim, double ofs)
        {
            munDim = aunDim;
            mValueA = new MatrixMN(aunDim, aunDim); ;
            mValueB = new VectorN((uint)aunDim);
            mValueC = 0;
            mdArea = 0;
            mDistOfs = ofs;
            mRamda = new double[aunDim];
            for (int i = 0; i < aunDim; i++)
                mRamda[i] = 1.0;
        }

        public double DistOffset
        {
            get { return mDistOfs; }
            set { mDistOfs = value; }
        }

        int Dim
        {
            get { return munDim; }
            set { munDim = value; }
        }

        public bool IsNaN()
        {
            return mValueA.IsNaN() || mValueB.IsNaN() || double.IsNaN(mValueC);
        }

        public void Compute(VectorN p1, VectorN p2, VectorN p3, QemPolicy policy)
        {
            //Clear();
            // スケール
            for (int i = 0; i < Dim; i++)
            {
                p1[i] = p1[i] * mRamda[i];
                p2[i] = p2[i] * mRamda[i];
                p3[i] = p3[i] * mRamda[i];
            }

            // 次元を保持
            munDim = (int)p1.Dim;

            var e1 = new VectorN(p2);
            e1 -= p1;
            if (e1.Dot(e1) > 0.0f)
                e1 = e1.Normalize();    // e1 = p2-p1; unitize

            var e2 = new VectorN(p3);
            e2 -= p1;        // e2 = p3-p1
            var t = new VectorN(e1);
            t *= e1.Dot(e2);    // e1*(e1*(p3-p1) )
            e2 -= t;             // e2 = p3-p1-e1*(e1*(p3-p1));
            if (e2.Dot(e2) > 0.0f)
                e2 = e2.Normalize();

            double p1e1 = p1.Dot(e1);
            double p1e2 = p1.Dot(e2);

            mValueA.Identity();

            // a = 1- e1e1^t - e2e2^t
            symmetric_subfrom(mValueA, e1, e1);
            symmetric_subfrom(mValueA, e2, e2);
            // b = e1*p1e1 + e2*p1e2 - p1
            mValueB.Set(e1);
            mValueB *= p1e1;
            t = e2; t *= p1e2;
            mValueB += t;
            mValueB -= p1;
            // c = p1*p1 - (p1*e1)^2 - (p1*e2)^2
            mValueC = p1.Dot(p1) - p1e1 * p1e1 - p1e2 * p1e2;
        }

        public double Evaluate(VectorN v, double w)
        {
            return (double )(v.Dot((mValueA * w) * v) + 2 * ((mValueB * w).Dot(v)) + mValueC * w);
        }

        public void Clear()
        {
            mValueA.SetZero();
            mValueB.SetZero();
            mValueC = 0;
            mdArea = 0;
        }

        /// <summary>
        /// 最適化位置を求める
        /// </summary>
        /// <param name="vOpt"></param>
        /// <returns></returns>
        public bool Optimize(ref VectorN vOpt)
        {
            MatrixMN invA;
            double det = mValueA.GetInverse(out invA);
            // 行列式がゼロ同然
            if (Math.Abs(det) <= 0.000001f || double.IsNaN(det)) { return false; }
            vOpt = invA * (-mValueB);
            return true;
        }

        public static Quadric operator +(Quadric lhs, Quadric rhs)
        {
            Quadric result = new Quadric(lhs.munDim, lhs.mDistOfs);

            result.mValueA = lhs.mValueA + rhs.mValueA;
            result.mValueB = lhs.mValueB + rhs.mValueB;
            result.mValueC = lhs.mValueC + rhs.mValueC;

            result.mdArea = lhs.mdArea + rhs.mdArea;

            return result;
        }

        public static Quadric operator *(Quadric lhs, double arRight)
        {
            Quadric result = new Quadric(lhs.Dim, lhs.mDistOfs);

            result.mValueA = lhs.mValueA * arRight;
            result.mValueB = lhs.mValueB * arRight;
            result.mValueC = lhs.mValueC * arRight;

            return result;
        }
    }
#else
    /// <summary>
    /// QuadricError演算行列(Hoppe版
    /// </summary>
    public class Quadric
    {
        private int mDim = 0;
        private MatrixMN mA;
        private VectorN mB;
        private double mC = 0.0;

        //private double mArea = 0.0;
        private double mDistOfs = 0.0;
        private double[] mRamda = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="d">次元数</param>
        public Quadric(int d, double ofs)
        {
            mDim = d;
            mDistOfs = ofs;
            mA = new MatrixMN(mDim, mDim);
            mB = new VectorN((uint)mDim);
            mC = 0.0f;
            mRamda = new double[mDim];
            for (int i = 0; i < mDim; i++)
            {
                mRamda[i] = 1.0;
            }
        }

        /// <summary>
        /// 値をクリアする
        /// </summary>
        public void Clear()
        {
            mA.SetZero();
            mB.SetZero();
            mC = 0;
            mDistOfs = 0;
            for (int i = 0; i < mDim; i++)
            {
                mRamda[i] = 1.0;
            }
        }
        /// <summary>
        /// 面積で重みを付けるか？
        /// </summary>
        public static bool AreaWeighted { get; set; }

        /// <summary>
        /// 距離オフセット（平面の多いオブジェクトに対応）
        /// </summary>
        public double DistOffset
        {
            get { return mDistOfs; }
            set { mDistOfs = value; }
        }

        /// <summary>
        /// 値をセットする
        /// </summary>
        /// <param name="rhs"></param>
        public void Set(Quadric rhs)
        {
            if (rhs.mDim == mDim)
            {
                mA.Set(rhs.mA);
                mB.Set(rhs.mB);
                mC = rhs.mC;

                mDim = rhs.mDim;
                mDistOfs = rhs.mDistOfs;
                Array.Copy(rhs.mRamda, mRamda, mRamda.Length);
            }
            else
            {
                int dim = Math.Min(mDim, rhs.mDim);
                for (int di = 0; di < dim; di++)
                {
                    for (int dj = 0; dj < dim; dj++)
                    {
                        mA[di, dj] = rhs.mA[di, dj];
                    }
                    mB[di] = rhs.mB[di];
                }
                mC = rhs.mC;
                mDistOfs = rhs.mDistOfs;
                Array.Copy(rhs.mRamda, mRamda, dim);
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <returns></returns>
        public bool IsNaN()
        {
            return mA.IsNaN() || mB.IsNaN() || double.IsNaN(mC);
        }

        /// <summary>
        /// A,B,Cの３つと、面積のみがたされる
        /// </summary>
        public static Quadric operator +(Quadric lhs, Quadric rhs)
        {
            if (lhs.mDim != rhs.mDim)
            {
                throw ExcepHandle.CreateException("The rank of quadrics is different.");
            }

            Quadric result = new Quadric(lhs.mDim, (lhs.mDistOfs + lhs.mDistOfs) * 0.5);
            result.mA = lhs.mA + rhs.mA;
            result.mB = lhs.mB + rhs.mB;
            result.mC = lhs.mC + rhs.mC;
            return result;
        }

        /// <summary>
        /// A,B,Cの３つと、面積のみがたされる
        /// </summary>
        public static Quadric operator *(Quadric lhs, double rhs)
        {
            Quadric result = new Quadric(lhs.mDim, (lhs.mDistOfs + lhs.mDistOfs) * 0.5);
            result.mA = lhs.mA * rhs;
            result.mB = lhs.mB * rhs;
            result.mC = lhs.mC * rhs;
            return result;
        }

        /// <summary>
        /// 重みを設定する。
        /// </summary>
        /// <param name="i"></param>
        /// <param name="val"></param>
        public void ExternalWeight(int i, double val)
        {
            mRamda[i] = val;
        }

        /// <summary>
        /// 与えられたベクトルに対して、QuadricError値を計算します
        /// </summary>
        public double Evaluate(VectorN v, double w)
        {
            return (v.Dot(mA * v) + 2 * mB.Dot(v) + mC) * w;
        }

        [Conditional("_DEBUG")]
        private void test_at_compute(Vector3 e1, Vector3 e2, Vector3 n)
        {
            double a1 = n.Dot(e1);
            double a2 = n.Dot(e2);
            if (Math.Abs(a1) > 1e-8 || Math.Abs(a2) > 1e-8)
            {
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(@"Computational error of cross product. ");
            }
        }

        private bool GaussJ(double[] x)
        {
            const double Epsilon = (double)1e-3;
            int i, j, k;

            double[][] C = new double[mDim][];

            for (int r = 0; r < mDim; r++)
            {
                C[r] = new double[mDim + 1];
            }

            for (int r = 0; r < mDim; r++)
            {
                for (int c = 0; c < mDim; c++)
                {
                    C[r][c] = mA[r, c];
                }
            }

            for (int r = 0; r < mDim; r++)
            {
                C[r][mDim] = -mB[r];
            }

            double eps;
            eps = Math.Abs(C[0][0]);
            for (i = 1; i < mDim; ++i)
            {
                double t = Math.Abs(C[i][i]);
                if (eps < t) { eps = t; }
            }
            eps *= Epsilon;

            for (i = 0; i < mDim - 1; ++i)
            {
                int ma = i;
                double vma = Math.Abs(C[i][i]);
                for (k = i + 1; k < mDim; k++)
                {
                    double t = Math.Abs(C[k][i]);
                    if (t > vma)
                    {
                        vma = t;
                        ma = k;
                    }
                }
                if (vma < eps) { return false; }

                if (i != ma)
                {
                    for (k = 0; k <= mDim; k++)
                    {
                        double t = C[i][k];
                        C[i][k] = C[ma][k];
                        C[ma][k] = t;
                    }
                }

                for (k = i + 1; k < mDim; k++)
                {
                    double s;
                    s = C[k][i] / C[i][i];
                    for (j = i + 1; j <= mDim; j++)
                    {
                        C[k][j] -= C[i][j] * s;
                    }

                    C[k][i] = 0.0;
                }
            }

            if (Math.Abs(C[mDim - 1][mDim - 1]) < eps) { return false; }

            for (i = mDim - 1; i >= 0; i--)
            {
                double t;
                for (t = 0.0, j = i + 1; j < mDim; j++)
                {
                    t += C[i][j] * x[j];
                }
                x[i] = (C[i][mDim] - t) / C[i][i];
            }
            return true;
        }

        /// <summary>
        /// 最適化位置を求める
        /// </summary>
        /// <param name="vOpt"></param>
        /// <returns></returns>
        public bool Optimize(ref VectorN vOpt)
        {
            MatrixMN invA;
            var det = mA.GetInverse(out invA);
            // 行列式がゼロ同然
            if (det <= 1e-10 || double.IsNaN(det)) { return false; }

            vOpt = new VectorN((uint)mDim);
            vOpt.Set(invA * (-mB));
            return true;
        }

        /// <summary>
        /// Quadric 行列を計算する
        /// </summary>
        /// <param name="p1">フェースの位置座標</param>
        /// <param name="p2">フェースの位置座標</param>
        /// <param name="p3">フェースの位置座標</param>
        public void Compute(VectorN p1, VectorN p2, VectorN p3)
        {
            Vector3 pos1 = new Vector3(), pos2 = new Vector3(), pos3 = new Vector3();
            Vector3 e1 = new Vector3(), e2 = new Vector3();

            // 法線を使う場合は、最後の一行は使わない
            int dim = mDim;
            //位置座標のみを切り出す
            for (int i = 0; i < 3; i++)
            {
                pos1[i] = p1[i];
                pos2[i] = p2[i];
                pos3[i] = p3[i];
            }

            // 法線を計算する
            e1 = pos2 - pos1;
            e2 = pos3 - pos1;
            Vector3 n = e1.Cross(e2);
            double area = n.Norm() * 0.5f;
            //n.Normalize();
            n = Mesh.CalculateTriangleNormal(pos1, pos2, pos3);
            var bc = (pos1 + pos2 + pos3) / 3;

            double d = -n.Dot(bc) + mDistOfs;
            //位置のみのQuadricを計算する
            //A
            mA[0, 0] = n[0] * n[0];
            mA[0, 1] = n[0] * n[1];
            mA[0, 2] = n[0] * n[2];
            mA[1, 0] = n[1] * n[0];
            mA[1, 1] = n[1] * n[1];
            mA[1, 2] = n[1] * n[2];
            mA[2, 0] = n[2] * n[0];
            mA[2, 1] = n[2] * n[1];
            mA[2, 2] = n[2] * n[2];

            //B
            mB[0] = n[0] * d;
            mB[1] = n[1] * d;
            mB[2] = n[2] * d;
            //C
            mC = d * d;

            // プロパティがある場合の計算
            if (dim > 3)
            {
                for (int j = 3; j < dim; j++)
                {
                    for (int i = 3; i < dim; i++)
                    {
                        mA[j, i] = 0.0f;
                    }
                }
                n.Normalize();

                Matrix44 mat = new Matrix44();
                Vector4 v0 = new Vector4();
                bool success;

                mat[0, 0] = p1[0];
                mat[0, 1] = p1[1];
                mat[0, 2] = p1[2];
                mat[0, 3] = 1.0;
                mat[1, 0] = p2[0];
                mat[1, 1] = p2[1];
                mat[1, 2] = p2[2];
                mat[1, 3] = 1.0;
                mat[2, 0] = p3[0];
                mat[2, 1] = p3[1];
                mat[2, 2] = p3[2];
                mat[2, 3] = 1.0;
                mat[3, 0] = n[0];
                mat[3, 1] = n[1];
                mat[3, 2] = n[2];
                mat[3, 3] = 0.0;

                Matrix44 inv = mat.GetInverse(out success);
                if (!success) { return; }

                Vector4 gj = new Vector4();
                for (int i = 3; i < dim; i++)
                {
                    double rmd = Math.Abs(mRamda[i]);
                    //Double rmd = 1.0;
                    double rmd2 = rmd * rmd;
                    gj.SetZero();

                    v0[0] = p1[i];
                    v0[1] = p2[i];
                    v0[2] = p3[i];
                    v0[3] = 0;

                    gj = inv * v0;

                    // 3x3 部分に追加
                    mA[0, 0] += rmd2 * gj[0] * gj[0];
                    mA[0, 1] += rmd2 * gj[0] * gj[1];
                    mA[0, 2] += rmd2 * gj[0] * gj[2];

                    mA[1, 0] += rmd2 * gj[1] * gj[0];
                    mA[1, 1] += rmd2 * gj[1] * gj[1];
                    mA[1, 2] += rmd2 * gj[1] * gj[2];

                    mA[2, 0] += rmd2 * gj[2] * gj[0];
                    mA[2, 1] += rmd2 * gj[2] * gj[1];
                    mA[2, 2] += rmd2 * gj[2] * gj[2];

                    mB[0] += rmd2 * gj[0] * gj[3];
                    mB[1] += rmd2 * gj[1] * gj[3];
                    mB[2] += rmd2 * gj[2] * gj[3];

                    //代入
                    mA[0, i] = -rmd2 * gj[0];
                    mA[1, i] = -rmd2 * gj[1];
                    mA[2, i] = -rmd2 * gj[2];

                    mA[i, 0] = -rmd2 * gj[0];
                    mA[i, 1] = -rmd2 * gj[1];
                    mA[i, 2] = -rmd2 * gj[2];

                    mB[i] = -rmd2 * gj[3];

                    mC += (rmd2 * gj[3] * gj[3]);

                    mA[i, i] = rmd2;
                    //                    mA[i, i] = rmd2 < double.Epsilon ? double.Epsilon : rmd2;
                }
            }
            // 最後に重みを乗せる
            if (AreaWeighted)
            {
                mA = mA * area * area;
                mB = mB * area * area;
                mC = mC * area * area;
            }
        }

        /// <summary>
        /// 特徴的な稜線を考慮する項目を追加する
        /// </summary>
        /// <param name="g">gradient</param>
        /// <param name="p"></param>
        public void AddFeature(Vector3 g, double d)
        {
            int dim = mDim - 1;
            for (int i = 0; i < 3; i++)
            {
                mA[dim, i] = g[i];
                mA[i, dim] = g[i];
            }
            mA[dim, dim] = 0;
            mB[dim] = d;
        }
    }
#endif
}
