﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Dynamic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;
using nw.g3d.iflib;
using nw4f.tinymathlib;
using Vector2 = nw4f.tinymathlib.Vector2;
using Vector3 = nw4f.tinymathlib.Vector3;

namespace nw4f.meshlib
{
    public class PolygonReductionProfiler
    {
        public class ProfileResult
        {
            /// <summary>
            /// 削除可能な最大面数、プロファイルパラメータによって結果は変わる
            ///
            /// ただし、簡略化しながら状況が変化するので、削除可能面数は減少してしまう可能性が高い
            /// その為、あくまで参考値
            /// </summary>
            public int AvailableFaceCount { get; internal set; }

            /// <summary>
            /// qaulityコスト以内に存在するエッジ数
            /// ただし、簡略化しながら状況が変化するので、削除可能面数は減少してしまう可能性が高い
            /// その為、あくまで参考値
            /// </summary>
            public int AvailableEdgeCountWithinQuality { get; internal set; }

            /// <summary>
            /// qaulityコスト以内に存在するポリゴン数
            /// ただし、簡略化しながら状況が変化するので、削除可能面数は減少してしまう可能性が高い
            /// その為、あくまで参考値
            /// </summary>
            public int AvailableFaceCountWithinQuality { get; internal set; }

            /// <summary>
            /// 設定可能な最小簡略化率
            /// ただし、簡略化しながら状況は変化するので、最小は変化して上がる可能性が高い
            /// その為、あくまで参考値で、設定値としての最小値と思ってください。
            /// </summary>
            public double AvailableSimplifyRate { get; internal set; }

            /// <summary>
            /// ラミナを含む非多様体構造を構成する頂点のリスト
            /// </summary>
            public List<int> NonManifoldVerteces { get; internal set; } = new ResizableList<int>();

            /// <summary>
            /// ラミナを含む非多様体構造を構成する面のリスト
            /// </summary>
            public List<int> NonManifoldFaces { get; internal set; } = new ResizableList<int>();

            /// <summary>
            /// サブメッシュ境界上の頂点のリスト
            /// </summary>
            public List<int> SubMeshBorderVerteces { get; internal set; } = new ResizableList<int>();
            /// <summary>
            /// サブメッシュ境界に隣接する面のリスト
            /// </summary>
            public List<int> SubMeshFaces { get; internal set; } = new ResizableList<int>();

            /// <summary>
            /// 開放端に繋がる頂点のリスト
            /// </summary>
            public List<int> OpenEdgeVerteces { get; internal set; } = new ResizableList<int>();
            /// <summary>
            /// 開放端をエッジに持つ様な面のリスト
            /// </summary>
            public List<int> OpenEdgeFaces { get; internal set; } = new ResizableList<int>();

            /// <summary>
            /// 現在の設定で、特徴的なエッジとみなされるエッジを構成している頂点
            /// </summary>
            public List<int> FeatEdgeVerteces { get; internal set; } = new ResizableList<int>();

            /// <summary>
            /// 現在の設定で、特徴的なエッジとみなされるエッジに接する面
            /// </summary>
            public List<int> FeatEdgeFaces { get; internal set; } = new ResizableList<int>();

            /// <summary>
            /// 現在の簡略化設定で保護される開放端の頂点
            /// </summary>
            public List<int> LockedOpenEdgeVerteces { get; internal set; } = new ResizableList<int>();
            /// <summary>
            /// 現在の簡略化設定で保護される開放端の面
            /// </summary>
            public List<int> LockedOpenEdgeFaces { get; internal set; } = new ResizableList<int>();
            /// <summary>
            /// 現在の簡略化設定で保護されるUVハードエッジ上の頂点
            /// </summary>
            public List<int> LockedUvEdgeVerteces { get; internal set; } = new ResizableList<int>();
            /// <summary>
            /// 現在の簡略化設定で保護されるUVハードエッジ上の面
            /// </summary>
            public List<int> LockedUvEdgeFaces { get; internal set; } = new ResizableList<int>();

            /// <summary>
            /// 開放端の劣角によるヒストグラム
            /// </summary>
            public Histgram OpenEdgeHistgram { get; internal set; } = new Histgram();

            /// <summary>
            /// UVハードエッジの劣角によるヒストグラム
            /// </summary>
            public Histgram UvHardEdgeHistgram { get; internal set; } = new Histgram();

            /// <summary>
            /// UVハードエッジの劣角によるヒストグラム
            /// </summary>
            public Histgram FeatureEdgeHistgram { get; internal set; } = new Histgram();

            /// <summary>
            /// 全てのエッジの key ％を含むことが可能な、Quality値を　value に保存します。
            /// 最小コスト、最大コストは簡略化しながら変化しますので、初期値で全ては決まりません。
            /// ですが、quality 値を決定する際には、おおよその参考になります。
            /// key % の簡略化を行う場合には、 value を設定して下さい。
            /// ただし、削除不可能な頂点が多すぎる場合には、必ずしも合致しません。
            /// key ; パーセンテージ
            /// value : エッジ数
            /// </summary>
            public Dictionary<double, double> EdgeCountForCostRate { get; internal set; } = new Dictionary<double, double>();

            /// <summary>
            /// 現在の設定値で、指定されたポリゴン数に達する最大のquality値
            ///
            /// 初期値での評価なのであくまで参考値。簡略化しつつ周辺状況が変わるために正確には特定できない。
            /// </summary>
            public double RecomendedQualityValue { get; internal set; }
        }

        /// <summary>
        /// ヒストグラム
        /// </summary>
        public class Histgram
        {
            /// <summary>
            /// ヒストグラムのビンの最大値
            /// </summary>
            public double BinMax { get; private set; }
            /// <summary>
            /// ヒストグラムのビンの最小値
            /// </summary>
            public double BinMin { get; private set; }
            /// <summary>
            /// ヒストグラムのビンの刻み幅
            /// </summary>
            public double BinSize { get; private set; }
            /// <summary>
            /// ヒストグラムのビンの数
            /// </summary>
            public int BinCount { get; private set; }

            /// <summary>
            /// ヒストグラムのデータ
            /// </summary>
            public List<List<int>> Bins { get; private set; }
            /// <summary>
            /// ヒストグラムを作成する
            /// </summary>
            /// <param name="size"></param>
            public void Create(double maxSize, double minSize, double size)
            {
                BinSize = size;
                BinMax = maxSize;
                BinMin = minSize;

                BinCount = (int)((maxSize - minSize) / size);
                Bins = new List<List<int>>();
                for (var i = 0; i < BinCount; i++)
                {
                    Bins.Add(new List<int>());
                }
            }

            /// <summary>
            /// エッジIDを角度共に与えて、ビンに登録する
            /// </summary>
            /// <param name="angle"></param>
            /// <param name="edgeId"></param>
            public void Add(double val, int edgeId)
            {
                if (double.IsNaN(val))
                {
                    return;
                }

                val = Math.Min(BinMax - 1e-5, val);
                val = Math.Max(BinMin, val);
                var address = (int)(val / BinSize);
                if (!Bins[address].Contains(edgeId))
                {
                    Bins[address].Add(edgeId);
                }
            }

            /// <summary>
            /// 指定のパーセンテージに達するヒストグラムのビンを取得
            /// </summary>
            /// <param name="rate">0.0 - 1.0 の範囲で指定</param>
            /// <returns>ビンID</returns>
            public int GetBinForRate(double rate)
            {
                var totalCount = Bins.Sum(o => o.Count());
                var sum     = 0;
                var result  = -1;
                for (int index = 0; index < Bins.Count; index++)
                {
                    var bin = Bins[index];

                    if (sum >= totalCount * rate && result == -1)
                    {
                        result = index;
                    }
                    sum += bin.Count();
                }
                return result;
            }

            /// <summary>
            /// 指定のパーセンテージに達する値の範囲を取得
            /// </summary>
            /// <param name="rate"></param>
            /// <returns></returns>
            public Tuple<double, double> GetRangeForRate(double rate)
            {
                var binId = GetBinForRate(rate);
                return Tuple.Create(binId * BinSize, (binId + 1) * BinSize);
            }

            /// <summary>
            /// 出力するための文字列の生成
            /// </summary>
            /// <returns></returns>
            public string GeneratePrintMessage()
            {
                string result = string.Empty;

                for (int index = 0; index < Bins.Count; index++)
                {
                    var bin = Bins[index];
                    result += $@"Range[{BinSize*index}:{BinSize*(index + 1)}] EdgeCount:{bin.Count()}" + Environment.NewLine;
                }
                return result;
            }
        }

        /// <summary>
        /// プロファイラを実行する場合のオプション
        /// </summary>
        public class ProfileOptions
        {
            /// <summary>
            /// ヒストグラムを作成する必要があるか？ trueの場合はヒストグラムを作成する
            /// </summary>
            public bool ComputeHistgrams { get; set; } = false;
            /// <summary>
            /// 簡略化コストのマップを作成する必要があるか？trueの場合は作成する
            /// </summary>
            public bool ComputeCostMap { get; set; } = false;
        }

        /// <summary>
        /// 入力メッシュ
        /// </summary>
        private Mesh SourceMesh { get; set; }
        /// <summary>
        /// プロファイル結果
        /// </summary>
        private ProfileResult Result { get; set; } = new ProfileResult();

        /// <summary>
        /// プロファイラを実行します。
        /// </summary>
        /// <param name="mesh">入力メッシュ</param>
        /// <param name="simpOptions">簡略化オプション</param>
        /// <param name="profileOptions">プロファイラオプション</param>
        /// <returns></returns>
        public ProfileResult Profile(Mesh mesh, QemSimp.QemSimpParam simpOptions, ProfileOptions profileOptions)
        {
            SourceMesh = mesh;
            QemSimp qemSimp = new QemSimp();
            qemSimp.Params = simpOptions;

            qemSimp.Initialize(mesh);

            var vtxProfile = qemSimp.GetVertexProfile();

            // 非多様体に接する面と頂点を列挙する
            Result.NonManifoldVerteces = PickVertices(vtxProfile[0], (int)QemVertexType.Nonmanifold);
            Result.NonManifoldFaces = PickFaces(Result.NonManifoldVerteces);

            // サブメッシュ境界に接する頂点と面を列挙
            Result.SubMeshBorderVerteces = PickVertices(vtxProfile[0], (int)QemVertexType.SubMesh);
            Result.SubMeshFaces = PickFaces(Result.SubMeshBorderVerteces);

            // 開放端に接している頂点と面を列挙します
            Result.OpenEdgeVerteces = PickVertices(vtxProfile[0], (int)QemVertexType.OpenEdge);
            Result.OpenEdgeFaces = PickFaces(Result.OpenEdgeVerteces);

            // 現在のパラメータで保護される開放端の頂点と面をピックする
            Result.LockedOpenEdgeVerteces = PickVertices(vtxProfile[0], (int)QemVertexType.Corner);
            Result.LockedOpenEdgeFaces = PickFaces(Result.LockedOpenEdgeVerteces);
            if (profileOptions.ComputeHistgrams)
            {
                CreateOpenEdgeHistgram(qemSimp);
            }

            // 現在のパラメータで保護されるUVエッジの頂点と面をピックする
            Result.LockedUvEdgeVerteces = PickVertices(vtxProfile[0], (int)QemVertexType.UvCorner);
            Result.LockedUvEdgeFaces = PickFaces(Result.LockedUvEdgeVerteces);
            if (profileOptions.ComputeHistgrams)
            {
                CreateUvHardEdgeHistgram(qemSimp);
            }

            // Featエッジに連結してる頂点と面を取得して、ヒストグラムを作成する
            Result.FeatEdgeVerteces = PickVertices(vtxProfile[0], (int)QemVertexType.FeatEdge);
            Result.FeatEdgeFaces = PickFaces(Result.FeatEdgeVerteces);
            if (profileOptions.ComputeHistgrams)
            {
                CreateFeatureEdgeHistgram(qemSimp);
            }

            // エッジのコストを一回計算して、quality値を決定するのに参考になるデータをとる
            // ただし、概算しかしようがない
            if (profileOptions.ComputeCostMap)
            {
                CreateCostMap(qemSimp, vtxProfile);
            }

            //取得した結果から、最大削除可能面数を計算する
            var lockedFaces = new List<int>();
            lockedFaces.AddRange(Result.NonManifoldFaces);
            lockedFaces.AddRange(Result.SubMeshFaces);
            lockedFaces.AddRange(Result.LockedOpenEdgeFaces);
            lockedFaces.AddRange(Result.LockedUvEdgeFaces);

            // 重複を削除して、ユニークなリストを作る
            lockedFaces = lockedFaces.Distinct().ToList();

            // 全体から削除不可能な面数を引いた、削除可能な面数
            Result.AvailableFaceCount = SourceMesh.FaceN - lockedFaces.Count;

            // 到達可能な最大削除レート（だけれど、制約があるので、これが最大値になることはない）
            Result.AvailableSimplifyRate = 1.0 - (double)Result.AvailableFaceCount / (double)SourceMesh.FaceN;
            return Result;
        }

        /// <summary>
        /// 非多様体ポリゴンを構成している頂点をピックします
        ///
        /// ここでピックされる頂点は動かすことが出来ません。頂点は必ず残ります。
        /// </summary>
        /// <param name="vtxProfiles"></param>
        private List<int> PickVertices(int[] vtxProfiles, int flag)
        {
            var tmpVerteces = new List<int>();
            for (var i = 0; i < vtxProfiles.Length; i++)
            {
                var vtxFlag = vtxProfiles[i];
                if ((vtxFlag & flag) > 0)
                {
                    tmpVerteces.Add(i);
                }
            }
            return tmpVerteces;
        }

        /// <summary>
        /// 指定の頂点リストに含まれる頂点を使用している面を全てピックします
        ///
        /// ここでピックされる面は、簡略化が進むと、大きさなどが変更する事はありますが
        /// 非多様体に接する頂点が削除できないので、面その物が消える事はありません。
        /// したがって、そのものが残らない場合でも、その面自身が削除されていないものと考えます。
        /// </summary>
        private List<int> PickFaces(List<int> vtxList)
        {
            var tmpFaces = new List<int>();
            var faceN = SourceMesh.FaceN;
            var sourceN = SourceMesh.SourceN;
            var posOffset = SourceMesh.GetSourceOfset(SourceType.Position, null);

            for (var fi = 0; fi < faceN; fi++)
            {
                var vtxCount = SourceMesh.VNums[fi];
                var vtxOffset = SourceMesh.VOffset[fi];

                for (var vi = 0; vi < vtxCount; vi++)
                {
                    var vtxIndex = SourceMesh.VBuffer[ (vtxOffset + vi) * sourceN + posOffset];
                    // 非多様体頂点に繋がっている場合は、この面は非多様体を構成する面としてカウントする
                    // 面の構成自体が正しくとも関係ない
                    if (vtxList.Contains(vtxIndex))
                    {
                        tmpFaces.Add(fi);
                        break;
                    }
                }
            }
            return tmpFaces.Distinct().ToList();
        }

        /// <summary>
        /// 開放端を構成するエッジが、隣接するエッジと連結する角度を計算して
        /// ヒストグラムとして返します。
        /// </summary>
        /// <param name="qemSimp"></param>
        private void CreateOpenEdgeHistgram(QemSimp qemSimp)
        {
            var wingedEdgeMesh = qemSimp.WingEdedgeMesh;

            Result.OpenEdgeHistgram.Create(180, 0, 5);
            // 開放端でループして、両端で接続するエッジのうち
            // 劣角の小さい方を拾って、ヒストグラムに登録する
            foreach (WingedEdge we in wingedEdgeMesh.RepresentativeEdges)
            {
                if (we.Boder)
                {
                    var borders0 = qemSimp.GetNeighborOpenEdges(we, 0);
                    var angle = 180.0;
                    for (int j = 0; j < borders0.Count; j++)
                    {
                        angle = Math.Min(angle, WingedEdgeMesh.ComputeAngle(wingedEdgeMesh, borders0[j], we));
                    }
                    var borders1 = qemSimp.GetNeighborOpenEdges(we, 1);
                    for (int j = 0; j < borders1.Count; j++)
                    {
                        angle = Math.Min(angle, WingedEdgeMesh.ComputeAngle(wingedEdgeMesh, borders1[j], we));
                    }
                    Result.OpenEdgeHistgram.Add(angle, we.ID);
                }
            }
        }

        /// <summary>
        /// UVハードエッジになっている、エッジをピックして、隣接の辺との角度によってヒストグラムを作成します
        /// </summary>
        /// <param name="qemSimp"></param>
        private void CreateUvHardEdgeHistgram(QemSimp qemSimp)
        {
            var wingEdedgeMesh = qemSimp.WingEdedgeMesh;
            Result.UvHardEdgeHistgram.Create(180, 0, 5);
            // 開放端でループして、両端で接続するエッジのうち
            // 劣角の小さい方を拾って、ヒストグラムに登録する
            foreach (WingedEdge we in wingEdedgeMesh.RepresentativeEdges)
            {
                foreach (var cep in we.ChildEdges)
                {
                    foreach (var ce in cep.Value)
                    {
                        if (ce.IsOpen)
                        {
                            if (cep.Key.type == SourceType.Texture)
                            {
                                var borders0 = qemSimp.GetNeighborOpenEdges(ce, 0);
                                var angle = 180.0;
                                for (int i = 0; i < borders0.Count; i++)
                                {
                                    angle = Math.Min(angle, WingedEdgeMesh.ComputeAngle(wingEdedgeMesh, borders0[i], ce));
                                }
                                var borders1 = qemSimp.GetNeighborOpenEdges(ce, 1);
                                for (int i = 0; i < borders1.Count; i++)
                                {
                                    angle = Math.Min(angle, WingedEdgeMesh.ComputeAngle(wingEdedgeMesh, borders1[i], ce));
                                }
                                Result.UvHardEdgeHistgram.Add(angle, we.ID);
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// エッジの両側の面などによって、エッジがFeatureEdgeとして判定される角度を、
        /// エッジごとに計算しHistgram化します。
        ///
        /// </summary>
        /// <param name="qemSimp"></param>
        private void CreateFeatureEdgeHistgram(QemSimp qemSimp)
        {
            var vtxTriMesh = qemSimp.VtxTriMesh;
            var wingEdedgeMesh = qemSimp.WingEdedgeMesh;
            var edgeFeatAngles = new double[wingEdedgeMesh.GetWingedEdgeN()];
            Array.Clear(edgeFeatAngles, 0, wingEdedgeMesh.GetWingedEdgeN());
            Result.FeatureEdgeHistgram.Create(180, 0, 5);

            for (int vi = 0; vi < SourceMesh.VertexN; vi++)
            {
                var vtl = vtxTriMesh.VTL[vi];
                var vnormal = new Vector3();
                foreach (var faceId in vtl.TriList)
                {
                    vnormal += SourceMesh.mFaceNormal[faceId] * SourceMesh.mFaceArea[faceId];
                }
                vnormal.Normalize();
                var maxAngle = 0.0;
                foreach (var faceId in vtl.TriList)
                {
                    var angle = Math.Acos(vnormal.Dot(SourceMesh.mFaceNormal[faceId]));
                    angle = (angle / Math.PI) * 180.0;

                    maxAngle = Math.Max(angle, maxAngle);
                }

                foreach (var we in wingEdedgeMesh.RepVEL[vi].Edges)
                {
                    edgeFeatAngles[we.ID] = Math.Max(edgeFeatAngles[we.ID], maxAngle);
                }
            }

            // エッジヒストグラムを構築する
            for (int ei = 0;  ei < edgeFeatAngles.Length; ei++)
            {
                Result.FeatureEdgeHistgram.Add(edgeFeatAngles[ei], ei);
            }
        }

        /// <summary>
        /// 簡略化コストのマップを作成します。頂点ごとに計算される簡略化のコストをリスト化して格納します。
        /// </summary>
        /// <param name="qemSimp"></param>
        /// <param name="vtxProfile"></param>
        private void CreateCostMap(QemSimp qemSimp, List<int[]> vtxProfile)
        {
            ResizableList<WingedEdge> edges = qemSimp.WingEdedgeMesh.RepresentativeEdges;
            ResizableList<QemHeapData> edgeHeapData = new ResizableList<QemHeapData>();

            edgeHeapData.Resize(edges.Count, null);
            // 初期のヒープ(min-heap)を構築
            qemSimp.BuildHeap(qemSimp.WingEdedgeMesh, vtxProfile, ref edgeHeapData);

            //var costArray = edgeHeapData.Select(o => o.VCost).ToList();
            //costArray.Sort();

            //下位X％を含むことが可能な、Quality値を計算します。
            var split = edgeHeapData.Count / 10;

            Result.AvailableEdgeCountWithinQuality = 0;
            var availableFaces = new List<int>();
            var targetFaces = new List<int>();
            var maxQuality = 0.0;
            for (int i = 0; i < edgeHeapData.Count; i++)
            {
                var heapData = edgeHeapData[i];
                var rate = (double)i / (double)edgeHeapData.Count;
                var localQuality = (heapData.VCost - qemSimp.MinCost) / qemSimp.MaxCost;

                if (i % split == 0 && i > 0)
                {
                    Result.EdgeCountForCostRate.Add( rate, localQuality);
                }

                var edge = qemSimp.WingEdedgeMesh.RepresentativeEdges[heapData.EdgeID];
                // 設定Quality値以内のポリゴン数をカウントアップする
                // この値が正解にはならないけれど、初期値として参考に
                if (localQuality < qemSimp.Quality)
                {
                    availableFaces.Add(edge.Face0);
                    availableFaces.Add(edge.Face1);

                    Result.AvailableEdgeCountWithinQuality++;
                }

                // 指定された簡略化レート以内のポリゴン数であれば、削減可能ポリゴンに追加
                if (targetFaces.Count() < (1.0 - qemSimp.TargetRate) * SourceMesh.FaceN)
                {
                    maxQuality = localQuality;
                    if (!targetFaces.Contains(edge.Face0))
                    {
                        targetFaces.Add(edge.Face0);
                    }
                    if (!targetFaces.Contains(edge.Face1))
                    {
                        targetFaces.Add(edge.Face1);
                    }
                }
            }
            //　現在の設定で、指定の簡略化レートに達するのに必要なquality値
            Result.RecomendedQualityValue = maxQuality;
            // 現在の設定で削除可能な面数を列挙します
            Result.AvailableFaceCountWithinQuality = availableFaces.Distinct().Count();
        }
    }
}
