﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    internal class OctreeNode<T>
    {
        internal ResizableList<T> Objects { get; set; } = new ResizableList<T>();
        internal BoundingBox mBoundingBox { get; set; } = new BoundingBox();
    }

    public class Octree<T>
    {
        private const int Inner8TreeManagerMaxlevel = 8;
        private bool[] dirtyFlag = null;
        private uint mnLevel = 0; ///ツリーの深さ
        private Vector3 mLtb = new Vector3(); ///x,y,z  左上奥
        private Vector3 mRbf = new Vector3(); ///x,y,z　右下手前
        private Vector3 mUnitLen = new Vector3(); ///x,y,z  単位長さ

        private ResizableList<int> mStartIndex = new ResizableList<int>(); ///ツリーノードのオフセットインデックス
        private ResizableList<int> mUnitCount = new ResizableList<int>(); ///各軸上の分割数

        private ResizableList<OctreeNode<T>> mNodePtrArray = new ResizableList<OctreeNode<T>>();

        internal ResizableList<OctreeNode<T>> Nodes
        {
            get { return mNodePtrArray; }
        }

        ///線形8分木ノードリスト
        public int NodeCount
        {
            get { return mNodePtrArray.Count; }
        }

        /// <summary>
        /// 指定レベルの軸上のノード数を取得します
        /// </summary>
        /// <param name="aunLevel"></param>
        /// <returns></returns>
        public int GetUnitNodeCount(int aunLevel) { return mUnitCount[aunLevel]; }

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public Octree() { }

        //! 要素数を取得
        public int GetElemIndex(int aunLevel, int aunLocIndex)
        {
            Nintendo.Foundation.Contracts.Ensure.Argument.True(mStartIndex.Count > aunLevel);
            return (mStartIndex[aunLevel] + aunLocIndex);
        }

        /// <summary>
        /// 3ビットトビに分割
        /// </summary>
        /// <param name="n"></param>
        /// <returns></returns>
        private uint Part1By2(uint n)
        {
            n = (n ^ (n << 16)) & 0xFF0000FF;
            n = (n ^ (n << 8)) & 0x0300F00F;
            n = (n ^ (n << 4)) & 0x030C30C3;
            n = (n ^ (n << 2)) & 0x09249249;
            return n;
        }

        /// <summary>
        /// モートンコードを求める
        /// </summary>
        /// <returns></returns>
        private uint GetMortonCode(uint x, uint y, uint z)
        {
            //z--z--z--z-- ; Part1By2(z)<<2
            //-y--y--y--y- ; Part1By2(y)<<1
            //--x--x--x--x ; Part1By2(x)
            return Part1By2(x) << 2 | Part1By2(y) << 1 | Part1By2(z);
        }

        /// <summary>
        /// 最上位空間の空間レベルと番号を取得する
        /// </summary>
        /// <param name="arLevelN"></param>
        /// <param name="arExorN"></param>
        /// <param name="min"></param>
        /// <param name="max"></param>
        private void QueryHighlevel(ref uint arLevelN, ref uint arExorN, Vector3 min, Vector3 max)
        {
            uint ltb = 0;
            if (max == min || max == null)
            {
                ltb = GetPtMortonCode(min.x, min.y, min.z);
                // 最上位空間の空間レベルと番号を取得する
                arExorN = ltb;
                arLevelN = 1;
            }
            else
            {
                uint rbf = 0;
                ltb = GetPtMortonCode(min.x, min.y, min.z);
                rbf = GetPtMortonCode(max.x, max.y, max.z);
                // 最上位空間の空間レベルと番号を取得する
                arExorN = rbf ^ ltb;
                arLevelN = 1;
            }

            for (int i = 0; i < (int)mnLevel; i++)
            {
                int t = (i * 3);
                uint tmp = (arExorN >> t) & 0x7;
                if (tmp != 0)
                {
                    arLevelN = (uint)i + 1;
                }
            }
        }

        /// <summary>
        /// Octree を作成する
        /// </summary>
        /// <param name="level"> ツリーのレベル</param>
        /// <param name="min">BB-min</param>
        /// <param name="max">BB-max</param>
        public void Create(int level, Vector3 min, Vector3 max)
        {
            // オクツリーのオブジェクト初期化
            try
            {
                // 各ノードを取得する際のオフセットの計算
                mStartIndex.Resize(Inner8TreeManagerMaxlevel + 2, 0);
                mUnitCount.Resize(Inner8TreeManagerMaxlevel + 2, 0);
                for (int i = 0; i < Inner8TreeManagerMaxlevel + 1; i++)
                {
                    mStartIndex[i] = (((int)Math.Pow(8.0, (double)i) - 1) / 7);
                    mUnitCount[i] = (int)Math.Pow(2.0, (double)i);
                }
                // ノードを生成する
                int len = GetElemIndex(level + 1, 1);
                mnLevel = (uint)level;
                mNodePtrArray.Resize(len, null);
                for (int i = 0; i < len; i++)
                {
                    mNodePtrArray[i] = new OctreeNode<T>();
                }
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Fail to initialize octree", ex);
            }

            // オクツリーのサイズ更新
            try
            {
                dirtyFlag = new bool[NodeCount];

                for (int i = 0; i < 3; i++)
                {
                    if (min[i] < mLtb[i])
                    {
                        mLtb[i] = min[i];
                    }
                    if (max[i] > mRbf[i])
                    {
                        mRbf[i] = max[i];
                    }
                }
                for (int i = 0; i < 3; i++)
                {
                    mUnitLen[i] = Math.Abs(mRbf[i] - mLtb[i]) / GetUnitNodeCount((int)mnLevel);
                }

                Vector3 ltb = mLtb;
                // 位置座標を設定する

                OctreeNode<T> pNode = mNodePtrArray[0];
                pNode.mBoundingBox.Update(mLtb);
                pNode.mBoundingBox.Update(mRbf);
                QueryLowerlevelSpaces(ltb, true, 0, 0, null);
            }
            catch (Exception ex)
            {
                throw ExcepHandle.CreateException("Fail to update octree size", ex);
            }
        }

        /// <summary>
        /// バウンディングボックスの最上位モートンコードを取得
        /// </summary>
        /// <param name="min"></param>
        /// <param name="max"></param>
        /// <returns></returns>
        private uint GetBBMortonCode(Vector3 min, Vector3 max)
        {
            // 最上位空間の空間レベルと番号を取得する
            uint ltn = GetPtMortonCode(min[0], min[1], min[2]);
            uint rbf = GetPtMortonCode(max[0], max[1], max[2]);
            if (ltn == rbf)
            {
                return rbf;
            }
            return rbf ^ ltn;
        }

        /// <summary>
        /// 指定座標のモートン順序のコードを取得
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="z"></param>
        /// <returns></returns>
        private uint GetPtMortonCode(double x, double y, double z)
        {
            int dx = (int)((x - mLtb.x) / mUnitLen.x);
            int dy = (int)((y - mLtb.y) / mUnitLen.y);
            int dz = (int)((z - mLtb.z) / mUnitLen.z);
            int elem = GetUnitNodeCount((int)mnLevel);
            dx = dx < 0 ? 0 : dx;
            dy = dy < 0 ? 0 : dy;
            dz = dz < 0 ? 0 : dz;
            dx = dx >= elem ? elem - 1 : dx;
            dy = dy >= elem ? elem - 1 : dy;
            dz = dz >= elem ? elem - 1 : dz;
            return GetMortonCode((uint)dx, (uint)dy, (uint)dz);
        }

        private void QueryLowerlevelSpaces(Vector3 ltb, bool initAABB, uint aunParentLevelN, uint aunParentSpace, bool[] dirtyFlags)
        {
            if (aunParentLevelN < mnLevel)
            {
                uint shift = (mnLevel - (aunParentLevelN + 1));
                uint unit = (uint)mUnitCount[(int)shift];
                uint _tmpCode = 0;
                uint tmpCode = 0;

                uint z = 0;
                uint y = 0;
                uint x = 0;

                Vector3 nxtLtb = new Vector3();
                Vector3 size = new Vector3();

                // オクタントサイズ
                size.x = mUnitLen.x * unit;
                size.y = mUnitLen.y * unit;
                size.z = mUnitLen.z * unit;

                for (uint j = 0; j < 8; j++)
                {
                    _tmpCode = (aunParentSpace << 3) | j;
                    tmpCode = (uint)GetElemIndex((int)aunParentLevelN + 1, (int)_tmpCode);

                    z = (j & 0x1) >> 0;
                    y = (j & 0x2) >> 1;
                    x = (j & 0x4) >> 2;

                    // オクタントのAABB
                    nxtLtb.x = ltb.x + size.x * x;
                    nxtLtb.y = ltb.y + size.y * y;
                    nxtLtb.z = ltb.z + size.z * z;

                    // 初期化字
                    if (initAABB)
                    {
                        OctreeNode<T> node = mNodePtrArray[(int)tmpCode];
                        if (node == null)
                        {
                            continue;
                        }

                        node.mBoundingBox.Update(nxtLtb);
                        node.mBoundingBox.Update(nxtLtb + size);
                    }

                    if (dirtyFlags != null)
                    {
                        if (!dirtyFlags[tmpCode])
                        {
                            dirtyFlags[tmpCode] = true;
                        }
                    }
                    QueryLowerlevelSpaces(nxtLtb, initAABB, aunParentLevelN + 1, _tmpCode, dirtyFlags);
                }
            }
        }

        /// <summary>
        /// 指定のBBに含まれる空間を取得します
        /// </summary>
        /// <param name="arvSpaces">空間リスト</param>
        /// <param name="min">最少位置</param>
        /// <param name="max">最大位置</param>
        /// <returns></returns>
        internal bool WindowQueryForBB(ResizableList<OctreeNode<T>> arvSpaces, Vector3 min, Vector3 max)
        {
            // 最上位空間の空間レベルと番号を取得する
            uint shiftN = 0;
            uint highspace = 0;
            uint ltf = GetPtMortonCode(min.x, min.y, min.z);
            int nodeN = NodeCount;

            QueryHighlevel(ref shiftN, ref highspace, min, max);

            // 最上位空間ののローカルコードの取得
            uint parentLevel = mnLevel - shiftN;
            uint parentSpace = ltf >> ((int)shiftN * 3);

            if (parentLevel < 0)
            {
                parentLevel = 0;
            }

            // ルート空間なので
            if (parentLevel == 0)
            {
                parentSpace = 0;
            }
            if (parentLevel == mnLevel)
            {
                parentSpace = ltf;
            }

            int index = GetElemIndex((int)parentLevel, (int)parentSpace);

            arvSpaces.Clear();

            BoundingBox aabb = new BoundingBox();

            aabb.Update(min);
            aabb.Update(max);

            Array.Clear(dirtyFlag, 0, NodeCount);

            if (parentLevel < mnLevel)
            {
                OctreeNode<T> node = mNodePtrArray[index];
                Vector3 minVec = node.mBoundingBox.Min;
                QueryLowerlevelSpaces(minVec, false, parentLevel, parentSpace, dirtyFlag);
            }

            for (int i = 0; i < nodeN; i++)
            {
                if (dirtyFlag[i])
                {
                    OctreeNode<T> tmp = mNodePtrArray[i];
                    if (tmp.mBoundingBox.Intersect(aabb))
                    {
                        arvSpaces.Add(tmp);
                    }
                }
            }
            if (arvSpaces.Count == 0)
            {
                return false;
            }
            return true;
        }

        /// <summary>
        /// オブジェクトを追加します
        /// </summary>
        /// <param name="obj"></param>
        /// <param name="box"></param>
        /// <returns></returns>
        public bool AddObject(T obj, BoundingBox box)
        {
            if (mUnitLen.Norm() < double.Epsilon)
            {
                return false;
            }

            ResizableList<OctreeNode<T>> tmpObjectArray = new ResizableList<OctreeNode<T>>();
            // BoundingBoxに含まれる全てのノードを取得します
            WindowQueryForBB(tmpObjectArray, box.Min, box.Max);
            int nodeN = tmpObjectArray.Count;
            if (nodeN == 0)
            {
                return false;
            }

            foreach (var octreeNode in tmpObjectArray)
            {
                octreeNode.Objects.Add(obj);
            }
            return true;
        }

        /// <summary>
        /// オブジェクトを追加します
        /// </summary>
        /// <param name="obj"></param>
        /// <param name="box"></param>
        /// <returns></returns>
        public bool AddObject(T obj, Vector3 pos)
        {
            if (mUnitLen.Norm() < double.Epsilon)
            {
                return false;
            }

            ResizableList<OctreeNode<T>> tmpObjectArray = new ResizableList<OctreeNode<T>>();
            // BoundingBoxに含まれる全てのノードを取得します
            WindowQueryForBB(tmpObjectArray, pos, null);
            int nodeN = tmpObjectArray.Count;
            if (nodeN == 0)
            {
                return false;
            }

            foreach (var octreeNode in tmpObjectArray)
            {
                octreeNode.Objects.Add(obj);
            }
            return true;
        }
    }
}
