﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using nw.g3d.iflib;
using nw.g3d.toollib;
using nw.g3d.nw4f_3dif;
using System.Linq.Expressions;
using System;
using nw.g3d.toollib.Ext;

namespace nw.g3d.optcvtr
{
    // オプティマイザ
    internal abstract class Optimizer
    {
        internal static Optimizer Create(g3doptcvtrParams fileOption)
        {
            // 入力ファイルが存在するか
            if (!File.Exists(fileOption.Path))
            {
                Strings.Throw("Error_InputFileNotFound", fileOption.Path);
            }

            // 入力ファイルが中間ファイルか
            if (!G3dPath.IsPath(fileOption.Path))
            {
                Strings.Throw("Error_InputFileIsNotIf", fileOption.Path);
            }

            if (G3dPath.IsModelPath(fileOption.Path))
            {
                return new ModelOptimizer(fileOption);
            }
            else if (G3dPath.IsSkeletalAnimPath(fileOption.Path))
            {
                return new SkeletalAnimOptimizer(fileOption);
            }
            else if (G3dPath.IsMaterialAnimPath(fileOption.Path))
            {
                return new MaterialAnimOptimizer(fileOption);
            }
            else if (G3dPath.IsShaderParamAnimGroupPath(fileOption.Path))
            {
                return new ShaderParamAnimOptimizer(fileOption);
            }
            else if (G3dPath.IsTexPatternAnimPath(fileOption.Path))
            {
                return new TexPatternAnimOptimizer(fileOption);
            }
            else if (G3dPath.IsBoneVisibilityAnimPath(fileOption.Path))
            {
                return new BoneVisibilityAnimOptimizer(fileOption);
            }
            else if (G3dPath.IsMatVisibilityAnimPath(fileOption.Path))
            {
                return new MatVisibilityAnimOptimizer(fileOption);
            }
            else if (G3dPath.IsShapeAnimPath(fileOption.Path))
            {
                return new ShapeAnimOptimizer(fileOption);
            }
            else if (G3dPath.IsSceneAnimPath(fileOption.Path))
            {
                return new SceneAnimOptimizer(fileOption);
            }
            else
            {
                // 最適化に対応していない中間ファイル
                Strings.Throw("Error_InputIfIsNotSupported", fileOption.Path);
            }
            return null;
        }

        internal Optimizer(g3doptcvtrParams fileOption)
        {
            this.Param = fileOption;

            if (!string.IsNullOrEmpty(fileOption.Output))
            {
                this.OutputFilePath = fileOption.Output;
            }
        }

        // 非サポートオプション値のチェック
        protected void CheckUnsupportedOptionValue(ArgumentOption option)
        {
            if (option.HasValue)
            {
                Strings.Throw("Error_InputIfUnsupportedOptionValue", option);
            }
        }

        protected void CheckUnsupportedOption<T>(T owner, Expression<Func<bool>> property)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(owner != null);
            var body = property.Body as MemberExpression;
            var propName = body.Member.Name;
            var prop = typeof(T).GetProperty(propName);
            bool value = (bool)prop.GetValue(owner);
            if (value)
            {
                Strings.Throw("Error_InputIfUnsupportedOptionValue", owner.GetOptionName(property));
            }
        }

        protected void CheckUnsupportedOption<T, U>(T owner, Expression<Func<U>> property)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(owner != null);
            var body = property.Body as MemberExpression;
            var propName = body.Member.Name;
            var prop = typeof(T).GetProperty(propName);
            var value = prop.GetValue(owner);
            if (value != null)
            {
                Strings.Throw("Error_InputIfUnsupportedOptionValue", owner.GetOptionName(property));
            }
        }

        // 圧縮系オプションの無効化
        protected void DisableCompressOption()
        {
            // マテリアル圧縮
            CheckUnsupportedOption(Param, () => Param.CompressMaterial);

            // リジッドスキン変換
            CheckUnsupportedOption(Param, () => Param.RigidBodyToRigidSkinning);

            // 刈り込みボーン圧縮
            CheckUnsupportedOption(Param, () => Param.CompressBoneCull);

            // マージボーン圧縮
            CheckUnsupportedOption(Param, () => Param.CompressBoneMerge);

            // 親子合体ボーン圧縮
            CheckUnsupportedOption(Param, () => Param.CompressBoneUniteChild);

            // 合体ボーン圧縮
            CheckUnsupportedOption(Param, () => Param.CompressBoneUnite);

            // 完全合体ボーン圧縮
            CheckUnsupportedOption(Param, () => Param.CompressBoneUniteAll);

            // シェイプ圧縮
            CheckUnsupportedOption(Param, () => Param.CompressShape);
        }

        //---------------------------------------------------------------------
        // 最適化
        internal abstract void Optimize();

        // 読み込み
        protected void Read()
        {
            this.ReadStopwatch.Start();

            // 読み込み
            this.nw4f_3dif = IfReadUtility.Read(
                this.Streams, this.Param.Path, this.XsdBasePath);

            // チェック
            IfCheckContext context = new IfCheckContext(this.Param.Path);
            IfChecker.Check(context, this.nw4f_3dif, this.Streams);
            context.Throw();

            this.ReadStopwatch.Stop();
        }

        // 書き出し
        protected void Write()
        {
            this.WriteStopwatch.Start();

            // ファイル情報の更新
            if (!this.DisableFileInfo)
            {
                IfFileLogUtility.SetModify(this.nw4f_3dif);
            }

            IfCheckContext context = new IfCheckContext();
            IfChecker.Check(context, this.nw4f_3dif, this.Streams);

            try
            {
                context.Throw();
            }
            catch(Exception exception)
            {
                System.Text.StringBuilder builder = new System.Text.StringBuilder();
                builder.Append(GetLog());
                builder.Append(exception.Message);

                Exception optimizerException = new Exception(builder.ToString());
                throw optimizerException;
            }

            // 書き出し
            string output = this.OutputFilePath;
            if (output == null) { output = this.Param.Path; }
            IfWriteUtility.Write(this.nw4f_3dif, this.Streams, output);

            this.WriteStopwatch.Stop();
        }

        //---------------------------------------------------------------------
        // ログの取得
        internal virtual string GetLog()
        {
            return this.Param.Path;
        }

        // メモリ解放
        internal virtual void Dispose()
        {
            this.nw4f_3dif = null;
            this.Streams.Clear();
        }

        // 読み込み時間の取得
        protected string GetReadLog()
        {
            return Strings.Get("ReadLog",
                this.ReadStopwatch.ElapsedMilliseconds, this.Param.Path);
        }

        // 書き出し時間の取得
        protected string GetWriteLog()
        {
            string output = this.OutputFilePath;
            if (output == null) { output = this.Param.Path; }
            return Strings.Get("WriteLog",
                this.WriteStopwatch.ElapsedMilliseconds, output);
        }

        //---------------------------------------------------------------------
        // スキーマベースパス
        internal string XsdBasePath { get; set; }
        // <file_info> 更新の無効化
        internal bool DisableFileInfo { get; set; }

        // オプション
        protected readonly g3doptcvtrParams Param;
        // 出力ファイルパス
        protected readonly string OutputFilePath;

        // 中間ファイル
        protected nw4f_3difType nw4f_3dif { get; set; }
        protected List<G3dStream> Streams { get; set; } = new List<G3dStream>();

        // ストップウォッチ
        protected readonly Stopwatch ReadStopwatch = new Stopwatch();
        protected readonly Stopwatch WriteStopwatch = new Stopwatch();
    }
}
