﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace nw.g3d.iflib
{
    using System;
    using System.Diagnostics;

    /// <summary>
    /// 1 次元のベクトルを表します。
    /// </summary>
    public sealed class Vector1 : ICloneable, IEquatable<Vector1>, IVector
    {
        //-----------------------------------------------------------------
        private float x;

        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public Vector1()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元のベクトルです。</param>
        public Vector1(Vector1 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// X座標を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="x">X座標です。</param>
        public Vector1(float x)
        {
            this.Set(x);
        }

        /// <summary>
        /// X座標を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="source">X座標の配列です。</param>
        public Vector1(float[] source)
        {
            this.Set(source);
        }

        //-----------------------------------------------------------------
        // プロパティの取得または設定
        //-----------------------------------------------------------------

        //-----------------------------------------------------------------

        /// <summary>
        /// 次元数を取得します。
        /// </summary>
        public int Dimension
        {
            get
            {
                return 1;
            }
        }

        /// <summary>
        /// X 成分を取得または設定します。
        /// </summary>
        public float X
        {
            get
            {
                return this.x;
            }

            set
            {
                this.x = value;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// ベクトルの長さを取得します。
        /// </summary>
        /// <returns>ベクトルの長さです。</returns>
        public float Length
        {
            get { return this.X; }
        }

        /// <summary>
        /// 長さの 2 乗を取得します。
        /// </summary>
        /// <returns>ベクトルの長さの２乗です。</returns>
        public float LengthSquared
        {
            get { return this.X * this.X; }
        }

        /// <summary>
        /// 指定したインデックス位置の成分を取得、または設定します。
        /// </summary>
        /// <param name="i">インデックスです。</param>
        /// <returns>成分です。</returns>
        public float this[int i]
        {
            get
            {
                switch (i)
                {
                    case 0:
                        return this.X;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }

            set
            {
                switch (i)
                {
                    case 0:
                        this.X = value;
                        break;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
        }

        //-----------------------------------------------------------------
        // オペレータ
        //-----------------------------------------------------------------

        /// <summary>
        /// 2 つのベクトルを加算します。
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです。</param>
        /// <param name="rhs">演算の右のベクトルです。</param>
        /// <returns>lhsとrhsを加算した結果のベクトルです。</returns>
        public static Vector1 operator +(Vector1 lhs, Vector1 rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new Vector1(lhs.X + rhs.X);
        }

        /// <summary>
        /// 2 つのベクトルを減算します。
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです。</param>
        /// <param name="rhs">演算の右のベクトルです。</param>
        /// <returns>lhsとrhsを減算した結果のベクトルです。</returns>
        public static Vector1 operator -(Vector1 lhs, Vector1 rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return new Vector1(lhs.X - rhs.X);
        }

        /// <summary>
        /// ベクトルにスカラーを乗算します。
        /// </summary>
        /// <param name="vec">演算の左のベクトルです。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>vecとscalarを乗算した結果のベクトルです。</returns>
        public static Vector1 operator *(Vector1 vec, float scalar)
        {
            Ensure.Argument.NotNull(vec);

            return new Vector1(vec.X * scalar);
        }

        /// <summary>
        /// ベクトルをスカラーで除算します。
        /// </summary>
        /// <param name="vec">演算の左のベクトルです。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>vecとscalarを除算した結果のベクトルです。</returns>
        public static Vector1 operator /(Vector1 vec, float scalar)
        {
            Ensure.Argument.NotNull(vec);
            Ensure.Operation.DividerNotZero(scalar);

            return new Vector1(vec.X / scalar);
        }

        /// <summary>
        /// 2 つのベクトルの内積を計算します。
        /// </summary>
        /// <param name="lhs">演算の左のベクトルです。</param>
        /// <param name="rhs">演算の右のベクトルです。</param>
        /// <returns>lhsとrhsの外積を計算した結果のスカラーです。</returns>
        public static float Dot(Vector1 lhs, Vector1 rhs)
        {
            Ensure.Argument.NotNull(lhs);
            Ensure.Argument.NotNull(rhs);

            return lhs.X * rhs.X;
        }

        /// <summary>
        /// X座標を格納した配列を返します。
        /// </summary>
        /// <returns>X座標の配列です。</returns>
        public float[] ToArray()
        {
            return new float[] { this.X };
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            Ensure.Argument.True(source is IVector);
            this.Set(source as IVector);
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。同じ次元でなければなりません。</param>
        public void Set(IVector source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(this.Dimension == source.Dimension);

            this.Set(source[0]);
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public void Set(Vector1 source)
        {
            Ensure.Argument.NotNull(source);

            this.Set(source.X);
        }

        /// <summary>
        /// X座標を指定の値に設定します。
        /// </summary>
        /// <param name="x">X座標です。</param>
        public void Set(float x)
        {
            this.X = x;
        }

        /// <summary>
        /// X座標を指定の値に設定します。
        /// </summary>
        /// <param name="source">X座標の配列です。</param>
        public void Set(float[] source)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.True(this.Dimension == source.Length);

            this.Set(source[0]);
        }

        /// <summary>
        /// 成分を 0 に設定します。
        /// </summary>
        public void SetZero()
        {
            this.X = 0f;
        }

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public object Clone()
        {
            return new Vector1(this);
        }

        //-----------------------------------------------------------------
        // 数学演算
        //-----------------------------------------------------------------

        /// <summary>
        /// ベクトルの長さが FloatUtility.Epsilon 以下か示します。
        /// </summary>
        /// <returns>ベクトルの長さが FloatUtility.Epsilon 以下ならば true です。
        /// それ以外の場合は false です。</returns>
        public bool IsZero()
        {
            return this.IsZero(this.LengthSquared);
        }

        /// <summary>
        /// 指定したベクトルとの内積を計算します。
        /// </summary>
        /// <param name="vec">掛けるベクトルです。</param>
        /// <returns>vecとの内積を計算した結果のスカラーです。</returns>
        public float Dot(Vector1 vec)
        {
            return Dot(this, vec);
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as Vector1);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Vector1 other)
        {
            if (other == this)
            {
                return true;
            }

            if (other == null)
            {
                return false;
            }

            return this.X == other.X;
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <param name="tolerance">各成分毎の差の最大許容値です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Vector1 other, float tolerance)
        {
            if (other == this)
            {
                return true;
            }

            if (other == null)
            {
                return false;
            }

            return
                FloatUtility.NearlyEqual(this.X, other.X, tolerance);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return
                this.X.GetHashCode();
        }

        //-----------------------------------------------------------------
        // 文字列化
        //-----------------------------------------------------------------

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>現在のオブジェクトを表す文字列です。</returns>
        public override string ToString()
        {
            return string.Format("[{0}]", this.X);
        }

        // 長さの 2 乗を受け取り、長さがほぼゼロかどうか判定します。
        private bool IsZero(float lengthSquared)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(lengthSquared == this.LengthSquared, "Unexpected case!");

            float epsilonSquared = FloatUtility.Epsilon * FloatUtility.Epsilon;
            return lengthSquared <= epsilonSquared;
        }

        // 長さの 2 乗を受け取り長さを１に正規化
        private void Normalize(float lengthSquared)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(lengthSquared == this.LengthSquared, "Unexpected case!");

            if (lengthSquared == 1f)
            {
                return;
            }

            float inverseLength = 1f / (float)Math.Sqrt(lengthSquared);
            this.Set(this * inverseLength);
        }
    }
}
