﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    //-----------------------------------------------------------------------------
    internal class StreamCheckerContext
    {
        private int ReferenceIndex = 0;
        private int StreamCount;
        private bool[] UsedFlags;
        public string LogMessage { get; private set; }

        public StreamCheckerContext(int streamCount)
        {
            this.StreamCount = streamCount;
            if (streamCount > 0)
            {
                this.UsedFlags = new bool[streamCount];
                for (int i = 0; i < streamCount; i++)
                {
                    this.UsedFlags[i] = false;
                }
            }

            LogMessage = string.Empty;
        }

        // ストリームインデックスの順番が適切か確認します。
        private bool IsStreamIndexSuitableOrder(int index)
        {
            bool valid = (index == this.ReferenceIndex);
            // ストリームインデックスとリファレンスインデックスを比較した後に、
            // リファレンスインデックスをインクリメントし、次のインデックスの比較に備える。
            this.ReferenceIndex++;

            return valid;
        }

        // ストリームインデックスの範囲が適切か確認します。
        private bool IsStreamIndexSuitableRange(int index)
        {
            return (index < StreamCount);
        }

        // ストリームの参照が重複していないか確認します。
        // 重複していない時には true を返します。
        private bool IsNotOverlappedStreamIndex(int index)
        {
            // 確認されたストリームのフラグを true に変更する。
            // 次回同じストリームを確認するとフラグが true になっているので、
            // 重複している事を判別できる。
            bool used = this.UsedFlags[index];
            return !used;
        }

        // 使用されていないストリームが存在していないか確認します。
        public void CheckAllStreamsUsed()
        {
            if (this.UsedFlags == null) { return; }

            int index = Array.FindIndex(this.UsedFlags,
                delegate(bool flag) { return (flag == false); });
            if (index != -1)
            {
                Log("There is a stream which is not used.\n\n");
            }
        }

        // ログメッセージを追加します。
        public void Log(string format, params object[] args)
        {
            this.LogMessage += string.Format(format, args);
        }

        // ストリームインデックのチェック
        public void CheckStreamIndex(int streamIndex, string item)
        {
            // ストリームの範囲チェック
            bool rangeResult = IsStreamIndexSuitableRange(streamIndex);
            if (!rangeResult)
            {
                Log("stream_index is outside the range at {0}.\n\n", item);
            }
            else
            {
                if (!IsNotOverlappedStreamIndex(streamIndex))
                {
                    // ストリームの重複チェック
                    Log("stream_index overlaps other references at {0}.\n\n", item);
                }
                // 参照するストリームが重複している場合は必ず順序も正しくないので以下のチェックはスキップ
                else if (!IsStreamIndexSuitableOrder(streamIndex))
                {
                    // ストリームの順序チェック
                    Log("stream_index has not pointed out the stream in the right order at {0}.\n\n", item);
                }

                // 重複した参照を検出するためにストリームが使用されているフラグを立てる
                this.UsedFlags[streamIndex] = true;
            }
        }
    }

    //-----------------------------------------------------------------------------
    public class StreamChecker
    {
        //-----------------------------------------------------------------------------
        // ユーザーデータのストリームインデックスの正当性を確認します。
        // ストリームインデックスに問題がなければ true を返します。
        private static void CheckStreamIndex(
            user_data_arrayType userDataArray,
            StreamCheckerContext context)
        {
            // ユーザーデータがない場合は問題なしとする。
            if (userDataArray == null || userDataArray.user_data == null)
            {
                return;
            }

            foreach (user_dataType userData in userDataArray.user_data)
            {
                if (userData == null) { continue; }

                if (userData.Item is user_streamType)
                {
                    IG3dStreamReference streamRef = userData.Item as IG3dStreamReference;
                    int streamIndex = streamRef.stream_index;
                    if (streamIndex >= 0)
                    {
                        context.CheckStreamIndex(
                            streamIndex,
                            string.Format("<user_data name=\"{0}\">", userData.name));
                    }
                }
            }
        }

        //-----------------------------------------------------------------------------
        // モデルのストリームインデックスの正当性を確認します。
        public static bool CheckStreamIndex(
            out string result,
            modelType model,
            List<G3dStream> streams)
        {
            StreamCheckerContext context = new StreamCheckerContext(streams.Count);

            // マテリアルのユーザーデータのストリームインデックスを確認する。
            if (model.material_array != null &&
                model.material_array.material != null)
            {
                foreach (materialType mat in model.material_array.material)
                {
                    CheckStreamIndex(mat.user_data_array, context);
                }
            }

            // スケルトンのユーザーデータのストリームインデックスを確認する。
            foreach (boneType bone in model.skeleton.bone_array.bone)
            {
                CheckStreamIndex(bone.user_data_array, context);
            }

            // vtx_attrib.stream_index を確認する。
            if (model.vertex_array != null &&
                model.vertex_array.vertex != null)
            {
                foreach (vertexType vertex in model.vertex_array.vertex)
                {
                    foreach (vtx_attribType vtxAttr in vertex.vtx_attrib_array.vtx_attrib)
                    {
                        int streamIndex = vtxAttr.stream_index;
                        context.CheckStreamIndex(
                            streamIndex,
                            string.Format("<vtx_attrib attrib_index=\"{0}\"> in <vertex vertex_index=\"{1}\">",
                            vtxAttr.attrib_index, vertex.vertex_index));
                    }
                }
            }

            // mesh.stream_index を確認する。
            if (model.shape_array != null &&
                model.shape_array.shape != null)
            {
                foreach (shapeType shape in model.shape_array.shape)
                {
                    int streamIndex = shape.mesh_array.mesh[0].stream_index;
                        context.CheckStreamIndex(
                            streamIndex,
                        string.Format("<mesh> in <shape index=\"{0}\">", shape.index));
                }
            }

            // ユーザーデータのストリームインデックスを確認する。
            CheckStreamIndex(model.user_data_array, context);
            // 全てのストリームが使用されているか確認する。
            context.CheckAllStreamsUsed();

            //　ログメッセージを記録する。
            result = context.LogMessage;
            // ログメッセージがあるかどうかで問題の有無を判断する。
            return result.Length == 0;
        }

        //-----------------------------------------------------------------------------
        // スケルタルアニメーションのストリームインデックスの正当性を確認します。
        public static bool CheckStreamIndex(
            out string result,
            skeletal_animType skeletalAnim,
            List<G3dStream> streams)
        {
            StreamCheckerContext context = new StreamCheckerContext(streams.Count);

            // <bone_anim> のカーブが参照するストリームインデックスの正当性を確認する
            if (skeletalAnim.bone_anim_array != null &&
                skeletalAnim.bone_anim_array.bone_anim != null)
            {
                foreach (bone_animType boneAnim in skeletalAnim.bone_anim_array.bone_anim)
                {
                    foreach (bone_anim_targetType animTarget in boneAnim.bone_anim_target)
                    {
                        if (animTarget.Curve == null) { continue; }

                        int streamIndex = animTarget.Curve.stream_index;
                        context.CheckStreamIndex(
                            streamIndex,
                            string.Format("<bone_anim_target target=\"{0}\"> in <bone_anim bone_name=\"{1}\">",
                                animTarget.target, boneAnim.bone_name));
                    }
                }
            }

            // ユーザーデータのストリームインデックスを確認する。
            CheckStreamIndex(skeletalAnim.user_data_array, context);
            // 全てのストリームが使用されているか確認する。
            context.CheckAllStreamsUsed();

            //　ログメッセージを記録する。
            result = context.LogMessage;
            // ログメッセージがあるかどうかで問題の有無を判断する。
            return result.Length == 0;
        }

        //-----------------------------------------------------------------------------
        // ボーンビジビリティアニメーションのストリームインデックスの正当性を確認します。
        public static bool CheckStreamIndex(
            out string result,
            bone_visibility_animType boneVisAnim,
            List<G3dStream> streams)
        {
            StreamCheckerContext context = new StreamCheckerContext(streams.Count);

            // <bone_vis_bone_anim> のカーブが参照するストリームインデックスの正当性を確認する
            if (boneVisAnim.bone_vis_bone_anim_array != null &&
                boneVisAnim.bone_vis_bone_anim_array.bone_vis_bone_anim != null)
            {
                foreach (bone_vis_bone_animType visAnim in
                    boneVisAnim.bone_vis_bone_anim_array.bone_vis_bone_anim)
                {
                    if (visAnim.Curve == null) { continue; }

                    int streamIndex = visAnim.Curve.stream_index;
                    context.CheckStreamIndex(
                        streamIndex,
                        string.Format("<bone_vis_bone_anim bone_name=\"{0}\">", visAnim.bone_name));
                }
            }

            // ユーザーデータのストリームインデックスを確認する。
            CheckStreamIndex(boneVisAnim.user_data_array, context);
            // 全てのストリームが使用されているか確認する。
            context.CheckAllStreamsUsed();

            //　ログメッセージを記録する。
            result = context.LogMessage;
            // ログメッセージがあるかどうかで問題の有無を判断する。
            return result.Length == 0;
        }

        //-----------------------------------------------------------------------------
        // マテリアルビジビリティアニメーションのストリームインデックスの正当性を確認します。
        public static bool CheckStreamIndex(
            out string result,
            mat_visibility_animType matVisAnim,
            List<G3dStream> streams)
        {
            StreamCheckerContext context = new StreamCheckerContext(streams.Count);

            // <mat_vis_mat_anim> のカーブが参照するストリームインデックスの正当性を確認する
            if (matVisAnim.mat_vis_mat_anim_array != null &&
                matVisAnim.mat_vis_mat_anim_array.mat_vis_mat_anim != null)
            {
                foreach (mat_vis_mat_animType visAnim in
                    matVisAnim.mat_vis_mat_anim_array.mat_vis_mat_anim)
                {
                    if (visAnim.Curve == null) { continue; }

                    int streamIndex = visAnim.Curve.stream_index;
                    context.CheckStreamIndex(
                        streamIndex,
                        string.Format("<mat_vis_mat_anim mat_name=\"{0}\">", visAnim.mat_name));
                }
            }

            // ユーザーデータのストリームインデックスを確認する。
            CheckStreamIndex(matVisAnim.user_data_array, context);
            // 全てのストリームが使用されているか確認する。
            context.CheckAllStreamsUsed();

            //　ログメッセージを記録する。
            result = context.LogMessage;
            // ログメッセージがあるかどうかで問題の有無を判断する。
            return result.Length == 0;
        }

        //-----------------------------------------------------------------------------
        // シーンアニメーションのストリームインデックスの正当性を確認します。
        public static bool CheckStreamIndex(
            out string result,
            scene_animType sceneAnim,
            List<G3dStream> streams)
        {
            StreamCheckerContext context = new StreamCheckerContext(streams.Count);

            // <camera_anim_target> のカーブが参照するストリームインデックスの正当性を確認する
            if (sceneAnim.camera_anim_array != null &&
                sceneAnim.camera_anim_array.camera_anim != null)
            {
                foreach (camera_animType cameraAnim in
                    sceneAnim.camera_anim_array.camera_anim)
                {
                    foreach (camera_anim_targetType cameraTarget in cameraAnim.camera_anim_target ?? Enumerable.Empty<camera_anim_targetType>())
                    {
                        if (cameraTarget.Curve == null) { continue; }

                        int streamIndex = cameraTarget.Curve.stream_index;
                        context.CheckStreamIndex(
                            streamIndex,
                            string.Format("<camera_anim_target target=\"{0}\"> in <camera_anim camera_name=\"{1}\">",
                                cameraTarget.target.ToString(), cameraAnim.camera_name));
                    }
                }
            }

            // <light_anim_target> のカーブが参照するストリームインデックスの正当性を確認する
            if (sceneAnim.light_anim_array != null &&
                sceneAnim.light_anim_array.light_anim != null)
            {
                foreach (light_animType lightAnim in
                    sceneAnim.light_anim_array.light_anim)
                {
                    foreach (light_anim_targetType lightTarget in lightAnim.light_anim_target ?? Enumerable.Empty<light_anim_targetType>())
                    {
                        if (lightTarget.Curve == null) { continue; }

                        int streamIndex = lightTarget.Curve.stream_index;
                        context.CheckStreamIndex(
                            streamIndex,
                            string.Format("<light_anim_target target=\"{0}\"> in <light_anim light_name=\"{1}\">",
                                lightTarget.target.ToString(), lightAnim.light_name));
                    }
                }
            }

            // <fog_anim_target> のカーブが参照するストリームインデックスの正当性を確認する
            if (sceneAnim.fog_anim_array != null &&
                sceneAnim.fog_anim_array.fog_anim != null)
            {
                foreach (fog_animType fogAnim in
                    sceneAnim.fog_anim_array.fog_anim)
                {
                    foreach (fog_anim_targetType fogTarget in fogAnim.fog_anim_target ?? Enumerable.Empty<fog_anim_targetType>())
                    {
                        if (fogTarget.Curve == null) { continue; }

                        int streamIndex = fogTarget.Curve.stream_index;
                        context.CheckStreamIndex(
                            streamIndex,
                            string.Format("<fog_anim_target target=\"{0}\"> in <fog_anim fog_name=\"{1}\">",
                                fogTarget.target.ToString(), fogAnim.fog_name));
                    }
                }
            }

            // ユーザーデータのストリームインデックスを確認する。
            CheckStreamIndex(sceneAnim.user_data_array, context);
            // 全てのストリームが使用されているか確認する。
            context.CheckAllStreamsUsed();

            //　ログメッセージを記録する。
            result = context.LogMessage;
            // ログメッセージがあるかどうかで問題の有無を判断する。
            return result.Length == 0;
        }
        //-----------------------------------------------------------------------------
        // マテリアルアニメーションのストリームインデックスの正当性を確認します。
        public static bool CheckStreamIndex(
            out string result,
            material_animType materialAnim,
            List<G3dStream> streams)
        {
            StreamCheckerContext context = new StreamCheckerContext(streams.Count);

            // <per_material_anim> のカーブが参照するストリームインデックスの正当性を確認する
            if (materialAnim.per_material_anim_array != null
                && materialAnim.per_material_anim_array.per_material_anim != null
                && materialAnim.per_material_anim_array.per_material_anim.Length > 0)
            {
                // shader_param_anim のカーブが参照するストリームインデックスの正当性を確認する
                var shaderAnims =
                    materialAnim.per_material_anim_array.per_material_anim.Where(
                        x =>
                        x.shader_param_anim_array != null && x.shader_param_anim_array.param_anim != null
                        && x.shader_param_anim_array.param_anim.Length > 0);
                foreach (var matAnim in shaderAnims)
                {
                    foreach (var paramAnim in matAnim.shader_param_anim_array.param_anim.Where(x => x.param_anim_target != null && x.param_anim_target.Length > 0))
                    {
                        foreach (var target in paramAnim.param_anim_target.Where(x => x.Curve != null))
                        {
                            var streamIndex = target.Curve.stream_index;
                            context.CheckStreamIndex(
                                streamIndex,
                                string.Format(
                                    "<param_anim_target component_index=\"{0}\"> in <param_anim id=\"{1}\"> in <shader_param_mat_anim mat_name=\"{2}\">",
                                    target.component_index,
                                    paramAnim.id,
                                    matAnim.mat_name));
                        }
                    }
                }
                // pattern_anim のカーブが参照するストリームインデックスの正当性を確認する
                var texPatternAnims =
                    materialAnim.per_material_anim_array.per_material_anim.Where(
                        x =>
                        x.tex_pattern_anim_array != null && x.tex_pattern_anim_array.pattern_anim != null
                        && x.tex_pattern_anim_array.pattern_anim.Length > 0);
                foreach (var matAnim in texPatternAnims)
                {
                    foreach (var target in matAnim.tex_pattern_anim_array.pattern_anim.Where(x => x.Curve != null))
                    {
                        var streamIndex = target.Curve.stream_index;
                        context.CheckStreamIndex(
                            streamIndex,
                            string.Format("<pattern_anim sampler_name=\"{0}\"> in <tex_pattern_mat_anim mat_name=\"{1}\">",
                                target.sampler_name, matAnim.mat_name));
                    }
                }

                // material_visibility_anim のカーブが参照するストリームインデックスの正当性を確認する
                var visibilityAnims =
                    materialAnim.per_material_anim_array.per_material_anim.Where(
                        x => x.material_visibility_anim != null && x.material_visibility_anim.step_curve != null);
                foreach (var visAnim in visibilityAnims)
                {
                    var streamIndex = visAnim.material_visibility_anim.step_curve.stream_index;
                    context.CheckStreamIndex(
                        streamIndex,
                        string.Format("<mat_vis_mat_anim mat_name=\"{0}\">", visAnim.mat_name));
                }
            }

            if (materialAnim.original_per_material_anim_array != null &&
                materialAnim.original_per_material_anim_array.original_per_material_anim != null)
            {
                foreach (original_material_animType orgMatAnim in
                    materialAnim.original_per_material_anim_array.original_per_material_anim)
                {
                    // <original_color_anim> のカーブが参照するストリームインデックスの正当性を確認する
                    if (orgMatAnim.original_color_anim_array != null &&
                        orgMatAnim.original_color_anim_array.original_color_anim != null)
                    {
                        foreach (var colorAnim in
                            orgMatAnim.original_color_anim_array.original_color_anim)
                        {
                            foreach (var colorTarget in
                                colorAnim.original_color_anim_target)
                            {
                                if (colorTarget.Curve == null) { continue; }

                                int streamIndex = colorTarget.Curve.stream_index;
                                context.CheckStreamIndex(
                                    streamIndex,
                                    string.Format("<original_color_anim_target target=\"{0}\"> in <original_color_anim hint=\"{1}\"> in <original_per_material_anim mat_name=\"{2}\">",
                                        colorTarget.target, colorAnim.hint, orgMatAnim.mat_name));
                            }
                        }
                    }

                    // <original_texsrt_anim> のカーブが参照するストリームインデックスの正当性を確認する
                    if (orgMatAnim.original_texsrt_anim_array != null &&
                        orgMatAnim.original_texsrt_anim_array.original_texsrt_anim != null)
                    {
                        foreach (original_texsrt_animType texsrtAnim in
                            orgMatAnim.original_texsrt_anim_array.original_texsrt_anim)
                        {
                            foreach (original_texsrt_anim_targetType texsrtTarget in
                                texsrtAnim.original_texsrt_anim_target)
                            {
                                if (texsrtTarget.Curve == null) { continue; }

                                int streamIndex = texsrtTarget.Curve.stream_index;
                                context.CheckStreamIndex(
                                    streamIndex,
                                    string.Format("<original_texsrt_anim_target target=\"{0}\"> in <original_texsrt_anim hint=\"{1}\"> in <original_material_anim mat_name=\"{2}\">",
                                        texsrtTarget.target, texsrtAnim.hint, orgMatAnim.mat_name));
                            }
                        }
                    }
                }
            }

            // ユーザーデータのストリームインデックスを確認する。
            CheckStreamIndex(materialAnim.user_data_array, context);
            // 全てのストリームが使用されているか確認する。
            context.CheckAllStreamsUsed();

            //　ログメッセージを記録する。
            result = context.LogMessage;
            // ログメッセージがあるかどうかで問題の有無を判断する。
            return result.Length == 0;
        }

        //-----------------------------------------------------------------------------
        // シェーダーパラメーターアニメーションのストリームインデックスの正当性を確認します。
        public static bool CheckStreamIndex(
            out string result,
            shader_param_animType shaderParamAnim,
            List<G3dStream> streams)
        {
            StreamCheckerContext context = new StreamCheckerContext(streams.Count);

            // <shader_param_mat_anim> のカーブが参照するストリームインデックスの正当性を確認する
            if (shaderParamAnim.shader_param_mat_anim_array != null &&
                shaderParamAnim.shader_param_mat_anim_array.shader_param_mat_anim != null)
            {
                foreach (shader_param_mat_animType matAnim in
                    shaderParamAnim.shader_param_mat_anim_array.shader_param_mat_anim)
                {
                    foreach (param_animType paramAnim in matAnim.param_anim_array.param_anim)
                    {
                        foreach (param_anim_targetType target in paramAnim.param_anim_target)
                        {
                            if (target.Curve == null) { continue; }

                            int streamIndex = target.Curve.stream_index;
                            context.CheckStreamIndex(
                                streamIndex,
                                string.Format("<param_anim_target component_index=\"{0}\"> in <param_anim id=\"{1}\"> in <shader_param_mat_anim mat_name=\"{2}\">",
                                    target.component_index, paramAnim.id, matAnim.mat_name));
                        }
                    }
                }
            }

            if (shaderParamAnim.original_material_anim_array != null &&
                shaderParamAnim.original_material_anim_array.original_material_anim != null)
            {
                foreach (original_material_animType orgMatAnim in
                    shaderParamAnim.original_material_anim_array.original_material_anim)
                {
                    // <original_color_anim> のカーブが参照するストリームインデックスの正当性を確認する
                    if (orgMatAnim.original_color_anim_array != null &&
                        orgMatAnim.original_color_anim_array.original_color_anim != null)
                    {
                        foreach (original_color_animType colorAnim in
                            orgMatAnim.original_color_anim_array.original_color_anim)
                        {
                            foreach (original_color_anim_targetType colorTarget in
                                colorAnim.original_color_anim_target)
                            {
                                if (colorTarget.Curve == null) { continue; }

                                int streamIndex = colorTarget.Curve.stream_index;
                                context.CheckStreamIndex(
                                    streamIndex,
                                    string.Format("<original_color_anim_target target=\"{0}\"> in <original_color_anim hint=\"{1}\"> in <original_material_anim mat_name=\"{2}\">",
                                        colorTarget.target, colorAnim.hint, orgMatAnim.mat_name));
                            }
                        }
                    }

                    // <original_texsrt_anim> のカーブが参照するストリームインデックスの正当性を確認する
                    if (orgMatAnim.original_texsrt_anim_array != null &&
                        orgMatAnim.original_texsrt_anim_array.original_texsrt_anim != null)
                    {
                        foreach (original_texsrt_animType texsrtAnim in
                            orgMatAnim.original_texsrt_anim_array.original_texsrt_anim)
                        {
                            foreach (original_texsrt_anim_targetType texsrtTarget in
                                texsrtAnim.original_texsrt_anim_target)
                            {
                                if (texsrtTarget.Curve == null) { continue; }

                                int streamIndex = texsrtTarget.Curve.stream_index;
                                context.CheckStreamIndex(
                                    streamIndex,
                                    string.Format("<original_texsrt_anim_target target=\"{0}\"> in <original_texsrt_anim hint=\"{1}\"> in <original_material_anim mat_name=\"{2}\">",
                                        texsrtTarget.target, texsrtAnim.hint, orgMatAnim.mat_name));
                            }
                        }
                    }
                }
            }

            // ユーザーデータのストリームインデックスを確認する。
            CheckStreamIndex(shaderParamAnim.user_data_array, context);
            // 全てのストリームが使用されているか確認する。
            context.CheckAllStreamsUsed();

            //　ログメッセージを記録する。
            result = context.LogMessage;
            // ログメッセージがあるかどうかで問題の有無を判断する。
            return result.Length == 0;
        }

        //-----------------------------------------------------------------------------
        // シェイプアニメーションのストリームインデックスの正当性を確認します。
        public static bool CheckStreamIndex(
            out string result,
            shape_animType shapeAnim,
            List<G3dStream> streams)
        {
            StreamCheckerContext context = new StreamCheckerContext(streams.Count);

            // <shape_anim_target> のカーブが参照するストリームインデックスの正当性を確認する
            if (shapeAnim.vertex_shape_anim_array != null &&
                shapeAnim.vertex_shape_anim_array.vertex_shape_anim != null)
            {
                foreach (vertex_shape_animType vertexShapeAnim in
                    shapeAnim.vertex_shape_anim_array.vertex_shape_anim)
                {
                    foreach (shape_anim_targetType target in
                        vertexShapeAnim.shape_anim_target)
                    {
                        if (target.Curve == null) { continue; }

                        int streamIndex = target.Curve.stream_index;
                        context.CheckStreamIndex(
                            streamIndex,
                            string.Format("<shape_anim_target key_shape_name=\"{0}\"> in <vertex_shape_anim shape_name=\"{1}\">",
                                target.key_shape_name, vertexShapeAnim.shape_name));
                    }
                }
            }

            // ユーザーデータのストリームインデックスを確認する。
            CheckStreamIndex(shapeAnim.user_data_array, context);
            // 全てのストリームが使用されているか確認する。
            context.CheckAllStreamsUsed();

            //　ログメッセージを記録する。
            result = context.LogMessage;
            // ログメッセージがあるかどうかで問題の有無を判断する。
            return result.Length == 0;
        }

        //-----------------------------------------------------------------------------
        // テクスチャパターンアニメーションのストリームインデックスの正当性を確認します。
        public static bool CheckStreamIndex(
            out string result,
            tex_pattern_animType texPatternAnim,
            List<G3dStream> streams)
        {
            StreamCheckerContext context = new StreamCheckerContext(streams.Count);

            // <mat_vis_mat_anim> のカーブが参照するストリームインデックスの正当性を確認する
            if (texPatternAnim.tex_pattern_mat_anim_array != null &&
                texPatternAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim != null)
            {
                foreach (tex_pattern_mat_animType matAnim in
                    texPatternAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim)
                {
                    foreach (pattern_anim_targetType target in matAnim.pattern_anim_target)
                    {
                        if (target.Curve == null) { continue; }

                        int streamIndex = target.Curve.stream_index;
                        context.CheckStreamIndex(
                            streamIndex,
                            string.Format("<pattern_anim_target sampler_name=\"{0}\"> in <tex_pattern_mat_anim mat_name=\"{1}\">",
                                target.sampler_name, matAnim.mat_name));
                    }
                }
            }

            // ユーザーデータのストリームインデックスを確認する。
            CheckStreamIndex(texPatternAnim.user_data_array, context);
            // 全てのストリームが使用されているか確認する。
            context.CheckAllStreamsUsed();

            //　ログメッセージを記録する。
            result = context.LogMessage;
            // ログメッセージがあるかどうかで問題の有無を判断する。
            return result.Length == 0;
        }
    }
}
