﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.ServiceModel.Configuration;

using Microsoft.Scripting.Utils;

using nw.g3d.iflib.Resources;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    internal class CombineInfo
    {
        public CombineInfo(string filename, material_animType matAnim, List<G3dStream> streams)
        {
            Filename = filename;
            MatAnim = matAnim;
            Streams = streams;
        }

        public string Filename { get; set; }
        public material_animType MatAnim { get; set; }
        public List<G3dStream> Streams { get; set; }
    }

    public class CombineSettings
    {
        public bool IgnoreFrameLoopUnmatch { get; set; }
        public bool IgnoreAnimationConflict { get; set; }
        public string XsdBasePath { get; set; }
        public string MessageLog { get; set; }
    }

    public static class IfMaterialAnimCombineUtility
    {
        public static bool Combine(material_animType baseAnim, List<G3dStream> baseStreams, string[] combineFilePaths, CombineSettings combineSettings)
        {
            if (combineSettings == null)
            {
                combineSettings = new CombineSettings();
            }
            combineSettings.MessageLog = string.Empty;
            // 統合対象のアニメーションのファイルパスとmaterial_animTypeのペア。
            var combineAnims = PrepareCombineAnims(combineFilePaths, combineSettings);

            return Combine(baseAnim, baseStreams, combineAnims.ToArray(), combineSettings);
        }

        internal static bool Combine(material_animType baseAnim, List<G3dStream> baseStreams, CombineInfo[] combineAnims, CombineSettings combineSettings)
        {
            // 各アニメーションのフレーム・ループが一致するかどうかチェック
            if (!combineSettings.IgnoreFrameLoopUnmatch)
            {
                var info = baseAnim.material_anim_info;
                foreach (var animTuple in combineAnims)
                {
                    var anim = animTuple.MatAnim;
                    if (anim.material_anim_info.frame_count != info.frame_count
                        || anim.material_anim_info.frame_resolution != info.frame_resolution
                        || anim.material_anim_info.loop != info.loop)
                    {
                        combineSettings.MessageLog += string.Format(StringResource.IfMaterialAnimCombineUtility_Combine_FrameLoopUnmatch, animTuple.Filename);
                        return false;
                    }
                }
            }

            // ベースのマテリアルアニメーションに各マテリアルアニメーションを統合
            var newStreams = new List<G3dStream>();
            if (baseStreams != null)
            {
                newStreams.AddRange(baseStreams);
            }
            var combinedMaterialAnims = new List<per_material_animType>();
            if (baseAnim.HasMaterials())
            {
                combinedMaterialAnims.AddRange(baseAnim.GetPerMaterialAnims());
            }
            var combinedOrgMaterials = new List<original_material_animType>();
            if (baseAnim.HasOriginalMaterials())
            {
                combinedOrgMaterials.AddRange(baseAnim.GetOriginalMaterials());
            }
            var combinedTexPatterns = new Dictionary<string, int>();
            if (baseAnim.HasTexturePattern())
            {
                for (var i = 0; i < baseAnim.tex_pattern_array.tex_pattern.Length; i++)
                {
                    var pattern = baseAnim.tex_pattern_array.tex_pattern[i];
                    combinedTexPatterns.Add(pattern.tex_name, i);
                }
            }

            foreach (var combineInfo in combineAnims)
            {
                var newAnim = combineInfo.MatAnim;
                var streamOffset = newStreams.Count;

                if (newAnim.HasMaterials())
                {
                    // oldMaterialAnim から newMaterialAnim にコピーしたテクスチャパターンの
                    // 新しいインデックスを記録した配列。
                    var newTexPatternIdx = new int[0];

                    // テクスチャパターンを統合する
                    if (newAnim.HasTexturePattern())
                    {
                        newTexPatternIdx = new int[newAnim.tex_pattern_array.tex_pattern.Length];
                        for (var i = 0; i < newAnim.tex_pattern_array.tex_pattern.Length; i++)
                        {
                            int index;
                            var pattern = newAnim.tex_pattern_array.tex_pattern[i];
                            if (combinedTexPatterns.TryGetValue(pattern.tex_name, out index))
                            {
                                newTexPatternIdx[i] = index;
                            }
                            else
                            {
                                index = combinedTexPatterns.Count;
                                newTexPatternIdx[i] = index;
                                combinedTexPatterns.Add(pattern.tex_name, index);
                            }
                        }
                    }

                    foreach (var newMat in newAnim.GetPerMaterialAnims())
                    {
                        // 同名のマテリアルがない場合は作成
                        var baseMat = combinedMaterialAnims.FirstOrDefault(x => x.mat_name == newMat.mat_name);
                        if (baseMat == null)
                        {
                            baseMat = new per_material_animType() { mat_name = newMat.mat_name };
                            combinedMaterialAnims.Add(baseMat);
                        }

                        // シェーダーパラメーターアニメーションの統合
                        if (newMat.HasShaderParameterAnim())
                        {
                            if (baseMat.HasShaderParameterAnim())
                            {
                                foreach (var paramAnim in newMat.shader_param_anim_array.param_anim)
                                {
                                    var baseParamAnim = baseMat.shader_param_anim_array.param_anim.FirstOrDefault(x => x.id == paramAnim.id);

                                    if (baseParamAnim != null)
                                    {
                                        combineSettings.MessageLog += string.Format(
                                            StringResource.IfMaterialAnimCombineUtility_Combine_ConflictShaderParamAnim,
                                            combineInfo.Filename,
                                            newMat.mat_name, paramAnim.id);
                                        if (!combineSettings.IgnoreAnimationConflict)
                                        {
                                            return false;
                                        }
                                    }
                                    else
                                    {
                                        var combinedAnims = new List<param_animType>(baseMat.shader_param_anim_array.param_anim);
                                        combinedAnims.Add(paramAnim);
                                        baseMat.shader_param_anim_array.param_anim = combinedAnims.ToArray();
                                        foreach (var target in paramAnim.param_anim_target)
                                        {
                                            if (target.Curve == null)
                                            {
                                                continue;
                                            }
                                            var idx = target.Curve.stream_index;
                                            target.Curve.stream_index = newStreams.Count;
                                            newStreams.Add(combineInfo.Streams[idx]);
                                        }
                                    }
                                }
                            }
                            else
                            {
                                baseMat.shader_param_anim_array = newMat.shader_param_anim_array;
                                foreach (var paramAnim in newMat.shader_param_anim_array.param_anim)
                                {
                                    foreach (var target in paramAnim.param_anim_target)
                                    {
                                        if (target.Curve == null)
                                        {
                                            continue;
                                        }
                                        var idx = target.Curve.stream_index;
                                        target.Curve.stream_index = newStreams.Count;
                                        newStreams.Add(combineInfo.Streams[idx]);
                                    }
                                }
                            }
                        }

                        // テクスチャパターンアニメーションの統合
                        if (newMat.HasTexturePatternAnim())
                        {
                            if (baseMat.HasTexturePatternAnim())
                            {
                                foreach (var patternAnim in newMat.tex_pattern_anim_array.pattern_anim)
                                {
                                    var basePatternAnim = baseMat.tex_pattern_anim_array.pattern_anim.FirstOrDefault(x => x.sampler_name == patternAnim.sampler_name);
                                    if (basePatternAnim != null)
                                    {
                                        combineSettings.MessageLog += string.Format(
                                            StringResource.IfMaterialAnimCombineUtility_Combine_ConflictTexPatternAnim,
                                            combineInfo.Filename, newMat.mat_name, patternAnim.sampler_name);
                                        if (!combineSettings.IgnoreAnimationConflict)
                                        {
                                            return false;
                                        }
                                    }
                                    else
                                    {
                                        var combinedAnims = new List<pattern_anim_targetType>(baseMat.tex_pattern_anim_array.pattern_anim);
                                        combinedAnims.Add(patternAnim);
                                        baseMat.tex_pattern_anim_array.pattern_anim = combinedAnims.ToArray();
                                        if (patternAnim.Curve != null)
                                        {
                                            var curve = patternAnim.Curve;
                                            var idx = curve.stream_index;
                                            curve.stream_index = newStreams.Count;
                                            // ストリームのテクスチャパターンのインデックスを置きかえる。
                                            var newStream = combineInfo.Streams[idx];
                                            ReorderTexPatternAnimCurve(newStream, newTexPatternIdx);
                                            newStreams.Add(newStream);
                                        }
                                    }
                                }
                            }
                            else
                            {
                                baseMat.tex_pattern_anim_array = newMat.tex_pattern_anim_array;
                                foreach (var patternAnim in newMat.tex_pattern_anim_array.pattern_anim.Where(x => x.Curve != null))
                                {
                                    var curve = patternAnim.Curve;
                                    var idx = curve.stream_index;
                                    curve.stream_index = newStreams.Count;
                                    // ストリームのテクスチャパターンのインデックスを置きかえる。
                                    var newStream = combineInfo.Streams[idx];
                                    ReorderTexPatternAnimCurve(newStream, newTexPatternIdx);
                                    newStreams.Add(newStream);
                                }
                            }
                        }

                        // マテリアルビジビリティアニメーションの統合
                        if (newMat.HasVisibilityAnim())
                        {
                            if (baseMat.HasVisibilityAnim())
                            {
                                combineSettings.MessageLog += string.Format(
                                    StringResource.IfMaterialAnimCombineUtility_Combine_ConflictMatVisibilityAnim,
                                    combineInfo.Filename, newMat.mat_name);
                                if (!combineSettings.IgnoreAnimationConflict)
                                {
                                    return false;
                                }
                            }
                            else
                            {
                                baseMat.material_visibility_anim = newMat.material_visibility_anim;
                                if (newMat.material_visibility_anim.step_curve != null)
                                {
                                    var idx = newMat.material_visibility_anim.step_curve.stream_index;
                                    newMat.material_visibility_anim.step_curve.stream_index = newStreams.Count;
                                    newStreams.Add(combineInfo.Streams[idx]);
                                }
                            }
                        }
                    }
                }

                if (newAnim.HasOriginalMaterials())
                {
                    foreach (var newOrgMat in newAnim.GetOriginalMaterials())
                    {
                        // 同名のオリジナルマテリアルがない場合は作成
                        var baseOrgMat = combinedOrgMaterials.FirstOrDefault(x => x.mat_name == newOrgMat.mat_name);
                        if (baseOrgMat == null)
                        {
                            baseOrgMat = new original_material_animType() { mat_name = newOrgMat.mat_name };
                            combinedOrgMaterials.Add(baseOrgMat);
                        }

                        if (newOrgMat.HasOriginalColorAnim())
                        {
                            if (baseOrgMat.HasOriginalColorAnim())
                            {
                                foreach (var colorAnim in newOrgMat.original_color_anim_array.original_color_anim)
                                {
                                    var baseColorAnim =
                                        baseOrgMat.original_color_anim_array.original_color_anim.FirstOrDefault(x => x.hint == colorAnim.hint);
                                    if (baseColorAnim != null)
                                    {
                                        combineSettings.MessageLog += string.Format(
                                            StringResource.IfMaterialAnimCombineUtility_Combine_ConflictOriginalColorAnim,
                                            combineInfo.Filename,
                                            newOrgMat.mat_name, colorAnim.hint);
                                        if (!combineSettings.IgnoreAnimationConflict)
                                        {
                                            return false;
                                        }
                                    }
                                    else
                                    {
                                        var combinedAnims = new List<original_color_animType>(baseOrgMat.original_color_anim_array.original_color_anim);
                                        combinedAnims.Add(colorAnim);
                                        baseOrgMat.original_color_anim_array.original_color_anim = combinedAnims.ToArray();
                                        foreach (var target in colorAnim.original_color_anim_target)
                                        {
                                            if (target.Curve == null)
                                            {
                                                continue;
                                            }
                                            var idx = target.Curve.stream_index;
                                            target.Curve.stream_index = newStreams.Count;
                                            newStreams.Add(combineInfo.Streams[idx]);
                                        }
                                    }
                                }
                            }
                            else
                            {
                                baseOrgMat.original_color_anim_array = newOrgMat.original_color_anim_array;
                                foreach (var colorAnim in newOrgMat.original_color_anim_array.original_color_anim)
                                {
                                    foreach (var target in colorAnim.original_color_anim_target)
                                    {
                                        if (target.Curve == null)
                                        {
                                            continue;
                                        }
                                        var idx = target.Curve.stream_index;
                                        target.Curve.stream_index = newStreams.Count;
                                        newStreams.Add(combineInfo.Streams[idx]);
                                    }
                                }
                            }
                        }

                        if (newOrgMat.HasOriginalTexsrtAnim())
                        {
                            if (baseOrgMat.HasOriginalTexsrtAnim())
                            {
                                foreach (var texsrtAnim in newOrgMat.original_texsrt_anim_array.original_texsrt_anim)
                                {
                                    var baseTexsrtAnim =
                                        baseOrgMat.original_texsrt_anim_array.original_texsrt_anim.FirstOrDefault(x => x.hint == texsrtAnim.hint);
                                    if (baseTexsrtAnim != null)
                                    {
                                        combineSettings.MessageLog += string.Format(
                                            StringResource.IfMaterialAnimCombineUtility_Combine_ConflictOriginalTexSRTAnim,
                                            combineInfo.Filename,
                                            newOrgMat.mat_name, texsrtAnim.hint);
                                        if (!combineSettings.IgnoreAnimationConflict)
                                        {
                                            return false;
                                        }
                                    }
                                    else
                                    {
                                        var combinedAnims = new List<original_texsrt_animType>(baseOrgMat.original_texsrt_anim_array.original_texsrt_anim);
                                        combinedAnims.Add(texsrtAnim);
                                        baseOrgMat.original_texsrt_anim_array.original_texsrt_anim = combinedAnims.ToArray();
                                        foreach (var target in texsrtAnim.original_texsrt_anim_target)
                                        {
                                            if (target.Curve == null)
                                            {
                                                continue;
                                            }
                                            var idx = target.Curve.stream_index;
                                            target.Curve.stream_index = newStreams.Count;
                                            newStreams.Add(combineInfo.Streams[idx]);
                                        }
                                    }
                                }
                            }
                            else
                            {
                                baseOrgMat.original_texsrt_anim_array = newOrgMat.original_texsrt_anim_array;
                                foreach (var texsrtAnim in newOrgMat.original_texsrt_anim_array.original_texsrt_anim)
                                {
                                    foreach (var target in texsrtAnim.original_texsrt_anim_target)
                                    {
                                        if (target.Curve == null)
                                        {
                                            continue;
                                        }
                                        var idx = target.Curve.stream_index;
                                        target.Curve.stream_index = newStreams.Count;
                                        newStreams.Add(combineInfo.Streams[idx]);
                                    }
                                }
                            }
                        }
                    }
                }
            }

            baseAnim.tex_pattern_array = null;
            if (combinedTexPatterns.Any())
            {
                baseAnim.tex_pattern_array = new tex_pattern_arrayType();
                baseAnim.tex_pattern_array.tex_pattern = combinedTexPatterns.Select(
                    x => new tex_patternType()
                    {
                        pattern_index = x.Value,
                        tex_name = x.Key
                    }).ToArray();
                baseAnim.tex_pattern_array.UpdateHint();
            }
            else
            {
                baseAnim.tex_pattern_array = null;
            }

            if (combinedMaterialAnims.Any())
            {
                combinedMaterialAnims.Sort((l, r) => string.CompareOrdinal(l.mat_name, r.mat_name));
                baseAnim.per_material_anim_array = new per_material_anim_arrayType();
                baseAnim.per_material_anim_array.per_material_anim = combinedMaterialAnims.ToArray();
                baseAnim.per_material_anim_array.UpdateHint();
            }
            else
            {
                baseAnim.per_material_anim_array = null;
            }

            if (combinedOrgMaterials.Any())
            {
                combinedOrgMaterials.Sort((l, r) => string.CompareOrdinal(l.mat_name, r.mat_name));
                baseAnim.original_per_material_anim_array = new original_per_material_anim_arrayType();
                baseAnim.original_per_material_anim_array.original_per_material_anim = combinedOrgMaterials.ToArray();
                baseAnim.original_per_material_anim_array.UpdateHint();
            }
            else
            {
                baseAnim.original_per_material_anim_array = null;
            }

            baseStreams.Clear();
            baseStreams.AddRange(newStreams);

            StreamUtility.SortStream(baseAnim, baseStreams);

            return true;
        }

        private static void ReorderTexPatternAnimCurve(G3dStream newStream, int[] newTexPatternIdx)
        {
            var numKey = newStream.FloatData.Count / newStream.column;
            for (var iKey = 0; iKey < numKey; iKey++)
            {
                //float time = newStream.FloatData[iKey * newStream.column];
                var value = (int)newStream.FloatData[iKey * newStream.column + 1];
                value = newTexPatternIdx[value];
                newStream.FloatData[iKey * newStream.column + 1] = (float)value;
            }
        }

        private static List<CombineInfo> PrepareCombineAnims(string[] combineFilePaths, CombineSettings combineSettings)
        {
            var combineAnims = new List<CombineInfo>();

            foreach (var path in combineFilePaths)
            {
                if (!File.Exists(path))
                {
                    combineSettings.MessageLog += string.Format(StringResource.IfMaterialAnimCombineUtility_PrepareCombineAnims_FileNotFound, path);
                    return null;
                }
                var streams = new List<G3dStream>();
                // 各種アニメーションをマテリアルアニメーションに変換してから登録する
                if (G3dPath.IsMaterialAnimPath(path))
                {
                    var nwif = IfReadUtility.Read(streams, path, combineSettings.XsdBasePath);
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(nwif != null && nwif.Item is material_animType);
                    combineAnims.Add(new CombineInfo(path, nwif.Item as material_animType, streams));
                }
                else if (G3dPath.IsShaderParamAnimGroupPath(path))
                {
                    var nwif = IfReadUtility.Read(streams, path, combineSettings.XsdBasePath);
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(nwif != null && nwif.Item is shader_param_animType);
                    var matAnim = ConvertToMaterialAnim(nwif.Item as shader_param_animType, streams);
                    combineAnims.Add(new CombineInfo(path, matAnim, streams));
                }
                else if (G3dPath.IsMatVisibilityAnimPath(path))
                {
                    var nwif = IfReadUtility.Read(streams, path, combineSettings.XsdBasePath);
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(nwif != null && nwif.Item is mat_visibility_animType);
                    var matAnim = ConvertToMaterialAnim(nwif.Item as mat_visibility_animType, streams);
                    combineAnims.Add(new CombineInfo(path, matAnim, streams));
                }
                else if (G3dPath.IsTexPatternAnimPath(path))
                {
                    var nwif = IfReadUtility.Read(streams, path, combineSettings.XsdBasePath);
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(nwif != null && nwif.Item is tex_pattern_animType);
                    var matAnim = ConvertToMaterialAnim(nwif.Item as tex_pattern_animType, streams);
                    combineAnims.Add(new CombineInfo(path, matAnim, streams));
                }
                else
                {
                    combineSettings.MessageLog += string.Format(StringResource.IfMaterialAnimCombineUtility_PrepareCombineAnims_InvalidFileType, path);
                    return null;
                }
            }
            return combineAnims;
        }

        #region マテリアルアニメーションの拡張メソッド
        // マテリアルアニメーションの拡張メソッド
        public static bool HasMaterials(this material_animType matAnim)
        {
            return matAnim != null
                && matAnim.per_material_anim_array != null
                && matAnim.per_material_anim_array.per_material_anim != null
                && matAnim.per_material_anim_array.per_material_anim.Length > 0;
        }

        public static per_material_animType[] GetPerMaterialAnims(this material_animType matAnim)
        {
            return matAnim.HasMaterials() ? matAnim.per_material_anim_array.per_material_anim : new per_material_animType[0];
        }

        public static per_material_animType GetPerMaterialAnim(this material_animType matAnim, string matName)
        {
            return matAnim.HasMaterials() ? matAnim.per_material_anim_array.per_material_anim.FirstOrDefault(x => x.mat_name == matName) : null;
        }

        public static bool HasOriginalMaterials(this material_animType matAnim)
        {
            return matAnim != null
                && matAnim.original_per_material_anim_array != null
                && matAnim.original_per_material_anim_array.original_per_material_anim != null
                && matAnim.original_per_material_anim_array.original_per_material_anim.Length > 0;
        }

        public static original_material_animType[] GetOriginalMaterials(this material_animType matAnim)
        {
            return matAnim.HasOriginalMaterials() ? matAnim.original_per_material_anim_array.original_per_material_anim : new original_material_animType[0];
        }

        public static bool HasOriginalColorAnim(this original_material_animType orgAnim)
        {
            return orgAnim != null
                && orgAnim.original_color_anim_array != null
                && orgAnim.original_color_anim_array.original_color_anim != null
                && orgAnim.original_color_anim_array.original_color_anim.Length > 0;
        }

        public static bool HasOriginalTexsrtAnim(this original_material_animType orgAnim)
        {
            return orgAnim != null
                && orgAnim.original_texsrt_anim_array != null
                && orgAnim.original_texsrt_anim_array.original_texsrt_anim != null
                && orgAnim.original_texsrt_anim_array.original_texsrt_anim.Length > 0;
        }

        public static bool HasStreams(this material_animType matAnim)
        {
            return matAnim != null
                && matAnim.stream_array != null
                && matAnim.stream_array.stream != null
                && matAnim.stream_array.stream.Length > 0;
        }

        public static streamType[] GetStreams(this material_animType matAnim)
        {
            return matAnim.HasStreams() ? matAnim.stream_array.stream : new streamType[0];
        }

        public static bool HasTexturePattern(this material_animType materialAnim)
        {
            return materialAnim.tex_pattern_array != null
                && materialAnim.tex_pattern_array.tex_pattern != null
                && materialAnim.tex_pattern_array.tex_pattern.Length > 0;
        }
        // シェーダーパラメーターアニメーションを持つか調べます。
        public static bool HasShaderParameterAnim(this material_animType materialAnim)
        {
            return materialAnim.per_material_anim_array != null
                   && materialAnim.per_material_anim_array.per_material_anim != null
                   && materialAnim.per_material_anim_array.per_material_anim.Length > 0
                   && materialAnim.per_material_anim_array.per_material_anim.Any(x => x.HasShaderParameterAnim());
        }

        public static bool HasShaderParameterAnim(this per_material_animType perMatAnim)
        {
            return perMatAnim != null
                   && perMatAnim.shader_param_anim_array != null
                   && perMatAnim.shader_param_anim_array.param_anim != null
                   && perMatAnim.shader_param_anim_array.param_anim.Length > 0;
        }

        // マテリアルビジビリティアニメーションを持つか調べます。
        public static bool HasVisibilityAnim(this material_animType materialAnim)
        {
            return materialAnim.per_material_anim_array != null
                   && materialAnim.per_material_anim_array.per_material_anim != null
                   && materialAnim.per_material_anim_array.per_material_anim.Length > 0
                   && materialAnim.per_material_anim_array.per_material_anim.Any(x => x.HasVisibilityAnim());
        }

        public static bool HasVisibilityAnim(this per_material_animType perMatAnim)
        {
            return perMatAnim != null
                   && perMatAnim.material_visibility_anim != null;
        }

        // テクスチャパターンアニメーションを持つか調べます。
        public static bool HasTexturePatternAnim(this material_animType materialAnim)
        {
            if (!materialAnim.HasTexturePattern())
            {
                return false;
            }

            return materialAnim.per_material_anim_array != null
                   && materialAnim.per_material_anim_array.per_material_anim != null
                   && materialAnim.per_material_anim_array.per_material_anim.Length > 0
                   && materialAnim.per_material_anim_array.per_material_anim.Any(x => x.HasTexturePatternAnim());
        }

        public static bool HasTexturePatternAnim(this per_material_animType perMatAnim)
        {
            return perMatAnim != null
                   && perMatAnim.tex_pattern_anim_array != null
                   && perMatAnim.tex_pattern_anim_array.pattern_anim != null
                   && perMatAnim.tex_pattern_anim_array.pattern_anim.Length > 0;
        }

        #endregion マテリアルアニメーションの拡張メソッド

        // シェーダーパラメーターアニメーションからマテリアルアニメーションに変換
        public static material_animType ConvertToMaterialAnim(shader_param_animType shaderParamAnim, List<G3dStream> streams)
        {
            if (shaderParamAnim == null)
            {
                return null;
            }

            var paramAnimInfo = shaderParamAnim.shader_param_anim_info;
            var matAnim = new material_animType
            {
                comment = shaderParamAnim.comment,
                process_log_array = shaderParamAnim.process_log_array,
                tool_data = shaderParamAnim.tool_data,
                user_tool_data = shaderParamAnim.user_tool_data,
                user_data_array = shaderParamAnim.user_data_array,
                //stream_array = shaderParamAnim.stream_array,
                material_anim_info = new material_anim_infoType
                {
                    frame_count = paramAnimInfo.frame_count,
                    frame_resolution = paramAnimInfo.frame_resolution,
                    loop = paramAnimInfo.loop,
                    bake_all = paramAnimInfo.bake_all,
                    dcc_fps = paramAnimInfo.dcc_fps,
                    dcc_start_frame = paramAnimInfo.dcc_start_frame,
                    dcc_end_frame = paramAnimInfo.dcc_end_frame,
                    dcc_preset = paramAnimInfo.dcc_preset,
                    bake_tolerance_color = paramAnimInfo.bake_tolerance_color,
                    bake_tolerance_tex_scale = paramAnimInfo.bake_tolerance_tex_scale,
                    bake_tolerance_tex_rotate = paramAnimInfo.bake_tolerance_tex_rotate,
                    bake_tolerance_tex_translate = paramAnimInfo.bake_tolerance_tex_translate,
                    quantize_tolerance_tex_scale = paramAnimInfo.quantize_tolerance_tex_scale,
                    quantize_tolerance_tex_rotate = paramAnimInfo.quantize_tolerance_tex_rotate,
                    quantize_tolerance_tex_translate = paramAnimInfo.quantize_tolerance_tex_translate,
                }
            };

            if (shaderParamAnim.shader_param_mat_anim_array != null
                && shaderParamAnim.shader_param_mat_anim_array.shader_param_mat_anim != null
                && shaderParamAnim.shader_param_mat_anim_array.shader_param_mat_anim.Length != 0)
            {
                // シェーダーパラメーターアニメーションをコピー
                var paramAnims = shaderParamAnim.shader_param_mat_anim_array.shader_param_mat_anim;
                var materialAnims = new List<per_material_animType>();
                foreach (var paramAnim in paramAnims)
                {
                    var perMatAnim = new per_material_animType
                    {
                        index = paramAnim.index,
                        index_hint = paramAnim.index_hint,
                        mat_name = paramAnim.mat_name
                    };
                    materialAnims.Add(perMatAnim);

                    if (paramAnim.param_anim_array == null || paramAnim.param_anim_array.param_anim == null
                        || paramAnim.param_anim_array.param_anim.Length == 0)
                    {
                        continue;
                    }

                    perMatAnim.shader_param_anim_array = new shader_param_anim_arrayType
                    {
                        length = paramAnim.param_anim_array.param_anim.Length,
                        param_anim = paramAnim.param_anim_array.param_anim
                    };
                }
                matAnim.per_material_anim_array = new per_material_anim_arrayType
                {
                    per_material_anim = materialAnims.ToArray(),
                    length = materialAnims.Count
                };
                matAnim.per_material_anim_array.UpdateHint();
            }

            // オリジナルアニメーションをコピー
            if (shaderParamAnim.original_material_anim_array != null
                && shaderParamAnim.original_material_anim_array.original_material_anim != null
                && shaderParamAnim.original_material_anim_array.original_material_anim.Length != 0)
            {
                matAnim.original_per_material_anim_array = new original_per_material_anim_arrayType
                {
                    length = shaderParamAnim.original_material_anim_array.length,
                    original_per_material_anim =
                        shaderParamAnim.original_material_anim_array.original_material_anim
                };
                matAnim.original_per_material_anim_array.UpdateHint();
            }

            return matAnim;
        }

        // テクスチャパターンアニメーションからマテリアルアニメーションに変換
        public static material_animType ConvertToMaterialAnim(tex_pattern_animType texPatAnim, List<G3dStream> streams)
        {
            if (texPatAnim == null)
            {
                return null;
            }
            var patternAnimInfo = texPatAnim.tex_pattern_anim_info;
            var matAnim = new material_animType
            {
                comment = texPatAnim.comment,
                process_log_array = texPatAnim.process_log_array,
                tex_pattern_array = texPatAnim.tex_pattern_array,
                tool_data = texPatAnim.tool_data,
                user_tool_data = texPatAnim.user_tool_data,
                user_data_array = texPatAnim.user_data_array,
                //stream_array = texPatAnim.stream_array,
                material_anim_info = new material_anim_infoType
                {
                    frame_count = patternAnimInfo.frame_count,
                    frame_resolution = patternAnimInfo.frame_resolution,
                    loop = patternAnimInfo.loop,
                    bake_all = patternAnimInfo.bake_all,
                    dcc_fps = patternAnimInfo.dcc_fps,
                    dcc_start_frame = patternAnimInfo.dcc_start_frame,
                    dcc_end_frame = patternAnimInfo.dcc_end_frame,
                    dcc_preset = patternAnimInfo.dcc_preset,
                    bake_tolerance_color = 0.001f,
                    bake_tolerance_tex_scale = 0.01f,
                    bake_tolerance_tex_rotate = 0.1f,
                    bake_tolerance_tex_translate = 0.01f,
                    quantize_tolerance_tex_scale = 0.01f,
                    quantize_tolerance_tex_rotate = 0.2f,
                    quantize_tolerance_tex_translate = 0.01f,
                }
            };
            if (texPatAnim.tex_pattern_mat_anim_array == null
                || texPatAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim == null
                || texPatAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim.Length == 0)
            {
                return matAnim;
            }

            //テクスチャパターンアニメーションをコピー
            var patternAnims = texPatAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim;
            var materialAnims = new List<per_material_animType>();
            foreach (var patternAnim in patternAnims)
            {
                var perMatAnim = new per_material_animType
                {
                    index = patternAnim.index,
                    index_hint = patternAnim.index_hint,
                    mat_name = patternAnim.mat_name
                };
                materialAnims.Add(perMatAnim);

                if (patternAnim.pattern_anim_target == null
                    || patternAnim.pattern_anim_target.Length == 0)
                {
                    continue;
                }

                perMatAnim.tex_pattern_anim_array = new tex_pattern_anim_arrayType()
                {
                    length = patternAnim.pattern_anim_target.Length,
                    pattern_anim = patternAnim.pattern_anim_target
                };
            }
            matAnim.per_material_anim_array = new per_material_anim_arrayType
            {
                per_material_anim = materialAnims.ToArray(),
                length = materialAnims.Count
            };
            matAnim.per_material_anim_array.UpdateHint();

            return matAnim;
        }

        // マテリアルビジビリティアニメーションからマテリアルアニメーションに変換
        public static material_animType ConvertToMaterialAnim(mat_visibility_animType matVisAnim, List<G3dStream> streams)
        {
            if (matVisAnim == null)
            {
                return null;
            }
            var visAnimInfo = matVisAnim.mat_visibility_anim_info;
            var matAnim = new material_animType
            {
                comment = matVisAnim.comment,
                process_log_array = matVisAnim.process_log_array,
                tool_data = matVisAnim.tool_data,
                user_tool_data = matVisAnim.user_tool_data,
                user_data_array = matVisAnim.user_data_array,
                //stream_array = matVisAnim.stream_array,
                material_anim_info = new material_anim_infoType
                {
                    frame_count = visAnimInfo.frame_count,
                    frame_resolution = visAnimInfo.frame_resolution,
                    loop = visAnimInfo.loop,
                    bake_all = visAnimInfo.bake_all,
                    dcc_fps = visAnimInfo.dcc_fps,
                    dcc_start_frame = visAnimInfo.dcc_start_frame,
                    dcc_end_frame = visAnimInfo.dcc_end_frame,
                    dcc_preset = visAnimInfo.dcc_preset,
                    bake_tolerance_color = 0.001f,
                    bake_tolerance_tex_scale = 0.01f,
                    bake_tolerance_tex_rotate = 0.1f,
                    bake_tolerance_tex_translate = 0.01f,
                    quantize_tolerance_tex_scale = 0.01f,
                    quantize_tolerance_tex_rotate = 0.2f,
                    quantize_tolerance_tex_translate = 0.01f,
                }
            };
            if (matVisAnim.mat_vis_mat_anim_array == null
                || matVisAnim.mat_vis_mat_anim_array.mat_vis_mat_anim == null
                || matVisAnim.mat_vis_mat_anim_array.mat_vis_mat_anim.Length == 0)
            {
                return matAnim;
            }

            // マテリアルビジビリティアニメーションをコピー
            var visAnims = matVisAnim.mat_vis_mat_anim_array.mat_vis_mat_anim;
            var materialAnims = new List<per_material_animType>();
            foreach (var visAnim in visAnims)
            {
                var perMatAnim = new per_material_animType
                {
                    index = visAnim.index,
                    index_hint = visAnim.index_hint,
                    mat_name = visAnim.mat_name
                };
                materialAnims.Add(perMatAnim);
                perMatAnim.material_visibility_anim = new material_visibility_animType()
                {
                    base_value = visAnim.base_value,
                    step_curve = visAnim.step_curve
                };
            }
            matAnim.per_material_anim_array = new per_material_anim_arrayType
            {
                per_material_anim = materialAnims.ToArray(),
                length = materialAnims.Count
            };
            matAnim.per_material_anim_array.UpdateHint();

            return matAnim;
        }

        public static bool CheckCombinePaths(string[] paths, out string errorPath)
        {
            if (paths != null)
            {
                foreach (var path in paths)
                {
                    if (!File.Exists(path) || !(G3dPath.IsMaterialAnimPath(path) || G3dPath.IsMaterialAnimConvertiblePath(path)))
                    {
                        errorPath = path;
                        return false;
                    }
                }
            }

            errorPath = null;
            return true;
        }
    }
}
