﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq.Expressions;
using System.Linq;
using System.Text;
using System.Diagnostics;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib.Resources;

namespace nw.g3d.iflib
{
    public static class IfShaderAssignUtility
    {
        /// <summary>
        /// シェーダー割り当てとシェーディングモデルの比較結果
        /// </summary>
        public enum CompareResult
        {
            /// <summary>
            /// 未割当のものがある
            /// サンプラ割り当てと頂点属性割り当て用
            /// </summary>
            NotFullyAssigned,
            /// <summary>
            /// 順番の相違
            /// </summary>
            NotSameOrder,
            /// <summary>
            /// それ以外の相違
            /// </summary>
            Conflict,
            /// <summary>
            /// 問題なし
            /// </summary>
            Ok,
        }

        /// <summary>
        /// リビジョンの比較
        /// </summary>
        public static CompareResult CompareRevision(int revision, shading_modelType shading_model)
        {
            return revision == shading_model.revision ? CompareResult.Ok : CompareResult.Conflict;
        }

        /// <summary>
        /// シェーダー割り当ての新規割り当て
        /// </summary>
        public static void AssignShaderAssign(
            modelType model,
            string shaderArchive,
            shading_modelType shading_model,
            bool force,
            bool keepExistingValues)
        {
            if (model.material_array == null)
            {
                return;
            }

            foreach (var material in model.material_array.GetItems())
            {
                if (force || material.shader_assign == null)
                {
                    UpdateShaderAssign(
                        model,
                        material,
                        shaderArchive,
                        shading_model,
                        initialize: !keepExistingValues,
                        allowEmptyString: false);
                }
            }
        }

        public static void AssignShaderAssign(
            modelType model,
            string[] targetMaterialNames,
            string shaderArchive,
            shading_modelType shading_model,
            bool force,
            bool keepExistingValues)
        {
            if (model.material_array == null)
            {
                return;
            }

            foreach (var material in model.material_array.GetItems())
            {
                if (!targetMaterialNames.Any(materialName => IsMatchWithWildCardPattern(material.name, materialName)))
                {
                    continue;
                }

                if (force || material.shader_assign == null)
                {
                    UpdateShaderAssign(
                        model,
                        material,
                        shaderArchive,
                        shading_model,
                        initialize: !keepExistingValues,
                        allowEmptyString: false);
                }
            }
        }

        public enum InconsistentMessageType
        {
            Plan,        // 予定
            Result,        // 結果
        }

        /// <summary>
        /// マテリアルのアップデート予定項目の取得
        /// </summary>
        public static List<string> UpdateMessages(
            modelType model,
            materialType material,
            shading_modelType shading_model,
            bool ignoreOrder,
            InconsistentMessageType type = InconsistentMessageType.Plan)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(material != null);

            List<string> messages = new List<string>();

            var shaderAssign = material.shader_assign;

            if (shaderAssign != null)
            {
                // リビジョン
                messages.AddRange(GetUpdateRevisionMessages(shaderAssign.revision, shading_model, type));

                // オプション
                messages.AddRange(IfShaderOptionUtility.GetUpdateOptionMessages(shaderAssign.shader_option_array.GetItems(), shading_model, ignoreOrder, type));

                // サンプラ割り当て
                messages.AddRange(IfSamplerAssignUtility.GetUpdateSamplerMessages(shaderAssign.sampler_assign_array.GetItems(), shading_model, material.sampler_array, ignoreOrder, type));

                // パラメーター
                var original_material = GetOriginalMaterial(model, material);
                messages.AddRange(IfShaderParameterUtility.GetUpdateParameterMessage(shaderAssign.shader_param_array.GetItems(), original_material, shading_model, ignoreOrder, type));

                // 頂点属性
                messages.AddRange(IfAttributeAssignUtility.GetUpdateAttributeAssignMessages(shaderAssign.attrib_assign_array.GetItems(), model, material.name, shading_model, ignoreOrder, type));

                // 描画情報
                messages.AddRange(IfRenderInfoUtility.GetUpdateRenderInfoMessages(shaderAssign.render_info_array.GetItems(), shading_model, ignoreOrder, type));
            }

            // 頂点属性
            var attribAssigns = material.shader_assign != null ?
                material.shader_assign.attrib_assign_array.GetItems() :
                Enumerable.Empty<attrib_assignType>();

            // 頂点バッファ
            messages.AddRange(IfVertexBufferUtility.GetUpdateVertexBufferMessages(model, material.name, attribAssigns, shading_model, type));

            return messages;
        }

        /// <summary>
        /// シェーダー割り当ての更新
        /// </summary>
        public static void UpdateShaderAssign(
            modelType model,
            materialType material,
            string shaderArchive,
            shading_modelType definition,
            bool initialize,
            bool allowEmptyString)
        {
            if (definition != null)
            {
                if (material.shader_assign == null || initialize)
                {
                    material.shader_assign = new shader_assignType();
                }

                var shader_assign = material.shader_assign;
                shader_assign.shader_archive = shaderArchive;
                shader_assign.shading_model = definition.name;

                // リビジョン
                shader_assign.revision = definition.revision;

                // オプション
                shader_assign.shader_option_array =
                    IfShaderOptionUtility.CreateOptions(
                        definition,
                        shader_assign.shader_option_array.GetItems()).ToG3dArrayElement<shader_optionType, shader_option_arrayType>();

                // サンプラ割り当て
                shader_assign.sampler_assign_array =
                    IfSamplerAssignUtility.CreateSamplerAssigns(
                        definition,
                        material.sampler_array.GetItems(),
                        shader_assign.sampler_assign_array.GetItems(),
                        true,
                        allowEmptyString).ToG3dArrayElement<sampler_assignType, sampler_assign_arrayType>();

                // パラメーター
                // fmt マテリアルへの割り当てのために model が null でも実行する。
                shader_assign.shader_param_array =
                    IfShaderParameterUtility.CreateParameters(
                        definition,
                        shader_assign.shader_param_array.GetItems(),
                        (model != null) ? GetOriginalMaterial(model, material) : null).ToG3dArrayElement<shader_paramType, shader_param_arrayType>();

                // 頂点属性割り当て
                if (model != null)
                {
                    shader_assign.attrib_assign_array =
                        IfAttributeAssignUtility.CreateAttribAssigns(
                            definition,
                            model,
                            material.name,
                            shader_assign.attrib_assign_array.GetItems(),
                            true,
                            allowEmptyString).ToG3dArrayElement<attrib_assignType, attrib_assign_arrayType>();
                }

                // 描画情報
                shader_assign.render_info_array =
                    IfRenderInfoUtility.CreateRenderInfo(
                        definition,
                        shader_assign.render_info_array.GetItems()).ToG3dArrayElement<render_infoType, render_info_arrayType>();

                // 存在しないパラメータに対するマテリアル参照情報を整理する
                var materialReference = IfToolData.GetMaterialReference(material);
                IfApplyBaseMaterialUtility.UpdateMaterialReference(materialReference, definition);
                IfToolData.SetMaterialReference(material, materialReference);
            }
            else
            {
                material.shader_assign = null;
                material.material_info.mesh_adjacency = false;
            }

            // 頂点バッファ
            if (model != null)
            {
                IfVertexBufferUtility.UpdateVertexBuffers(
                    model,
                    material,
                    definition);
            }
        }

        /// <summary>
        /// シェーダー割り当てのリビジョン変更のメッセージの取得
        /// </summary>
        internal static IEnumerable<string> GetUpdateRevisionMessages(int shaderAssignRevision, shading_modelType shading_model, InconsistentMessageType type)
        {
            // リビジョンの比較
            if (shaderAssignRevision < shading_model.revision)
            {
                yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Older_Material_Revision, type), shaderAssignRevision, shading_model.revision)
                    + string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Set_Revision, type), shading_model.revision);
            }
            else if (shaderAssignRevision > shading_model.revision)
            {
                yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Older_ShadingModel_Revision, type), shaderAssignRevision, shading_model.revision)
                    + string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Set_Revision, type), shading_model.revision);
            }
        }

        /// <summary>
        /// first が second の部分列になっているかを判定
        /// first second に重複が無いものとする
        /// </summary>
        internal static bool IsSubSequence<T>(IEnumerable<T> first, IEnumerable<T> second)
        {
            return first.SequenceEqual(second.Intersect(first));
        }

        /// <summary>
        /// 文字列へ変換
        /// 逆の操作をやる関数は
        /// nw.g3d.nw4f_3dif.G3dDataParser
        /// に実装されています。
        /// </summary>
        public static string MakeArrayString<T>(IEnumerable<T> values)
        {
            var sb = new StringBuilder();
            {
                foreach (T v in values)
                {
                    sb.Append(string.Format("{0}\t", v));
                }
            }
            return sb.ToString();
        }

        /// <summary>
        /// オリジナルマテリアルの取得
        /// </summary>
        internal static original_materialType GetOriginalMaterial(modelType model, materialType material)
        {
            if (model.original_material_array == null)
            {
                return null;
            }

            return model.original_material_array.original_material.FirstOrDefault(x => x.mat_name == material.name);
        }

        /// <summary>
        /// G3dArrayElement の要素を取得
        /// </summary>
        internal static IEnumerable<T> GetItems<T>(this G3dArrayElement<T> array) where T : IG3dIndexHintElement
        {
            if (array == null)
            {
                return Enumerable.Empty<T>();
            }

            return array.Items;
        }

        /// <summary>
        /// 要素の列をG3dArrayElement に変換
        /// </summary>
        internal static S ToG3dArrayElement<T, S>(this IEnumerable<T> items)
            where T : IG3dIndexHintElement
            where S : G3dArrayElement<T>, new()
        {
            if (items == null)
            {
                return null;
            }

            T[] array = items.ToArray();

            if (array.Length == 0)
            {
                return null;
            }

            return new S { Items = array };
        }

        /// <summary>
        /// 範囲から min max を取得
        /// </summary>
        public static bool TryParseIntRange(string choice, out int min, out int max)
        {
            min = 0;
            max = 0;
            if (choice.StartsWith("[") && choice.EndsWith("]"))
            {
                choice = choice.Substring(1, choice.Length - 2);
                string[] splited = choice.Split(new char[] { ',' });
                if (splited.Length == 2)
                {
                    return int.TryParse(splited[0], out min) && int.TryParse(splited[1], out max);
                }
            }

            return false;
        }

        /// <summary>
        /// 範囲から min max を取得
        /// </summary>
        public static bool TryParseFloatRange(string choice, out float min, out float max)
        {
            min = 0;
            max = 0;
            if (choice.StartsWith("[") && choice.EndsWith("]"))
            {
                choice = choice.Substring(1, choice.Length - 2);
                string[] splited = choice.Split(new char[] { ',' });
                if (splited.Length == 2)
                {
                    return float.TryParse(splited[0], out min) && float.TryParse(splited[1], out max);
                }
            }

            return false;
        }

        /// <summary>
        /// タイプごとのリソース文字列を得る
        /// </summary>
        internal static string GetResourceString<T>(Expression<Func<T>> nameExpression, InconsistentMessageType type)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(nameExpression.Body is MemberExpression);

            string str = null;
            {
                var memberExpression = nameExpression.Body as MemberExpression;
                var name = memberExpression.Member.Name;

                if (type == InconsistentMessageType.Result)
                {
                    str = StringResource.ResourceManager.GetString(name + "_Result", StringResource.Culture);
                    if (str != null)
                    {
                        return str;
                    }
                }

                str = StringResource.ResourceManager.GetString(name, StringResource.Culture);
                Nintendo.Foundation.Contracts.Assertion.Operation.True(str != null);
            }
            return str;
        }

        private static bool IsMatchWithWildCardPattern(string value, string pattern)
        {
            return System.Text.RegularExpressions.Regex.IsMatch(
                            value, ConvertWildCardToRegularExpression(pattern));
        }

        private static string ConvertWildCardToRegularExpression(string value)
        {
            return "^" + System.Text.RegularExpressions.Regex.Escape(value).Replace("\\*", ".*") + "$";
        }
    }
}
