﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public static class IfSamplerAssignUtility
    {
        /// <summary>
        /// 比較
        /// </summary>
        public static IfShaderAssignUtility.CompareResult Compare(IEnumerable<sampler_assignType> sampler_assigns, shading_modelType shadingModel, sampler_arrayType samplers)
        {
            IEnumerable<sampler_varType> sampler_vars = shadingModel.sampler_var_array.GetItems();

            // 値が不正
            var invalid = from samplerVar in sampler_vars
                          let samplerAssign = sampler_assigns.FirstOrDefault(x => x.id == samplerVar.id)
                          where samplerAssign != null && !string.IsNullOrEmpty(samplerAssign.sampler_name) &&
                                !samplers.GetItems().Any(x => x.name == samplerAssign.sampler_name)
                          select samplerAssign;
            if (invalid.Any())
            {
                return IfShaderAssignUtility.CompareResult.Conflict;
            }

            // 未割り当てで割り当て可能なものがある場合
            if (sampler_vars.Any(x =>
                !sampler_assigns.Any(y => y.id == x.id && !string.IsNullOrEmpty(y.sampler_name)) &&
                GetSamplerNameFromHint(samplers.GetItems(), x) != null))
            {
                return IfShaderAssignUtility.CompareResult.Conflict;
            }

            if (sampler_assigns.Count() != sampler_vars.Count())
            {
                if (sampler_assigns.All(x => sampler_vars.Any(y => y.id == x.id)))
                {
                    return IfShaderAssignUtility.CompareResult.NotFullyAssigned;
                }
                else
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }
            else if (!sampler_assigns.Select(x => x.id).SequenceEqual(sampler_vars.Select(x => x.id)))
            {
                if (sampler_assigns.Select(x => x.id).OrderBy(x => x).SequenceEqual(sampler_vars.Select(x => x.id).OrderBy(x => x)))
                {
                    return IfShaderAssignUtility.CompareResult.NotSameOrder;
                }
                else
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }

            return IfShaderAssignUtility.CompareResult.Ok;
        }

        /// <summary>
        /// サンプラ割り当てを生成
        /// </summary>
        public static IEnumerable<sampler_assignType> CreateSamplerAssigns(
            shading_modelType shadingModel,
            IEnumerable<samplerType> samplers,
            IEnumerable<sampler_assignType> oldAssigns,
            bool useHint,
            bool allowEmptySamplerName)
        {
            foreach (var sampler_var in shadingModel.sampler_var_array.GetItems())
            {
                // 生成
                var sampler_assign = new sampler_assignType()
                {
                    id = sampler_var.id,
                    sampler_name = string.Empty,
                };

                // 元の値が割り当て可能なら引き継ぐ
                var old = oldAssigns.FirstOrDefault(
                    x => x.id == sampler_var.id);
                bool oldName = false;
                if (old != null && (old.sampler_name == string.Empty || samplers.Any(x => x.name == old.sampler_name)))
                {
                    oldName = true;
                    sampler_assign.sampler_name = old.sampler_name;
                }

                // ヒントから割り当て
                if (useHint && sampler_var.hint != string.Empty && sampler_assign.sampler_name == string.Empty && !oldName)
                {
                    string samplerName = GetSamplerNameFromHint(samplers, sampler_var);

                    if (samplerName != null)
                    {
                        sampler_assign.sampler_name = samplerName;
                    }
                }

                // 出力
                if (allowEmptySamplerName || sampler_assign.sampler_name != string.Empty)
                {
                    yield return sampler_assign;
                }
            }
        }

        /// <summary>
        /// サンプラ割り当て変更メッセージの取得
        /// </summary>
        internal static IEnumerable<string> GetUpdateSamplerMessages(IEnumerable<sampler_assignType> samplerAssigns, shading_modelType shadingModel, sampler_arrayType samplers, bool ignoreOrder, IfShaderAssignUtility.InconsistentMessageType type)
        {
            // シェーダー定義側
            IEnumerable<sampler_varType> sampler_vars = shadingModel.sampler_var_array.GetItems();

            // 過剰分
            var over = samplerAssigns.Select(x => x.id).Except(sampler_vars.Select(x => x.id)).ToArray();
            foreach (var item in over)
            {
                yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Over_Sampler, type), item);
            }

            // 値が不正
            var invalid = from samplerVar in sampler_vars
                          let samplerAssign = samplerAssigns.FirstOrDefault(x => x.id == samplerVar.id)
                          where samplerAssign != null && !string.IsNullOrEmpty(samplerAssign.sampler_name) && !samplers.GetItems().Any(x => x.name == samplerAssign.sampler_name)
                          select new { samplerAssign, samplerVar };

            foreach (var item in invalid)
            {
                string samplerName = GetSamplerNameFromHint(samplers.GetItems(), item.samplerVar);
                string value = string.IsNullOrEmpty(samplerName) ?
                    IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Unspecified, type) :
                    string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Value, type), samplerName);
                yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Invalid_Sampler, type), item.samplerAssign.id, item.samplerAssign.sampler_name)
                    + value;
            }

            // 未割り当て分の設定
            var lack = sampler_vars.Where(x => !samplerAssigns.Any(y => y.id == x.id && !string.IsNullOrEmpty(y.sampler_name))).ToArray();
            foreach (var item in lack)
            {
                var samplerName = GetSamplerNameFromHint(samplers.GetItems(), item);

                if (samplerName != null)
                {
                    yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Lack_Sampler, type), item.id) +
                        string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Value, type), samplerName);
                }
            }

            // 順番を比較
            if (!ignoreOrder &&
                over.Length == 0 &&
                !IfShaderAssignUtility.IsSubSequence(samplerAssigns.Select(x => x.id), sampler_vars.Select(x => x.id)))
            {
                yield return IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Disorder_Sampler, type);
            }
        }

        /// <summary>
        /// ヒントをもとにサンプラを取得
        /// 適当なものがない場合は null
        /// </summary>
        public static string GetSamplerNameFromHint(IEnumerable<samplerType> samplers, sampler_varType sampler_var)
        {
            return samplers?.Where(sampler => !string.IsNullOrEmpty(sampler.hint) && sampler.hint == sampler_var.hint)
                .Select(sampler => sampler.name).FirstOrDefault();
        }
    }
}
