﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Text.RegularExpressions;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public static class IfRenderInfoUtility
    {
        /// <summary>
        /// 描画情報の比較
        /// </summary>
        public static IfShaderAssignUtility.CompareResult Compare(IEnumerable<render_infoType> renderInfos, shading_modelType shadingModel)
        {
            IEnumerable<render_info_slotType> slots = shadingModel.render_info_slot_array.GetItems();

            // 個数の比較
            if (renderInfos.Count() != slots.Count())
            {
                return IfShaderAssignUtility.CompareResult.Conflict;
            }

            // 項目の比較
            Func<render_infoType, render_info_slotType, bool> isOk =
                 (x, y) =>
                 {
                     // id と型の比較
                     if (x.name != y.name || x.type != y.Type())
                     {
                         return false;
                     }

                     var tokens = G3dDataParser.Tokenize(x.Value ?? string.Empty);

                     // 要素数が正しいか?
                     if (y.optional)
                     {
                         if (tokens.Length > y.count)
                         {
                             return false;
                         }
                     }
                     else
                     {
                         if (tokens.Length != y.count)
                         {
                             return false;
                         }
                     }

                     // 要素が choice に含まれるか
                     var contains = ChoiceContainsPredicate(y);
                     if (tokens.Any(z => !contains(z)))
                     {
                         return false;
                     }

                     return true;
                 };

            if (renderInfos.Zip(slots, isOk).Any(x => !x))
            {
                if (renderInfos.OrderBy(x => x.name).Zip(slots.OrderBy(x => x.name), isOk).All(x => x))
                {
                    return IfShaderAssignUtility.CompareResult.NotSameOrder;
                }
                else
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }

            return IfShaderAssignUtility.CompareResult.Ok;
        }

        /// <summary>
        /// render_info_slot から値を生成する。
        /// </summary>
        public static List<string> CreateRenderInfoValue(
            render_info_slotType slot,
            render_infoType old) // null でもよい
        {
            var defaults = IsDefaultInChoice(slot) ? G3dDataParser.Tokenize(slot.@default) : new string[0];
            List<string> values = defaults.Length <= slot.count ?
                                    defaults.ToList() :
                                    new List<string>();

            if (old != null)
            {
                var contains = ChoiceContainsPredicate(slot);
                values = G3dDataParser.Tokenize(old.Value ?? string.Empty)
                        .Where(x => contains(x))
                        .Take(slot.count).ToList();
            }

            // 不足分を埋める
            if (!slot.optional && values.Count < slot.count)
            {
                // default から
                for (int i = values.Count; i < slot.count; i++)
                {
                    if (defaults.Length <= i)
                    {
                        break;
                    }

                    values.Add(defaults[i]);
                }

                // choice から
                if (values.Count < slot.count && !string.IsNullOrEmpty(slot.choice))
                {
                    string value = null;
                    switch (slot.type)
                    {
                        case render_info_slot_typeType.@float:
                            {
                                float min;
                                float max;
                                if (IfShaderAssignUtility.TryParseFloatRange(slot.choice, out min, out max))
                                {
                                    value = min.ToString();
                                }
                                else
                                {
                                    value = IfShaderOptionUtility.Enum(slot.choice).FirstOrDefault();
                                }
                                break;
                            }
                        case render_info_slot_typeType.@int:
                            {
                                int min;
                                int max;
                                if (IfShaderAssignUtility.TryParseIntRange(slot.choice, out min, out max))
                                {
                                    value = min.ToString();
                                }
                                else
                                {
                                    value = IfShaderOptionUtility.Enum(slot.choice).FirstOrDefault();
                                }
                                break;
                            }
                        case render_info_slot_typeType.@string:
                            value = IfShaderOptionUtility.Enum(slot.choice).FirstOrDefault();
                            break;
                    }

                    if (value != null)
                    {
                        while (values.Count < slot.count)
                        {
                            values.Add(value);
                        }
                    }
                }
            }

            return values;
        }

        /// <summary>
        /// 描画情報の生成
        /// </summary>
        internal static IEnumerable<render_infoType> CreateRenderInfo(
            shading_modelType shadingModel,
            IEnumerable<render_infoType> oldRenderInfos)
        {
            foreach (var slot in shadingModel.render_info_slot_array.GetItems())
            {
                // 生成
                var renderInfo = new render_infoType()
                {
                    name = slot.name,
                    type = slot.Type(),
                };

                // 元の値を引き継ぐ
                var old = oldRenderInfos.FirstOrDefault(x => x.name == slot.name);
                var values = CreateRenderInfoValue(slot, old);

                Nintendo.Foundation.Contracts.Assertion.Operation.True(values.All(x => !string.IsNullOrWhiteSpace(x)));

                renderInfo.Value = IfShaderAssignUtility.MakeArrayString(values);

                // 要素数を設定
                renderInfo.count = values.Count;

                // Value は null を使う!
                if (renderInfo.Value == string.Empty)
                {
                    renderInfo.Value = null;
                }

                yield return renderInfo;
            }
        }

        private static Predicate<string> ChoiceContainsPredicate(render_info_slotType slot)
        {
            if (string.IsNullOrEmpty(slot.choice))
            {
                return x => true;
            }

            switch (slot.type)
            {
                case render_info_slot_typeType.@string:
                    {
                        var choices = IfShaderOptionUtility.Enum(slot.choice).ToArray();
                        return x => choices.Contains(x);
                    }
                case render_info_slot_typeType.@int:
                    {
                        int min;
                        int max;
                        if (IfShaderAssignUtility.TryParseIntRange(slot.choice, out min, out max))
                        {
                            return x => true;
                        }
                        var choices = IfShaderOptionUtility.Enum(slot.choice).Select(x => { int v; int.TryParse(x, out v); return v; }).ToArray();
                        return x => { int v; int.TryParse(x, out v); return choices.Contains(v); };
                    }
                case render_info_slot_typeType.@float:
                    {
                        float min;
                        float max;
                        if (IfShaderAssignUtility.TryParseFloatRange(slot.choice, out min, out max))
                        {
                            return x => true;
                        }
                        var choices = IfShaderOptionUtility.Enum(slot.choice).Select(x => { float v; float.TryParse(x, out v); return v; }).ToArray();
                        return x => { float v; float.TryParse(x, out v); return choices.Contains(v); };
                    }
            }

            throw new NotImplementedException();
        }

        /// <summary>
        /// 描画情報変更メッセージの取得
        /// </summary>
        internal static IEnumerable<string> GetUpdateRenderInfoMessages(IEnumerable<render_infoType> renderInfos, shading_modelType shadingModel, bool ignoreOrder, IfShaderAssignUtility.InconsistentMessageType type)
        {
            // シェーダー定義側
            IEnumerable<render_info_slotType> render_info_slots = shadingModel.render_info_slot_array.GetItems();

            // 過剰分
            var over = renderInfos.Select(x => x.name).Except(render_info_slots.Select(x => x.name)).ToArray();
            foreach (var item in over)
            {
                yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Over_RenderInfo, type), item);
            }

            // 不足分
            var lack = render_info_slots.Where(x => !renderInfos.Any(y => y.name == x.name)).ToArray();
            foreach (var item in lack)
            {
                var builder = new StringBuilder();
                builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Lack_RenderInfo, type), item.name);
                yield return builder.ToString();
            }

            // タイプ, 個数, 項目のチェック
            var typeOrCountErrors = from var in render_info_slots
                                    let contains = ChoiceContainsPredicate(var)
                                    from info in renderInfos
                                    let values = G3dDataParser.Tokenize(info.Value ?? string.Empty)
                                    where var.name == info.name && (var.Type() != info.type || (!var.optional && var.count != values.Length) || var.count < values.Length || values.Any(x => !contains(x)))
                                    select new { var, param = info, values, contains };
            foreach (var item in typeOrCountErrors)
            {
                if (item.var.Type() != item.param.type)
                {
                    var builder = new StringBuilder();
                    builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Type_RenderInfo, type), item.param.name, item.param.type.ToString(), item.var.type.ToString());
                    yield return builder.ToString();
                }
                else if (item.values.Any(x => !item.contains(x)))
                {
                    var builder = new StringBuilder();
                    foreach (var value in item.values.Where(x => !item.contains(x)))
                    {
                        if (builder.Length > 0)
                        {
                            builder.Append(", ");
                        }
                        builder.Append(value);
                    }
                    var tmp = item.var.choice;
                    yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_NotInChoice_RenderInfo, type), item.param.name, builder.ToString(), item.var.choice ?? string.Empty);
                }
                else if (item.var.count < item.values.Length)
                {
                    var builder = new StringBuilder();
                    for (int i = item.var.count; i < item.values.Length; i++)
                    {
                        var value = item.values[i];
                        if (builder.Length > 0)
                        {
                            builder.Append(", ");
                        }
                        builder.Append(value);
                    }
                    yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Count_RenderInfo, type), item.param.name, item.values.Length, item.var.count, builder.ToString());
                }
                else
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(!item.var.optional && item.var.count > item.values.Length);
                    var builder = new StringBuilder();
                    builder.AppendFormat(Resources.StringResource.Shader_Assign_NotOptionalCount_RenderInfo, item.param.name, item.values.Length, item.var.count);
                    var values = CreateRenderInfoValue(item.var, item.param);
                    if (values.Count == item.var.count)
                    {
                        var value = IfShaderAssignUtility.MakeArrayString(values).Replace('\t', ' ');
                        builder.AppendFormat(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Convert, type), value);
                    }
                    yield return builder.ToString();
                }
            }

            // 順番のチェック
            if (!ignoreOrder &&
                over.Length == 0 &&
                lack.Length == 0 &&
                renderInfos.Zip(render_info_slots, (x, y) => x.name == y.name).Any(x => !x))
            {
                yield return IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Disorder_RenderInfo, type);
            }
        }

        /// <summary>
        /// render_info_slot に対応する render_info_typeType の取得
        /// </summary>
        internal static render_info_typeType Type(this render_info_slotType slot)
        {
            switch (slot.type)
            {
                case render_info_slot_typeType.@float:
                    return render_info_typeType.@float;
                case render_info_slot_typeType.@int:
                    return render_info_typeType.@int;
                case render_info_slot_typeType.@string:
                    return render_info_typeType.@string;
            }

            Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
            throw new NotImplementedException();
        }

        /// <summary>
        /// render_info_slot のエラーの種類
        /// </summary>
        public enum RenderInfoSlotErrorType
        {
            /// <summary>
            /// 問題なし
            /// </summary>
            OK,

            /// <summary>
            /// 数値型の choice がパース出来ない
            /// 文字列型の choice が不正な文字を使用している
            /// </summary>
            ChoiceCannotParse,

            /// <summary>
            /// min > max
            /// </summary>
            InvalidRange,

            /// <summary>
            /// string 型で choice が重複
            /// </summary>
            DuplicateChoice,

            /// <summary>
            /// string 型で alias が重複
            /// </summary>
            DuplicateAlias,

            /// <summary>
            /// 数値型の default がパース出来ない
            /// 文字列型の default が不正な文字を使用している
            /// </summary>
            DefaultCannotParse,

            /// <summary>
            /// default に含まれる要素数が不正
            /// </summary>
            InvalidDefaultCount,

            /// <summary>
            /// default が　choice に含まれない
            /// </summary>
            DefaultNotInChoice,
        }

        private static Regex stringChoiceItemRegex = new Regex("[a-zA-Z0-9_]+");

        // [^:,]+ は選択肢
        // (:[^:,]+)? はエイリアス
        // (,[^:,]+(:[^:,]+)?)* は2つ目以降の項目
        //private static Regex stringChoicePunctuationRegex = new Regex("^([^:,]+(:[^:,]+)?(,[^:,]+(:[^:,]+)?)*)$");

        // TODO: 空白の考慮
        // [^:,]+ は選択肢
        // (:[^,]+)? はエイリアス
        // (,[^:,]+(:[^,]+)?)* は2つ目以降の項目
        private static Regex stringChoicePunctuationRegex = new Regex("^([^:,]+(:[^,]+)?(,[^:,]+(:[^,]+)?)*)$");

        private static RenderInfoSlotErrorType IsValidChoiceList(render_info_slotType slot, Predicate<string> predicate, out string[] choices)
        {
            choices = null;

            // choice の区切り文字チェック
            {
                var match = stringChoicePunctuationRegex.Match(slot.choice);
                if (!match.Success || match.Length != slot.choice.Length)
                {
                    return RenderInfoSlotErrorType.ChoiceCannotParse;
                }
            }

            choices = IfShaderOptionUtility.Enum(slot.choice);

            // choice の構成文字
            foreach (var choice in choices)
            {
                if (predicate != null)
                {
                    if (!predicate(choice))
                    {
                        return RenderInfoSlotErrorType.ChoiceCannotParse;
                    }
                }
            }

            // choice の重複
            if (choices.Distinct().Count() != choices.Length)
            {
                return RenderInfoSlotErrorType.DuplicateChoice;
            }

            // alias の重複
            var alias = IfShaderOptionUtility.Alias(slot.choice);
            if (choices.Select(x => alias.ContainsKey(x) ? alias[x] : x).Distinct().Count() != choices.Length)
            {
                return RenderInfoSlotErrorType.DuplicateAlias;
            }

            return RenderInfoSlotErrorType.OK;
        }

        /// <summary>
        /// render_info_slot のチェック
        /// </summary>
        public static RenderInfoSlotErrorType IsValidRenderInfoSlot(render_info_slotType slot)
        {
            switch (slot.type)
            {
                case render_info_slot_typeType.@string:
                    {
                        // choice のチェック
                        string[] choices = null;
                        if (!string.IsNullOrEmpty(slot.choice))
                        {
                            var error = IsValidChoiceList(slot, x => { var match = stringChoiceItemRegex.Match(x); return x.Length != 0 && match.Success && match.Length == x.Length; }, out choices);
                            if (error != RenderInfoSlotErrorType.OK)
                            {
                                return error;
                            }
                        }

                        // default のチェック
                        if (!string.IsNullOrEmpty(slot.@default))
                        {
                            var defaults = G3dDataParser.Tokenize(slot.@default);

                            if (choices != null)
                            {
                                // default が choice に含まれるか
                                foreach (var value in defaults)
                                {
                                    if (!choices.Contains(value))
                                    {
                                        return RenderInfoSlotErrorType.DefaultNotInChoice;
                                    }
                                }
                            }
                            else
                            {
                                // default の構成文字
                                foreach (var value in defaults)
                                {
                                    var match = stringChoiceItemRegex.Match(value);
                                    if (value.Length == 0 || !match.Success || match.Length != value.Length)
                                    {
                                        return RenderInfoSlotErrorType.DefaultCannotParse;
                                    }
                                }
                            }

                            // default の数
                            if (defaults.Length > slot.count)
                            {
                                return RenderInfoSlotErrorType.InvalidDefaultCount;
                            }
                        }
                        break;
                    }
                case render_info_slot_typeType.@int:
                    {
                        bool hasMinMax = false;
                        int min = 0;
                        int max = 0;

                        int[] choices = null;
                        // choice のチェック
                        if (!string.IsNullOrEmpty(slot.choice))
                        {
                            // パース出来るか
                            if (IfShaderAssignUtility.TryParseIntRange(slot.choice, out min, out max))
                            {
                                // 範囲が正しいか
                                if (min > max)
                                {
                                    return RenderInfoSlotErrorType.InvalidRange;
                                }

                                hasMinMax = true;
                            }
                            else
                            {
                                string[] stringChoices = null;
                                var error = IsValidChoiceList(slot, x => { int o; return int.TryParse(x, out o); }, out stringChoices);
                                if (error != RenderInfoSlotErrorType.OK)
                                {
                                    return error;
                                }
                                choices = stringChoices.Select(x => { int o; int.TryParse(x, out o); return o; }).ToArray();
                            }
                        }

                        // default のチェック
                        if (!string.IsNullOrEmpty(slot.@default))
                        {
                            int[] values;

                            // パース出来るか
                            if (TryParseIntArray(slot.@default, out values))
                            {
                                // default が choice に含まれるか
                                if (hasMinMax)
                                {
                                    foreach (var value in values)
                                    {
                                        if (value < min || max < value)
                                        {
                                            return RenderInfoSlotErrorType.DefaultNotInChoice;
                                        }
                                    }
                                }
                                else if (choices != null)
                                {
                                    foreach (var value in values)
                                    {
                                        if (!choices.Contains(value))
                                        {
                                            return RenderInfoSlotErrorType.DefaultNotInChoice;
                                        }
                                    }
                                }

                                // default の数
                                if (values.Length > slot.count)
                                {
                                    return RenderInfoSlotErrorType.InvalidDefaultCount;
                                }
                            }
                            else
                            {
                                return RenderInfoSlotErrorType.DefaultCannotParse;
                            }
                        }
                        break;
                    }
                case render_info_slot_typeType.@float:
                    {
                        bool hasMinMax = false;
                        float min = 0;
                        float max = 0;
                        float[] choices = null;

                        // choice のチェック
                        if (!string.IsNullOrEmpty(slot.choice))
                        {
                            // パース出来るか
                            if (IfShaderAssignUtility.TryParseFloatRange(slot.choice, out min, out max))
                            {
                                // 範囲が正しいか
                                if (min > max)
                                {
                                    return RenderInfoSlotErrorType.InvalidRange;
                                }

                                hasMinMax = true;
                            }
                            else
                            {
                                string[] stringChoices = null;
                                var error = IsValidChoiceList(slot, x => { float o; return float.TryParse(x, out o); }, out stringChoices);
                                if (error != RenderInfoSlotErrorType.OK)
                                {
                                    return error;
                                }
                                choices = stringChoices.Select(x => { float o; float.TryParse(x, out o); return o; }).ToArray();
                            }
                        }

                        // default のチェック
                        if (!string.IsNullOrEmpty(slot.@default))
                        {
                            float[] values;

                            // パース出来るか
                            // float.TryParse で ',' がパース出来てしまうのでチェック。
                            if (!slot.@default.Contains(',') && TryParseFloatArray(slot.@default, out values))
                            {
                                // default が choice に含まれるか
                                if (hasMinMax)
                                {
                                    foreach (var value in values)
                                    {
                                        if (value < min || max < value)
                                        {
                                            return RenderInfoSlotErrorType.DefaultNotInChoice;
                                        }
                                    }
                                }
                                else if (choices != null)
                                {
                                    foreach (var value in values)
                                    {
                                        if (!choices.Contains(value))
                                        {
                                            return RenderInfoSlotErrorType.DefaultNotInChoice;
                                        }
                                    }
                                }

                                // default の数
                                if (values.Length > slot.count)
                                {
                                    return RenderInfoSlotErrorType.InvalidDefaultCount;
                                }
                            }
                            else
                            {
                                return RenderInfoSlotErrorType.DefaultCannotParse;
                            }
                        }
                        break;
                    }
            }

            return RenderInfoSlotErrorType.OK;
        }

        /// <summary>
        /// デフォルト値が選択肢内にあるかどうか？
        /// </summary>
        public static bool IsDefaultInChoice(render_info_slotType slot)
        {
            if (!string.IsNullOrEmpty(slot.choice))
            {
                switch (slot.type)
                {
                    case render_info_slot_typeType.@int:
                        {
                            int min;
                            int max;
                            if (IfShaderAssignUtility.TryParseIntRange(slot.choice, out min, out max))
                            {
                                foreach (var value in SplitInt(slot.@default))
                                {
                                    if (value < min || max < value)
                                    {
                                        return false;
                                    }
                                }
                            }
                            break;
                        }
                    case render_info_slot_typeType.@float:
                        {
                            float min;
                            float max;
                            if (IfShaderAssignUtility.TryParseFloatRange(slot.choice, out min, out max))
                            {
                                foreach (var value in SplitFloat(slot.@default))
                                {
                                    if (value < min || max < value)
                                    {
                                        return false;
                                    }
                                }
                            }
                            break;
                        }
                    case render_info_slot_typeType.@string:
                        {
                            var choices = IfShaderOptionUtility.Enum(slot.choice);
                            foreach (var value in G3dDataParser.Tokenize(slot.@default))
                            {
                                if (!choices.Contains(value))
                                {
                                    return false;
                                }
                            }
                            break;
                        }
                }
            }

            return true;
        }

        /// <summary>
        /// 文字列を int 列に変換
        /// </summary>
        private static IEnumerable<int> SplitInt(string value)
        {
            return G3dDataParser.Tokenize(value).Select(x => { int result; int.TryParse(x, out result); return result; });
        }

        /// <summary>
        /// 文字列を float 列に変換
        /// </summary>
        private static IEnumerable<float> SplitFloat(string value)
        {
            return G3dDataParser.Tokenize(value).Select(x => { float result; float.TryParse(x, out result); return result; });
        }

        /// <summary>
        /// 文字列を int 列に変換
        /// </summary>
        private static bool TryParseIntArray(string value, out int[] values)
        {
            bool ok = true;
            List<int> valueList = new List<int>();
            foreach (var token in G3dDataParser.Tokenize(value))
            {
                int v;
                ok &= int.TryParse(token, out v);
                valueList.Add(v);
            }

            values = valueList.ToArray();
            return ok;
        }

        /// <summary>
        /// 文字列を float 列に変換
        /// </summary>
        private static bool TryParseFloatArray(string value, out float[] values)
        {
            bool ok = true;
            List<float> valueList = new List<float>();
            foreach (var token in G3dDataParser.Tokenize(value))
            {
                float v;
                ok &= float.TryParse(token, out v);
                valueList.Add(v);
            }

            values = valueList.ToArray();
            return ok;
        }
    }
}
