﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public static class IfAttributeAssignUtility
    {
        /// <summary>
        /// 比較
        /// </summary>
        public static IfShaderAssignUtility.CompareResult Compare(IEnumerable<attrib_assignType> assigns, modelType model, string materialName, shading_modelType shadingModel)
        {
            IEnumerable<attrib_varType> attrib_vars = shadingModel.attrib_var_array.GetItems();

            // 値が不正
            var invalid = from attrib_var in attrib_vars
                          let assign = assigns.FirstOrDefault(x => x.id == attrib_var.id)
                          where assign != null && !string.IsNullOrEmpty(assign.attrib_name) &&
                                !AssignableAttribs(model, materialName, attrib_var).Any(x => x.Key == assign.attrib_name)
                          select assign;

            if (invalid.Any())
            {
                return IfShaderAssignUtility.CompareResult.Conflict;
            }

            // 未割り当てで割り当て可能なものがある場合
            if (attrib_vars.Any(x =>
                !assigns.Any(y => y.id == x.id && !string.IsNullOrEmpty(y.attrib_name)) &&
                GetAttribNameFromHint(model, materialName, x) != null))
            {
                return IfShaderAssignUtility.CompareResult.Conflict;
            }

            if (attrib_vars.Count() != assigns.Count())
            {
                if (assigns.All(x => attrib_vars.Any(y => y.id == x.id)))
                {
                    return IfShaderAssignUtility.CompareResult.NotFullyAssigned;
                }
                else
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }
            else if (!assigns.Select(x => x.id).SequenceEqual(attrib_vars.Select(x => x.id)))
            {
                if (assigns.Select(x => x.id).OrderBy(x => x).SequenceEqual(attrib_vars.Select(x => x.id).OrderBy(x => x)))
                {
                    return IfShaderAssignUtility.CompareResult.NotSameOrder;
                }
                else
                {
                    return IfShaderAssignUtility.CompareResult.Conflict;
                }
            }

            return IfShaderAssignUtility.CompareResult.Ok;
        }

        /// <summary>
        /// 頂点属性割り当てを生成
        /// </summary>
        public static IEnumerable<attrib_assignType> CreateAttribAssigns(
            shading_modelType shadingModel,
            modelType model,
            string materialName,
            IEnumerable<attrib_assignType> oldAssigns,
            bool useHint,
            bool allowEmptyName)
        {
            foreach (var attrib_var in shadingModel.attrib_var_array.GetItems())
            {
                // 生成
                var assign = new attrib_assignType()
                {
                    id = attrib_var.id,
                    attrib_name = string.Empty,
                };

                // 元の値を引き継ぐ
                var old = oldAssigns.FirstOrDefault(x => x.id == attrib_var.id);
                if (old != null && AssignableAttribs(model, materialName, attrib_var).Any(x => x.Key == old.attrib_name))
                {
                    assign.attrib_name = old.attrib_name;
                }

                // ヒントから割り当て
                if (useHint && assign.attrib_name == string.Empty && (old == null || old.attrib_name != string.Empty))
                {
                    string attribName = GetAttribNameFromHint(model, materialName, attrib_var);

                    if (attribName != null)
                    {
                        assign.attrib_name = attribName;
                    }
                }

                // 出力
                if (allowEmptyName || assign.attrib_name != string.Empty)
                {
                    yield return assign;
                }
            }
        }

        /// <summary>
        /// 頂点属性割り当て変更メッセージの取得
        /// </summary>
        internal static IEnumerable<string> GetUpdateAttributeAssignMessages(IEnumerable<attrib_assignType> attribAssigns, modelType model, string materialName, shading_modelType shadingModel, bool ignoreOrder, IfShaderAssignUtility.InconsistentMessageType type)
        {
            // シェーダー定義側
            IEnumerable<attrib_varType> attrib_vars = shadingModel.attrib_var_array.GetItems();

            // 過剰
            var over = attribAssigns.Select(x => x.id).Except(attrib_vars.Select(x => x.id)).ToArray();
            foreach (var item in over)
            {
                yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Over_Attrib, type), item);
            }

            // 値が不正
            var invalid = from attrib_var in attrib_vars
                          let assign = attribAssigns.FirstOrDefault(x => x.id == attrib_var.id)
                          where assign != null && !string.IsNullOrEmpty(assign.attrib_name) &&
                                !AssignableAttribs(model, materialName, attrib_var).Any(x => x.Key == assign.attrib_name)
                          select new { assign, attrib_var };

            foreach (var item in invalid)
            {
                string attribName = GetAttribNameFromHint(model, materialName, item.attrib_var);
                string value = string.IsNullOrEmpty(attribName) ?
                    IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Unspecified, type) :
                    string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Value, type), attribName);
                yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Invalid_Attrib, type), item.assign.id, item.assign.attrib_name) +
                    value;
            }

            // 未割り当て分の設定
            var lack = attrib_vars.Where(x => !attribAssigns.Any(y => y.id == x.id && !string.IsNullOrEmpty(y.attrib_name))).ToArray();
            foreach (var item in lack)
            {
                string attribName = GetAttribNameFromHint(model, materialName, item);

                if (attribName != null)
                {
                    yield return string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Lack_AttribAssign, type), item.id) +
                        string.Format(IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Add_Value, type), attribName);
                }
            }

            // 順番
            if (!ignoreOrder &&
                over.Length == 0 &&
                !IfShaderAssignUtility.IsSubSequence(attribAssigns.Select(x => x.id), attrib_vars.Select(x => x.id)))
            {
                yield return IfShaderAssignUtility.GetResourceString(() => Resources.StringResource.Shader_Assign_Disorder_Attrib, type);
            }
        }

        /// <summary>
        /// attrib_var に適合する vtx_attribType の名前毎にまとめられたグループを列挙
        /// </summary>
        public static IEnumerable<IGrouping<string, vtx_attribType>> AssignableAttribs(modelType model, string materialName, attrib_varType attrib_var)
        {
            // マテリアルに関連する頂点属性を名前毎にまとめる
            var shapes = model.shape_array != null ? model.shape_array.shape : Enumerable.Empty<shapeType>();
            var vertices = model.vertex_array != null ? model.vertex_array.vertex : new vertexType[0];
            var vtx_attribGroups = from shape in shapes
                                   where shape.shape_info.mat_name == materialName
                                   let vertex = vertices[shape.shape_info.vertex_index]
                                   from vtx_attrib in vertex.vtx_attrib_array != null ? vertex.vtx_attrib_array.vtx_attrib : Enumerable.Empty<vtx_attribType>()
                                   group vtx_attrib by vtx_attrib.name;

            // 許容される型のみ返す
            return from attribs in vtx_attribGroups
                   where attribs.Any(x => IsFloatType(x.type) == IsFloatType(attrib_var.type))
                   select attribs;
        }

        /// <summary>
        /// ヒントをもとに頂点属性名を取得
        /// 適当なものがない場合は null
        /// </summary>
        public static string GetAttribNameFromHint(modelType model, string materialName, attrib_varType attrib_var)
        {
            return (from attribGroup in AssignableAttribs(model, materialName, attrib_var)
                    where attribGroup.All(x => x.hint == attrib_var.hint)
                    select attribGroup.Key).FirstOrDefault();
        }

        private static bool IsFloatType(vtx_attrib_typeType type)
        {
            switch (type)
            {
                case vtx_attrib_typeType.@float:
                case vtx_attrib_typeType.float2:
                case vtx_attrib_typeType.float3:
                case vtx_attrib_typeType.float4:
                    return true;
            }

            return false;
        }
    }
}
