﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // スケルタルアニメーションの量子化分析
    public class IfSkeletalAnimQuantizationAnalysisOptimizer
        : IfSkeletalAnimOptimizer
    {
        // コンストラクタ
        public IfSkeletalAnimQuantizationAnalysisOptimizer() :
            base("IfSkeletalAnimQuantizationAnalysisOptimizer_Log") { }

        // プロセス
        public override string Process
        {
            get { return "quantization_analysis"; }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Format(
                "frame[{0}/{1}({2:f0}%)]",
                QuantizedFrameSize, FrameSize, 100f * QuantizedFrameSize / FrameSize);
        }

        private int FrameSize;
        private int QuantizedFrameSize;

        private class Bone
        {
            public string name { get; set; } = null;
            public Bone parent { get; set; } = null;
            public IList<Bone> children { get; set; } = new List<Bone>();
            public int lengthToLeaf { get; set; } = 1;
            public bool hasRotateCurve { get; set; } = false;

            public IDictionary<bone_anim_targetType, IfAnimCurve> curveDic { get; set; }
                = new Dictionary<bone_anim_targetType, IfAnimCurve>();
        }

        private void SetLengthToLeaf(Bone bone)
        {
            if (bone.parent != null)
            {
                int lengthToLeaf = bone.lengthToLeaf;
                if (bone.hasRotateCurve)
                {
                    ++lengthToLeaf;
                }

                if (bone.parent.lengthToLeaf < lengthToLeaf)
                {
                    bone.parent.lengthToLeaf = lengthToLeaf;
                }

                SetLengthToLeaf(bone.parent);
            }
        }

        // 最適化
        protected override void Optimize()
        {
            if (this.Target.bone_anim_array == null) { return; }

            skeletal_animType skeletalAnim = this.Target;
            skeletal_anim_infoType skeletalAnimInfo = skeletalAnim.skeletal_anim_info;
            IList<Bone> bones = new List<Bone>();

            // ボーンの親子構造とカーブデータを構築する
            foreach (bone_animType bone_anim in skeletalAnim.bone_anim_array.bone_anim)
            {
                Bone current = new Bone();
                current.name = bone_anim.bone_name;

                foreach (bone_anim_targetType target in bone_anim.bone_anim_target)
                {
                    if (target.Item != null)
                    {
                        IfAnimCurve animCurve = new IfAnimCurve();
                        animCurve.curveType = IfAnimCurve.CurveType.Float;
                        bool isDegreeValue =
                            target.target == bone_anim_target_targetType.rotate_x ||
                            target.target == bone_anim_target_targetType.rotate_y ||
                            target.target == bone_anim_target_targetType.rotate_z ||
                            target.target == bone_anim_target_targetType.rotate_w;

                        if (isDegreeValue)
                        {
                            current.hasRotateCurve = true;
                        }

                        animCurve.BuildIfAnimCurve(target.Item, isDegreeValue, this.Streams);
                        current.curveDic.Add(target, animCurve);
                    }
                }

                // 親ボーンを探索します。
                foreach (Bone bone in bones)
                {
                    if (bone.name == bone_anim.parent_name)
                    {
                        current.parent = bone;
                        current.lengthToLeaf = 1;
                        SetLengthToLeaf(current);
                        bone.children.Add(current);

                        break;
                    }
                }

                bones.Add(current);
            }

            // frame の量子化を評価して type を選択する。
            foreach (var bone in bones)
            {
                foreach (var curve in bone.curveDic)
                {
                    // frametype が none 以外の場合は frame 量子化の判定を行わない。
                    if (curve.Value.frameType != curve_frame_typeType.none)
                    {
                        continue;
                    }
                    this.EnableProcessLog = true;

                    curve.Value.EvalQuantizeFrame(0.5f / skeletalAnimInfo.frame_resolution);
                }
            }

            // key の量子化を評価して type を選択する。
            foreach (var bone in bones)
            {
                foreach (var curve in bone.curveDic)
                {
                    // keytype が none 以外の場合は key 量子化の判定を行わない。
                    if (curve.Value.keyType != curve_key_typeType.none ||
                        !curve.Value.HasCurve())
                    {
                        continue;
                    }
                    this.EnableProcessLog = true;

                    // カーブを量子化します。
                    curve.Value.QuantizeCurve();

                    // 量子化したカーブを評価して誤差を計算します。

                    // 回転値の量子化
                    if (curve.Key.target == bone_anim_target_targetType.rotate_x ||
                        curve.Key.target == bone_anim_target_targetType.rotate_y ||
                        curve.Key.target == bone_anim_target_targetType.rotate_z ||
                        curve.Key.target == bone_anim_target_targetType.rotate_w)
                    {
                        // 回転量の誤差視認性はアニメーション全体における変化量との依存が
                        // 小さいと考えられるため、量子化誤差チェックには絶対誤差を用いる。
                        bool useAbsEval = true;
                        float tolerance_rotate =
                            FloatUtility.DegreeToRadian(skeletalAnimInfo.quantize_tolerance_rotate);
                        curve.Value.EvalQuantizeFloat(
                            tolerance_rotate / bone.lengthToLeaf,
                            useAbsEval,
                            skeletalAnimInfo.frame_resolution);
                    }
                    else if (curve.Key.target == bone_anim_target_targetType.scale_x ||
                             curve.Key.target == bone_anim_target_targetType.scale_y ||
                             curve.Key.target == bone_anim_target_targetType.scale_z)
                    {
                        bool useAbsEval = false;
                        curve.Value.EvalQuantizeFloat(
                            skeletalAnimInfo.quantize_tolerance_scale,
                            useAbsEval,
                            skeletalAnimInfo.frame_resolution);
                    }
                    else if (curve.Key.target == bone_anim_target_targetType.translate_x ||
                             curve.Key.target == bone_anim_target_targetType.translate_y ||
                             curve.Key.target == bone_anim_target_targetType.translate_z)
                    {
                        bool useAbsEval = false;
                        curve.Value.EvalQuantizeFloat(
                            skeletalAnimInfo.quantize_tolerance_translate,
                            useAbsEval,
                            skeletalAnimInfo.frame_resolution);
                    }
                }
            }

            // 結果を中間ファイルに適用します。
            foreach (var bone in bones)
            {
                foreach (var curve in bone.curveDic)
                {
                    ApplyQuantize(curve.Key, curve.Value);
                }
            }
        }

        /// <summary>
        /// 量子化タイプを適応します。
        /// </summary>
        private void ApplyQuantize(bone_anim_targetType target, IfAnimCurve animCurve)
        {
            curve_frame_typeType frameType = curve_frame_typeType.none;
            curve_key_typeType keyType = curve_key_typeType.none;
            int frameCount;

            if (target.Item is hermite_curveType)
            {
                hermite_curveType hermiteCurve = target.Item as hermite_curveType;
                hermiteCurve.frame_type = animCurve.frameType;
                hermiteCurve.key_type = animCurve.keyType;
                hermiteCurve.scale = animCurve.scale;
                hermiteCurve.offset = animCurve.offset;

                frameType = hermiteCurve.frame_type;
                keyType = hermiteCurve.key_type;
                frameCount = hermiteCurve.count;
            }
            else if (target.Item is linear_curveType)
            {
                linear_curveType linearCurve = target.Item as linear_curveType;
                linearCurve.frame_type = animCurve.frameType;
                linearCurve.key_type = animCurve.keyType;
                linearCurve.scale = animCurve.scale;
                linearCurve.offset = animCurve.offset;

                frameType = linearCurve.frame_type;
                keyType = linearCurve.key_type;
                frameCount = linearCurve.count;
            }
            else if (target.Item is step_curveType)
            {
                step_curveType stepCurve = target.Item as step_curveType;
                stepCurve.frame_type = animCurve.frameType;
                stepCurve.key_type = animCurve.keyType;
                stepCurve.scale = animCurve.scale;
                stepCurve.offset = animCurve.offset;

                frameType = stepCurve.frame_type;
                keyType = stepCurve.key_type;
                frameCount = stepCurve.count;
            }
            else
            {
                throw new NotImplementedException();
            }

            this.FrameSize += frameCount * sizeof(float);
            switch (frameType)
            {
                case curve_frame_typeType.none:
                case curve_frame_typeType.frame32:
                    this.QuantizedFrameSize += frameCount * sizeof(float);
                    break;
                case curve_frame_typeType.frame16:
                    this.QuantizedFrameSize += frameCount * sizeof(ushort);
                    break;
                case curve_frame_typeType.frame8:
                    this.QuantizedFrameSize += frameCount * sizeof(byte);
                    break;
            }
        }
    }
}
