﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;

namespace nw.g3d.iflib
{
    /// <summary>
    /// 三角形の情報を管理するためのクラスです。
    /// </summary>
    public class IfTriangle
    {
        /// <summary>
        /// 頂点数です。
        /// </summary>
        public const int VertexCount = 3;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public IfTriangle()
            : this(0, 0, 0)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="p0">頂点0の Id です。</param>
        /// <param name="p1">頂点1の Id です。</param>
        /// <param name="p2">頂点2の Id です。</param>
        public IfTriangle(int p0, int p1, int p2)
        {
            this.PositionIds = new int[VertexCount];

            this.PositionIds[0] = p0;
            this.PositionIds[1] = p1;
            this.PositionIds[2] = p2;
        }

        /// <summary>
        /// 頂点の Id です。
        /// </summary>
        public int[] PositionIds { get; private set; }

        /// <summary>
        /// 頂点の Id を設定します。
        /// </summary>
        /// <param name="p0">頂点0の Id です。</param>
        /// <param name="p1">頂点1の Id です。</param>
        /// <param name="p2">頂点2の Id です。</param>
        public void SetPositionIds(int p0, int p1, int p2)
        {
            this.PositionIds[0] = p0;
            this.PositionIds[1] = p1;
            this.PositionIds[2] = p2;
        }

        /// <summary>
        /// トライアングルに共有する頂点があるか調べます。
        /// </summary>
        /// <param name="IfTriangle">もう一方のトライアングルです。</param>
        /// <returns>共有する頂点があれば true を返します。</returns>
        public bool ContainsSharedVertex(IfTriangle IfTriangle)
        {
            if (IfTriangle != null)
            {
                for (int i = 0; i < VertexCount; ++i)
                {
                    for (int j = 0; j < VertexCount; ++j)
                    {
                        if (this.PositionIds[i] == IfTriangle.PositionIds[j])
                        {
                            return true;
                        }
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// トライアングルに共有する頂点の数を取得します。
        /// </summary>
        /// <param name="IfTriangle">もう一方のトライアングルです。</param>
        /// <returns>共有する頂点の数を返します。</returns>
        public int GetSharedVertexCount(IfTriangle IfTriangle)
        {
            int count = 0;

            if (IfTriangle != null)
            {
                for (int i = 0; i < VertexCount; ++i)
                {
                    for (int j = 0; j < VertexCount; ++j)
                    {
                        if (this.PositionIds[i] == IfTriangle.PositionIds[j])
                        {
                            ++count;
                        }
                    }
                }
            }

            return count;
        }

        /// <summary>
        /// トライアングルに共有していない頂点を取得します。
        /// </summary>
        /// <param name="IfTriangle">もう一方のトライアングルです。</param>
        /// <returns>
        /// オブジェクトの持つ頂点のうち、比較対象のトライアングルと
        /// 共有していない頂点のリストを返します。
        /// IfTriangle に null を指定した場合は自身の頂点のリストを返します。
        /// </returns>
        public List<int> GetUnsharedVertex(IfTriangle IfTriangle)
        {
            List<int> unsharedPositionIds = new List<int>();

            if (IfTriangle != null)
            {
                for (int i = 0; i < VertexCount; ++i)
                {
                    bool found = false;
                    for (int j = 0; j < VertexCount; ++j)
                    {
                        if (this.PositionIds[i] == IfTriangle.PositionIds[j])
                        {
                            found = true;
                            break;
                        }
                    }

                    if (!found)
                    {
                        unsharedPositionIds.Add(this.PositionIds[i]);
                    }
                }
            }
            else
            {
                for (int i = 0; i < VertexCount; ++i)
                {
                    unsharedPositionIds.Add(this.PositionIds[i]);
                }
            }

            return unsharedPositionIds;
        }
    }
}
