﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Collections.ObjectModel;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;

namespace nw.g3d.iflib
{
    internal class IfSubmeshPrimitiveIndexSortOptimizer : ISubmeshPrimitiveOptimizer
    {
        private readonly modelType model;
        private readonly List<G3dStream> Streams;
        private readonly shapeType shape;
        private readonly List<int> IndexStream;
        private readonly submeshType submesh;
        private readonly string Mode;
        private ReadOnlyCollection<int> source = null;
        private readonly List<IfOptimizePrimitiveAlgorithm> algorithms =
            new List<IfOptimizePrimitiveAlgorithm>();
        private IfOptimizePrimitiveAlgorithm algorithm = null;

        // コンストラクタ
        internal IfSubmeshPrimitiveIndexSortOptimizer(
            modelType model, List<G3dStream> streams, shapeType shape,
            List<int> indexStream, submeshType submesh,
            string mode)
        {
            this.model = model;
            this.Streams = streams;
            this.shape = shape;
            this.IndexStream = indexStream;
            this.submesh = submesh;
            this.Mode = mode;
        }

        public int TriangleCount { get; private set; }
        public int SourceProcessingVertexCount { get; private set; }
        public int ProcessingVertexCount { get; private set; }

        //---------------------------------------------------------------------
        // 初期化
        public void Initialize()
        {
            // 最適化対象のインデックスストリームを取り出す
            int[] source = new int[this.submesh.count];
            for (int i = 0; i < this.submesh.count; i++)
            {
                source[i] = this.IndexStream[i + submesh.offset];
            }
            this.source = Array.AsReadOnly(source);

            this.TriangleCount = this.source.Count / 3;
            this.SourceProcessingVertexCount = G3dProcessingVertexCounter.CountTrianglesNX(this.source);
            this.ProcessingVertexCount = this.SourceProcessingVertexCount;
        }

        //---------------------------------------------------------------------
        // 最適化
        public void Optimize()
        {
            BuildAlgorithms();

            // 大量に並列化するとメモリがあふれるので、Default のみ並列化する
            if (this.algorithms.Count == 2)
            {
                G3dParallel.ForEach(this.algorithms,
                    delegate(IfOptimizePrimitiveAlgorithm algorithm)
                    {
                        algorithm.Optimize(int.MaxValue);
                    });
            }
            else
            {
                int minProcessingVertexCount = int.MaxValue;
                int count = this.algorithms.Count;
                for (int i = 0; i < count; i++)
                {
                    IfOptimizePrimitiveAlgorithm algorithm = this.algorithms[i];
                    algorithm.Optimize(minProcessingVertexCount);
                    if (algorithm.ProcessingVertexCount < minProcessingVertexCount)
                    {
                        minProcessingVertexCount = algorithm.ProcessingVertexCount;
                    }
                }
            }

            ApplyResult();
        }

        // アルゴリズムの構築
        private void BuildAlgorithms()
        {
            IEnumerable<IfOptimizePrimitiveAlgorithm> enumerable;
            if (this.Mode == "force")
            {
                enumerable = IfOptimizePrimitiveAlgorithmCombo.EnumerateForce(this.source);
            }
            else if (this.Mode == "brute_force")
            {
                enumerable = IfOptimizePrimitiveAlgorithmCombo.EnumerateBruteForce(this.source);
            }
            else
            {
                enumerable = IfOptimizePrimitiveAlgorithmCombo.EnumerateDefault(this.source);
            }

            foreach (IfOptimizePrimitiveAlgorithm algorithm in enumerable)
            {
                this.algorithms.Add(algorithm);
            }
        }

        // 結果の適用
        private void ApplyResult()
        {
            // 最も処理頂点数の少ないアルゴリズムを選択
            foreach (IfOptimizePrimitiveAlgorithm algorithm in this.algorithms)
            {
                if (algorithm.ProcessingVertexCount < this.ProcessingVertexCount)
                {
                    this.algorithm = algorithm;
                    this.ProcessingVertexCount = algorithm.ProcessingVertexCount;
                }
            }

            int[] result = this.source.ToArray();

#if DEBUG
            int preCount = result.Length;
#endif

            // インデックスストリームへの最適化結果書き込み
            if (IsSuccess())
            {
                result = this.algorithm.Result;
            }
            SubmeshClusterSet clusterSet = new SubmeshClusterSet(this.model, this.Streams, this.shape, result, this.submesh, this.Mode);
            clusterSet.MakeClusters();
            clusterSet.Calc();
            result = clusterSet.Source.ToArray();
            this.ProcessingVertexCount = G3dProcessingVertexCounter.CountTrianglesNX(result);
#if DEBUG
            int postCount = result.Length;
            Nintendo.Foundation.Contracts.Assertion.Operation.True(preCount == postCount);
#endif

            for (int i = 0; i < this.submesh.count; i++)
            {
                this.IndexStream[i + submesh.offset] = result[i];
            }
        }

        //---------------------------------------------------------------------
        // 最適化が成功したか
        public bool IsSuccess() { return (this.algorithm != null); }
    }
}
