﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
#define SMOOTH_MERGE
//#define POLYGON_TEST

using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;
using System.Linq;

namespace nw.g3d.iflib
{
    // 同一頂点の削除
    public class IfModelDeleteEqualVertexOptimizer : IfModelOptimizer
    {
        public class AttribTarget
        {
            private enum Target
            {
                Position,
                Normal,
                Tangent,
                Binormal,
                Color,
                Uv,
                BlendIndex,
                BlendWeight,
                Others,
                NumTarget,
            }

            private static float[] TargetThreshold =
            {
                1.0f / 4096.0f,
                0.1f / 180.0f * (float)Math.PI, // 0.1°
                0.1f / 180.0f * (float)Math.PI, // 0.1°
                0.1f / 180.0f * (float)Math.PI, // 0.1°
                1.0f / 255.0f,
                1.0f / 256.0f,
                FloatUtility.Epsilon,
                FloatUtility.Epsilon,
                FloatUtility.Epsilon,
            };

            public AttribTarget(G3dStream stream, int vertexCount, string hint)
            {
                this.Stream = stream;
                this.VectorStream = null;
                this.HintTarget = HintToTarget(hint);
                switch (Stream.type)
                {
                    case stream_typeType.@float:
                        {
                            this.CompCount = Stream.FloatData.Count / vertexCount;
                        }
                        break;
                    case stream_typeType.@int:
                        {
                            this.CompCount = Stream.IntData.Count / vertexCount;
                        }
                        break;
                    default:
                        throw new Exception("Internal error.");
                }

                switch (HintTarget)
                {
                    case Target.Position:
                        {
                            Vector3[] vectorStream = new Vector3[vertexCount];
                            for (int i = 0; i < vertexCount; ++i)
                            {
                                vectorStream[i] = new Vector3();
                                for (int j = 0; j < this.CompCount; j++)
                                {
                                    vectorStream[i][j] = this.Stream.FloatData[i * this.CompCount + j];
                                }
                            }
                            VectorStream = vectorStream;
                            break;
                        }
                    case Target.Normal:
                        {
                            Vector3[] vectorStream = new Vector3[vertexCount];
                            for (int i = 0; i < vertexCount; ++i)
                            {
                                vectorStream[i] = new Vector3();
                                for (int j = 0; j < this.CompCount; j++)
                                {
                                    vectorStream[i][j] = this.Stream.FloatData[i * this.CompCount + j];
                                }
                            }
                            VectorStream = vectorStream;
                            break;
                        }
                    case Target.Tangent:
                        {
                            Vector4[] vectorStream = new Vector4[vertexCount];
                            for (int i = 0; i < vertexCount; ++i)
                            {
                                vectorStream[i] = new Vector4();
                                for (int j = 0; j < this.CompCount; j++)
                                {
                                    vectorStream[i][j] = this.Stream.FloatData[i * this.CompCount + j];
                                }
                            }
                            VectorStream = vectorStream;
                            break;
                        }
                    case Target.Binormal:
                        {
                            Vector4[] vectorStream = new Vector4[vertexCount];
                            for (int i = 0; i < vertexCount; ++i)
                            {
                                vectorStream[i] = new Vector4();
                                for (int j = 0; j < this.CompCount; j++)
                                {
                                    vectorStream[i][j] = this.Stream.FloatData[i * this.CompCount + j];
                                }
                            }
                            VectorStream = vectorStream;
                            break;
                        }
                    default:
                        {
                            break;
                        }
                }
            }

            public bool NearlyEqualPosition(int lhsIdx, int rhsIdx)
            {
                Vector3[] posStream = VectorStream as Vector3[];
                // 頂点間の距離を計算
                Vector3 pos = posStream[lhsIdx];
                Vector3 tPos = posStream[rhsIdx];
                float length = (tPos - pos).Length;
                return length < TargetThreshold[(int)HintTarget];
            }

            public bool NearlyEqual(int lhsIdx, int rhsIdx)
            {
                // int の場合は完全一致
                if (Stream.type == stream_typeType.@int)
                {
                    bool isSimilar = true;
                    int lhsAttribIdx = lhsIdx * CompCount;
                    int rhsAttribIdx = rhsIdx * CompCount;
                    for (int i = 0; i < CompCount; ++i)
                    {
                        List<int> stream = Stream.IntData;
                        if (stream[lhsAttribIdx + i] != stream[rhsAttribIdx + i])
                        {
                            isSimilar = false;
                            break;
                        }
                    }

                    return isSimilar;
                }

                switch (HintTarget)
                {
                    case Target.Normal:
                        {
                            Vector3[] normStream = VectorStream as Vector3[];
                            // 法線の角度の差を求める
                            Vector3 pos = normStream[lhsIdx];
                            Vector3 tPos = normStream[rhsIdx];
                            double cosTheta = pos.Dot(tPos) / Math.Sqrt(pos.LengthSquared * tPos.LengthSquared);
                            double radian = Math.Acos(Math.Max(Math.Min(cosTheta, 1.0), -1.0));
                            return radian < TargetThreshold[(int)HintTarget];
                        }
                    case Target.Tangent:
                    case Target.Binormal:
                        {
                            Vector4[] tanStream = VectorStream as Vector4[];
                            // 接線・従法線の角度の差を求める
                            Vector4 pos = tanStream[lhsIdx];
                            Vector4 tPos = tanStream[rhsIdx];
                            if (pos.W != tPos.W)
                            {
                                return false;
                            }
                            double cosTheta = (pos.Dot(tPos) - 1) / Math.Sqrt((pos.LengthSquared - 1) * (tPos.LengthSquared - 1));
                            double radian = Math.Acos(Math.Max(Math.Min(cosTheta, 1.0), -1.0));
                            return radian < TargetThreshold[(int)HintTarget];
                        }
                    default:
                        {
                            bool isSimilar = true;
                            for (int i = 0; i < CompCount; ++i)
                            {
                                if (!FloatUtility.NearlyEqual(
                                    Stream.FloatData[lhsIdx * CompCount + i],
                                    Stream.FloatData[rhsIdx * CompCount + i],
                                    TargetThreshold[(int)HintTarget]))
                                {
                                    isSimilar = false;
                                    break;
                                }
                            }
                            return isSimilar;
                        }
                }
            }

            public static float GetTargetThreshold(string hint)
            {
                Target target = HintToTarget(hint);
                if (target == Target.Normal ||
                    target == Target.Tangent ||
                    target == Target.Binormal)
                {
                    return TargetThreshold[(int)target] * 180.0f / (float)Math.PI;
                }
                return TargetThreshold[(int)target];
            }

            public static void SetTargetThreshold(string hint, float value)
            {
                Target target = HintToTarget(hint);
                if (target != Target.Others)
                {
                    if (target == Target.Normal ||
                        target == Target.Tangent ||
                        target == Target.Binormal)
                    {
                        value = value * (float)Math.PI / 180.0f;
                    }
                    TargetThreshold[(int)target] = value;
                }
            }

            private static Target HintToTarget(string hint)
            {
                Target target;
                if (hint.StartsWith("position"))
                {
                    target = Target.Position;
                }
                else if (hint.StartsWith("normal"))
                {
                    target = Target.Normal;
                }
                else if (hint.StartsWith("tangent"))
                {
                    target = Target.Tangent;
                }
                else if (hint.StartsWith("binormal"))
                {
                    target = Target.Binormal;
                }
                else if (hint.StartsWith("color"))
                {
                    target = Target.Color;
                }
                else if (hint.StartsWith("uv"))
                {
                    target = Target.Uv;
                }
                else if (hint.StartsWith("blendindex"))
                {
                    target = Target.BlendIndex;
                }
                else if (hint.StartsWith("blendweight"))
                {
                    target = Target.BlendWeight;
                }
                else
                {
                    target = Target.Others;
                }

                return target;
            }

            public G3dStream Stream { get; private set; }
            public IVector[] VectorStream { get; private set; }
            public int CompCount { get; private set; }
            private Target HintTarget { get; set; }
        }

        public class VertexNearList
        {
            public VertexNearList(int vertexIndex, int vertexOffset, int vertexCount)
            {
                this.VertexIndex = vertexIndex;
                this.VertexOffset = vertexOffset;
                this.VertexCount = vertexCount;
                this.integrationList = new int[this.VertexCount];
                this.vanishCountList = new int[this.VertexCount];
                this.bringVertexList = new List<int>[this.VertexCount];

                for (int i = 0; i < vertexCount; ++i)
                {
                    this.integrationList[i] = -1;
                    this.vanishCountList[i] = -1;
                    this.bringVertexList[i] = new List<int>();
                }
            }

            public int VertexIndex { get; private set; }
            public int VertexOffset { get; set; }
            public int VertexCount { get; private set; }
            public int[] IntegrationList
            {
                get { return integrationList; }
            }
            public int[] VanishCountList
            {
                get { return vanishCountList; }
            }
            public List<int>[] BringVertexList
            {
                get { return bringVertexList; }
            }

            private int[] integrationList = null;
            private int[] vanishCountList = null;
            private List<int>[] bringVertexList = null;
        }

        public class VertexSearchAccelerator
        {
            public VertexSearchAccelerator(float radius, Vector3[] positions, int offset, int count)
            {
                this.radius = radius;
                this.positions = positions;
                this.sortedIndices = new int[3][];
                for (int axis = 0; axis < 3; ++axis)
                {
                    this.sortedIndices[axis] = this.positions.Skip(offset).Take(count).Select((p, i) =>
                        new KeyValuePair<int, float>(i, p[axis])).OrderBy(p => p.Value).Select(p => p.Key).ToArray();
                }
            }

            public IEnumerable<int> GetNearVertices(int vtxIdx)
            {
                var pos = this.positions[vtxIdx];
                List<int>[] axisNear = new List<int>[3];
                for (int axis = 0; axis < 3; ++axis)
                {
                    axisNear[axis] = new List<int>();
                    int lower = ~sortedIndices[axis].BinarySearch(pos[axis] - radius, (i, p) => this.positions[i][axis] < p ? -1 : 1);
                    int upper = ~sortedIndices[axis].BinarySearch(pos[axis] + radius, (i, p) => this.positions[i][axis] > p ? 1 : -1);
                    for (int idx = lower; idx < upper; ++idx)
                    {
                        axisNear[axis].Add(sortedIndices[axis][idx]);
                    }
                }

                return axisNear[0].Intersect(axisNear[1]).Intersect(axisNear[2]).Where(p => p != vtxIdx);
            }

            private float radius;
            private Vector3[] positions;
            private int[][] sortedIndices;
        }

        // コンストラクタ
        public IfModelDeleteEqualVertexOptimizer(string argument) :
            base("IfModelDeleteEqualVertexOptimizer_Log", argument)
        {
        }

        public override string ToString()
        {
            return string.Format(
                "{0} {1}({2}<{3})", base.ToString(),
                Process, AfterVertexCount, BeforeVertexCount);
        }

        // プロセス
        public override string Process
        {
            get
            {
                return "delete_near_vertex";
            }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Format(
                "{0}[{1}({2:f0}%)/{3}]",
                Process, AfterVertexCount, 100.0 * AfterVertexCount / BeforeVertexCount, BeforeVertexCount);
        }

        private int BeforeVertexCount { get; set; }
        private int AfterVertexCount { get; set; }

        // 最適化
        protected override void Optimize()
        {
            // 頂点情報が存在しない場合はなにも行わない
            if (this.Target.vertex_array == null ||
                this.Target.shape_array == null)
            {
                return;
            }

            string arg = this.Argument;
            string[] splitArg = arg.Split(new[] { ',' }, StringSplitOptions.RemoveEmptyEntries);

            foreach (string targetStr in splitArg)
            {
                string[] targetValue = targetStr.Split(':');
                if (targetValue.Length != 2)
                {
                    throw new Exception("Argument error.");
                }
                string target = targetValue[0].Trim();

                float value = 0.0f;
                if (!float.TryParse(targetValue[1].Trim(), out value))
                {
                    throw new Exception("Argument error.");
                }

                AttribTarget.SetTargetThreshold(target, value);
            }

            VertexNearList[] nearListArray = new VertexNearList[this.Target.vertex_array.Count];
            // 十分に近い頂点を統合する処理を行います。
            for (int vertexIndex = 0; vertexIndex < this.Target.vertex_array.Count; ++vertexIndex)
            {
                vertexType vertex = this.Target.vertex_array.vertex[vertexIndex];
                if (vertex.vtx_attrib_array.Count <= 0)
                {
                    continue;
                }

                int vertexCount = vertex.vtx_attrib_array.vtx_attrib[0].count;
                if (vertexCount <= 0)
                {
                    continue;
                }

                BeforeVertexCount += vertexCount;

                bool hasPosTarget = false;
                foreach (vtx_attribType vtx_attrib in vertex.vtx_attrib_array.Enumerate())
                {
                    if (vtx_attrib.hint.StartsWith("position"))
                    {
                        hasPosTarget = true;
                        break;
                    }
                }

                if (!hasPosTarget)
                {
                    continue;
                }

                // vertex を参照している shape を検索する
                shapeType refShape = Array.Find(this.Target.shape_array.shape, delegate (shapeType shape)
                {
                    return shape.shape_info.vertex_index == vertexIndex;
                });
                if (refShape == null)
                {
                    continue;
                }

                // 過去の処理と同じ引数が指定された時は何も行わない
                if (refShape.shape_info.delete_near_vertex_mode == null ||
                    refShape.shape_info.delete_near_vertex_mode.CompareTo(this.Argument) != 0)
                {
                    List<int> lod_offsets = new List<int> { 0 };
                    if (vertex.lod_offset != null)
                    {
                        lod_offsets.AddRange(G3dDataParser.ParseIntArray(vertex.lod_offset.Value));
                    }
                    if (lod_offsets.Count != refShape.mesh_array.mesh.Length)
                    {
                        IfStrings.Throw("IfModelDeleteEqualVertexOptimizer_Error_InvalidLodOffset", refShape.name);
                    }
                    lod_offsets.Add(vertexCount);

                    // LOD へは Unite 前にかけられている前提なので、ベースにのみ適用する
                    // もし LOD にもかける場合、Vertex 単位で並列化されていることに注意し、
                    // 後の lod_offset の調整やインデクスの付け替えも変更する必要がある
                    nearListArray[vertexIndex] = new VertexNearList(
                        vertexIndex, lod_offsets[0], lod_offsets.Min(p => p > lod_offsets[0] ? p : vertexCount) - lod_offsets[0]);
                }
            }

#if POLYGON_TEST
            foreach(vertexType vertex in this.Target.vertex_array.Enumerate())
            {
                List<float> normalStream = null;
                List<float> tangentStream = null;
                List<float> binormalStream = null;
                foreach (vtx_attribType vtx_attrib in vertex.vtx_attrib_array.Enumerate())
                {
                    if (vtx_attrib.hint.StartsWith("normal"))
                    {
                        normalStream = this.Streams[vtx_attrib.stream_index].FloatData;
                    }
                    else if (vtx_attrib.hint.StartsWith("tangent"))
                    {
                        tangentStream = this.Streams[vtx_attrib.stream_index].FloatData;
                    }
                    else if (vtx_attrib.hint.StartsWith("binormal"))
                    {
                        binormalStream = this.Streams[vtx_attrib.stream_index].FloatData;
                    }
                }

                const float TORERANCE = 0.0001f;
                if (normalStream != null &&
                    tangentStream != null &&
                    binormalStream != null)
                {
                    for (int posIdx = 0; posIdx < normalStream.Count / 3; ++posIdx)
                    {
                        Vector3 normal = new Vector3(normalStream[posIdx * 3], normalStream[posIdx * 3 + 1], normalStream[posIdx * 3 + 2]);
                        Vector3 tangent = new Vector3(tangentStream[posIdx * 4], tangentStream[posIdx * 4 + 1], tangentStream[posIdx * 4 + 2]);
                        Vector3 binormal = new Vector3(binormalStream[posIdx * 4], binormalStream[posIdx * 4 + 1], binormalStream[posIdx * 4 + 2]);

                        if (!FloatUtility.NearlyEqual(normal.Length, 1.0f, TORERANCE))
                        {
                            Debug.WriteLine("normal is not length 1.");
                            Debug.WriteLine(string.Format("{0}: {1}, {2}, {3}: {4}", posIdx, normal.X, normal.X, normal.Z, normal.Length));
                        }

                        if (!FloatUtility.NearlyEqual(tangent.Length, 1.0f, TORERANCE))
                        {
                            Debug.WriteLine("tangent is not length 1.");
                            Debug.WriteLine(string.Format("{0}: {1}, {2}, {3}: {4}", posIdx, tangent.X, tangent.X, tangent.Z, tangent.Length));
                        }

                        if (!FloatUtility.NearlyEqual(binormal.Length, 1.0f, TORERANCE))
                        {
                            Debug.WriteLine("binormal is not length 1.");
                            Debug.WriteLine(string.Format("{0}: {1}, {2}, {3}: {4}", posIdx, binormal.X, binormal.X, binormal.Z, binormal.Length));
                        }

                        if (!FloatUtility.NearlyEqual(normal.Dot(tangent), 0.0f, TORERANCE))
                        {
                            Debug.WriteLine("normal is not orthogonal to tangent.");
                            Debug.WriteLine(string.Format("{0}: {1}", posIdx, normal.Dot(tangent)));
                        }

                        if (!FloatUtility.NearlyEqual(normal.Dot(binormal), 0.0f, TORERANCE))
                        {
                            Debug.WriteLine("normal is not orthogonal to binormal.");
                            Debug.WriteLine(string.Format("{0}: {1}", posIdx, normal.Dot(binormal)));
                        }

                        if (!FloatUtility.NearlyEqual(tangent.Dot(binormal), 0.0f, TORERANCE))
                        {
                            Debug.WriteLine("tangent is not orthogonal to binormal.");
                            Debug.WriteLine(string.Format("{0}: {1}", posIdx, tangent.Dot(binormal)));
                        }
                    }
                }
            }
#endif

            G3dParallel.ForEach(nearListArray, delegate (VertexNearList vertexNearList)
            {
                if (vertexNearList == null) { return; }

                vertexType vertex = this.Target.vertex_array.vertex[vertexNearList.VertexIndex];
                int[] iList = vertexNearList.IntegrationList;

                AttribTarget posTarget = null;
                List<AttribTarget> targets = new List<AttribTarget>();
                foreach (vtx_attribType vtx_attrib in vertex.vtx_attrib_array.Enumerate())
                {
                    string hint = vtx_attrib.hint;
                    AttribTarget target = new AttribTarget(Streams[vtx_attrib.stream_index], vtx_attrib.count, hint);
                    if (vtx_attrib.hint.StartsWith("position"))
                    {
                        posTarget = target;
                    }
                    else
                    {
                        targets.Add(target);
                    }
                }

                VertexSearchAccelerator accelerator = new VertexSearchAccelerator(
                    AttribTarget.GetTargetThreshold("position"), posTarget.VectorStream as Vector3[],
                    vertexNearList.VertexOffset, vertexNearList.VertexCount);

                for (int vtxIdx = 0; vtxIdx < vertexNearList.VertexCount; ++vtxIdx)
                {
                    int vtxIdxAbsolute = vtxIdx + vertexNearList.VertexOffset;
                    List<int> bringVertex = vertexNearList.BringVertexList[vtxIdx];
                    // 既に統合リストに登録されている場合はスキップ
                    if (iList[vtxIdx] != -1)
                    {
                        continue;
                    }

                    List<int> similarVtxIndices = new List<int>();
                    foreach (int tVtxIdx in accelerator.GetNearVertices(vtxIdx))
                    {
                        int tVtxIdxAbsolute = tVtxIdx + vertexNearList.VertexOffset;
                        if (posTarget.NearlyEqualPosition(vtxIdxAbsolute, tVtxIdxAbsolute))
                        {
                            similarVtxIndices.Add(tVtxIdx);
                        }
                    }

                    // 頂点位置が近い Vertex について、さらに詳しく調べる
                    foreach (int similarIdx in similarVtxIndices)
                    {
                        bool isSimilar = true;
                        foreach (AttribTarget target in targets)
                        {
                            int similarIdxAbsolute = similarIdx + vertexNearList.VertexOffset;
                            if (!target.NearlyEqual(vtxIdxAbsolute, similarIdxAbsolute))
                            {
                                isSimilar = false;
                                break;
                            }
                        }

                        if (isSimilar)
                        {
                            // 近い頂点を登録する
                            iList[similarIdx] = vtxIdx;
                            bringVertex.Add(similarIdx);
                        }
                    }
                }
            });

#if SMOOTH_MERGE
            // 同一頂点の平均をとって統合
            G3dParallel.ForEach(nearListArray, delegate (VertexNearList vertexNearList)
            {
                if (vertexNearList == null) { return; }

                vertexType vertex = this.Target.vertex_array.vertex[vertexNearList.VertexIndex];
                for (int vtxIdx = 0; vtxIdx < vertexNearList.VertexCount; ++vtxIdx)
                {
                    int vtxIdxAbsolute = vtxIdx + vertexNearList.VertexOffset;
                    List<int> bringVertex = vertexNearList.BringVertexList[vtxIdx];
                    if (bringVertex.Count != 0)
                    {
                        Vector3 normal = new Vector3();
                        Vector3 tangent = new Vector3();
                        Vector3 binormal = new Vector3();
                        float tangentW = 1.0f;
                        foreach (vtx_attribType vtx_attrib in vertex.vtx_attrib_array.Enumerate())
                        {
                            if (vtx_attrib.hint.StartsWith("normal"))
                            {
                                G3dStream stream = Streams[vtx_attrib.stream_index];
                                int compCount = stream.FloatData.Count / vtx_attrib.count;
                                normal.X = stream.FloatData[vtxIdxAbsolute * compCount + 0];
                                normal.Y = stream.FloatData[vtxIdxAbsolute * compCount + 1];
                                normal.Z = stream.FloatData[vtxIdxAbsolute * compCount + 2];
                            }
                            if (vtx_attrib.hint.StartsWith("binormal"))
                            {
                                G3dStream stream = Streams[vtx_attrib.stream_index];
                                int compCount = stream.FloatData.Count / vtx_attrib.count;
                                binormal.X = stream.FloatData[vtxIdxAbsolute * compCount + 0];
                                binormal.Y = stream.FloatData[vtxIdxAbsolute * compCount + 1];
                                binormal.Z = stream.FloatData[vtxIdxAbsolute * compCount + 2];
                            }
                        }

                        foreach (vtx_attribType vtx_attrib in vertex.vtx_attrib_array.Enumerate())
                        {
                            if (vtx_attrib.hint.StartsWith("tangent"))
                            {
                                G3dStream stream = Streams[vtx_attrib.stream_index];
                                int compCount = stream.FloatData.Count / vtx_attrib.count;
                                Vector3 baseVtx = new Vector3(
                                    stream.FloatData[vtxIdxAbsolute * compCount + 0],
                                    stream.FloatData[vtxIdxAbsolute * compCount + 1],
                                    stream.FloatData[vtxIdxAbsolute * compCount + 2]);
                                tangentW = stream.FloatData[vtxIdxAbsolute * compCount + 3];
                                Vector3 center = new Vector3(baseVtx);
                                foreach (int bVtxIdx in bringVertex)
                                {
                                    Vector3 bVtx = new Vector3(
                                        stream.FloatData[bVtxIdx * compCount + 0],
                                        stream.FloatData[bVtxIdx * compCount + 1],
                                        stream.FloatData[bVtxIdx * compCount + 2]);
                                    center += bVtx;
                                }
                                if (!(center.Length < FloatUtility.Epsilon))
                                {
                                    center.Normalize();
                                    tangent.Set(center);
                                }
                                if (center.Length < FloatUtility.Epsilon ||
                                    1.0f - Math.Abs(center.Dot(normal)) < FloatUtility.Epsilon)
                                {
                                    // tangent の方向が計算できなくなる場合（長さが 0 または normal と平行）は
                                    // normal ともっとも直交に近い tangent を採用する。
                                    float minDot = Math.Abs(normal.Dot(baseVtx));
                                    tangent.Set(baseVtx);
                                    for (int idxVtx = 0; idxVtx < bringVertex.Count; ++idxVtx)
                                    {
                                        int bVtxIdx = bringVertex[idxVtx];
                                        Vector3 bVtx = new Vector3(
                                            stream.FloatData[bVtxIdx * compCount + 0],
                                            stream.FloatData[bVtxIdx * compCount + 1],
                                            stream.FloatData[bVtxIdx * compCount + 2]);
                                        float dot = Math.Abs(normal.Dot(bVtx));
                                        if (dot < minDot)
                                        {
                                            minDot = dot;
                                            tangent.Set(bVtx);
                                        }
                                    }
                                    tangent.Normalize();

                                    // 少なくとも base の normal と tangent は異なっている。
                                    if (normal.Equals(tangent, FloatUtility.Epsilon))
                                    {
                                        throw new Exception("tangent and normal are same : Vertex " +
                                            vertex.index_hint.ToString() + " (" + vtxIdxAbsolute.ToString() + ")");
                                    }
                                }

                                stream.FloatData[vtxIdxAbsolute * compCount + 0] = tangent.X;
                                stream.FloatData[vtxIdxAbsolute * compCount + 1] = tangent.Y;
                                stream.FloatData[vtxIdxAbsolute * compCount + 2] = tangent.Z;
                                break;
                            }
                        }

                        foreach (vtx_attribType vtx_attrib in vertex.vtx_attrib_array.Enumerate())
                        {
                            if (vtx_attrib.hint.StartsWith("binormal"))
                            {
                                // binormal 再計算
                                G3dStream stream = Streams[vtx_attrib.stream_index];
                                int compCount = stream.FloatData.Count / vtx_attrib.count;
                                Vector3 nBinormal = normal.Cross(tangent);
                                nBinormal.Normalize();
                                stream.FloatData[vtxIdxAbsolute * compCount + 0] = tangentW * nBinormal.X;
                                stream.FloatData[vtxIdxAbsolute * compCount + 1] = tangentW * nBinormal.Y;
                                stream.FloatData[vtxIdxAbsolute * compCount + 2] = tangentW * nBinormal.Z;
                                break;
                            }
                        }
                    }
                }
            });
#endif

            // 不要な頂点を削除
            // Stream を同時に書き換えるので複数の vertex が同じ stream を参照しないことを前提としている。
            G3dParallel.ForEach(nearListArray, delegate (VertexNearList vertexNearList)
            {
                if (vertexNearList == null) { return; }

                int[] iList = vertexNearList.IntegrationList;
                int[] vList = vertexNearList.VanishCountList;
                vertexType vertex = this.Target.vertex_array.vertex[vertexNearList.VertexIndex];

                int vertexCount = vertexNearList.VertexCount;
                int vanishCount = 0;
                for (int idx = 0; idx < vertexCount; ++idx)
                {
                    if (iList[idx] != -1)
                    {
                        int tIdxAbsolute = idx - vanishCount + vertexNearList.VertexOffset;
                        foreach (vtx_attribType vtx_attrib in vertex.vtx_attrib_array.Enumerate())
                        {
                            G3dStream stream = Streams[vtx_attrib.stream_index];

                            switch (stream.type)
                            {
                                case stream_typeType.@float:
                                    {
                                        int compCount = stream.FloatData.Count / vtx_attrib.count;
                                        for (int i = 0; i < compCount; ++i)
                                        {
                                            // 削除していくのでインクリメントしない。
                                            int posIdx = tIdxAbsolute * compCount;
                                            stream.FloatData.RemoveAt(posIdx);
                                        }
                                    }
                                    break;
                                case stream_typeType.@int:
                                    {
                                        int compCount = stream.IntData.Count / vtx_attrib.count;
                                        for (int i = 0; i < compCount; ++i)
                                        {
                                            // 削除していくのでインクリメントしない。
                                            int posIdx = tIdxAbsolute * compCount;
                                            stream.IntData.RemoveAt(posIdx);
                                        }
                                    }
                                    break;
                                default:
                                    throw new Exception("Internal error.");
                            }
                            // 頂点数を変更
                            --vtx_attrib.count;
                        }
                        ++vanishCount;
                    }

                    vList[idx] = vanishCount;
                }

                // lod_offset の調整
                var vtx = this.Target.vertex_array.vertex[vertexNearList.VertexIndex];
                if (vtx.lod_offset != null)
                {
                    var lod_offsets = G3dDataParser.ParseIntArray(vtx.lod_offset.Value).Select(p => p == 0 ? 0 : p - vanishCount);
                    StringBuilder sb = new StringBuilder(64);
                    foreach (var offset in lod_offsets)
                    {
                        sb.Append(offset).Append(' ');
                    }
                    vtx.lod_offset.Value = sb.ToString().TrimEnd();
                }
            });

            // インデクスの付け替え
            // Stream を同時に書き換えるので複数の shape が同じ stream を参照しないことを前提としている。
            G3dParallel.ForEach(this.Target.shape_array.shape, delegate (shapeType shape)
            {
                foreach (var vertexNearList in nearListArray)
                {
                    if (vertexNearList != null &&
                        shape.shape_info.vertex_index == vertexNearList.VertexIndex)
                    {
                        int[] iList = vertexNearList.IntegrationList;
                        int[] vList = vertexNearList.VanishCountList;
                        var vertex = this.Target.vertex_array.vertex[shape.shape_info.vertex_index];
                        List<int> lod_offsets = new List<int> { 0 };
                        if (vertex.lod_offset != null)
                        {
                            lod_offsets.AddRange(G3dDataParser.ParseIntArray(vertex.lod_offset.Value));
                        }

                        for (int iMesh = 0; iMesh < shape.mesh_array.mesh.Length; ++iMesh)
                        {
                            var mesh = shape.mesh_array.mesh[iMesh];
                            if (lod_offsets[iMesh] == 0 && shape.mesh_array.mesh.First(
                                p => p.stream_index == mesh.stream_index) == mesh)
                            {
                                G3dStream stream = this.Streams[mesh.stream_index];
                                for (int i = 0; i < stream.IntData.Count; i++)
                                {
                                    int iIdx = iList[stream.IntData[i]];
                                    if (iIdx != -1)
                                    {
                                        // 消失したインデクスを詰めていく
                                        stream.IntData[i] = iIdx - vList[iIdx];
                                    }
                                    else
                                    {
                                        // 消失したインデクスを詰めていく
                                        stream.IntData[i] = stream.IntData[i] - vList[stream.IntData[i]];
                                    }
                                }
                            }
                        }

                        // 頂点削除の処理モードを shape_info に記録しておく
                        shape.shape_info.delete_near_vertex_mode = this.Argument;

                        shape.shape_info.optimize_primitive_mode = string.Empty;
                    }
                }

                string hash = shape.shape_info.original_shape_hash;
                if (hash == string.Empty)
                {
                    hash = IfOptimizePrimitiveHashCalculator.Calculate(
                        this.Target, this.Streams, shape);
                    shape.shape_info.original_shape_hash = hash;
                }
            });
#if POLYGON_TEST
            foreach (shapeType shape in this.Target.shape_array.Enumerate())
            {
                vertexType vertex = this.Target.vertex_array.vertex[shape.shape_info.vertex_index];
                int[] stream = Streams[shape.mesh.stream_index].IntData.ToArray();
                for (int i = 0; i < shape.mesh.count / 3; ++i)
                {
                    // 存在しない頂点を参照しているポリゴンがある
                    if (stream[i * 3] >= vertex.vtx_attrib_array.vtx_attrib[0].count ||
                        stream[i * 3 + 1] >= vertex.vtx_attrib_array.vtx_attrib[0].count ||
                        stream[i * 3 + 2] >= vertex.vtx_attrib_array.vtx_attrib[0].count)
                    {
                        Debug.WriteLine("error vertex");
                    }
                    // 同じ頂点を参照するポリゴンがある
                    if (stream[i * 3] == stream[i * 3 + 1] ||
                        stream[i * 3] == stream[i * 3 + 2] ||
                        stream[i * 3 + 1] == stream[i * 3 + 2])
                    {
                            Debug.WriteLine("error vertex");
                    }
                    // 同一ポリゴンが存在している
                    for (int j = i + 1; j < shape.mesh.count / 3; ++j)
                    {
                        if (stream[i * 3] == stream[j * 3] &&
                            stream[i * 3 + 1] == stream[j * 3 + 1] &&
                            stream[i * 3 + 2] == stream[j * 3 + 2])
                        {
                            Debug.WriteLine("error vertex");
                        }
                    }
                }
            }
#endif

            foreach (vertexType vertex in this.Target.vertex_array.Enumerate())
            {
                if (vertex.vtx_attrib_array.Count <= 0)
                {
                    continue;
                }

                int vertexCount = vertex.vtx_attrib_array.vtx_attrib[0].count;
                if (vertexCount <= 0)
                {
                    continue;
                }

                AfterVertexCount += vertexCount;
            }

            if (AfterVertexCount != BeforeVertexCount)
            {
                EnableProcessLog = true;
            }
        }
    }

    public static class Utility
    {
        public delegate int DifferentTypeComparison<in T, in U>(T x, U y);
        public static int BinarySearch<T, U>(this T[] ary, U value, DifferentTypeComparison<T, U> comparer)
        {
            int min = 0;
            int max = ary.Length - 1;
            while (min <= max)
            {
                int middle = (min + max) / 2;
                int comp = comparer(ary[middle], value);
                if (comp < 0)
                {
                    min = middle + 1;
                }
                else if (comp > 0)
                {
                    max = middle - 1;
                }
                else
                {
                    return middle;
                }
            }
            return ~min;
        }
    }
}
