﻿using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public class IfMaterialAnimQuantizationAnalysis
    {
        public enum Srt2dIndex
        {
            ScaleX = 0,
            ScaleY,
            Rotate,
            TranslateX,
            TranslateY
        }

        public enum TexsrtIndex
        {
            Mode = 0,
            ScaleX,
            ScaleY,
            Rotate,
            TranslateX,
            TranslateY
        }

        public enum Srt3dIndex
        {
            ScaleX = 0,
            ScaleY,
            ScaleZ,
            RotateX,
            RotateY,
            RotateZ,
            TranslateX,
            TranslateY,
            TranslateZ
        }

        // シェーダーパラメーターアニメーションの量子化分析
        public static IfQuantizationAnalysisResult AnalyseShaderParameterCurve(
            material_anim_infoType materialAnimInfo,
            shader_param_typeType type,
            int componentIndex,
            IfAnimCurve curve,
            bool forceAnalyse)
        {
            // 量子化判定を途中で打ち切っても元の量子化設定が反映されるように
            // 最初の段階の設定を結果にコピーしておく。
            var result = new IfQuantizationAnalysisResult
                             {
                                 frameType = curve.frameType,
                                 keyType = curve.keyType,
                                 scale = curve.scale,
                                 offset = curve.offset,
                                 skipped = true
                             };

            // frame の量子化を評価して type を選択する。
            // frametype が none 以外の場合は frame 量子化の判定を行わない。
            if (!forceAnalyse &&
                curve.frameType != curve_frame_typeType.none)
            {
                return result;
            }
            result.skipped = false;
            curve.EvalQuantizeFrame(0.5f / materialAnimInfo.frame_resolution);
            result.frameType = curve.frameType;

            // keytype が none 以外の場合は key 量子化の判定を行わない。
            if (!forceAnalyse &&
                (curve.keyType != curve_key_typeType.none || !curve.HasCurve()))
            {
                return result;
            }
            // カーブを量子化します。
            curve.QuantizeCurve();

            if (type == shader_param_typeType.@int ||
                type == shader_param_typeType.int2 ||
                type == shader_param_typeType.int3 ||
                type == shader_param_typeType.int4 ||
                type == shader_param_typeType.@uint ||
                type == shader_param_typeType.uint2 ||
                type == shader_param_typeType.uint3 ||
                type == shader_param_typeType.uint4)
            {
                curve.EvalQuantizeInt();
            }
            else if (
                type == shader_param_typeType.texsrt)
            {
                // 量子化結果を判定し反映します。
                if (componentIndex == (int)TexsrtIndex.Mode)
                {
                    curve.EvalQuantizeInt();
                }
                else if (componentIndex == (int)TexsrtIndex.Rotate)
                {
                    // 回転量の誤差視認性はアニメーション全体における変化量との依存が
                    // 小さいと考えられるため、量子化誤差チェックには絶対誤差を用いる。
                    var useAbsEval = true;
                    float tolerance_tex_rotate;

                    // texsrt, texsrt_ex の場合は回転の量子化解析をランタイムの動作に合わせて Radian
                    // に変換してから行うため tolerance の値を Radian に変換する必要がある。
                    tolerance_tex_rotate =
                        FloatUtility.DegreeToRadian(materialAnimInfo.quantize_tolerance_tex_rotate);

                    curve.EvalQuantizeFloat(
                        tolerance_tex_rotate,
                        useAbsEval,
                        materialAnimInfo.frame_resolution);
                }
                else if (
                    componentIndex == (int)TexsrtIndex.ScaleX ||
                    componentIndex == (int)TexsrtIndex.ScaleY)
                {
                    var useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        materialAnimInfo.quantize_tolerance_tex_scale,
                        useAbsEval,
                        materialAnimInfo.frame_resolution);
                }
                else if (
                    componentIndex == (int)TexsrtIndex.TranslateX ||
                    componentIndex == (int)TexsrtIndex.TranslateY)
                {
                    var useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        materialAnimInfo.quantize_tolerance_tex_translate,
                        useAbsEval,
                        materialAnimInfo.frame_resolution);
                }
            }
            else if (type == shader_param_typeType.srt2d)
            {
                // 量子化結果を判定し反映します。
                // 回転値の量子化
                if (componentIndex == (int)Srt2dIndex.Rotate)
                {
                    // 回転量の誤差視認性はアニメーション全体における変化量との依存が
                    // 小さいと考えられるため、量子化誤差チェックには絶対誤差を用いる。
                    var useAbsEval = true;
                    float tolerance_tex_rotate;

                    // srt2d の場合は回転の量子化解析をランタイムの動作に合わせて Radian
                    // に変換してから行うため tolerance の値を Radian に変換する必要がある。
                    tolerance_tex_rotate =
                        FloatUtility.DegreeToRadian(materialAnimInfo.quantize_tolerance_tex_rotate);

                    curve.EvalQuantizeFloat(
                        tolerance_tex_rotate,
                        useAbsEval,
                        materialAnimInfo.frame_resolution);
                }
                else if (
                    componentIndex == (int)Srt2dIndex.ScaleX ||
                    componentIndex == (int)Srt2dIndex.ScaleY)
                {
                    var useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        materialAnimInfo.quantize_tolerance_tex_scale,
                        useAbsEval,
                        materialAnimInfo.frame_resolution);
                }
                else if (
                    componentIndex == (int)Srt2dIndex.TranslateX ||
                    componentIndex == (int)Srt2dIndex.TranslateY)
                {
                    var useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        materialAnimInfo.quantize_tolerance_tex_translate,
                        useAbsEval,
                        materialAnimInfo.frame_resolution);
                }
            }
            else if (type == shader_param_typeType.srt3d)
            {
                if (
                    componentIndex == (int)Srt3dIndex.RotateX ||
                    componentIndex == (int)Srt3dIndex.RotateY ||
                    componentIndex == (int)Srt3dIndex.RotateZ)
                {
                    // 回転量の誤差視認性はアニメーション全体における変化量との依存が
                    // 小さいと考えられるため、量子化誤差チェックには絶対誤差を用いる。
                    var useAbsEval = true;
                    // 回転の量子化解析はランタイムの動作に合わせてキーを Radian に変換してから行うため
                    // tolerance の値を Radian に変換する必要がある。
                    var tolerance_rotate =
                        FloatUtility.DegreeToRadian(materialAnimInfo.quantize_tolerance_tex_rotate);
                    curve.EvalQuantizeFloat(
                        tolerance_rotate,
                        useAbsEval,
                        materialAnimInfo.frame_resolution);
                }
                else if (
                    componentIndex == (int)Srt3dIndex.ScaleX ||
                    componentIndex == (int)Srt3dIndex.ScaleY ||
                    componentIndex == (int)Srt3dIndex.ScaleZ)
                {
                    var useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        materialAnimInfo.quantize_tolerance_tex_scale,
                        useAbsEval,
                        materialAnimInfo.frame_resolution);
                }
                else if (
                    componentIndex == (int)Srt3dIndex.TranslateX ||
                    componentIndex == (int)Srt3dIndex.TranslateY ||
                    componentIndex == (int)Srt3dIndex.TranslateZ)
                {
                    var useAbsEval = false;
                    curve.EvalQuantizeFloat(
                        materialAnimInfo.quantize_tolerance_tex_translate,
                        useAbsEval,
                        materialAnimInfo.frame_resolution);
                }
            }

            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;

            return result;
        }

        // テクスチャパターンアニメーションの量子化分析
        public static IfQuantizationAnalysisResult AnalyseTexPatternCurve(
            material_anim_infoType matAnimInfo,
            IfAnimCurve curve,
            bool forceAnalyse)
        {
            var result = new IfQuantizationAnalysisResult();

            // 量子化判定を途中で打ち切っても元の量子化設定が反映されるように
            // 最初の段階の設定を結果にコピーしておく。
            result.frameType = curve.frameType;
            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;
            result.skipped = true;

            // frametype が none 以外の場合は frame 量子化の判定を行わない。
            if (!forceAnalyse &&
                curve.frameType != curve_frame_typeType.none)
            {
                return result;
            }
            result.skipped = false;

            curve.EvalQuantizeFrame(0.5f / matAnimInfo.frame_resolution);
            result.frameType = curve.frameType;

            // keytype が none 以外の場合は key 量子化の判定を行わない。
            if (!forceAnalyse &&
                (curve.keyType != curve_key_typeType.none || !curve.HasCurve()))
            {
                return result;
            }

            // カーブを量子化します。
            curve.QuantizeCurve();
            curve.EvalQuantizeInt();

            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;

            return result;
        }

        // マテリアルビジビリティアニメーションの量子化分析
        public static IfQuantizationAnalysisResult AnalyseMaterialVisibilityCurve(
            material_anim_infoType matAnimInfo,
            IfAnimCurve curve,
            bool forceAnalyse)
        {
            var result = new IfQuantizationAnalysisResult();

            // 量子化判定を途中で打ち切っても元の量子化設定が反映されるように
            // 最初の段階の設定を結果にコピーしておく。
            result.frameType = curve.frameType;
            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;
            result.skipped = true;

            // keytype が none 以外の場合は key 量子化の判定を行わない。
            if (!forceAnalyse &&
                (curve.keyType != curve_key_typeType.none || !curve.HasCurve()))
            {
                return result;
            }
            result.skipped = false;

            curve.EvalQuantizeFrame(0.5f / matAnimInfo.frame_resolution);
            result.frameType = curve.frameType;

            return result;
        }
    }
}
