﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;
using System.IO;
using System.Linq;
using Nintendo.G3dTool.Entities;

namespace nw.g3d.iflib
{
    // バイナリ中間ファイル書き出しユーティリティ
    public static class IfBinaryWriteUtility
    {
        public static void WriteIntermediateFile(IntermediateFile file, string filePath, string xsdBasePath)
        {
            // ストリーム以外をシリアライズ可能データに変換
            nw4f_3difType fileData;
            {
                bool originalStreamSerializationEnabled = file.RootEntity.IsStreamSerializationEnabled;
                file.RootEntity.IsStreamSerializationEnabled = false;
                fileData = file.CreateSerializableData();
                file.RootEntity.IsStreamSerializationEnabled = originalStreamSerializationEnabled;
            }

            var streams = ConvertToG3dStream(file.RootEntity.Streams);
            IfBinaryWriteUtility.Write(fileData, streams, filePath, xsdBasePath);
            file.Path = filePath;
        }

        public static void WriteIntermediateFile(IntermediateFile file, System.IO.Stream outputStream, string xsdBasePath)
        {
            // ストリーム以外をシリアライズ可能データに変換
            nw4f_3difType fileData;
            {
                bool originalStreamSerializationEnabled = file.RootEntity.IsStreamSerializationEnabled;
                file.RootEntity.IsStreamSerializationEnabled = false;
                fileData = file.CreateSerializableData();
                file.RootEntity.IsStreamSerializationEnabled = originalStreamSerializationEnabled;
            }

            var streams = ConvertToG3dStream(file.RootEntity.Streams);
            byte[] fileImage = IfBinaryFormatter.FormatStream(fileData, streams, xsdBasePath);
            outputStream.Write(fileImage, 0, fileImage.Length);
        }

        private static List<G3dStream> ConvertToG3dStream(IEnumerable<Nintendo.G3dTool.Entities.Stream> sourceStreams)
        {
            List<G3dStream> streams = new List<G3dStream>();
            foreach (var sourceStream in sourceStreams)
            {
                G3dStream stream = new G3dStream()
                {
                    column = sourceStream.Column,
                    type = sourceStream.Type,
                };

                switch (sourceStream.Type)
                {
                    case stream_typeType.@byte:
                        stream.ByteData.AddRange((sourceStream as StreamByte).Values);
                        break;
                    case stream_typeType.@float:
                        stream.FloatData.AddRange((sourceStream as StreamFloat).Values);
                        break;
                    case stream_typeType.@int:
                        stream.IntData.AddRange((sourceStream as StreamInt).Values);
                        break;
                    case stream_typeType.@string:
                        stream.StringData = (sourceStream as StreamString).Value;
                        break;
                    case stream_typeType.wstring:
                        stream.StringData = (sourceStream as StreamWstring).Value;
                        break;
                    default:
                        break;
                }

                streams.Add(stream);
            }

            return streams;
        }

        // バイナリ中間ファイルの書き出し
        // 重いので非推奨
        public static void Write(nw4f_3difType nw4f_3dif, string filePath, string xsdBasePath)
        {
            // ストリーム配列の取得、パースするので重い
            List<G3dStream> streams = new List<G3dStream>();
            if (G3dStreamUtility.HasStreamArray(nw4f_3dif))
            {
                G3dStreamUtility.ToStreams(streams, nw4f_3dif.RootElement.stream_array);
                nw4f_3dif.RootElement.stream_array = null;
            }

            Write(nw4f_3dif, streams, filePath, xsdBasePath);
            nw4f_3dif.RootElement.stream_array = G3dStreamUtility.ToStreamArray(streams);
        }

        // バイナリ中間ファイルの書き出し
        public static void Write(
            nw4f_3difType nw4f_3dif, List<G3dStream> streamArray, string filePath, string xsdBasePath)
        {
            byte[] fileImage = IfBinaryFormatter.FormatStream(nw4f_3dif, streamArray, xsdBasePath);
            IfWriteUtility.WriteFileImage(fileImage, filePath);
        }

        // バイナリ中間ファイルの書き出し
        public static void Write(
            nw4f_3difType nw4f_3dif, List<IntermediateFileStream> streamArray, string filePath, string xsdBasePath)
        {
            byte[] fileImage = IfBinaryFormatter.FormatStream(nw4f_3dif, streamArray, xsdBasePath);
            IfWriteUtility.WriteFileImage(fileImage, filePath);
        }

        // バイナリ中間ファイルの書き出し(ストリームバイナリ部分はバイト列とオフセットとして渡す)
        public static void Write(
            nw4f_3difType nw4f_3dif, byte[] streamImage, string filePath, string xsdBasePath, int streamOffset = 0)
        {
            // テキストフォーマットを施す
            var textData = IfTextFormatter.FormatBytes(nw4f_3dif, xsdBasePath);

            // テキスト部分はnullを追加
            var textImagelength = textData.Length + 1;

            // バイナリ部分はアライメントを考慮
            int binaryOffset = (textImagelength +
                (G3dConstant.BinaryAlignment - 1)) &
                ~(G3dConstant.BinaryAlignment - 1);

            var streamLength = streamImage.Length - streamOffset;
            var fileSize = binaryOffset + streamLength;
            using (var fileStream = new FileStream(filePath, FileMode.Create))
            {
                fileStream.Write(textData, 0, textData.Length);
                for (var i = 0; i < binaryOffset - textData.Length; i++)
                {
                    fileStream.WriteByte(0);
                }
                fileStream.Write(streamImage, streamOffset, streamLength);
            }
        }

        // ストリームデータの書き出し
        public static byte[] WriteStreamArray(List<G3dStream> stream_array)
        {
            MemoryStream result = new MemoryStream();
            using (BinaryWriter resultWriter = new BinaryWriter(result))
            {
                // ストリームデータのフォーマット
                IfBinaryFormatter.FormatStream(resultWriter, stream_array);
            }
            result.Close();
            return result.ToArray();
        }
    }
}
