﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Diagnostics;
using System.IO;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // テキストシェーダー定義中間ファイルフォーマッタ
    internal static class IfTextShaderDefinitionFormatter
    {
        // shader_definition のフォーマット
        internal static void Format(TextReader rd, TextWriter wt)
        {
            // shader_definition
            wt.WriteLine($"<{G3dConstant.ShaderDefinitionElementName} {rd.ReadLine()}");

            // shader_definition_info
            string shader_definition_info = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(shader_definition_info == "<shader_definition_info");
            wt.WriteLine("{0}", shader_definition_info);
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
            wt.WriteLine("/>");

            // force_include
            string force_include_array = rd.ReadLine();
            string shading_model_array;
            if (force_include_array == "<force_include_array")
            {
                IfTextShaderDefinitionFormatter.Format_force_include_array(rd, wt);
                shading_model_array = rd.ReadLine();
            }
            else
            {
                shading_model_array = force_include_array;
            }

            // shading_model
            Nintendo.Foundation.Contracts.Assertion.Operation.True(shading_model_array == "<shading_model_array");
            IfTextShaderDefinitionFormatter.Format_shading_model_array(rd, wt);

            // shader_src
            string shader_src_array = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(shader_src_array == "<shader_src_array");
            IfTextShaderDefinitionFormatter.Format_shader_src_array(rd, wt);

            // stream
            string stream_array = rd.ReadLine();
            string tool_data;
            if (stream_array == "<stream_array")
            {
                IfTextFormatterUtility.Format_stream(rd, wt);
                tool_data = rd.ReadLine();
            }
            else
            {
                tool_data = stream_array;
            }

            // tool_data
            string close_shader_definition;
            if (tool_data == "<tool_data>")
            {
                IfTextFormatterUtility.FormatToolData(rd, wt, string.Empty);
                close_shader_definition = rd.ReadLine();
            }
            else
            {
                close_shader_definition = tool_data;
            }

            Nintendo.Foundation.Contracts.Assertion.Operation.True(close_shader_definition == "</shader_definition>");
            wt.WriteLine(close_shader_definition);
        }

        //---------------------------------------------------------------------
        // force_include_array のフォーマット
        private static void Format_force_include_array(TextReader rd, TextWriter wt)
        {
            // force_include_array
            wt.WriteLine("<force_include_array {0}", rd.ReadLine());

            while (true)
            {
                // force_include
                string force_include = rd.ReadLine();
                if (force_include != "<force_include")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(force_include == "</force_include_array>");
                    wt.WriteLine("{0}", force_include);
                    break;
                }

                wt.WriteLine("\t{0} {1} {2}", force_include, rd.ReadLine(), rd.ReadLine());
            }
        }

        //---------------------------------------------------------------------
        // shading_model_array のフォーマット
        private static void Format_shading_model_array(TextReader rd, TextWriter wt)
        {
            // shading_model_array
            wt.WriteLine("<shading_model_array {0}", rd.ReadLine());

            while (true)
            {
                // shading_model
                string shading_model = rd.ReadLine();
                if (shading_model != "<shading_model")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(shading_model == "</shading_model_array>");
                    wt.WriteLine("{0}", shading_model);
                    break;
                }

                // shading_model は必ず子要素を持つ
                wt.WriteLine("\t{0} {1} {2} {3} {4}",
                    shading_model, rd.ReadLine(), rd.ReadLine(), rd.ReadLine(), rd.ReadLine());

                // vertex_stage
                string vertex_stage = rd.ReadLine();
                string geometry_stage;
                if (vertex_stage == "<vertex_stage")
                {
                    wt.WriteLine("\t\t{0} {1}", vertex_stage, rd.ReadLine());
                    geometry_stage = rd.ReadLine();
                }
                else
                {
                    geometry_stage = vertex_stage;
                }

                // geometry_stage
                string fragment_stage;
                if (geometry_stage == "<geometry_stage")
                {
                    wt.WriteLine("\t\t{0} {1}", geometry_stage, rd.ReadLine());
                    fragment_stage = rd.ReadLine();
                }
                else
                {
                    fragment_stage = geometry_stage;
                }

                // fragment_stage
                string compute_stage;
                if (fragment_stage == "<fragment_stage")
                {
                    wt.WriteLine("\t\t{0} {1}", fragment_stage, rd.ReadLine());
                    compute_stage = rd.ReadLine();
                }
                else
                {
                    compute_stage = fragment_stage;
                }

                // compute_stage
                string macro_array;
                if (compute_stage == "<compute_stage")
                {
                    wt.WriteLine("\t\t{0} {1}", compute_stage, rd.ReadLine());
                    macro_array = rd.ReadLine();
                }
                else
                {
                    macro_array = compute_stage;
                }

                // macro_array
                string option_var_array;
                if (macro_array == "<macro_array")
                {
                    IfTextFormatterUtility.Format_macro_array(rd, wt, "\t\t");
                    option_var_array = rd.ReadLine();
                }
                else
                {
                    option_var_array = macro_array;
                }

                // option_var_array
                string attrib_var_array;
                if (option_var_array == "<option_var_array")
                {
                    IfTextShaderDefinitionFormatter.Format_option_var_array(rd, wt);
                    attrib_var_array = rd.ReadLine();
                }
                else
                {
                    attrib_var_array = option_var_array;
                }

                // attrib_var_array
                string sampler_var_array;
                if (attrib_var_array == "<attrib_var_array")
                {
                    IfTextShaderDefinitionFormatter.Format_attrib_var_array(rd, wt);
                    sampler_var_array = rd.ReadLine();
                }
                else
                {
                    sampler_var_array = attrib_var_array;
                }

                // sampler_var_array
                string block_var_array;
                if (sampler_var_array == "<sampler_var_array")
                {
                    IfTextShaderDefinitionFormatter.Format_sampler_var_array(rd, wt);
                    block_var_array = rd.ReadLine();
                }
                else
                {
                    block_var_array = sampler_var_array;
                }

                // block_var_array
                string shader_storage_block_var_array;
                if (block_var_array == "<block_var_array")
                {
                    IfTextShaderDefinitionFormatter.Format_block_var_array(rd, wt);
                    shader_storage_block_var_array = rd.ReadLine();
                }
                else
                {
                    shader_storage_block_var_array = block_var_array;
                }

                // shader_storage_block_var_array
                string render_info_slot_array;
                if (shader_storage_block_var_array == "<shader_storage_block_var_array")
                {
                    IfTextShaderDefinitionFormatter.Format_shader_storage_block_var_array(rd, wt);
                    render_info_slot_array = rd.ReadLine();
                }
                else
                {
                    render_info_slot_array = shader_storage_block_var_array;
                }

                // render_info_slot_array
                string textblock_array;
                if (render_info_slot_array == "<render_info_slot_array")
                {
                    IfTextShaderDefinitionFormatter.Format_render_info_slot_array(rd, wt);
                    textblock_array = rd.ReadLine();
                }
                else
                {
                    textblock_array = render_info_slot_array;
                }

                // textblock_array
                string interleave_array;
                if (textblock_array == "<textblock_array")
                {
                    IfTextShaderDefinitionFormatter.Format_textblock_array(rd, wt);
                    interleave_array = rd.ReadLine();
                }
                else
                {
                    interleave_array = textblock_array;
                }

                // interleave_array
                string group_array;
                if (interleave_array == "<interleave_array")
                {
                    IfTextShaderDefinitionFormatter.Format_interleave_array(rd, wt);
                    group_array = rd.ReadLine();
                }
                else
                {
                    group_array = interleave_array;
                }

                // group_array
                string page_array;
                if (group_array == "<group_array")
                {
                    IfTextShaderDefinitionFormatter.Format_group_array(rd, wt);
                    page_array = rd.ReadLine();
                }
                else
                {
                    page_array = group_array;
                }

                // page_array
                string streamout;
                if (page_array == "<page_array")
                {
                    IfTextShaderDefinitionFormatter.Format_page_array(rd, wt);
                    streamout = rd.ReadLine();
                }
                else
                {
                    streamout = page_array;
                }

                // streamout
                string close_shading_model;
                if (streamout == "<streamout")
                {
                    wt.WriteLine("\t\t{0} {1}", streamout, rd.ReadLine());
                    close_shading_model = rd.ReadLine();
                }
                else
                {
                    close_shading_model = streamout;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_shading_model == "</shading_model>");
                wt.WriteLine("\t{0}", close_shading_model);
            }
        }

        //---------------------------------------------------------------------
        // option_var_array のフォーマット
        private static void Format_option_var_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<option_var_array {0}", rd.ReadLine());
            while (true)
            {
                // option_var
                string option_var = rd.ReadLine();
                if (option_var != "<option_var")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(option_var == "</option_var_array>");
                    wt.WriteLine("\t\t{0}", option_var);
                    break;
                }

                // option_var は必ず子要素を持つ
                wt.WriteLine("\t\t\t{0} {1} {2}",
                    option_var, rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}",
                    IfTextFormatterUtility.RemoveBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t>");

                // vertex_symbol
                string vertex_symbol = rd.ReadLine();
                string geometry_symbol;
                if (vertex_symbol == "<vertex_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", vertex_symbol, rd.ReadLine());
                    geometry_symbol = rd.ReadLine();
                }
                else
                {
                    geometry_symbol = vertex_symbol;
                }

                // geometry_symbol
                string fragment_symbol;
                if (geometry_symbol == "<geometry_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", geometry_symbol, rd.ReadLine());
                    fragment_symbol = rd.ReadLine();
                }
                else
                {
                    fragment_symbol = geometry_symbol;
                }

                // fragment_symbol
                string compute_symbol;
                if (fragment_symbol == "<fragment_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", fragment_symbol, rd.ReadLine());
                    compute_symbol = rd.ReadLine();
                }
                else
                {
                    compute_symbol = fragment_symbol;
                }

                // compute_symbol
                string ui_label;
                if (compute_symbol == "<compute_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", compute_symbol, rd.ReadLine());
                    ui_label = rd.ReadLine();
                }
                else
                {
                    ui_label = compute_symbol;
                }

                // ui_label
                string ui_comment;
                if (ui_label == "<ui_label")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_label, rd.ReadLine());
                    ui_comment = rd.ReadLine();
                }
                else
                {
                    ui_comment = ui_label;
                }

                // ui_comment
                string ui_group;
                if (ui_comment == "<ui_comment")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_comment, rd.ReadLine());
                    ui_group = rd.ReadLine();
                }
                else
                {
                    ui_group = ui_comment;
                }

                // ui_group
                string ui_order;
                if (ui_group == "<ui_group")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_group, rd.ReadLine());
                    ui_order = rd.ReadLine();
                }
                else
                {
                    ui_order = ui_group;
                }

                // ui_order
                string ui_item;
                if (ui_order == "<ui_order")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_order, rd.ReadLine());
                    ui_item = rd.ReadLine();
                }
                else
                {
                    ui_item = ui_order;
                }

                // ui_item
                string ui_visible;
                if (ui_item == "<ui_item")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_item, rd.ReadLine());
                    ui_visible = rd.ReadLine();
                }
                else
                {
                    ui_visible = ui_item;
                }

                // ui_visible
                string close_option_var;
                if (ui_visible == "<ui_visible")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_visible, rd.ReadLine());
                    close_option_var = rd.ReadLine();
                }
                else
                {
                    close_option_var = ui_visible;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_option_var == "</option_var>");
                wt.WriteLine("\t\t\t{0}", close_option_var);
            }
        }

        //---------------------------------------------------------------------
        // attrib_var_array のフォーマット
        private static void Format_attrib_var_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<attrib_var_array {0}", rd.ReadLine());
            while (true)
            {
                // attrib_var
                string attrib_var = rd.ReadLine();
                if (attrib_var != "<attrib_var")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(attrib_var == "</attrib_var_array>");
                    wt.WriteLine("\t\t{0}", attrib_var);
                    break;
                }

                // attrib_var は必ず子要素を持つ
                wt.WriteLine("\t\t\t{0} {1} {2}",
                    attrib_var, rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}",
                    IfTextFormatterUtility.RemoveBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t>");

                // vertex_symbol
                string vertex_symbol = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(vertex_symbol == "<vertex_symbol");
                wt.WriteLine("\t\t\t\t{0} {1}", vertex_symbol, rd.ReadLine());

                // ui_label
                string ui_label = rd.ReadLine();
                string ui_comment;
                if (ui_label == "<ui_label")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_label, rd.ReadLine());
                    ui_comment = rd.ReadLine();
                }
                else
                {
                    ui_comment = ui_label;
                }

                // ui_comment
                string ui_group;
                if (ui_comment == "<ui_comment")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_comment, rd.ReadLine());
                    ui_group = rd.ReadLine();
                }
                else
                {
                    ui_group = ui_comment;
                }

                // ui_group
                string ui_order;
                if (ui_group == "<ui_group")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_group, rd.ReadLine());
                    ui_order = rd.ReadLine();
                }
                else
                {
                    ui_order = ui_group;
                }

                // ui_order
                string ui_editable;
                if (ui_order == "<ui_order")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_order, rd.ReadLine());
                    ui_editable = rd.ReadLine();
                }
                else
                {
                    ui_editable = ui_order;
                }

                // ui_editable
                string close_attrib_var;
                if (ui_editable == "<ui_editable")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_editable, rd.ReadLine());
                    close_attrib_var = rd.ReadLine();
                }
                else
                {
                    close_attrib_var = ui_editable;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_attrib_var == "</attrib_var>");
                wt.WriteLine("\t\t\t{0}", close_attrib_var);
            }
        }

        //---------------------------------------------------------------------
        // sampler_var_array のフォーマット
        private static void Format_sampler_var_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<sampler_var_array {0}", rd.ReadLine());
            while (true)
            {
                // sampler_var
                string sampler_var = rd.ReadLine();
                if (sampler_var != "<sampler_var")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(sampler_var == "</sampler_var_array>");
                    wt.WriteLine("\t\t{0}", sampler_var);
                    break;
                }

                // sampler_var は必ず子要素を持つ
                wt.WriteLine("\t\t\t{0} {1} {2}",
                    sampler_var, rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}",
                    IfTextFormatterUtility.RemoveBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t>");

                // vertex_symbol
                string vertex_symbol = rd.ReadLine();
                string geometry_symbol;
                if (vertex_symbol == "<vertex_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", vertex_symbol, rd.ReadLine());
                    geometry_symbol = rd.ReadLine();
                }
                else
                {
                    geometry_symbol = vertex_symbol;
                }

                // geometry_symbol
                string fragment_symbol;
                if (geometry_symbol == "<geometry_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", geometry_symbol, rd.ReadLine());
                    fragment_symbol = rd.ReadLine();
                }
                else
                {
                    fragment_symbol = geometry_symbol;
                }

                // fragment_symbol
                string compute_symbol;
                if (fragment_symbol == "<fragment_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", fragment_symbol, rd.ReadLine());
                    compute_symbol = rd.ReadLine();
                }
                else
                {
                    compute_symbol = fragment_symbol;
                }

                // compute_symbol
                string ui_label;
                if (compute_symbol == "<compute_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", compute_symbol, rd.ReadLine());
                    ui_label = rd.ReadLine();
                }
                else
                {
                    ui_label = compute_symbol;
                }

                // ui_label
                string ui_comment;
                if (ui_label == "<ui_label")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_label, rd.ReadLine());
                    ui_comment = rd.ReadLine();
                }
                else
                {
                    ui_comment = ui_label;
                }

                // ui_comment
                string ui_group;
                if (ui_comment == "<ui_comment")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_comment, rd.ReadLine());
                    ui_group = rd.ReadLine();
                }
                else
                {
                    ui_group = ui_comment;
                }

                // ui_group
                string ui_order;
                if (ui_group == "<ui_group")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_group, rd.ReadLine());
                    ui_order = rd.ReadLine();
                }
                else
                {
                    ui_order = ui_group;
                }

                // ui_order
                string ui_visible;
                if (ui_order == "<ui_order")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_order, rd.ReadLine());
                    ui_visible = rd.ReadLine();
                }
                else
                {
                    ui_visible = ui_order;
                }

                // ui_visible
                string close_sampler_var;
                if (ui_visible == "<ui_visible")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_visible, rd.ReadLine());
                    close_sampler_var = rd.ReadLine();
                }
                else
                {
                    close_sampler_var = ui_visible;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_sampler_var == "</sampler_var>");
                wt.WriteLine("\t\t\t{0}", close_sampler_var);
            }
        }

        //---------------------------------------------------------------------
        // block_var_array のフォーマット
        private static void Format_block_var_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<block_var_array {0}", rd.ReadLine());
            while (true)
            {
                // block_var
                string block_var = rd.ReadLine();
                if (block_var != "<block_var")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(block_var == "</block_var_array>");
                    wt.WriteLine("\t\t{0}", block_var);
                    break;
                }

                // block_var は必ず子要素を持つ
                wt.WriteLine("\t\t\t{0} {1} {2}",
                    block_var, rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}",
                    IfTextFormatterUtility.RemoveBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t>");

                // vertex_symbol
                string vertex_symbol = rd.ReadLine();
                string geometry_symbol;
                if (vertex_symbol == "<vertex_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", vertex_symbol, rd.ReadLine());
                    geometry_symbol = rd.ReadLine();
                }
                else
                {
                    geometry_symbol = vertex_symbol;
                }

                // geometry_symbol
                string fragment_symbol;
                if (geometry_symbol == "<geometry_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", geometry_symbol, rd.ReadLine());
                    fragment_symbol = rd.ReadLine();
                }
                else
                {
                    fragment_symbol = geometry_symbol;
                }

                // fragment_symbol
                string compute_symbol;
                if (fragment_symbol == "<fragment_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", fragment_symbol, rd.ReadLine());
                    compute_symbol = rd.ReadLine();
                }
                else
                {
                    compute_symbol = fragment_symbol;
                }

                // compute_symbol
                string uniform_var_array;
                if (compute_symbol == "<compute_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", compute_symbol, rd.ReadLine());
                    uniform_var_array = rd.ReadLine();
                }
                else
                {
                    uniform_var_array = compute_symbol;
                }

                // uniform_var_array
                string close_block_var;
                if (uniform_var_array == "<uniform_var_array")
                {
                    IfTextShaderDefinitionFormatter.Format_uniform_var_array(rd, wt);
                    close_block_var = rd.ReadLine();
                }
                else
                {
                    close_block_var = uniform_var_array;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_block_var == "</block_var>");
                wt.WriteLine("\t\t\t{0}", close_block_var);
            }
        }

        //---------------------------------------------------------------------
        // uniform_var_array のフォーマット
        private static void Format_uniform_var_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t\t\t<uniform_var_array {0}", rd.ReadLine());
            while (true)
            {
                // uniform_var
                string uniform_var = rd.ReadLine();
                if (uniform_var != "<uniform_var")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(uniform_var == "</uniform_var_array>");
                    wt.WriteLine("\t\t\t\t{0}", uniform_var);
                    break;
                }

                // uniform_var は必ず子要素を持つ
                wt.WriteLine("\t\t\t\t\t{0} {1} {2}",
                    uniform_var, rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t\t\t{0}",
                    IfTextFormatterUtility.RemoveBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t\t\t>");

                // vertex_symbol
                string vertex_symbol = rd.ReadLine();
                string geometry_symbol;
                if (vertex_symbol == "<vertex_symbol")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", vertex_symbol, rd.ReadLine());
                    geometry_symbol = rd.ReadLine();
                }
                else
                {
                    geometry_symbol = vertex_symbol;
                }

                // geometry_symbol
                string fragment_symbol;
                if (geometry_symbol == "<geometry_symbol")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", geometry_symbol, rd.ReadLine());
                    fragment_symbol = rd.ReadLine();
                }
                else
                {
                    fragment_symbol = geometry_symbol;
                }

                // fragment_symbol
                string compute_symbol;
                if (fragment_symbol == "<fragment_symbol")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", fragment_symbol, rd.ReadLine());
                    compute_symbol = rd.ReadLine();
                }
                else
                {
                    compute_symbol = fragment_symbol;
                }

                // compute_symbol
                string ui_label;
                if (compute_symbol == "<compute_symbol")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", compute_symbol, rd.ReadLine());
                    ui_label = rd.ReadLine();
                }
                else
                {
                    ui_label = compute_symbol;
                }

                // ui_label
                string ui_comment;
                if (ui_label == "<ui_label")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", ui_label, rd.ReadLine());
                    ui_comment = rd.ReadLine();
                }
                else
                {
                    ui_comment = ui_label;
                }

                // ui_comment
                string ui_group;
                if (ui_comment == "<ui_comment")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", ui_comment, rd.ReadLine());
                    ui_group = rd.ReadLine();
                }
                else
                {
                    ui_group = ui_comment;
                }

                // ui_group
                string ui_order;
                if (ui_group == "<ui_group")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", ui_group, rd.ReadLine());
                    ui_order = rd.ReadLine();
                }
                else
                {
                    ui_order = ui_group;
                }

                // ui_order
                string ui_item;
                if (ui_order == "<ui_order")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", ui_order, rd.ReadLine());
                    ui_item = rd.ReadLine();
                }
                else
                {
                    ui_item = ui_order;
                }

                // ui_item
                string ui_min;
                if (ui_item == "<ui_item")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", ui_item, rd.ReadLine());
                    ui_min = rd.ReadLine();
                }
                else
                {
                    ui_min = ui_item;
                }

                // ui_min
                string ui_max;
                if (ui_min == "<ui_min")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", ui_min, rd.ReadLine());
                    ui_max = rd.ReadLine();
                }
                else
                {
                    ui_max = ui_min;
                }

                // ui_max
                string ui_visible;
                if (ui_max == "<ui_max")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", ui_max, rd.ReadLine());
                    ui_visible = rd.ReadLine();
                }
                else
                {
                    ui_visible = ui_max;
                }

                // ui_visible
                string ui_default_min;
                if (ui_visible == "<ui_visible")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", ui_visible, rd.ReadLine());
                    ui_default_min = rd.ReadLine();
                }
                else
                {
                    ui_default_min = ui_visible;
                }

                // ui_default_min
                string ui_default_max;
                if (ui_default_min == "<ui_default_min")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", ui_default_min, rd.ReadLine());
                    ui_default_max = rd.ReadLine();
                }
                else
                {
                    ui_default_max = ui_default_min;
                }

                // ui_default_max
                string close_uniform_var;
                if (ui_default_max == "<ui_default_max")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", ui_default_max, rd.ReadLine());
                    close_uniform_var = rd.ReadLine();
                }
                else
                {
                    close_uniform_var = ui_default_max;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_uniform_var == "</uniform_var>");
                wt.WriteLine("\t\t\t\t\t{0}", close_uniform_var);
            }
        }

        //---------------------------------------------------------------------
        // shader_storage_block_var_array のフォーマット
        private static void Format_shader_storage_block_var_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<shader_storage_block_var_array {0}", rd.ReadLine());
            while (true)
            {
                // block_var
                string block_var = rd.ReadLine();
                if (block_var != "<shader_storage_block_var")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(block_var == "</shader_storage_block_var_array>");
                    wt.WriteLine("\t\t{0}", block_var);
                    break;
                }

                // block_var は必ず子要素を持つ
                wt.WriteLine("\t\t\t{0} {1} {2}",
                    block_var, rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}",
                    IfTextFormatterUtility.RemoveBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t>");

                // vertex_symbol
                string vertex_symbol = rd.ReadLine();
                string geometry_symbol;
                if (vertex_symbol == "<vertex_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", vertex_symbol, rd.ReadLine());
                    geometry_symbol = rd.ReadLine();
                }
                else
                {
                    geometry_symbol = vertex_symbol;
                }

                // geometry_symbol
                string fragment_symbol;
                if (geometry_symbol == "<geometry_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", geometry_symbol, rd.ReadLine());
                    fragment_symbol = rd.ReadLine();
                }
                else
                {
                    fragment_symbol = geometry_symbol;
                }

                // fragment_symbol
                string compute_symbol;
                if (fragment_symbol == "<fragment_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", fragment_symbol, rd.ReadLine());
                    compute_symbol = rd.ReadLine();
                }
                else
                {
                    compute_symbol = fragment_symbol;
                }

                // compute_symbol
                string buffer_var_array;
                if (compute_symbol == "<compute_symbol")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", compute_symbol, rd.ReadLine());
                    buffer_var_array = rd.ReadLine();
                }
                else
                {
                    buffer_var_array = compute_symbol;
                }

                // buffer_var_array
                string close_shader_storage_block_var;
                if (buffer_var_array == "<buffer_var_array")
                {
                    IfTextShaderDefinitionFormatter.Format_buffer_var_array(rd, wt);
                    close_shader_storage_block_var = rd.ReadLine();
                }
                else
                {
                    close_shader_storage_block_var = buffer_var_array;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_shader_storage_block_var == "</shader_storage_block_var>");
                wt.WriteLine("\t\t\t{0}", close_shader_storage_block_var);
            }
        }

        //---------------------------------------------------------------------
        // buffer_var_array のフォーマット
        private static void Format_buffer_var_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t\t\t<buffer_var_array {0}", rd.ReadLine());
            while (true)
            {
                // buffer_var
                string buffer_var = rd.ReadLine();
                if (buffer_var != "<buffer_var")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(buffer_var == "</buffer_var_array>");
                    wt.WriteLine("\t\t\t\t{0}", buffer_var);
                    break;
                }

                // buffer_var は必ず子要素を持つ
                wt.WriteLine("\t\t\t\t\t{0} {1} {2}",
                    buffer_var, rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t\t\t{0}",
                    IfTextFormatterUtility.RemoveBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t\t\t>");

                // vertex_symbol
                string vertex_symbol = rd.ReadLine();
                string geometry_symbol;
                if (vertex_symbol == "<vertex_symbol")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", vertex_symbol, rd.ReadLine());
                    geometry_symbol = rd.ReadLine();
                }
                else
                {
                    geometry_symbol = vertex_symbol;
                }

                // geometry_symbol
                string fragment_symbol;
                if (geometry_symbol == "<geometry_symbol")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", geometry_symbol, rd.ReadLine());
                    fragment_symbol = rd.ReadLine();
                }
                else
                {
                    fragment_symbol = geometry_symbol;
                }

                // fragment_symbol
                string compute_symbol;
                if (fragment_symbol == "<fragment_symbol")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", fragment_symbol, rd.ReadLine());
                    compute_symbol = rd.ReadLine();
                }
                else
                {
                    compute_symbol = fragment_symbol;
                }

                // compute_symbol
                string close_buffer_var;
                if (compute_symbol == "<compute_symbol")
                {
                    wt.WriteLine("\t\t\t\t\t\t{0} {1}", compute_symbol, rd.ReadLine());
                    close_buffer_var = rd.ReadLine();
                }
                else
                {
                    close_buffer_var = compute_symbol;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_buffer_var == "</buffer_var>");
                wt.WriteLine("\t\t\t\t\t{0}", close_buffer_var);
            }
        }

        //---------------------------------------------------------------------
        // render_info_slot_array のフォーマット
        private static void Format_render_info_slot_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<render_info_slot_array {0}", rd.ReadLine());
            while (true)
            {
                // render_info_slot
                string render_info_slot = rd.ReadLine();
                if (render_info_slot != "<render_info_slot")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(render_info_slot == "</render_info_slot_array>");
                    wt.WriteLine("\t\t{0}", render_info_slot);
                    break;
                }

                wt.WriteLine("\t\t\t{0} {1} {2}",
                    render_info_slot, rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());

                string default_ = rd.ReadLine();
                if (default_.EndsWith(" />"))
                {
                    wt.WriteLine("\t\t\t\t{0}",
                        IfTextFormatterUtility.RemoveSlashBracket(default_));
                    wt.WriteLine("\t\t\t/>");
                    continue;
                }
                else
                {
                    wt.WriteLine("\t\t\t\t{0}",
                        IfTextFormatterUtility.RemoveBracket(default_));
                    wt.WriteLine("\t\t\t>");
                }

                // ui_label
                string ui_label = rd.ReadLine();
                string ui_comment;
                if (ui_label == "<ui_label")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_label, rd.ReadLine());
                    ui_comment = rd.ReadLine();
                }
                else
                {
                    ui_comment = ui_label;
                }

                // ui_comment
                string ui_group;
                if (ui_comment == "<ui_comment")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_comment, rd.ReadLine());
                    ui_group = rd.ReadLine();
                }
                else
                {
                    ui_group = ui_comment;
                }

                // ui_group
                string ui_order;
                if (ui_group == "<ui_group")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_group, rd.ReadLine());
                    ui_order = rd.ReadLine();
                }
                else
                {
                    ui_order = ui_group;
                }

                // ui_order
                string ui_item;
                if (ui_order == "<ui_order")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_order, rd.ReadLine());
                    ui_item = rd.ReadLine();
                }
                else
                {
                    ui_item = ui_order;
                }

                // ui_item
                string ui_visible;
                if (ui_item == "<ui_item")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_item, rd.ReadLine());
                    ui_visible = rd.ReadLine();
                }
                else
                {
                    ui_visible = ui_item;
                }

                // ui_visible
                string close_render_info_slot;
                if (ui_visible == "<ui_visible")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_visible, rd.ReadLine());
                    close_render_info_slot = rd.ReadLine();
                }
                else
                {
                    close_render_info_slot = ui_visible;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_render_info_slot == "</render_info_slot>");
                wt.WriteLine("\t\t\t{0}", close_render_info_slot);
            }
        }

        //---------------------------------------------------------------------
        // textblock_array のフォーマット
        private static void Format_textblock_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<textblock_array {0}", rd.ReadLine());
            while (true)
            {
                // textblock
                string textblock = rd.ReadLine();
                if (textblock != "<textblock")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(textblock == "</textblock_array>");
                    wt.WriteLine("\t\t{0}", textblock);
                    break;
                }

                wt.WriteLine("\t\t\t{0} {1}",
                    textblock, rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());

                string default_ = rd.ReadLine();
                if (default_.EndsWith(" />"))
                {
                    wt.WriteLine("\t\t\t\t{0}",
                        IfTextFormatterUtility.RemoveSlashBracket(default_));
                    wt.WriteLine("\t\t\t/>");
                    continue;
                }
                else
                {
                    wt.WriteLine("\t\t\t\t{0}",
                        IfTextFormatterUtility.RemoveBracket(default_));
                    wt.WriteLine("\t\t\t>");
                }

                // ui_group
                string ui_group = rd.ReadLine();
                string ui_order;
                if (ui_group == "<ui_group")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_group, rd.ReadLine());
                    ui_order = rd.ReadLine();
                }
                else
                {
                    ui_order = ui_group;
                }

                // ui_order
                string close_textblock;
                if (ui_order == "<ui_order")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_order, rd.ReadLine());
                    close_textblock = rd.ReadLine();
                }
                else
                {
                    close_textblock = ui_order;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_textblock == "</textblock>");
                wt.WriteLine("\t\t\t{0}", close_textblock);
            }
        }


        //---------------------------------------------------------------------
        // interleave_array のフォーマット
        private static void Format_interleave_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<interleave_array {0}", rd.ReadLine());
            while (true)
            {
                // interleave
                string interleave = rd.ReadLine();
                if (interleave != "<interleave")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(interleave == "</interleave_array>");
                    wt.WriteLine("\t\t{0}", interleave);
                    break;
                }
                wt.WriteLine("\t\t\t{0} {1} {2} {3}",
                    interleave, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
            }
        }

        //---------------------------------------------------------------------
        // group_array のフォーマット
        private static void Format_group_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<group_array {0}", rd.ReadLine());
            while (true)
            {
                // group
                string group = rd.ReadLine();
                if (group != "<group")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(group == "</group_array>");
                    wt.WriteLine("\t\t{0}", group);
                    break;
                }
                string index = rd.ReadLine();
                string name = rd.ReadLine();
                string page_name = rd.ReadLine();
                string condition = rd.ReadLine();
                wt.WriteLine("\t\t\t{0} {1} {2} {3} {4}",
                    group, index, name, page_name, condition);
                if (condition.EndsWith(" />")) { continue; }

                // ui_label
                string ui_label = rd.ReadLine();
                string ui_comment;
                if (ui_label == "<ui_label")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_label, rd.ReadLine());
                    ui_comment = rd.ReadLine();
                }
                else
                {
                    ui_comment = ui_label;
                }

                // ui_comment
                string ui_order;
                if (ui_comment == "<ui_comment")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_comment, rd.ReadLine());
                    ui_order = rd.ReadLine();
                }
                else
                {
                    ui_order = ui_comment;
                }

                // ui_order
                string ui_group;
                if (ui_order == "<ui_order")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_order, rd.ReadLine());
                    ui_group = rd.ReadLine();
                }
                else
                {
                    ui_group = ui_order;
                }

                // ui_group
                string close_group;
                if (ui_group == "<ui_group")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_group, rd.ReadLine());
                    close_group = rd.ReadLine();
                }
                else
                {
                    close_group = ui_group;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_group == "</group>");
                wt.WriteLine("\t\t\t{0}", close_group);
            }
        }

        //---------------------------------------------------------------------
        // page_array のフォーマット
        private static void Format_page_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<page_array {0}", rd.ReadLine());
            while (true)
            {
                // page
                string page = rd.ReadLine();
                if (page != "<page")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(page == "</page_array>");
                    wt.WriteLine("\t\t{0}", page);
                    break;
                }
                string index = rd.ReadLine();
                string name = rd.ReadLine();
                wt.WriteLine("\t\t\t{0} {1} {2}",
                    page, index, name);
                if (name.EndsWith(" />")) { continue; }

                // ui_label
                string ui_label = rd.ReadLine();
                string ui_comment;
                if (ui_label == "<ui_label")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_label, rd.ReadLine());
                    ui_comment = rd.ReadLine();
                }
                else
                {
                    ui_comment = ui_label;
                }

                // ui_comment
                string ui_order;
                if (ui_comment == "<ui_comment")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_comment, rd.ReadLine());
                    ui_order = rd.ReadLine();
                }
                else
                {
                    ui_order = ui_comment;
                }

                // ui_order
                string close_page;
                if (ui_order == "<ui_order")
                {
                    wt.WriteLine("\t\t\t\t{0} {1}", ui_order, rd.ReadLine());
                    close_page = rd.ReadLine();
                }
                else
                {
                    close_page = ui_order;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_page == "</page>");
                wt.WriteLine("\t\t\t{0}", close_page);
            }
        }

        //---------------------------------------------------------------------
        // shader_src_array のフォーマット
        private static void Format_shader_src_array(TextReader rd, TextWriter wt)
        {
            // shader_src_array
            wt.WriteLine("<shader_src_array {0}", rd.ReadLine());

            while (true)
            {
                // shader_src
                string shader_src = rd.ReadLine();
                if (shader_src != "<shader_src")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(shader_src == "</shader_src_array>");
                    wt.WriteLine("{0}", shader_src);
                    break;
                }

                wt.WriteLine("\t{0} {1}", shader_src, rd.ReadLine());
                wt.WriteLine("\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t{0}",
                    IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
                wt.WriteLine("\t/>");
            }
        }
    }
}
