﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Diagnostics;
using System.IO;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // テキストシェーダー設定中間ファイルフォーマッタ
    internal static class IfTextShaderConfigFormatter
    {
        // shader_config のフォーマット
        internal static void Format(TextReader rd, TextWriter wt)
        {
            // shader_config
            wt.WriteLine($"<{G3dConstant.ShaderConfigElementName} {rd.ReadLine()}");

            // shader_config_info
            string shader_config_info = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(shader_config_info == "<shader_config_info");
            wt.WriteLine("{0}", shader_config_info);
            wt.WriteLine("\t{0}", IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
            wt.WriteLine("/>");

            // include_path
            string include_path_array = rd.ReadLine();
            string force_include_file_array;
            if (include_path_array == "<include_path_array")
            {
                IfTextShaderConfigFormatter.Format_include_path_array(rd, wt);
                force_include_file_array = rd.ReadLine();
            }
            else
            {
                force_include_file_array = include_path_array;
            }

            // force_include_file
            string shader_array;
            if (force_include_file_array == "<force_include_file_array")
            {
                IfTextShaderConfigFormatter.Format_force_include_file_array(rd, wt);
                shader_array = rd.ReadLine();
            }
            else
            {
                shader_array = force_include_file_array;
            }

            // shader
            Nintendo.Foundation.Contracts.Assertion.Operation.True(shader_array == "<shader_array");
            IfTextShaderConfigFormatter.Format_shader_array(rd, wt);

            // stream
            string stream_array = rd.ReadLine();
            string tool_data;
            if (stream_array == "<stream_array")
            {
                IfTextFormatterUtility.Format_stream(rd, wt);
                tool_data = rd.ReadLine();
            }
            else
            {
                tool_data = stream_array;
            }

            // tool_data
            string close_shader_config;
            if (tool_data == "<tool_data>")
            {
                IfTextFormatterUtility.FormatToolData(rd, wt, string.Empty);
                close_shader_config = rd.ReadLine();
            }
            else
            {
                close_shader_config = tool_data;
            }

            Nintendo.Foundation.Contracts.Assertion.Operation.True(close_shader_config == "</shader_config>");
            wt.WriteLine(close_shader_config);
        }

        //---------------------------------------------------------------------
        // include_path_array のフォーマット
        private static void Format_include_path_array(TextReader rd, TextWriter wt)
        {
            // include_path_array
            wt.WriteLine("<include_path_array {0}", rd.ReadLine());

            while (true)
            {
                // include_path
                string include_path = rd.ReadLine();
                if (include_path != "<include_path")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(include_path == "</include_path_array>");
                    wt.WriteLine("{0}", include_path);
                    break;
                }

                wt.WriteLine("\t{0} {1} {2}", include_path, rd.ReadLine(), rd.ReadLine());
            }
        }

        //---------------------------------------------------------------------
        // force_include_file_array のフォーマット
        private static void Format_force_include_file_array(TextReader rd, TextWriter wt)
        {
            // force_include_file_array
            wt.WriteLine("<force_include_file_array {0}", rd.ReadLine());

            while (true)
            {
                // force_include_file
                string force_include_file = rd.ReadLine();
                if (force_include_file != "<force_include_file")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(force_include_file == "</force_include_file_array>");
                    wt.WriteLine("{0}", force_include_file);
                    break;
                }

                wt.WriteLine("\t{0} {1} {2}", force_include_file, rd.ReadLine(), rd.ReadLine());
            }
        }

        //---------------------------------------------------------------------
        // shader_array のフォーマット
        private static void Format_shader_array(TextReader rd, TextWriter wt)
        {
            // shader_array
            wt.WriteLine("<shader_array {0}", rd.ReadLine());

            while (true)
            {
                // shader
                string shader = rd.ReadLine();
                if (shader != "<shader")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(shader == "</shader_array>");
                    wt.WriteLine("{0}", shader);
                    break;
                }

                // shader は必ず子要素を持つ
                wt.WriteLine("\t{0} {1} {2} {3}",
                    shader, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());

                // vertex_shader
                string vertex_shader = rd.ReadLine();
                string geometry_shader;
                if (vertex_shader == "<vertex_shader")
                {
                    wt.WriteLine("\t\t{0} {1}", vertex_shader, rd.ReadLine());
                    geometry_shader = rd.ReadLine();
                }
                else
                {
                    geometry_shader = vertex_shader;
                }

                // geometry_shader
                string fragment_shader;
                if (geometry_shader == "<geometry_shader")
                {
                    wt.WriteLine("\t\t{0} {1}", geometry_shader, rd.ReadLine());
                    fragment_shader = rd.ReadLine();
                }
                else
                {
                    fragment_shader = geometry_shader;
                }

                // fragment_shader
                string compute_shader;
                if (fragment_shader == "<fragment_shader")
                {
                    wt.WriteLine("\t\t{0} {1}", fragment_shader, rd.ReadLine());
                    compute_shader = rd.ReadLine();
                }
                else
                {
                    compute_shader = fragment_shader;
                }

                // compute_shader
                string macro_array;
                if (compute_shader == "<compute_shader")
                {
                    wt.WriteLine("\t\t{0} {1}", compute_shader, rd.ReadLine());
                    macro_array = rd.ReadLine();
                }
                else
                {
                    macro_array = compute_shader;
                }

                // macro_array
                string variation_array;
                if (macro_array == "<macro_array")
                {
                    IfTextFormatterUtility.Format_macro_array(rd, wt, "\t\t");
                    variation_array = rd.ReadLine();
                }
                else
                {
                    variation_array = macro_array;
                }

                // variation_array
                string close_shader;
                if (variation_array == "<variation_array")
                {
                    IfTextShaderConfigFormatter.Format_variation_array(rd, wt);
                    close_shader = rd.ReadLine();
                }
                else
                {
                    close_shader = variation_array;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_shader == "</shader>");
                wt.WriteLine("\t{0}", close_shader);
            }
        }

        //---------------------------------------------------------------------
        // variation_array のフォーマット
        private static void Format_variation_array(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<variation_array {0}", rd.ReadLine());
            while (true)
            {
                // variation
                string variation = rd.ReadLine();
                if (variation != "<variation")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(variation == "</variation_array>");
                    wt.WriteLine("\t\t{0}", variation);
                    break;
                }
                wt.WriteLine("\t\t\t{0} {1} {2}",
                    variation, rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}",
                    IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t/>");
            }
        }
    }
}
