﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Diagnostics;
using System.IO;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // テキストモデル中間ファイルフォーマッタ
    internal static class IfTextModelFormatter
    {
        // model のフォーマット
        internal static void Format(TextReader rd, TextWriter wt)
        {
            // model
            wt.WriteLine($"<{G3dConstant.ModelElementName} {rd.ReadLine()}");

            // process_log
            string process_log_array = rd.ReadLine();
            string model_info;
            if (process_log_array == "<process_log_array")
            {
                IfTextFormatterUtility.Format_process_log_array(rd, wt);
                model_info = rd.ReadLine();
            }
            else
            {
                model_info = process_log_array;
            }

            // model_info
            Nintendo.Foundation.Contracts.Assertion.Operation.True(model_info == "<model_info");
            wt.WriteLine("{0}", model_info);
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", rd.ReadLine());
            wt.WriteLine("\t{0}", IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
            wt.WriteLine("/>");

            // material
            string material_array = rd.ReadLine();
            string skeleton;
            if (material_array == "<material_array")
            {
                FormatMaterialArray(rd, wt);
                skeleton = rd.ReadLine();
            }
            else
            {
                skeleton = material_array;
            }

            // skeleton
            Nintendo.Foundation.Contracts.Assertion.Operation.True(skeleton == "<skeleton>");
            FormatSkeleton(rd, wt);

            // vertex
            string vertex_array = rd.ReadLine();
            string shape_array;
            if (vertex_array == "<vertex_array")
            {
                FormatVertex(rd, wt);
                shape_array = rd.ReadLine();
            }
            else
            {
                shape_array = vertex_array;
            }

            // shape
            string original_material_array;
            if (shape_array == "<shape_array")
            {
                FormatShape(rd, wt);
                original_material_array = rd.ReadLine();
            }
            else
            {
                original_material_array = shape_array;
            }

            // original_material
            string user_data_array;
            if (original_material_array == "<original_material_array")
            {
                FormatOriginalMaterial(rd, wt);
                user_data_array = rd.ReadLine();
            }
            else
            {
                user_data_array = original_material_array;
            }

            // user_data
            string stream_array;
            if (user_data_array == "<user_data_array")
            {
                IfTextFormatterUtility.Format_user_data_array(rd, wt, string.Empty);
                stream_array = rd.ReadLine();
            }
            else
            {
                stream_array = user_data_array;
            }

            // stream
            string comment;
            if (stream_array == "<stream_array")
            {
                IfTextFormatterUtility.Format_stream(rd, wt);
                comment = rd.ReadLine();
            }
            else
            {
                comment = stream_array;
            }

            // comment
            string tool_data;
            if (comment == "<comment")
            {
                IfTextFormatterUtility.FormatComment(rd, wt, string.Empty);
                tool_data = rd.ReadLine();
            }
            else
            {
                tool_data = comment;
            }

            // tool_data
            string user_tool_data;
            if (tool_data == "<tool_data>")
            {
                IfTextFormatterUtility.FormatToolData(rd, wt, string.Empty);
                user_tool_data = rd.ReadLine();
            }
            else
            {
                user_tool_data = tool_data;
            }

            // user_tool_data
            string close_model;
            if (user_tool_data == "<user_tool_data>")
            {
                IfTextFormatterUtility.FormatUserToolData(rd, wt, string.Empty);
                close_model = rd.ReadLine();
            }
            else
            {
                close_model = user_tool_data;
            }

            Nintendo.Foundation.Contracts.Assertion.Operation.True(close_model == "</model>");
            wt.WriteLine("{0}", close_model);
        }

        //---------------------------------------------------------------------
        #region material

        private static void FormatMaterialArray(TextReader rd, TextWriter wt)
        {
            // material_array
            wt.WriteLine("<material_array {0}", rd.ReadLine());

            while (true)
            {
                string material = rd.ReadLine();
                if (material != "<material")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(material == "</material_array>");
                    wt.WriteLine(material);
                    break;
                }

                FormatMaterial(rd, wt);
            }
        }

        // material のフォーマット
        internal static void FormatMaterial(TextReader rd, TextWriter wt)
        {
            // material
            wt.Write($"\t<{G3dConstant.MaterialElementName} {rd.ReadLine()} {rd.ReadLine()}");

            // マテリアル中間ファイルを試験するために version を非必須属性にしてあるため、ない場合も考慮する
            string version = rd.ReadLine();
            string process_log_array;
            if (version.StartsWith("version"))
            {
                wt.WriteLine($" {version}");
                process_log_array = rd.ReadLine();
            }
            else
            {
                wt.WriteLine();
                process_log_array = version;
            }

            // process_log
            string material_info;
            if (process_log_array == "<process_log_array")
            {
                IfTextFormatterUtility.Format_process_log_array(rd, wt, "\t\t");
                material_info = rd.ReadLine();
            }
            else
            {
                material_info = process_log_array;
            }

            // material_info
            Nintendo.Foundation.Contracts.Assertion.Operation.True(material_info == "<material_info", material_info);
            wt.WriteLine("\t\t{0}", material_info);
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t{0}",
                IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
            wt.WriteLine("\t\t/>");

            // render_state
            string render_state = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(render_state == "<render_state", render_state);
            FormatRenderState(rd, wt);

            // sampler_array
            string sampler_array = rd.ReadLine();
            string shader_assign;
            if (sampler_array == "<sampler_array")
            {
                wt.WriteLine("\t\t{0} {1}", sampler_array, rd.ReadLine());
                FormatSampler(rd, wt);
                shader_assign = rd.ReadLine();
            }
            else
            {
                shader_assign = sampler_array;
            }

            string user_data_array;
            if (shader_assign == "<shader_assign")
            {
                string shader_archive = rd.ReadLine();
                string shader_name = rd.ReadLine();
                string revision = rd.ReadLine();
                wt.WriteLine("\t\t{0} {1} {2} {3}",
                    shader_assign, shader_archive, shader_name, revision);

                if (revision.EndsWith(" />"))
                {
                    user_data_array = rd.ReadLine();
                }
                else
                {
                    // render_info_array
                    string render_info_array = rd.ReadLine();
                    string shader_option_array;
                    if (render_info_array == "<render_info_array")
                    {
                        FormatRenderInfo(rd, wt);
                        shader_option_array = rd.ReadLine();
                    }
                    else
                    {
                        shader_option_array = render_info_array;
                    }

                    // shader_option
                    string sampler_assign_array;
                    if (shader_option_array == "<shader_option_array")
                    {
                        wt.WriteLine("\t\t\t{0} {1}", shader_option_array, rd.ReadLine());
                        FormatShaderOption(rd, wt);
                        sampler_assign_array = rd.ReadLine();
                    }
                    else
                    {
                        sampler_assign_array = shader_option_array;
                    }

                    // sampler_assign
                    string shader_param_array;
                    if (sampler_assign_array == "<sampler_assign_array")
                    {
                        wt.WriteLine("\t\t\t{0} {1}", sampler_assign_array, rd.ReadLine());
                        FormatSamplerAssign(rd, wt);
                        shader_param_array = rd.ReadLine();
                    }
                    else
                    {
                        shader_param_array = sampler_assign_array;
                    }

                    // shader_param
                    string attrib_assign_array;
                    if (shader_param_array == "<shader_param_array")
                    {
                        wt.WriteLine("\t\t\t{0} {1}", shader_param_array, rd.ReadLine());
                        FormatShaderParam(rd, wt);
                        attrib_assign_array = rd.ReadLine();
                    }
                    else
                    {
                        attrib_assign_array = shader_param_array;
                    }

                    // attrib_assign
                    string close_shader_assign;
                    if (attrib_assign_array == "<attrib_assign_array")
                    {
                        wt.WriteLine("\t\t\t{0} {1}", attrib_assign_array, rd.ReadLine());
                        FormatAttribAssign(rd, wt);
                        close_shader_assign = rd.ReadLine();
                    }
                    else
                    {
                        close_shader_assign = attrib_assign_array;
                    }

                    Nintendo.Foundation.Contracts.Assertion.Operation.True(close_shader_assign == "</shader_assign>");
                    wt.WriteLine("\t\t{0}", close_shader_assign);

                    user_data_array = rd.ReadLine();
                }
            }
            else
            {
                user_data_array = shader_assign;
            }

            // user_data
            string stream_array;
            if (user_data_array == "<user_data_array")
            {
                IfTextFormatterUtility.Format_user_data_array(rd, wt, "\t\t");
                stream_array = rd.ReadLine();
            }
            else
            {
                stream_array = user_data_array;
            }

            // stream
            string comment;
            if (stream_array == "<stream_array")
            {
                IfTextFormatterUtility.Format_stream(rd, wt, "\t\t");
                comment = rd.ReadLine();
            }
            else
            {
                comment = stream_array;
            }

            // comment
            string tool_data;
            if (comment == "<comment")
            {
                IfTextFormatterUtility.FormatComment(rd, wt, "\t\t");
                tool_data = rd.ReadLine();
            }
            else
            {
                tool_data = comment;
            }

            // tool_data
            string user_tool_data;
            if (tool_data == "<tool_data>")
            {
                IfTextFormatterUtility.FormatToolData(rd, wt, "\t\t");
                user_tool_data = rd.ReadLine();
            }
            else
            {
                user_tool_data = tool_data;
            }

            // user_tool_data
            string close_material;
            if (user_tool_data == "<user_tool_data>")
            {
                IfTextFormatterUtility.FormatUserToolData(rd, wt, "\t\t");
                close_material = rd.ReadLine();
            }
            else
            {
                close_material = user_tool_data;
            }

            Nintendo.Foundation.Contracts.Assertion.Operation.True(close_material == "</material>");
            wt.WriteLine("\t{0}", close_material);
        }

        // render_info のフォーマット
        private static void FormatRenderInfo(TextReader rd, TextWriter wt)
        {
            // render_info_array
            wt.WriteLine("\t\t\t<render_info_array {0}", rd.ReadLine());

            while (true)
            {
                // render_info
                string render_info = rd.ReadLine();
                if (render_info != "<render_info")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(render_info == "</render_info_array>");
                    wt.WriteLine("\t\t\t{0}", render_info);
                    break;
                }
                string index = rd.ReadLine();
                string name = rd.ReadLine();
                string type = rd.ReadLine();
                string count = rd.ReadLine();
                wt.WriteLine("\t\t\t\t{0} {1} {2} {3} {4}",
                    render_info, index, name, type, count);
                if (count.EndsWith(" />") || count.EndsWith("</render_info>")) { continue; }

                while (true)
                {
                    string ref_name = rd.ReadLine();
                    wt.WriteLine(ref_name);
                    if (ref_name.EndsWith("</render_info>")) { break; }
                }
            }
        }

        // render_state のフォーマット
        private static void FormatRenderState(TextReader rd, TextWriter wt)
        {
            // render_state
            wt.WriteLine("\t\t<render_state {0} {1} {2}",
                rd.ReadLine(), rd.ReadLine(), rd.ReadLine());

            // depth_test
            string depth_test = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(depth_test == "<depth_test");
            wt.WriteLine("\t\t\t{0} {1} {2} {3}",
                depth_test, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());

            // alpha_test
            string alpha_test = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(alpha_test == "<alpha_test");
            wt.WriteLine("\t\t\t{0} {1} {2} {3}",
                alpha_test, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());

            // color_blend
            string color_blend = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(color_blend == "<color_blend");
            wt.WriteLine("\t\t\t{0}", color_blend);
            wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t\t{0}",
                IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
            wt.WriteLine("\t\t\t/>");

            // logical_blend
            string logical_blend = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(logical_blend == "<logical_blend");
            wt.WriteLine("\t\t\t{0} {1}", logical_blend, rd.ReadLine());

            string close_render_state = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(close_render_state == "</render_state>");
            wt.WriteLine("\t\t{0}", close_render_state);
        }

        // sampler のフォーマット
        private static void FormatSampler(TextReader rd, TextWriter wt)
        {
            while (true)
            {
                // sampler
                string sampler = rd.ReadLine();
                if (sampler != "<sampler")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(sampler == "</sampler_array>");
                    wt.WriteLine("\t\t{0}", sampler);
                    break;
                }
                wt.WriteLine("\t\t\t{0} {1} {2} {3} {4}",
                    sampler, rd.ReadLine(), rd.ReadLine(), rd.ReadLine(), rd.ReadLine());

                // wrap
                string wrap = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(wrap == "<wrap");
                wt.WriteLine("\t\t\t\t{0} {1} {2} {3}",
                    wrap, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());

                // filter
                string filter = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(filter == "<filter");
                wt.WriteLine("\t\t\t\t{0} {1} {2} {3} {4}",
                    filter, rd.ReadLine(), rd.ReadLine(), rd.ReadLine(), rd.ReadLine());

                // lod
                string lod = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(lod == "<lod");
                wt.WriteLine("\t\t\t\t{0} {1} {2} {3}",
                    lod, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());

                string close_sampler = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_sampler == "</sampler>");
                wt.WriteLine("\t\t\t{0}", close_sampler);
            }
        }

        // shader_option のフォーマット
        private static void FormatShaderOption(TextReader rd, TextWriter wt)
        {
            while (true)
            {
                // shader_option
                string shader_option = rd.ReadLine();
                if (shader_option != "<shader_option")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(shader_option == "</shader_option_array>");
                    wt.WriteLine("\t\t\t{0}", shader_option);
                    break;
                }
                wt.WriteLine("\t\t\t\t{0} {1} {2} {3}",
                    shader_option, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
            }
        }

        // sampler_assign のフォーマット
        private static void FormatSamplerAssign(TextReader rd, TextWriter wt)
        {
            while (true)
            {
                // sampler_assign
                string sampler_assign = rd.ReadLine();
                if (sampler_assign != "<sampler_assign")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(sampler_assign == "</sampler_assign_array>");
                    wt.WriteLine("\t\t\t{0}", sampler_assign);
                    break;
                }
                wt.WriteLine("\t\t\t\t{0} {1} {2} {3}",
                    sampler_assign, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
            }
        }

        // shader_param のフォーマット
        private static void FormatShaderParam(TextReader rd, TextWriter wt)
        {
            while (true)
            {
                // shader_param
                string shader_param = rd.ReadLine();
                if (shader_param != "<shader_param")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(shader_param == "</shader_param_array>");
                    wt.WriteLine("\t\t\t{0}", shader_param);
                    break;
                }
                wt.WriteLine("\t\t\t\t{0} {1} {2} {3}",
                    shader_param, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t\t{0}", IfTextFormatterUtility.RemoveBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t\t>");

                while (true)
                {
                    string close_shader_param = rd.ReadLine();
                    wt.WriteLine(close_shader_param);
                    if (close_shader_param.EndsWith("</shader_param>")) { break; }
                }
            }
        }

        // attrib_assign のフォーマット
        private static void FormatAttribAssign(TextReader rd, TextWriter wt)
        {
            while (true)
            {
                // attrib_assign
                string attrib_assign = rd.ReadLine();
                if (attrib_assign != "<attrib_assign")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(attrib_assign == "</attrib_assign_array>");
                    wt.WriteLine("\t\t\t{0}", attrib_assign);
                    break;
                }
                wt.WriteLine("\t\t\t\t{0} {1} {2} {3}",
                    attrib_assign, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
            }
        }
        #endregion

        //---------------------------------------------------------------------
        #region skeleton
        // skeleton のフォーマット
        private static void FormatSkeleton(TextReader rd, TextWriter wt)
        {
            // skeleton
            wt.WriteLine("<skeleton>");

            // skeleton_info
            string skeleton_info = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(skeleton_info == "<skeleton_info");
            wt.WriteLine("\t{0}", skeleton_info);
            wt.WriteLine("\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t{0}", IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
            wt.WriteLine("\t/>");

            // bone_array
            string bone_array = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(bone_array == "<bone_array");
            wt.WriteLine("\t{0} {1}", bone_array, rd.ReadLine());

            while (true)
            {
                string bone = rd.ReadLine();
                if (bone != "<bone")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(bone == "</bone_array>");
                    wt.WriteLine("\t{0}", bone);
                    break;
                }

                // bone
                Nintendo.Foundation.Contracts.Assertion.Operation.True(bone == "<bone");
                FormatBone(rd, wt);
            }

            // skeleton
            string close_skeleton = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(close_skeleton == "</skeleton>");
            wt.WriteLine(close_skeleton);
        }

        // bone のフォーマット
        private static void FormatBone(TextReader rd, TextWriter wt)
        {
            wt.WriteLine("\t\t<bone {0} {1} {2}",
                rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());
            wt.WriteLine("\t\t\t{0}", rd.ReadLine());

            // inv_model_matrix 対策
            string type = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(type.StartsWith("type="));
            bool hasChild = !type.EndsWith(" />");
            if (hasChild)
            {
                wt.WriteLine("\t\t\t{0}", IfTextFormatterUtility.RemoveBracket(type));
                wt.WriteLine("\t\t>");

                // inv_model_matrix
                string inv_model_matrix = rd.ReadLine();
                string user_data_array;
                if (inv_model_matrix == "<inv_model_matrix>")
                {
                    wt.WriteLine("\t\t\t{0}", inv_model_matrix);
                    wt.WriteLine("{0}", rd.ReadLine());
                    wt.WriteLine("{0}", rd.ReadLine());
                    wt.WriteLine("{0}", rd.ReadLine());
                    wt.WriteLine("{0}", rd.ReadLine());

                    user_data_array = rd.ReadLine();
                }
                else
                {
                    user_data_array = inv_model_matrix;
                }

                // user_data
                string comment;
                if (user_data_array == "<user_data_array")
                {
                    IfTextFormatterUtility.Format_user_data_array(rd, wt, "\t\t\t");
                    comment = rd.ReadLine();
                }
                else
                {
                    comment = user_data_array;
                }

                // comment
                string tool_data;
                if (comment == "<comment")
                {
                    IfTextFormatterUtility.FormatComment(rd, wt, "\t\t\t");
                    tool_data = rd.ReadLine();
                }
                else
                {
                    tool_data = comment;
                }

                // tool_data
                string user_tool_data;
                if (tool_data == "<tool_data>")
                {
                    IfTextFormatterUtility.FormatToolData(rd, wt, "\t\t\t");
                    user_tool_data = rd.ReadLine();
                }
                else
                {
                    user_tool_data = tool_data;
                }

                string close_bone;
                if (user_tool_data == "<user_tool_data>")
                {
                    IfTextFormatterUtility.FormatUserToolData(rd, wt, "\t\t\t");
                    close_bone = rd.ReadLine();
                }
                else
                {
                    close_bone = user_tool_data;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_bone == "</bone>");
                wt.WriteLine("\t\t{0}", close_bone);
            }
            else
            {
                wt.WriteLine("\t\t\t{0}",
                    IfTextFormatterUtility.RemoveSlashBracket(type));
                wt.WriteLine("\t\t/>");
            }
        }
        #endregion

        //---------------------------------------------------------------------
        #region vertex
        // vertex のフォーマット
        private static void FormatVertex(TextReader rd, TextWriter wt)
            {
            // vertex_array
            wt.WriteLine("<vertex_array {0}", rd.ReadLine());

            while (true)
            {
                // vertex
                string vertex = rd.ReadLine();
                if (vertex != "<vertex")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(vertex == "</vertex_array>");
                    wt.WriteLine(vertex);
                    break;
                }
                wt.WriteLine("\t{0} {1}", vertex, rd.ReadLine());

                // lod_offset
                string lod_offset_or_vertex_attrib = rd.ReadLine();
                if (lod_offset_or_vertex_attrib == "<lod_offset")
                {
                    FormatLodOffset(rd, wt);

                    lod_offset_or_vertex_attrib = rd.ReadLine();
                }

                // vtx_attrib
                Nintendo.Foundation.Contracts.Assertion.Operation.True(lod_offset_or_vertex_attrib == "<vtx_attrib_array");
                FormatVtxAttrib(rd, wt);

                // vtx_buffer
                string vtx_buffer_array = rd.ReadLine();
                string close_vertex;
                if (vtx_buffer_array == "<vtx_buffer_array")
                {
                    FormatVtxBuffer(rd, wt);
                    close_vertex = rd.ReadLine();
                }
                else
                {
                    close_vertex = vtx_buffer_array;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_vertex == "</vertex>");
                wt.WriteLine("\t{0}", close_vertex);
            }
        }

        // lod_offset のフォーマット
        private static void FormatLodOffset(TextReader rd, TextWriter wt)
        {
            string count = rd.ReadLine();
            wt.WriteLine("\t\t<lod_offset {0}", count);
            if (count.EndsWith(" />") || count.EndsWith("</lod_offset>")) { return; }
            while (true)
            {
                string close_lod_offset = rd.ReadLine();
                wt.WriteLine(close_lod_offset);
                if (close_lod_offset.EndsWith("</lod_offset>")) { return; }
            }
        }

        // vtx_attrib のフォーマット
        private static void FormatVtxAttrib(TextReader rd, TextWriter wt)
        {
            // vtx_attrib_array
            wt.WriteLine("\t\t<vtx_attrib_array {0}", rd.ReadLine());

            while (true)
            {
                // vtx_attrib
                string vtx_attrib = rd.ReadLine();
                if (vtx_attrib != "<vtx_attrib")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(vtx_attrib == "</vtx_attrib_array>");
                    wt.WriteLine("\t\t{0}", vtx_attrib);
                    break;
                }
                wt.WriteLine("\t\t\t{0} {1} {2} {3}",
                    vtx_attrib, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}",
                    IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t/>");
            }
        }

        // vtx_buffer のフォーマット
        private static void FormatVtxBuffer(TextReader rd, TextWriter wt)
        {
            // vtx_buffer_array
            wt.WriteLine("\t\t<vtx_buffer_array {0}", rd.ReadLine());

            while (true)
            {
                // vtx_buffer
                string vtx_buffer = rd.ReadLine();
                if (vtx_buffer != "<vtx_buffer")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(vtx_buffer == "</vtx_buffer_array>");
                    wt.WriteLine("\t\t{0}", vtx_buffer);
                    break;
                }
                wt.WriteLine("\t\t\t{0} {1}", vtx_buffer, rd.ReadLine());

                // input
                FormatInput(rd, wt);

                string close_vtx_buffer = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_vtx_buffer == "</vtx_buffer>");
                wt.WriteLine("\t\t\t{0}", close_vtx_buffer);
            }
        }

        // input のフォーマット
        private static void FormatInput(TextReader rd, TextWriter wt)
        {
            // input_array
            string input_array = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(input_array == "<input_array");
            wt.WriteLine("\t\t\t\t{0} {1}", input_array, rd.ReadLine());

            while (true)
            {
                // input
                string input = rd.ReadLine();
                if (input != "<input")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(input == "</input_array>");
                    wt.WriteLine("\t\t\t\t{0}", input);
                    break;
                }
                wt.WriteLine("\t\t\t\t\t{0} {1} {2}",
                    input, rd.ReadLine(), rd.ReadLine());
            }
        }
        #endregion

        //---------------------------------------------------------------------
        #region shape
        // shape のフォーマット
        private static void FormatShape(TextReader rd, TextWriter wt)
        {
            // shape_array
            wt.WriteLine("<shape_array {0}", rd.ReadLine());

            while (true)
            {
                // shape
                string shape = rd.ReadLine();
                if (shape != "<shape")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(shape == "</shape_array>");
                    wt.WriteLine(shape);
                    break;
                }
                wt.WriteLine("\t{0} {1} {2}",
                    shape, rd.ReadLine(), rd.ReadLine());

                // shape_info
                string shape_info = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(shape_info == "<shape_info");
                wt.WriteLine("\t\t{0}", shape_info);
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t{0}",
                    IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
                wt.WriteLine("\t\t/>");

                // mesh
                FormatMesh(rd, wt);

                // bounding
                string bounding_array = rd.ReadLine();
                string key_shape_array;
                if (bounding_array == "<bounding_array")
                {
                    FormatBounding(rd, wt);
                    key_shape_array = rd.ReadLine();
                }
                else
                {
                    key_shape_array = bounding_array;
                }

                // key_shape
                string close_shape;
                if (key_shape_array == "<key_shape_array")
                {
                    FormatKeyShape(rd, wt);
                    close_shape = rd.ReadLine();
                }
                else
                {
                    close_shape = key_shape_array;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_shape == "</shape>");
                wt.WriteLine("\t{0}", close_shape);
            }
        }

        // mesh のフォーマット
        private static void FormatMesh(TextReader rd, TextWriter wt)
        {
            // mesh_array
            string mesh_array = rd.ReadLine();
            Nintendo.Foundation.Contracts.Assertion.Operation.True(mesh_array == "<mesh_array");
            wt.WriteLine("\t\t{0} {1}", mesh_array, rd.ReadLine());

            while (true)
            {
                // mesh
                string mesh = rd.ReadLine();
                if (mesh != "<mesh")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(mesh == "</mesh_array>");
                    wt.WriteLine("\t\t{0}", mesh);
                    break;
                }
                wt.WriteLine("\t\t\t{0} {1}", mesh, rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", IfTextFormatterUtility.RemoveBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t>");

                // submesh_array
                string submesh_array = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(submesh_array == "<submesh_array");
                wt.WriteLine("\t\t\t\t{0} {1}", submesh_array, rd.ReadLine());

                while (true)
                {
                    // submesh
                    string submesh = rd.ReadLine();
                    if (submesh != "<submesh")
                    {
                        Nintendo.Foundation.Contracts.Assertion.Operation.True(submesh == "</submesh_array>");
                        wt.WriteLine("\t\t\t\t{0}", submesh);
                        break;
                    }
                    wt.WriteLine("\t\t\t\t\t{0} {1} {2} {3}",
                        submesh, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
                }

                string close_mesh = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_mesh == "</mesh>");
                wt.WriteLine("\t\t\t{0}", close_mesh);
            }
        }

        // bounding のフォーマット
        private static void FormatBounding(TextReader rd, TextWriter wt)
        {
            // bounding_array
            wt.WriteLine("\t\t<bounding_array {0}", rd.ReadLine());

            while (true)
            {
                // bounding
                string bounding = rd.ReadLine();
                if (bounding != "<bounding")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(bounding == "</bounding_array>");
                    wt.WriteLine("\t\t{0}", bounding);
                    break;
                }
                wt.WriteLine("\t\t\t{0} {1} {2} {3}",
                    bounding, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
            }
        }

        // key_shape のフォーマット
        private static void FormatKeyShape(TextReader rd, TextWriter wt)
        {
            // key_shape_array
            wt.WriteLine("\t\t<key_shape_array {0}", rd.ReadLine());

            while (true)
            {
                // key_shape
                string key_shape = rd.ReadLine();
                if (key_shape != "<key_shape")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(key_shape == "</key_shape_array>");
                    wt.WriteLine("\t\t{0}", key_shape);
                    break;
                }
                wt.WriteLine("\t\t\t{0} {1} {2}", key_shape, rd.ReadLine(), rd.ReadLine());

                // target_attrib_array
                string target_attrib_array = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(target_attrib_array == "<target_attrib_array");
                wt.WriteLine("\t\t\t\t{0} {1}", target_attrib_array, rd.ReadLine());

                while (true)
                {
                    // target_attrib
                    string target_attrib = rd.ReadLine();
                    if (target_attrib != "<target_attrib")
                    {
                        Nintendo.Foundation.Contracts.Assertion.Operation.True(target_attrib == "</target_attrib_array>");
                        wt.WriteLine("\t\t\t\t{0}", target_attrib);
                        break;
                    }
                    wt.WriteLine("\t\t\t\t\t{0} {1} {2}",
                        target_attrib, rd.ReadLine(), rd.ReadLine());
                }

                string close_key_shape = rd.ReadLine();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_key_shape == "</key_shape>");
                wt.WriteLine("\t\t\t{0}", close_key_shape);
            }
        }
        #endregion

        //---------------------------------------------------------------------
        #region original_material
        // original_material のフォーマット
        private static void FormatOriginalMaterial(TextReader rd, TextWriter wt)
        {
            // original_material_array
            wt.WriteLine("<original_material_array {0}", rd.ReadLine());

            while (true)
            {
                // original_material
                string original_material = rd.ReadLine();
                if (original_material != "<original_material")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(original_material == "</original_material_array>");
                    wt.WriteLine(original_material);
                    break;
                }
                string index = rd.ReadLine();
                string name = rd.ReadLine();
                wt.WriteLine("\t{0} {1} {2}", original_material, index, name);
                if (name.EndsWith(" />")) { continue; }

                // original_color
                string original_color_array = rd.ReadLine();
                string original_texsrt_array;
                if (original_color_array == "<original_color_array")
                {
                    FormatOriginalColor(rd, wt);
                    original_texsrt_array = rd.ReadLine();
                }
                else
                {
                    original_texsrt_array = original_color_array;
                }

                // original_texsrt
                string close_original_material;
                if (original_texsrt_array == "<original_texsrt_array")
                {
                    FormatOriginalTexSrt(rd, wt);
                    close_original_material = rd.ReadLine();
                }
                else
                {
                    close_original_material = original_texsrt_array;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(close_original_material == "</original_material>");
                wt.WriteLine("\t{0}", close_original_material);
            }
        }

        // original_color のフォーマット
        private static void FormatOriginalColor(TextReader rd, TextWriter wt)
        {
            // original_color_array
            wt.WriteLine("\t\t<original_color_array {0}", rd.ReadLine());

            while (true)
            {
                // original_color
                string original_color = rd.ReadLine();
                if (original_color != "<original_color")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(original_color == "</original_color_array>");
                    wt.WriteLine("\t\t{0}", original_color);
                    break;
                }
                wt.WriteLine("\t\t\t{0} {1} {2}",
                    original_color, rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}",
                    IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t/>");
            }
        }

        // original_texsrt のフォーマット
        private static void FormatOriginalTexSrt(TextReader rd, TextWriter wt)
        {
            // original_texsrt_array
            wt.WriteLine("\t\t<original_texsrt_array {0}", rd.ReadLine());

            while (true)
            {
                // original_texsrt
                string original_texsrt = rd.ReadLine();
                if (original_texsrt != "<original_texsrt")
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(original_texsrt == "</original_texsrt_array>");
                    wt.WriteLine("\t\t{0}", original_texsrt);
                    break;
                }
                wt.WriteLine("\t\t\t{0} {1} {2} {3}",
                    original_texsrt, rd.ReadLine(), rd.ReadLine(), rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}", rd.ReadLine());
                wt.WriteLine("\t\t\t\t{0}",
                    IfTextFormatterUtility.RemoveSlashBracket(rd.ReadLine()));
                wt.WriteLine("\t\t\t/>");
            }
        }
        #endregion
    }
}
