﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Xml;
using IronPython.Hosting;
using Microsoft.Scripting.Hosting;
using IronPython.Runtime;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // フィルタ対象
    public class IfFilterTarget
    {
        // コンストラクタ
        public IfFilterTarget(
            string filePath, string xsdBasePath, string argument, bool disableFileInfo)
        {
            this.FilePath = filePath;
            Nintendo.Foundation.Contracts.Assertion.Operation.True((this.FilePath != null) && File.Exists(this.FilePath));
            Nintendo.Foundation.Contracts.Assertion.Operation.True(G3dPath.IsPath(this.FilePath));

            this.XsdBasePath = xsdBasePath;

            this.Argument = argument;
            if (this.Argument == null) { this.Argument = string.Empty; }

            this._disableFileInfo = disableFileInfo;
        }

        //---------------------------------------------------------------------
        // フィルタ
        public bool Filter(IfFilterScript script)
        {
            if (!script.HasFilter) { return true; }

            ScriptEngine engine = ScriptEngines.GetEngine();
            ScriptSource source = engine.CreateScriptSourceFromFile(
                script.ScriptPath, Encoding.UTF8);
            ScriptScope scope = engine.CreateScope();
            source.Execute(scope);

            scope.SetVariable("ScriptArgument", script.Argument);
            scope.SetVariable("ScriptDictionary", script.Dictionary);
            scope.SetVariable("Result", true);

            scope.SetVariable("filter_target", this);

            engine.Execute("Filter(filter_target)", scope);

            Unload();
            ScriptEngines.PushEngine(engine);

            return scope.GetVariable<bool>("Result");
        }

        //---------------------------------------------------------------------
        // 書き出し
        public void Write() { Write(this.FilePath); }

        // 書き出し
        public void Write(string filePath)
        {
            if (!G3dPath.IsPath(filePath))
            {
                IfStrings.Throw("IfFilterTarget_Error_WritePath", filePath);
            }

            /// TODO: ソート
            if (!this._disableFileInfo)
            {
                IfFileLogUtility.SetModify(this.nw4f_3dif);
            }
            IfWriteUtility.Write(this.nw4f_3dif, this.Streams, filePath);
        }

        //---------------------------------------------------------------------
        // nw4f_3dif の取得
        public nw4f_3difType nw4f_3dif
        {
            get
            {
                // 遅延ロード
                if (_streams == null)
                {
                    _streams = new List<G3dStream>();
                    _nw4f_3dif = IfReadUtility.Read(
                        _streams, this.FilePath, this.XsdBasePath);

                    // チェック
                    IfCheckContext context = new IfCheckContext(this.FilePath);
                    IfChecker.Check(context, _nw4f_3dif, _streams);
                    context.Throw();
                }
                Nintendo.Foundation.Contracts.Assertion.Operation.True(_streams != null);

                // XmlDocument → nw4f_3difType
                if (_nw4f_3dif == null)
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(_document != null);
                    using (MemoryStream memStream = new MemoryStream())
                    {
                        _document.Save(memStream);

                        // スキーマチェックをここで行うと、要素順の制限でエラーになってしまうので
                        // (中間ファイルフィルタスクリプトで要素順を守って要素を追加するのが困難なため)
                        // xsd パスは指定しない。
                        // この後、Write() で自動的にフォーマットされる
                        _nw4f_3dif = IfTextReadUtility.Read(
                            memStream.ToArray(), null);
                    }
                    _document = null;
                }
                Nintendo.Foundation.Contracts.Assertion.Operation.True((_nw4f_3dif != null) && (_document == null));
                return _nw4f_3dif;
            }
        }

        // ドキュメントの取得
        public XmlDocument Document
        {
            get
            {
                // 遅延ロード
                // 問題のあるファイルを修正できるようにするため、スキーマを通さない
                if (_streams == null)
                {
                    // 拡張子に合わせて読み込む
                    if (G3dPath.IsTextPath(this.FilePath))
                    {
                        _document = new XmlDocument();
                        _document.Load(this.FilePath);

                        // ドキュメントから stream を取り出す
                        _streams = new List<G3dStream>();
                        G3dStreamUtility.GetStreams(_streams, _document);
                    }
                    else
                    {
                        // テキスト部とバイナリ部の分割
                        byte[] textImage;
                        byte[] binaryImage;
                        IfBinaryReadUtility.Separate(File.ReadAllBytes(this.FilePath),
                            out textImage, out binaryImage);

                        // テキスト部読み込み
                        _document = new XmlDocument();
                        using (MemoryStream stream = new MemoryStream(textImage))
                        {
                            _document.Load(stream);
                        }

                        // バイナリ部読み込み
                        _streams = new List<G3dStream>();
                        if (binaryImage != null)
                        {
                            IfBinaryReadUtility.ReadStreamArray(_streams, binaryImage);
                        }
                    }
                }
                Nintendo.Foundation.Contracts.Assertion.Operation.True(_streams != null);

                // nw4f_3difType → XmlDocument
                if (_document == null)
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(_nw4f_3dif != null);
                    using (MemoryStream memStream = new MemoryStream())
                    {
                        IfTextWriteUtility.Write(_nw4f_3dif, memStream, this.XsdBasePath);
                        memStream.Seek(0, SeekOrigin.Begin);
                        _document = new XmlDocument();
                        _document.Load(memStream);
                    }
                    _nw4f_3dif = null;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True((_document != null) && (_nw4f_3dif == null));
                return _document;
            }
        }

        // バイナリストリームの取得
        public List<G3dStream> Streams
        {
            get
            {
                // 遅延ロード
                if (_streams == null) { XmlDocument document = this.Document; }
                Nintendo.Foundation.Contracts.Assertion.Operation.True(_streams != null);
                return _streams;
            }
        }

        // ドキュメントをアンロードする
        public void Unload()
        {
            _streams = null;
            _nw4f_3dif = null;
            _document = null;
        }

        //---------------------------------------------------------------------
        public readonly string FilePath;
        public string FileName { get { return Path.GetFileName(this.FilePath); } }
        public string FileNameWithoutExtension
        {
            get { return Path.GetFileNameWithoutExtension(this.FilePath); }
        }
        public string Extension { get { return Path.GetExtension(this.FilePath); } }
        public string FullPath { get { return Path.GetFullPath(this.FilePath); } }
        public string DirectoryName { get { return Path.GetDirectoryName(this.FilePath); } }

        public readonly string XsdBasePath;
        public readonly string Argument;

        public object Result { get; set; }

        private nw4f_3difType _nw4f_3dif;
        private List<G3dStream> _streams;
        private XmlDocument _document;
        private bool _disableFileInfo;
    }

    // ScriptEngine を動的に生成・再利用します。
    public class ScriptEngines
    {
        private static Stack<ScriptEngine> _engines = new Stack<ScriptEngine>();

        public static ScriptEngine GetEngine()
        {
            lock (_engines)
            {
                if (_engines.Count > 0)
                {
                    return _engines.Pop();
                }
                return IfScriptingUtility.CreateEngine();
            }
        }

        public static void PushEngine(ScriptEngine engine)
        {
            lock (_engines)
            {
                _engines.Push(engine);
            }
        }

        public static void ClearEngines()
        {
            lock (_engines)
            {
                _engines.Clear();
            }
        }
    }
}
