﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // モデルのマテリアル圧縮
    public class IfModelMaterialCompressor : IfModelCompressor
    {
        // コンストラクタ
        public IfModelMaterialCompressor() :
            base("IfModelMaterialCompressor_Log") { }

        // プロセス
        public override string Process
        {
            get { return "compress_material"; }
        }

        // カウントの取得
        public override int GetCount()
        {
            if (this.Target.material_array != null &&
                this.Target.material_array.material != null)
            {
                return this.Target.material_array.material.Length;
            }
            else
            {
                return 0;
            }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Format(
                "material[{0}/{1}({2:f0}%)]",
                this.PostCount, this.PreCount, 100f * this.PostCount / this.PreCount);
        }

        // 圧縮
        protected override void Compress()
        {
            // TODO: Unite モデル対応
            // これより下は Unite されたモデルにまだ対応していない。
            if (this.Target.shape_array != null)
            {
                foreach (shapeType shape in this.Target.shape_array.shape)
                {
                    if (shape.mesh_array.mesh.Length > 1)
                    {
                        IfStrings.Throw("IfModelOptimizer_Error_UnitedModelUnsupported");
                    }
                }
            }

            // material_array が存在しない場合は圧縮を行わない
            if (this.Target.material_array == null ||
                this.Target.material_array.material == null) { return; }
            if (this.Target.shape_array == null ||
                this.Target.shape_array.shape == null) { return; }
            if (this.Target.original_material_array == null ||
                this.Target.original_material_array.original_material == null) { return; }

            int isSorted = IfModelSortUtility.IsMaterialSorted(this.Target);
            if (isSorted != -1)
            {
                /// TODO: DCC 対応後
                //IfStrings.Throw("IfModelMaterialCompressor_Error_NotSorted",
                //    materials[isSorted].name, materials[isSorted + 1].name);
            }

            // <original_material> の初期化
            original_materialType[] orgMaterials = this.Target.original_material_array.original_material;
            int orgMaterialCount = orgMaterials.Length;
            Dictionary<string, OriginalMaterialInfo> orgMateDict =
                    new Dictionary<string, OriginalMaterialInfo>();
            OriginalMaterialInfo[] orgMaterialInfos = new OriginalMaterialInfo[orgMaterialCount];
            for (int i = 0; i < orgMaterialCount; i++)
            {
                original_materialType orgMate = orgMaterials[i];
                orgMaterialInfos[i] = new OriginalMaterialInfo(orgMate, i);
                orgMateDict.Add(orgMaterialInfos[i].OriginalMaterial.mat_name, orgMaterialInfos[i]);
            }

            // <material> の初期化
            materialType[] materials = this.Target.material_array.material;
            int materialCount = materials.Length;
            Dictionary<string, MaterialInfo> mateDict = new Dictionary<string, MaterialInfo>();
            MaterialInfo[] materialInfos = new MaterialInfo[materialCount];
            for (int i = 0; i < materialCount; i++)
            {
                materialType mate = materials[i];
                // モデルをマージしても新しいマテリアルが作成されないので、
                // <material> と同じ名前の <original_material> が存在しない事はありえない。
                Nintendo.Foundation.Contracts.Assertion.Operation.True(orgMateDict.ContainsKey(mate.name));
                OriginalMaterialInfo orgMate = orgMateDict[mate.name];

                materialInfos[i] = new MaterialInfo(mate, orgMate);
                mateDict.Add(materialInfos[i].Material.name, materialInfos[i]);
            }

            // 削除可能なマテリアルの調査
            for (int i = 0; i < materialCount; i++)
            {
                if (materialInfos[i].RemoveFlag) { continue; }
                OriginalMaterialInfo uniteTo = materialInfos[i].OriginalMaterial;
                // 削除しないマテリアルには自身のインデックスを ReplaceIndex に設定する
                materialInfos[i].ReplaceIndex = i;
                uniteTo.ReplaceIndex = uniteTo.Index;

                for (int j = i + 1; j < materialCount; j++)
                {
                    if (materialInfos[j].RemoveFlag) { continue; }
                    OriginalMaterialInfo orgMateInfo = materialInfos[j].OriginalMaterial;
                    if (CanCompress(uniteTo.OriginalMaterial, orgMateInfo.OriginalMaterial) &&
                        CanCompress(materials[i], materials[j]))
                    {
                        // 削除するマテリアルでは統合する先のマテリアルのインデックスを
                        // ReplaceIndex に設定する
                        materialInfos[j].ReplaceIndex = i;
                        materialInfos[j].RemoveFlag = true;

                        orgMateInfo.ReplaceIndex = uniteTo.Index;
                        orgMateInfo.RemoveFlag = true;
                    }
                }
            }

            // 新しい <material> 配列の構築
            List<materialType> newMaterials = new List<materialType>();
            for (int i = 0; i < materialCount; i++)
            {
                MaterialInfo materialInfo = materialInfos[i];
                if (!materialInfo.RemoveFlag)
                {
                    newMaterials.Add(materialInfo.Material);
                }
            }
            this.Target.material_array.length = newMaterials.Count;
            this.Target.material_array.material = newMaterials.ToArray();

            // 新しい <original_material> 配列の構築
            List<original_materialType> newOriginalMaterials = new List<original_materialType>();
            for (int i = 0; i < orgMaterialCount; i++)
            {
                OriginalMaterialInfo orgMaterialInfo = orgMaterialInfos[i];
                if (!orgMaterialInfo.RemoveFlag)
                {
                    newOriginalMaterials.Add(orgMaterialInfo.OriginalMaterial);
                }
            }
            this.Target.original_material_array.length = newOriginalMaterials.Count;
            this.Target.original_material_array.original_material = newOriginalMaterials.ToArray();

            foreach (shapeType shape in this.Target.shape_array.shape)
            {
                shape_infoType shapeInfo = shape.shape_info;
                if (!mateDict.ContainsKey(shapeInfo.mat_name))
                {
                    // 参照するマテリアルが存在しない場合はエラー
                    IfStrings.Throw("ModelCompressUtility_Error_BoneNotFound",
                        shape.name, shapeInfo.mat_name);
                }

                MaterialInfo mateInfo = mateDict[shapeInfo.mat_name];
                shapeInfo.mat_name = materialInfos[mateInfo.ReplaceIndex].Material.name;
            }

            // 不要なストリームの削除とソートを行う。
            StreamUtility.SortStream(this.Target, this.Streams);

            //for (int i = 0; i < materialCount; i++)
            //{
            //    Debug.WriteLine(materialInfos[i]);
            //}
        }

        // 圧縮可能か
        private bool CanCompress(original_materialType lhs, original_materialType rhs)
        {
            if (lhs == null)
            {
                return (rhs == null) ? true : false;
            }
            else
            {
                if (rhs == null) { return false; }
            }

            // <original_color> の比較
            if (lhs.original_color_array != null)
            {
                if (rhs.original_color_array == null) { return false; }
                original_colorType[] lhsColors = lhs.original_color_array.original_color;
                original_colorType[] rhsColors = rhs.original_color_array.original_color;
                int color_count = lhsColors.Length;
                if (rhsColors.Length != color_count) { return false; }

                for (int i = 0; i < color_count; i++)
                {
                    original_colorType lhsColor = lhsColors[i];
                    original_colorType rhsColor = rhsColors[i];

                    if (lhsColor.hint != rhsColor.hint) { return false; }
                    for (int j = 0; j < 3; j++)
                    {
                        if (lhsColor.color[j] != rhsColor.color[j]) { return false; }
                    }
                }
            }
            else
            {
                if (rhs.original_color_array != null) { return false; }
            }

            // <original_texsrt> の比較
            if (lhs.original_texsrt_array != null)
            {
                if (rhs.original_texsrt_array == null) { return false; }
                original_texsrtType[] lhsTexSrts = lhs.original_texsrt_array.original_texsrt;
                original_texsrtType[] rhsTexSrts = rhs.original_texsrt_array.original_texsrt;
                int texsrt_count = lhsTexSrts.Length;
                if (rhsTexSrts.Length != texsrt_count) { return false; }

                for (int i = 0; i < texsrt_count; i++)
                {
                    original_texsrtType lhsTexSrt = lhsTexSrts[i];
                    original_texsrtType rhsTexSrt = rhsTexSrts[i];

                    if (lhsTexSrt.hint != rhsTexSrt.hint) { return false; }
                    if (lhsTexSrt.mode != rhsTexSrt.mode) { return false; }
                    if (lhsTexSrt.scale[0] != rhsTexSrt.scale[0]) { return false; }
                    if (lhsTexSrt.scale[1] != rhsTexSrt.scale[1]) { return false; }
                    if (lhsTexSrt.rotate != rhsTexSrt.rotate) { return false; }
                    if (lhsTexSrt.translate[0] != rhsTexSrt.translate[0]) { return false; }
                    if (lhsTexSrt.translate[1] != rhsTexSrt.translate[1]) { return false; }
                }
            }
            else
            {
                if (rhs.original_texsrt_array != null) { return false; }
            }

            return true;
        }

        // 圧縮可能か
        private bool CanCompress(materialType lhs, materialType rhs)
        {
            material_infoType lhsInfo = lhs.material_info;
            material_infoType rhsInfo = rhs.material_info;
            if (!lhsInfo.compress_enable) { return false; }
            if (!rhsInfo.compress_enable) { return false; }

            // <render_state> の比較
            {
                render_stateType lhsState = lhs.render_state;
                render_stateType rhsState = rhs.render_state;
                if (lhsState.depth_test.enable != rhsState.depth_test.enable) { return false; }
                if (lhsState.depth_test.func != rhsState.depth_test.func) { return false; }
                if (lhsState.depth_test.write != rhsState.depth_test.write) { return false; }
                if (lhsState.alpha_test.enable != rhsState.alpha_test.enable) { return false; }
                if (lhsState.alpha_test.func != rhsState.alpha_test.func) { return false; }
                if (lhsState.alpha_test.value != rhsState.alpha_test.value) { return false; }
                if (lhsState.color_blend.alpha_dst_func != rhsState.color_blend.alpha_dst_func) { return false; }
                if (lhsState.color_blend.alpha_op != rhsState.color_blend.alpha_op) { return false; }
                if (lhsState.color_blend.alpha_src_func != rhsState.color_blend.alpha_src_func) { return false; }
                if (lhsState.color_blend.const_color.Length != rhsState.color_blend.const_color.Length) { return false; }
                for (int i = 0; i < lhsState.color_blend.const_color.Length; i++)
                {
                    if (lhsState.color_blend.const_color[i] != rhsState.color_blend.const_color[i]) { return false; }
                }
                if (lhsState.color_blend.rgb_dst_func != rhsState.color_blend.rgb_dst_func) { return false; }
                if (lhsState.color_blend.rgb_op != rhsState.color_blend.rgb_op) { return false; }
                if (lhsState.color_blend.rgb_src_func != rhsState.color_blend.rgb_src_func) { return false; }
                if (lhsState.logical_blend.op != rhsState.logical_blend.op) { return false; }
                if (lhsState.mode != rhsState.mode) { return false; }
                if (lhsState.display_face != rhsState.display_face) { return false; }
                if (lhsState.blend_mode != rhsState.blend_mode) { return false; }
            }

            // <sampler_array> の比較
            if (lhs.sampler_array != null)
            {
                if (rhs.sampler_array == null) { return false; }
                samplerType[] lhsSamplers = lhs.sampler_array.sampler;
                samplerType[] rhsSamplers = rhs.sampler_array.sampler;
                int sampler_count = lhsSamplers.Length;
                if (rhsSamplers.Length != sampler_count) { return false; }

                for (int i = 0; i < sampler_count; i++)
                {
                    samplerType lhsSampler = lhsSamplers[i];
                    samplerType rhsSampler = rhsSamplers[i];

                    if (lhsSampler.name != rhsSampler.name) { return false; }
                    if (lhsSampler.hint != rhsSampler.hint) { return false; }
                    if (lhsSampler.tex_name != rhsSampler.tex_name) { return false; }

                    if (lhsSampler.wrap.u != rhsSampler.wrap.u) { return false; }
                    if (lhsSampler.wrap.v != rhsSampler.wrap.v) { return false; }
                    if (lhsSampler.wrap.w != rhsSampler.wrap.w) { return false; }

                    if (lhsSampler.filter.mag != rhsSampler.filter.mag) { return false; }
                    if (lhsSampler.filter.min != rhsSampler.filter.min) { return false; }
                    if (lhsSampler.filter.mip != rhsSampler.filter.mip) { return false; }
                    if (lhsSampler.filter.max_aniso != rhsSampler.filter.max_aniso) { return false; }

                    if (lhsSampler.lod.min != rhsSampler.lod.min) { return false; }
                    if (lhsSampler.lod.max != rhsSampler.lod.max) { return false; }
                    if (lhsSampler.lod.bias != rhsSampler.lod.bias) { return false; }
                }
            }
            else
            {
                if (rhs.sampler_array != null) { return false; }
            }

            // <shader_assign> の比較
            if (!CanCompress(lhs.shader_assign, rhs.shader_assign)) { return false; }
            // <user_tool_data> の比較
            if (!CanCompress(lhs.user_tool_data, rhs.user_tool_data)) { return false; }

            return true;
        }

        // 圧縮可能か
        private bool CanCompress(shader_assignType lhs, shader_assignType rhs)
        {
            if (lhs == null)
            {
                return (rhs == null) ? true : false;
            }
            else
            {
                if (rhs == null) { return false; }
            }

            // <shader_option> の比較
            if (lhs.shader_option_array != null)
            {
                if (rhs.shader_option_array == null) { return false; }
                shader_optionType[] lhsOptions = lhs.shader_option_array.shader_option;
                shader_optionType[] rhsOptions = rhs.shader_option_array.shader_option;
                int shader_option_count = lhsOptions.Length;
                if (rhsOptions.Length != shader_option_count) { return false; }

                for (int i = 0; i < shader_option_count; i++)
                {
                    shader_optionType lhsOption = lhsOptions[i];
                    shader_optionType rhsOption = rhsOptions[i];

                    if (lhsOption.id != rhsOption.id) { return false; }
                    if (lhsOption.value != rhsOption.value) { return false; }
                }
            }
            else
            {
                if (rhs.shader_option_array != null) { return false; }
            }

            // <sampler_assign> の比較
            if (lhs.sampler_assign_array != null)
            {
                if (rhs.sampler_assign_array == null) { return false; }
                sampler_assignType[] lhsSamplerAssigns = lhs.sampler_assign_array.sampler_assign;
                sampler_assignType[] rhsSamplerAssigns = rhs.sampler_assign_array.sampler_assign;
                int sampler_assign_count = lhsSamplerAssigns.Length;
                if (rhsSamplerAssigns.Length != sampler_assign_count) { return false; }

                for (int i = 0; i < sampler_assign_count; i++)
                {
                    sampler_assignType lhsSamplerAssign = lhsSamplerAssigns[i];
                    sampler_assignType rhsSamplerAssign = rhsSamplerAssigns[i];

                    if (lhsSamplerAssign.id != rhsSamplerAssign.id) { return false; }
                    if (lhsSamplerAssign.sampler_name != rhsSamplerAssign.sampler_name) { return false; }
                }
            }
            else
            {
                if (rhs.sampler_assign_array != null) { return false; }
            }

            // <shader_param> の比較
            if (lhs.shader_param_array != null)
            {
                if (rhs.shader_param_array == null) { return false; }
                shader_paramType[] lhsParams = lhs.shader_param_array.shader_param;
                shader_paramType[] rhsParams = rhs.shader_param_array.shader_param;
                int shader_param_count = lhsParams.Length;
                if (rhsParams.Length != shader_param_count) { return false; }

                for (int i = 0; i < shader_param_count; i++)
                {
                    shader_paramType lhsParam = lhsParams[i];
                    shader_paramType rhsParam = rhsParams[i];

                    if (lhsParam.id != rhsParam.id) { return false; }
                    if (lhsParam.original_hint != rhsParam.original_hint) { return false; }
                    if (lhsParam.type != rhsParam.type) { return false; }
                    if (lhsParam.Value != rhsParam.Value) { return false; }
                }
            }
            else
            {
                if (rhs.shader_param_array != null) { return false; }
            }

            // <attrib_assign> の比較
            if (lhs.attrib_assign_array != null)
            {
                if (rhs.attrib_assign_array == null) { return false; }
                attrib_assignType[] lhsAttribAssigns = lhs.attrib_assign_array.attrib_assign;
                attrib_assignType[] rhsAttribAssigns = rhs.attrib_assign_array.attrib_assign;
                int attrib_assign_count = lhsAttribAssigns.Length;
                if (rhsAttribAssigns.Length != attrib_assign_count) { return false; }

                for (int i = 0; i < attrib_assign_count; i++)
                {
                    attrib_assignType lhsAttribAssign = lhsAttribAssigns[i];
                    attrib_assignType rhsAttribAssign = rhsAttribAssigns[i];

                    if (lhsAttribAssign.id != rhsAttribAssign.id) { return false; }
                    if (lhsAttribAssign.attrib_name != rhsAttribAssign.attrib_name) { return false; }
                }
            }
            else
            {
                if (rhs.attrib_assign_array != null) { return false; }
            }

            if (lhs.shader_archive != rhs.shader_archive) { return false; }
            if (lhs.shading_model != rhs.shading_model) { return false; }

            return true;
        }

        // 圧縮可能か
        private bool CanCompress(user_tool_dataType lhs, user_tool_dataType rhs)
        {
            if (lhs == null)
            {
                return (rhs == null) ? true : false;
            }
            else
            {
                if (rhs == null) { return false; }
            }

            if (lhs.Any != null)
            {
                if (rhs.Any == null) { return false; }
                System.Xml.XmlElement[] lhsAnys = lhs.Any;
                System.Xml.XmlElement[] rhsAnys = rhs.Any;
                int any_count = lhsAnys.Length;
                if (rhsAnys.Length != any_count) { return false; }

                for (int i = 0; i < any_count; i++)
                {
                    System.Xml.XmlElement lhsAny = lhsAnys[i];
                    System.Xml.XmlElement rhsAny = rhsAnys[i];
                    if (!CanCompress(lhsAny, rhsAny)) { return false; }
                }
            }
            else
            {
                if (rhs.Any != null) { return false; }
            }

            return true;
        }

        // 圧縮可能か
        private bool CanCompress(System.Xml.XmlNode lhs, System.Xml.XmlNode rhs)
        {
            if (lhs == null)
            {
                return (rhs == null) ? true : false;
            }
            else
            {
                if (rhs == null) { return false; }
            }

            if (lhs.Name != rhs.Name) { return false; }
            if (lhs.Value != rhs.Value) { return false; }

            // アトリビュートの比較
            System.Xml.XmlAttributeCollection lhsAttributes = lhs.Attributes;
            System.Xml.XmlAttributeCollection rhsAttributes = rhs.Attributes;
            if (lhsAttributes != null)
            {
                if (rhsAttributes == null) { return false; }
                if (lhsAttributes.Count != rhsAttributes.Count) { return false; }
                for (int iattr = 0; iattr < lhsAttributes.Count; iattr++)
                {
                    System.Xml.XmlAttribute lhsAttr = lhsAttributes[iattr];
                    System.Xml.XmlAttribute rhsAttr = rhsAttributes[iattr];
                    if (lhsAttr.Name != rhsAttr.Name) { return false; }
                    if (lhsAttr.Value != rhsAttr.Value) { return false; }
                }
            }
            else
            {
                if (rhsAttributes != null) { return false; }
            }

            // 子ノードの比較
            System.Xml.XmlNodeList lhsChildren = lhs.ChildNodes;
            System.Xml.XmlNodeList rhsChildren = rhs.ChildNodes;
            if (lhsChildren != null)
            {
                if (rhsChildren == null) { return false; }
                if (lhsChildren.Count != rhsChildren.Count) { return false; }
                for (int inode = 0; inode < lhsChildren.Count; inode++)
                {
                    if (!CanCompress(lhsChildren[inode], rhsChildren[inode])) { return false; }
                }
            }
            else
            {
                if (rhsChildren != null) { return false; }
            }

            return true;
        }

        // マテリアル情報
        private class MaterialInfo
        {
            public MaterialInfo(materialType material, OriginalMaterialInfo orgMaterial)
            {
                this.Material = material;
                this.OriginalMaterial = orgMaterial;
                this.ReplaceIndex = -1;
            }

            public override string ToString()
            {
                return string.Format(
                    "{0} {1} {2}",
                    !this.RemoveFlag ? "○" : "×",
                    this.ReplaceIndex,
                    this.Material.name);
            }

            public readonly materialType Material;
            public OriginalMaterialInfo OriginalMaterial { get; set; }
            public int ReplaceIndex { get; set; }
            public bool RemoveFlag { get; set; }
        }

        // マテリアル情報
        private class OriginalMaterialInfo
        {
            public OriginalMaterialInfo(original_materialType originalMaterial, int index)
            {
                this.OriginalMaterial = originalMaterial;
                this.Index = index;
                this.ReplaceIndex = -1;
            }

            public override string ToString()
            {
                return string.Format(
                    "{0} {1} {2}",
                    !this.RemoveFlag ? "○" : "×",
                    this.ReplaceIndex,
                    this.OriginalMaterial.mat_name);
            }

            public readonly original_materialType OriginalMaterial;
            public readonly int Index;
            public int ReplaceIndex { get; set; }
            public bool RemoveFlag { get; set; }
        }
    }
}
