﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using System.IO;
using System.Diagnostics;
using Nintendo.G3dTool.Entities;
using Nintendo.G3dTool.Extensions;

namespace nw.g3d.ifcvtr
{
    // obj 用中間ファイルビルダ
    internal class ObjIfBuilder
    {
        public static string MakeRelativePath(string basePath, string filePath)
        {
            if (!basePath.EndsWith("/") && !basePath.EndsWith("\\"))
            {
                basePath += "/";
            }
            var baseUri = new Uri(basePath);
            var fileUri = new Uri(filePath);

            // 相対パスを取得します。
            var relativeUri = baseUri.MakeRelativeUri(fileUri);
            // 相対パスを取得できない場合は元ファイルパスを返す
            if (relativeUri.IsAbsoluteUri)
            {
                return filePath;
            }
            var relativePathEscaped = relativeUri.ToString();

            // Uri クラス が +  エンコードしてくれない文字があるので、自分で置換します。
            relativePathEscaped = relativePathEscaped.Replace("+", "%2b");

            // エスケープされていない文字列に変換します
            var resultPath = System.Web.HttpUtility.UrlDecode(relativePathEscaped);

            return resultPath;
        }

        internal void Build(ObjContext context)
        {
            // マテリアルの名前テーブル作成
            InitializeMaterialNameTable(context);

            // シェイプのソート
            context.Shapes.Sort(delegate(ObjShape lhs, ObjShape rhs)
            {
                return string.CompareOrdinal(
                    this.MaterialNameTable[lhs.Material],
                    this.MaterialNameTable[rhs.Material]);
            });

            string bone_name = "nw4f_root";
            var file = new IntermediateFile(IntermediateFileKind.Model);
            context.File = file;

            var filePath = Path.GetFullPath(context.FilePath);
            if (!string.IsNullOrEmpty(context.ProjectRoot))
            {
                var projectRoot = context.ProjectRoot;
                if (projectRoot.Contains("%"))
                {
                    projectRoot = Environment.ExpandEnvironmentVariables(projectRoot);
                }
                filePath = MakeRelativePath(Path.GetFullPath(projectRoot), filePath);
            }

            file.FileInfo.SetCreationInfo(filePath.Replace('\\', '/'));

            var model = file.GetRootEntity<Model>();

            BuildStreams(model, context);
            BuildMaterials(model, context);
            BuildSkeleton(model.Skeleton, bone_name);
            BuildShapes(model, context, bone_name);
            BuildOriginalMaterials(model, context);
        }

        // マテリアル名テーブルの初期化
        private void InitializeMaterialNameTable(ObjContext context)
        {
            /// TODO: Materials に登録されているが、Shape から使われていない場合はここで削除
            this.MaterialNameTable = new Dictionary<string, string>();
            foreach (ObjMaterial material in context.Materials)
            {
                string original = material.Name;
                // 同じ名前のマテリアルは非対応
                Nintendo.Foundation.Contracts.Assertion.Operation.True(!this.MaterialNameTable.ContainsKey(original));
                // もう少し速度に影響を与えずに何とかしたい
                string replace = original.Replace('[', '_');
                replace = replace.Replace(']', '_');
                int number = 0;
                if (this.MaterialNameTable.ContainsValue(replace))
                {
                    while (true)
                    {
                        string numbered = replace + "_" + number;
                        number++;
                        if (!this.MaterialNameTable.ContainsValue(numbered))
                        {
                            replace = numbered;
                            break;
                        }
                    }
                }
                this.MaterialNameTable.Add(original, replace);
            }
        }

        //=====================================================================
        private void BuildMaterials(Model model, ObjContext context)
        {
            foreach(var objMaterial in context.Materials)
            {
                var material = new Material();
                material.Name = this.MaterialNameTable[objMaterial.Name];
                BuildSamplers(material, objMaterial);
                model.Materials.Add(material);
            }
        }

        private void BuildSamplers(Material material, ObjMaterial objMaterial)
        {
            int samplerCount = objMaterial.TextureCount;
            if (samplerCount == 0)
            {
                return;
            }

            if (objMaterial.AlbedoTexture != null)
            {
                material.Samplers.Add(
                    CreateSampler("_a0", "albedo0", objMaterial.AlbedoTexture));
            }
            if (objMaterial.OpacityTexture != null)
            {
                material.Samplers.Add(
                    CreateSampler("_o0", "opacity0", objMaterial.OpacityTexture));
            }
            if (objMaterial.EmissionTexture != null)
            {
                material.Samplers.Add(
                    CreateSampler("_e0", "emission0", objMaterial.EmissionTexture));
            }
            if (objMaterial.SpecularTexture != null)
            {
                material.Samplers.Add(
                    CreateSampler("_s0", "specular0", objMaterial.SpecularTexture));
            }

            Nintendo.Foundation.Contracts.Assertion.Operation.True(samplerCount == material.Samplers.Count);
        }

        private Sampler CreateSampler(string name, string hint, string texturePath)
        {
            var sampler = new Sampler();
            sampler.Name = name;
            sampler.Hint = hint;
            sampler.TexName = Path.GetFileNameWithoutExtension(texturePath);
            sampler.Wrap.U = wrap_uvwType.repeat;
            sampler.Wrap.V = wrap_uvwType.repeat;
            sampler.Wrap.W = wrap_uvwType.clamp;
            return sampler;
        }

        //=====================================================================
        private void BuildSkeleton(Skeleton skeleton, string bone_name)
        {
            skeleton.SkeletonInfo.ScaleMode = skeleton_info_scale_modeType.maya;
            skeleton.SkeletonInfo.RotateMode = skeleton_info_rotate_modeType.euler_xyz;

            var bone = new Bone();
            bone.Type = string.Empty;
            bone.Name = bone_name;
            bone.ScaleCompensate = true;
            skeleton.RootBone = bone;
        }

        //=====================================================================
        private void BuildShapes(Model model, ObjContext context, string bone_name)
        {
            string nameFormat = "{0}__{1}__{2:D" + context.Shapes.Count.ToString().Length + "}";
            for (int shapeIndex = 0; shapeIndex < context.Shapes.Count; ++shapeIndex)
            {
                ObjShape objShape = context.Shapes[shapeIndex];
                Material mat = model.Materials.First(x => x.Name == this.MaterialNameTable[objShape.Material]);
                Bone bone = model.Skeleton.EnumerateBones().First(x => x.Name == bone_name);
                int streamIndex = context.Shapes.Count * 3 + shapeIndex;
                StreamInt indexStream = model.Streams[streamIndex] as StreamInt;

                Shape shape = model.CreateShape(mat, bone, indexStream);
                shape.Name = string.Format(nameFormat, bone.Name, mat.Name, shapeIndex);
                shape.AddVertexAttributePosition(model.Streams[shapeIndex * 3 + 0] as StreamFloat);
                shape.AddVertexAttributeNormal(model.Streams[shapeIndex * 3 + 1] as StreamFloat);
                shape.AddVertexAttributeUv(model.Streams[shapeIndex * 3 + 2] as StreamFloat);
            }
        }

        //=====================================================================
        private void BuildOriginalMaterials(Model model, ObjContext context)
        {
            for (int materialIndex = 0; materialIndex < context.Materials.Count; materialIndex++)
            {
                ObjMaterial objMaterial = context.Materials[materialIndex];
                var originalMaterial = new OriginalMaterial();
                originalMaterial.MatName = this.MaterialNameTable[objMaterial.Name];

                var diffuse = new OriginalColor();
                diffuse.Hint = "diffuse";
                diffuse.Color.R = objMaterial.Diffuse[0];
                diffuse.Color.G = objMaterial.Diffuse[1];
                diffuse.Color.B = objMaterial.Diffuse[2];
                originalMaterial.OriginalColors.Add(diffuse);

                var opacity = new OriginalColor();
                opacity.Hint = "opacity";
                diffuse.Color.R = objMaterial.Opacity;
                diffuse.Color.G = objMaterial.Opacity;
                diffuse.Color.B = objMaterial.Opacity;
                originalMaterial.OriginalColors.Add(opacity);

                var ambient = new OriginalColor();
                ambient.Hint = "ambient";
                diffuse.Color.R = objMaterial.Ambient[0];
                diffuse.Color.G = objMaterial.Ambient[1];
                diffuse.Color.B = objMaterial.Ambient[2];
                originalMaterial.OriginalColors.Add(ambient);

                var specular = new OriginalColor();
                specular.Hint = "specular";
                diffuse.Color.R = objMaterial.Specular[0];
                diffuse.Color.G = objMaterial.Specular[1];
                diffuse.Color.B = objMaterial.Specular[2];
                originalMaterial.OriginalColors.Add(specular);

                model.OriginalMaterials.Add(originalMaterial);
            }
        }

        //=====================================================================
        // ストリームの構築
        private void BuildStreams(Model model, ObjContext context)
        {
            // Stream を生成しておく
            int shapeCount = context.Shapes.Count;
            model.Streams.Clear();
            int indexStreamOffset = shapeCount * 3;
            for (int i = 0; i < shapeCount; ++i)
            {
                // 位置
                model.Streams.Add(new StreamFloat());
                // 法線
                model.Streams.Add(new StreamFloat());
                // UV
                model.Streams.Add(new StreamFloat());
            }

            for (int i = 0; i < shapeCount; ++i)
            {
                // インデックス
                model.Streams.Add(new StreamInt());
            }

            for (int shapeIndex = 0; shapeIndex < shapeCount; shapeIndex++)
            {
                var positionStream = model.Streams[shapeIndex * 3 + 0] as StreamFloat;
                var normalStream = model.Streams[shapeIndex * 3 + 1] as StreamFloat;
                var uvStream = model.Streams[shapeIndex * 3 + 2] as StreamFloat;
                var indexStream = model.Streams[indexStreamOffset + shapeIndex] as StreamInt;

                ObjShape objShape = context.Shapes[shapeIndex];
                int triangleCount = objShape.Faces.Count;
                int vertexIndex = 0;
                for (int j = 0; j < triangleCount; j++)
                {
                    // p0 u0 n0 p1 u1 n1 p2 u2 n2
                    int[] triangle = objShape.Faces[j];

                    BuildVertex(context, positionStream, normalStream, uvStream,
                        triangle[0], triangle[1], triangle[2]);
                    indexStream.Values.Add(vertexIndex);
                    vertexIndex++;

                    BuildVertex(context, positionStream, normalStream, uvStream,
                        triangle[3], triangle[4], triangle[5]);
                    indexStream.Values.Add(vertexIndex);
                    vertexIndex++;

                    BuildVertex(context, positionStream, normalStream, uvStream,
                        triangle[6], triangle[7], triangle[8]);
                    indexStream.Values.Add(vertexIndex);
                    vertexIndex++;
                }

                // position のストリームにスケールを乗算する
                for (int i = 0; i < positionStream.Values.Count; ++i)
                {
                    positionStream.Values[i] = positionStream.Values[i] * context.Magnify;
                }

                // UV 座標が左下原点で上方向が正の場合、垂直に反転させる
                if (context.InvertUvVertical)
                {
                    for (int i = 1; i < uvStream.Values.Count; i += 2)
                    {
                        uvStream.Values[i] = 1.0f - uvStream.Values[i];
                    }
                }
            }
        }

        // 頂点の構築
        private void BuildVertex(ObjContext context,
            StreamFloat position, StreamFloat normal, StreamFloat uv,
            int posIndex, int uvIndex, int nrmIndex)
        {
            position.Values.Add(context.Positions[posIndex]);

            if (nrmIndex != -1)
            {
                normal.Values.Add(context.Normals[nrmIndex]);
            }
            else
            {
                normal.Values.Add(0f);
                normal.Values.Add(1f);
                normal.Values.Add(0f);
            }

            if (uvIndex != -1)
            {
                uv.Values.Add(context.Uvs[uvIndex]);
            }
            else
            {
                uv.Values.Add(0f);
                uv.Values.Add(0f);
            }
        }

        //=====================================================================
        private Dictionary<string, string> MaterialNameTable;
    }
}
