﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace nw.g3d.ifcvtr
{
    public class ProcessExecutor
    {
        public string LogMessage { get; set; } = string.Empty;
        public string ErrorMessage { get; set; } = string.Empty;

        public int Execute(string cmd, string args)
        {
            return this.ExecuteCommand(cmd, args);
        }

        private void WriteLog(string message)
        {
            this.LogMessage += $"{message}";
        }

        private void WriteError(string message)
        {
            this.ErrorMessage += $"{message}";
        }

        private void WriteLogLine(string message)
        {
            WriteLog($"{message}\n");
        }

        private void WriteErrorLine(string message)
        {
            WriteError($"{message}\n");
        }

        private int ExecuteCommand(string cmd, string args = "", bool useShellExecute = false)
        {
            string stdout, stderr;
            ExecuteCommand(out stdout, out stderr, cmd, args, Encoding.Default);

            var info = new ProcessStartInfo(cmd, args) { UseShellExecute = useShellExecute };
            if (!info.UseShellExecute)
            {
                info.RedirectStandardError = true;
                info.RedirectStandardOutput = true;
                info.CreateNoWindow = true;
            }

            var process = new Process { StartInfo = info };
            process.OutputDataReceived += (s, e) => this.WriteLogLine(e.Data);
            process.ErrorDataReceived += (s, e) => this.WriteErrorLine(e.Data);
            process.Start();

            if (!info.UseShellExecute)
            {
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
            }

            process.WaitForExit();

            return process.ExitCode;
        }

        private static void ExecuteCommand(
            out string standardOut,
            out string standardError,
            string command,
            string args,
            Encoding outputEncoding)
        {
            ExecuteCommand(
                out standardOut, out standardError,
                command, args, System.IO.Directory.GetCurrentDirectory(), outputEncoding);
        }

        private static void ExecuteCommand(
            out string standardOut,
            out string standardError,
            string command,
            string args,
            string workingDirectory,
            Encoding outputEncoding)
        {
            System.Diagnostics.Process process = new System.Diagnostics.Process();
            process.StartInfo.FileName = command;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.RedirectStandardInput = false;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.Arguments = args;
            process.StartInfo.WorkingDirectory = workingDirectory;
            process.StartInfo.StandardOutputEncoding = outputEncoding;
            process.StartInfo.StandardErrorEncoding = outputEncoding;

            process.Start();

            //出力を読み取る
            string tempStandardOut = process.StandardOutput.ReadToEnd();
            string tempStandardError = process.StandardError.ReadToEnd();
            process.WaitForExit();

            process.Close();

            standardOut = EncodeStringToDefault(tempStandardOut, outputEncoding);
            standardError = EncodeStringToDefault(tempStandardError, outputEncoding);
        }

        private static string EncodeStringToDefault(string log, Encoding sourceEncoding)
        {
            byte[] logBytes = sourceEncoding.GetBytes(log);
            byte[] encodedBytes = Encoding.Convert(sourceEncoding, Encoding.Default, logBytes);
            string encodedLog = Encoding.Default.GetString(encodedBytes);
            return encodedLog;
        }
    }
}
