﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.toollib;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using System.IO;
using System.Diagnostics;

namespace nw.g3d.ifassign
{
    internal class UpdateShaderUtility : AssignUtility
    {
        private bool Test;

        private string shaderDefinitionFolder = string.Empty;

        // コンストラクタ
        internal UpdateShaderUtility(g3difassignParams.UpdateModelSubCommand programOption)
            : base(programOption)
        {
            Test = programOption.Test;

            shaderDefinitionFolder = programOption.ShaderPath;
            if (!Directory.Exists(shaderDefinitionFolder))
            {
                throw new Exception(string.Format(Resources.StringResource.Error_Assign_ShaderDirectorNotFound, shaderDefinitionFolder));
            }

            Input = programOption.Path;
            Output = programOption.Output;
        }

        //=====================================================================
        // 処理
        internal override void Process()
        {
            string sourcePath = Input;

            // パスのチェック
            if (!File.Exists(sourcePath))
            {
                throw new Exception(string.Format(Resources.StringResource.Error_InputFileNotFound, sourcePath));
            }
            if (!G3dPath.IsModelPath(sourcePath))
            {
                throw new Exception(string.Format(Resources.StringResource.Error_InputFileIsNotModelFtpFma, sourcePath));
            }

            // 出力パスの設定
            string outputPath = sourcePath;
            if (Output != null)
            {
                outputPath = Output;

                // パスのチェック
                if (!G3dPath.IsModelPath(outputPath))
                {
                    throw new Exception(string.Format(Resources.StringResource.Error_OutputFileExt, outputPath));
                }
            }

            // "{sourcePath} に対する処理を行います。"
            MessageBuilder.AppendLine(string.Format(Resources.StringResource.UpdateExecute, sourcePath));

            // モデル読み込み
            List<G3dStream> streams = new List<G3dStream>();
            nw4f_3difType nw4f_3dif = IfReadUtility.Read(streams, sourcePath, g3difassign.XsdBasePath);
            modelType model = (modelType)nw4f_3dif.Item;

            // マテリアルごとに処理
            var materials = model.material_array != null ?
                model.material_array.material.Where(x => x.shader_assign != null) :
                Enumerable.Empty<materialType>();
            foreach (var material in materials)
            {
                string shader_archive = string.Empty;
                string shadingModelName = string.Empty;
                if (material.shader_assign != null)
                {
                    shader_archive = material.shader_assign.shader_archive;
                    shadingModelName = material.shader_assign.shading_model;
                }

                // シェーダ定義とシェーディングモデルの取得
                shader_definitionType shaderDefinition = null;
                shading_modelType shadingModel = null;
                if (shader_archive != string.Empty)
                {
                    shadingModel = GetShadingModel(material, out shaderDefinition);

                    // 見つからなかったときは中断
                    if (shaderDefinition == null)
                    {
                        MessageBuilder.AppendLine(string.Format(Resources.StringResource.ShaderDefinitionNotFound, material.name, shader_archive));
                        continue;
                    }
                    else if (shadingModel == null)
                    {
                        MessageBuilder.AppendLine(string.Format(Resources.StringResource.ShadingModelNotFound, material.name, shadingModelName));
                        continue;
                    }
                    else if (!shadingModel.material_shader)
                    {
                        // マテリアル用でなかった場合は中断
                        MessageBuilder.AppendLine(string.Format(Resources.StringResource.ShadingModelNotForMaterial, material.name, shadingModelName));
                        continue;
                    }

                    MessageBuilder.Append(DeprecatedMessage(shadingModel));
                }

                if (shader_archive != string.Empty)
                {
                    // 修正メッセージ
                    List<string> updateMessages = IfShaderAssignUtility.UpdateMessages(
                        model,
                        material,
                        shadingModel,
                        false);

                    if (updateMessages.Count > 0)
                    {
                        MessageBuilder.AppendLine(string.Format(Resources.StringResource.UpdateMaterial, material.name));
                    }

                    foreach (var line in updateMessages)
                    {
                        MessageBuilder.AppendLine(line);
                    }

                    // 更新
                    IfShaderAssignUtility.UpdateShaderAssign(
                        model,
                        material,
                        shader_archive,
                        shadingModel,
                        initialize: false,
                        allowEmptyString: false);
                }
            }

            // 中間ファイル出力
            if (!this.DisableFileInfo) { IfFileLogUtility.SetModify(nw4f_3dif); }
            if (!Test)
            {
                IfWriteUtility.Write(nw4f_3dif, streams, outputPath);
            }
        }

        // シェーダ定義を取得する
        private shading_modelType GetShadingModel(materialType material, out shader_definitionType shaderDefinition)
        {
            shaderDefinition = GetShaderDefinition(material.shader_assign.shader_archive);

            if (shaderDefinition != null &&
                shaderDefinition.shading_model_array != null)
            {
                return shaderDefinition.shading_model_array.shading_model.FirstOrDefault(
                    x => x.name == material.shader_assign.shading_model);
            }

            return null;
        }

        private shader_definitionType GetShaderDefinition(string shaderArchive)
        {
            shader_definitionType shaderDefinition;
            lock (shaderDefinitions)
            {
                if (shaderDefinitions.ContainsKey(shaderArchive))
                {
                    // 既に探していれば再利用
                    shaderDefinition = shaderDefinitions[shaderArchive];
                }
                else
                {
                    // シェーダ定義を探す
                    shaderDefinition = SearchShaderDefinition(shaderDefinitionFolder, shaderArchive);
                    shaderDefinitions.Add(shaderArchive, shaderDefinition);
                }
            }
            return shaderDefinition;
        }

        /// <summary>
        /// フォルダ以下から再帰的にシェーダ定義を探す
        /// </summary>
        private shader_definitionType SearchShaderDefinition(string folder, string shaderArchive)
        {
            // MEMO: fsda は現在非対応がだ処理する。
            var fsds = "ba".Select(x => Path.Combine(folder, shaderArchive + ".fsd" + x));
            foreach (var fsd in fsds)
            {
                if (File.Exists(fsd))
                {
                    List<G3dStream> streams = new List<G3dStream>();
                    nw4f_3difType nw4f_3dif = IfReadUtility.Read(streams, fsd, g3difassign.XsdBasePath);
                    return (shader_definitionType)nw4f_3dif.Item;
                }
            }

            foreach (var subFolder in Directory.GetDirectories(folder))
            {
                var shaderDefinition = SearchShaderDefinition(subFolder, shaderArchive);
                if (shaderDefinition != null)
                {
                    return shaderDefinition;
                }
            }

            return null;
        }

        private readonly string Input;
        private readonly string Output;

        /// <summary>
        /// マテリアルが参照するシェーダ定義ファイル
        /// シェーダアーカイブ名をキーにして検索
        /// </summary>
        private static Dictionary<string, shader_definitionType> shaderDefinitions = new Dictionary<string, shader_definitionType>();
    }
}
