﻿using System;
using System.Linq;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Reflection;
using System.Text;

namespace G3dCombinerShaderConverter
{
    public class Utility
    {
        public static string NormalizePath(string targetPath)
        {
            System.IO.FileInfo fi = new System.IO.FileInfo(targetPath);
            return fi.FullName.ToLower();
        }

        public static bool ArePathEqual(string path0, string path1)
        {
            return NormalizePath(path0) == NormalizePath(path1);
        }

        public static string GetTempFilePath(string fileName)
        {
            string tempFolder = Path.GetTempPath();
            string assemblyName = Path.GetFileNameWithoutExtension(Assembly.GetExecutingAssembly().Location);
            return Path.Combine(
                tempFolder,
                $"Nintendo_{assemblyName}.{fileName}");
        }

        public static string GetTempWorkFolderPath()
        {
            string tempFolder = Path.GetTempPath();
            string assemblyName = Path.GetFileNameWithoutExtension(Assembly.GetExecutingAssembly().Location);
            return Path.Combine(
                tempFolder,
                $"Nintendo_{assemblyName}");
        }

        public static string EncodeStringToDefault(string log, Encoding sourceEncoding)
        {
            byte[] logBytes = sourceEncoding.GetBytes(log);
            byte[] encodedBytes = Encoding.Convert(sourceEncoding, Encoding.Default, logBytes);
            string encodedLog = Encoding.Default.GetString(encodedBytes);
            return encodedLog;
        }

        public static void ExecuteProcess(string cmd, string args, bool useShellExecute,
            Action<string> writeMessageAction,
            Action<string> errorMessageAction)
        {
            writeMessageAction($"{cmd} {args}");
            try
            {
                int result = Utility.ExecuteProcessImpl(
                    cmd, args, useShellExecute,
                    writeMessageAction,
                    errorMessageAction);
                if (result != 0)
                {
                    throw new Exception($"{cmd} {args} の実行に失敗しました。");
                }
            }
            catch (Exception e)
            {
                throw new Exception($"{cmd} {args} の実行に失敗しました。\n{e.Message}");
            }
        }

        private static int ExecuteProcessImpl(
            string cmd, string args, bool useShellExecute,
            Action<string> messageAction,
            Action<string> errorMessageAction)
        {
            var info = new ProcessStartInfo(cmd, args) { UseShellExecute = useShellExecute };
            if (!info.UseShellExecute)
            {
                info.RedirectStandardError = true;
                info.RedirectStandardOutput = true;
                info.CreateNoWindow = true;
            }

            var process = new Process { StartInfo = info };
            if (messageAction != null)
            {
                process.OutputDataReceived += (s, e) =>
                {
                    if (e.Data != null)
                    {
                        messageAction(e.Data);
                    }
                };
            }

            if (errorMessageAction != null)
            {
                process.ErrorDataReceived += (s, e) =>
                {
                    if (e.Data != null)
                    {
                        errorMessageAction(e.Data);
                    }
                };
            }

            process.Start();

            if (!info.UseShellExecute)
            {
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
            }

            process.WaitForExit();

            return process.ExitCode;
        }

        public static string FindGraphicsToolPath(string relativePathFromGraphicsFolder)
        {
            string assemblyPath = System.Reflection.Assembly.GetExecutingAssembly().Location;
            string assemblyFolderPath = Path.GetDirectoryName(assemblyPath);

            {
                string toolPath = System.IO.Path.Combine(assemblyFolderPath, "../", relativePathFromGraphicsFolder);
                if (System.IO.File.Exists(toolPath) || System.IO.Directory.Exists(toolPath))
                {
                    return toolPath;
                }
            }

            {
                string debugToolPath = Path.Combine(
                    assemblyFolderPath,
                    "../../../../../../../../../Tools/Graphics",
                    relativePathFromGraphicsFolder);
                if (File.Exists(debugToolPath) || System.IO.Directory.Exists(debugToolPath))
                {
                    return debugToolPath;
                }
            }

            {
                string sdkRoot = Environment.GetEnvironmentVariable("NINTENDO_SDK_ROOT");
                string toolPath = System.IO.Path.Combine(sdkRoot, "Tools/Graphics", relativePathFromGraphicsFolder);
                if (System.IO.File.Exists(toolPath) || System.IO.Directory.Exists(toolPath))
                {
                    return toolPath;
                }
            }

            return null;
        }


        public static string FindCombinerEditorPath()
        {
            string newToolPath = FindGraphicsToolPath("CombinerEditor/CombinerEditor.exe");
            if (!string.IsNullOrEmpty(newToolPath))
            {
                return newToolPath;
            }

            string toolPath = FindGraphicsToolPath("EffectCombinerEditor/EffectCombinerEditor.exe");
            if (!string.IsNullOrEmpty(toolPath))
            {
                return toolPath;
            }

            throw new Exception($"\"EffectCombinerEditor.exe\"、または \"CombinerEditor.exe\" が見つかりません。");
        }

        public static string FindShaderConverterPath()
        {
            string toolPath = FindGraphicsToolPath("3dTools/3dShaderConverter.exe");
            if (!string.IsNullOrEmpty(toolPath))
            {
                return toolPath;
            }

            throw new Exception($"3dShaderConveter.exe が見つかりません。");
        }

        public static string FindXsdPath()
        {
            string xsdPath = FindGraphicsToolPath("3dTools/3dIntermediateFileXsd");
            if (!string.IsNullOrEmpty(xsdPath))
            {
                return xsdPath;
            }

            throw new Exception($"3dIntermediateFileXsd が見つかりません。");
        }


        public static IEnumerable<string> ExpandFilePathPatterns(string pathPatternsText, string directorySearchFilePattern = "*")
        {
            List<string> result = new List<string>();
            if (string.IsNullOrEmpty(pathPatternsText))
            {
                return result;
            }

            string[] pathPatterns = pathPatternsText.Split(';');
            foreach (string pathPattern in pathPatterns)
            {
                var expandedPathPattern = Environment.ExpandEnvironmentVariables(pathPattern);
                string fullPathPattern = expandedPathPattern;
                if (!System.IO.Path.IsPathRooted(expandedPathPattern))
                {
                    fullPathPattern = System.IO.Path.Combine(System.Environment.CurrentDirectory, expandedPathPattern);
                }

                if (!fullPathPattern.Contains('*'))
                {
                    if (System.IO.File.Exists(fullPathPattern))
                    {
                        result.Add(fullPathPattern);
                        continue;
                    }
                    else if (System.IO.Directory.Exists(fullPathPattern))
                    {
                        result.AddRange(System.IO.Directory.EnumerateFiles(fullPathPattern, directorySearchFilePattern));
                        continue;
                    }
                    else
                    {
                        throw new Exception($"path was not found: {fullPathPattern}");
                    }
                }
                else
                {
                    var root = System.IO.Directory.GetDirectoryRoot(fullPathPattern);
                    var searchPattern = fullPathPattern.Substring(root.Length);

                    string fileNamePattern = System.IO.Path.GetFileName(fullPathPattern);
                    bool isFilePathPattern = fileNamePattern.Contains("*");
                    if (isFilePathPattern)
                    {
                        IEnumerable<string> files = System.IO.Directory.EnumerateFiles(root, searchPattern);
                        result.AddRange(files);
                    }
                    else
                    {
                        IEnumerable<string> folders = System.IO.Directory.EnumerateDirectories(root, searchPattern);
                        foreach (string folder in folders)
                        {
                            result.AddRange(System.IO.Directory.EnumerateFiles(folder, directorySearchFilePattern, System.IO.SearchOption.TopDirectoryOnly));
                        }
                    }
                }
            }

            return result.Distinct();
        }
    }
}
