﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// CreateImageFileFromCtex OutputDdsFile

// TextureEncoder DdsHeader

//=============================================================================
// include
//=============================================================================
#include "Texture.h"

using namespace std;
using namespace nn::gfx::tool::dcc;

//=============================================================================
//! @brief DLL から関数を取得するためのマクロです。
//=============================================================================
#define PROC_ADDRESS(handle, name)                                   \
*reinterpret_cast<void**>(&name) = GetProcAddress(handle, #name);    \
    if (name == nullptr)                                             \
    {                                                                \
        cerr << "Error: Cannot find function: " << #name << endl;    \
    }                                                                \

//=============================================================================
// c2nn ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace g3dTool {
namespace c2nn {

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//=============================================================================
// constants
//=============================================================================

//-----------------------------------------------------------------------------
// ctex
const int CtexStandardBlockW = 8;
const int CtexStandardBlockH = 8;
const int CtexStandardBlockPixelCount = CtexStandardBlockW * CtexStandardBlockH;

const int CtexStandardBlockPixelPoss[CtexStandardBlockPixelCount][2] =
{
    { 0, 0 }, { 1, 0 }, { 0, 1 }, { 1, 1 },
    { 2, 0 }, { 3, 0 }, { 2, 1 }, { 3, 1 },
    { 0, 2 }, { 1, 2 }, { 0, 3 }, { 1, 3 },
    { 2, 2 }, { 3, 2 }, { 2, 3 }, { 3, 3 },
    { 4, 0 }, { 5, 0 }, { 4, 1 }, { 5, 1 },
    { 6, 0 }, { 7, 0 }, { 6, 1 }, { 7, 1 },
    { 4, 2 }, { 5, 2 }, { 4, 3 }, { 5, 3 },
    { 6, 2 }, { 7, 2 }, { 6, 3 }, { 7, 3 },
    { 0, 4 }, { 1, 4 }, { 0, 5 }, { 1, 5 },
    { 2, 4 }, { 3, 4 }, { 2, 5 }, { 3, 5 },
    { 0, 6 }, { 1, 6 }, { 0, 7 }, { 1, 7 },
    { 2, 6 }, { 3, 6 }, { 2, 7 }, { 3, 7 },
    { 4, 4 }, { 5, 4 }, { 4, 5 }, { 5, 5 },
    { 6, 4 }, { 7, 4 }, { 6, 5 }, { 7, 5 },
    { 4, 6 }, { 5, 6 }, { 4, 7 }, { 5, 7 },
    { 6, 6 }, { 7, 6 }, { 6, 7 }, { 7, 7 },
};

const int CtexEtcMetaW = 8;
const int CtexEtcMetaH = 8;
const int CtexEtcMetaBlockCount = 4;
const int CtexEtcBlockW = 4;
const int CtexEtcBlockH = 4;
const int CtexEtcBlockPixelCount = CtexEtcBlockW * CtexEtcBlockH;

//=============================================================================
//! @brief DDS ファイルヘッダのクラスです。
//=============================================================================
class DdsHeader
{
public:
    static const uint32_t DDSD_CAPS        = 0x00000001; // dwCaps/dwCaps2 が有効
    static const uint32_t DDSD_HEIGHT      = 0x00000002; // dwHeight が有効
    static const uint32_t DDSD_WIDTH       = 0x00000004; // dwWidth が有効
    static const uint32_t DDSD_PITCH       = 0x00000008; // dwPitchOrLinearSize が Pitch を表す
    static const uint32_t DDSD_PIXELFORMAT = 0x00001000; // dwPfSize/dwPfFlags/dwRGB～ 等の直接定義が有効
    static const uint32_t DDSD_MIPMAPCOUNT = 0x00020000; // dwMipMapCount が有効
    static const uint32_t DDSD_LINEARSIZE  = 0x00080000; // dwPitchOrLinearSize が LinearSize を表す
    static const uint32_t DDSD_DEPTH       = 0x00800000; // dwDepth が有効

    static const uint32_t DDPF_ALPHAPIXELS = 0x00000001; // RGB 以外に alpha が含まれている
    static const uint32_t DDPF_ALPHA       = 0x00000002; // pixel は Alpha 成分のみ
    static const uint32_t DDPF_FOURCC      = 0x00000004; // dwFourCC が有効
    static const uint32_t DDPF_RGB         = 0x00000040; // dwRGBBitCount/dwRBitMask/dwGBitMask/dwBBitMask/dwRGBAlphaBitMask によってフォーマットが定義されている
    static const uint32_t DDPF_LUMINANCE   = 0x00020000; // 1ch のデータが R G B すべてに展開される
    static const uint32_t DDPF_BUMPDUDV    = 0x00080000; // pixel が符号付であることを示す（本来は bump 用）

    static const uint32_t DDSCAPS_ALPHA    = 0x00000002; // Alpha が含まれている場合 (あまり参照されない)
    static const uint32_t DDSCAPS_COMPLEX  = 0x00000008; // 複数のデータが含まれている場合 Palette/Mipmap/Cube/Volume 等
    static const uint32_t DDSCAPS_TEXTURE  = 0x00001000; // 常に 1
    static const uint32_t DDSCAPS_MIPMAP   = 0x00400000; // MipMap が存在する場合

    static const uint32_t DDSCAPS2_CUBEMAP   = 0x00000200; // Cubemap が存在する場合
    static const uint32_t DDSCAPS2_CUBEFACES = 0x0000fc00; // Cubemap の各フェースの存在フラグ
    static const uint32_t DDSCAPS2_VOLUME    = 0x00200000; // VolumeTexture の場合

    static const uint32_t DDS_A16B16G16R16F = 0x71; // 113
    static const uint32_t DDS_A32B32G32R32F = 0x74; // 116
    static const uint32_t DDS_DX10          = 0x30315844; // DX10

    static const uint32_t DXGI_R32G32B32A32_FLOAT = 0x02; //  2
    static const uint32_t DXGI_R16G16B16A16_FLOAT = 0x0a; // 10
    static const uint32_t DXGI_R8G8B8A8_UNORM     = 0x1c; // 28

    static const uint32_t DDS_DIM_1D = 0x02;
    static const uint32_t DDS_DIM_2D = 0x03;
    static const uint32_t DDS_DIM_3D = 0x04;

    static const uint32_t DDS_MISC_TEXTURECUBE = 0x04;

    // 0x00
    uint32_t dwMagic;    // == 0x20534444  ' SDD'
    uint32_t dwSize;     // == 0x7c で固定
    uint32_t dwFlags;    // ヘッダ内の有効な情報 DDSD_* の組み合わせ
    uint32_t dwHeight;   // 画像の高さ x size
    // 0x10
    uint32_t dwWidth;    // 画像の幅   y size
    uint32_t dwPitchOrLinearSize;    // 横1 line の byte 数 (pitch)
                    // または 1面分の byte 数 (linearsize)
    uint32_t dwDepth;    // 画像の奥行き z size (Volume Texture 用)
    uint32_t dwMipMapCount;  // 含まれている mipmap レベル数
    // 0x20
    uint32_t dwReserved1[11];
    uint32_t dwPfSize;   // == 0x20 で固定
    // 0x50
    uint32_t dwPfFlags;  // pixel フォーマットを表す DDPF_* の組み合わせ
    uint32_t dwFourCC;   // フォーマットが FourCC であらわされる場合のみ
    uint32_t dwRGBBitCount;  // 1 pixel の bit 数
    uint32_t dwRBitMask; // RGB format 時の mask
    // 0x60
    uint32_t dwGBitMask; // RGB format 時の mask
    uint32_t dwBBitMask; // RGB format 時の mask
    uint32_t dwRGBAlphaBitMask;  // RGB format 時の mask
    uint32_t dwCaps;     // mipmap 等のフラグ指定用
    // 0x70
    uint32_t dwCaps2;    // cube/volume texture 等のフラグ指定用
    uint32_t dwReservedCaps[2];
    uint32_t dwReserved2;

public:
    //! コンストラクタです。
    DdsHeader()
    {
        //cerr << "DdsHeader(): " << sizeof(*this) << endl;
        memset(this, 0x00, sizeof(*this));
        dwMagic  = 0x20534444;
        dwSize   = 0x7c;
        dwPfSize = 0x20;
    }
};

//=============================================================================
//! @brief DDS ファイル DX10 ヘッダのクラスです。
//=============================================================================
class DdsDX10Header
{
public:
    uint32_t dxgiFormat;
    uint32_t resourceDimension;
    uint32_t miscFlag;
    uint32_t arraySize;
    uint32_t reserved;

public:
    //! コンストラクタです。
    DdsDX10Header()
    : dxgiFormat(DdsHeader::DXGI_R8G8B8A8_UNORM),
      resourceDimension(DdsHeader::DDS_DIM_2D),
      miscFlag(0),
      arraySize(1),
      reserved(0)
    {
    }
};

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの 1 フェースのビットマップを設定します。
//!
//! @param[out] pDst 出力ビットマップのポインタです。
//! @param[in] pSrc 入力ビットマップのポインタです。
//!                 1 ピクセルあたり 4 または 8 または 16 バイトで、RGBA の順に格納します。
//! @param[in] colBytes 出力ビットマップの 1 ピクセルあたりのバイト数です。
//! @param[in] pixelCount 1 フェースのピクセル数です。
//! @param[in] dxgiFormat DXGI フォーマットです。
//! @param[in] hasAlpha アルファ成分を持つなら true です。
//! @param[in] fourCC ヘッダのフォーマット情報です。
//-----------------------------------------------------------------------------
void SetDdsFaceBitmap(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int colBytes,
    const int pixelCount,
    const int dxgiFormat,
    const bool hasAlpha,
    const uint32_t fourCC
)
{
    if (dxgiFormat == DdsHeader::DXGI_R8G8B8A8_UNORM)
    {
        for (int pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            pDst[0] = pSrc[0]; // R
            pDst[1] = pSrc[1]; // G
            pDst[2] = pSrc[2]; // B
            pDst[3] = (hasAlpha) ? pSrc[3] : 0xff; // A
            pDst += colBytes;
            pSrc += R_RGBA_COUNT;
        }
    }
    else if (fourCC == 0)
    {
        for (int pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            pDst[0] = pSrc[2]; // B
            pDst[1] = pSrc[1]; // G
            pDst[2] = pSrc[0]; // R
            if (hasAlpha)
            {
                pDst[3] = pSrc[3]; // A
            }
            pDst += colBytes;
            pSrc += R_RGBA_COUNT;
        }
    }
    else if (fourCC == DdsHeader::DDS_A16B16G16R16F)
    {
        const uint16_t* pSrcU16 = reinterpret_cast<const uint16_t*>(pSrc);
        uint16_t* pDstU16 = reinterpret_cast<uint16_t*>(pDst);
        for (int pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            memcpy(pDstU16, pSrcU16, colBytes);
            pDstU16 += R_RGBA_COUNT;
            pSrcU16 += R_RGBA_COUNT;
        }
    }
    else
    {
        const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        for (int pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            memcpy(pDstF32, pSrcF32, colBytes);
            if (!hasAlpha)
            {
                pDstF32[3] = 1.0f;
            }
            pDstF32 += R_RGBA_COUNT;
            pSrcF32 += R_RGBA_COUNT;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルを出力します。
//!
//! @param[in] filePath DDS ファイルのパスです。
//! @param[in] pBitmap ビットマップへのポインタです。
//!                    1 ピクセルあたり 4 または 8 または 16 バイトで、RGBA の順に格納します。
//!                    ミップマップの場合、レベル 0 の全フェースの後にレベル 1 の全フェースを格納します。
//! @param[in] isCubeMap キューブマップなら true です。
//! @param[in] isVolume ボリュームテクスチャなら true です。
//! @param[in] is1D 1D テクスチャなら true です。
//! @param[in] imageW ビットマップの幅です。
//! @param[in] imageH ビットマップの高さです。
//! @param[in] imageD ビットマップの奥行きです。
//! @param[in] mipLevel ミップマップのレベル数です。
//! @param[in] hasAlpha アルファ成分を持つなら true です。
//! @param[in] fourCC ヘッダのフォーマット情報です。
//!                   0（アルファの有無に応じて R8G8B8 か A8R8G8B8 を選択）、
//!                   DDS_A16B16G16R16F、DDS_A32B32G32R32F のいずれかを指定します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OutputDdsFile(
    const std::string& filePath,
    const void* pBitmap,
    const bool isCubeMap,
    const bool isVolume,
    const bool is1D,
    const uint32_t imageW,
    const uint32_t imageH,
    const uint32_t imageD,
    const uint32_t mipLevel,
    const bool hasAlpha,
    const uint32_t fourCC
)
{
    const int RgbaFloat16Bytes = 4 * sizeof(uint16_t); //!< RGBA 各 16 ビット浮動小数点数のピクセルのバイト数です。

    RStatus status;

    //-----------------------------------------------------------------------------
    // ファイルをオープンします。
    std::ofstream ofs(filePath.c_str(), ios_base::binary);
    if (!ofs)
    {
        return RStatus(RStatus::FAILURE, "Can't open the file: " + filePath); // RShowError;
    }

    //-----------------------------------------------------------------------------
    // ヘッダを出力します。
    const bool isCubeArray = (isCubeMap && imageD > RImage::CUBE_FACE_COUNT);
    const bool isArray = (!isVolume && !isCubeMap && imageD >= 2) || isCubeArray;
    //const bool isDX10 = is1D || isCubeArray; // 2D 配列テクスチャを DX9 形式で出力するならこちら
    const bool isDX10 = (is1D || isArray);

    DdsHeader header;
    header.dwWidth  = imageW;
    header.dwHeight = imageH;
    if (!isDX10)
    {
        header.dwDepth  = (isVolume || isArray) ? imageD : 0;
    }
    else
    {
        header.dwDepth  = (isVolume) ? imageD : ((isArray) ? 1 : 0);
    }

    const bool isHalf  = (fourCC == DdsHeader::DDS_A16B16G16R16F);
    const bool isFloat = (fourCC == DdsHeader::DDS_A32B32G32R32F);
    int dxgiFormat = 0;
    if (isDX10)
    {
        dxgiFormat = (isFloat) ? DdsHeader::DXGI_R32G32B32A32_FLOAT :
            ((isHalf) ? DdsHeader::DXGI_R16G16B16A16_FLOAT : DdsHeader::DXGI_R8G8B8A8_UNORM);
    }

    const int colBytes = (isHalf || isFloat) ?
        ((isHalf) ? RgbaFloat16Bytes : R_RGBA_FLOAT_BYTES) :
        ((dxgiFormat == DdsHeader::DXGI_R8G8B8A8_UNORM || hasAlpha) ? R_RGBA_BYTES : R_RGB_BYTES);
    const int faceBytes = imageW * imageH * colBytes;
    header.dwPitchOrLinearSize = faceBytes; // 1 フェースのバイト数

    header.dwFlags = DdsHeader::DDSD_CAPS | DdsHeader::DDSD_HEIGHT | DdsHeader::DDSD_WIDTH |
        DdsHeader::DDSD_PIXELFORMAT | DdsHeader::DDSD_LINEARSIZE;
    if (header.dwDepth != 0)
    {
        header.dwFlags |= DdsHeader::DDSD_DEPTH;
    }

    header.dwCaps = DdsHeader::DDSCAPS_TEXTURE;
    if (mipLevel >= 2 || isCubeMap || isVolume)
    {
        header.dwCaps |= DdsHeader::DDSCAPS_COMPLEX;
    }

    if (isCubeMap)
    {
        header.dwCaps2 |= DdsHeader::DDSCAPS2_CUBEMAP | DdsHeader::DDSCAPS2_CUBEFACES;
    }
    else if (isVolume)
    {
        header.dwCaps2 |= DdsHeader::DDSCAPS2_VOLUME;
    }

    if (mipLevel >= 2)
    {
        header.dwFlags |= DdsHeader::DDSD_MIPMAPCOUNT;
        header.dwMipMapCount = mipLevel;
        header.dwCaps |= DdsHeader::DDSCAPS_MIPMAP;
    }

    if (isDX10)
    {
        header.dwPfFlags |= DdsHeader::DDPF_FOURCC;
        header.dwFourCC = DdsHeader::DDS_DX10;
    }
    else if (fourCC != 0)
    {
        header.dwPfFlags |= DdsHeader::DDPF_FOURCC;
        header.dwFourCC = fourCC;
    }
    else // RGB
    {
        header.dwPfFlags |= DdsHeader::DDPF_RGB;
        if (hasAlpha)
        {
            header.dwPfFlags |= DdsHeader::DDPF_ALPHAPIXELS;
        }
        header.dwRGBBitCount = colBytes * 8;
        header.dwRBitMask = 0x00ff0000;
        header.dwGBitMask = 0x0000ff00;
        header.dwBBitMask = 0x000000ff;
        header.dwRGBAlphaBitMask = (hasAlpha) ? 0xff000000 : 0;
    }

    ofs.write(reinterpret_cast<const char*>(&header), sizeof(header));

    //-----------------------------------------------------------------------------
    // DX10 ヘッダを出力します。
    if (isDX10)
    {
        DdsDX10Header dx10Header;
        dx10Header.dxgiFormat = dxgiFormat;
        dx10Header.resourceDimension = (is1D) ? DdsHeader::DDS_DIM_1D :
            ((isVolume) ? DdsHeader::DDS_DIM_3D : DdsHeader::DDS_DIM_2D);
        if (isCubeMap)
        {
            dx10Header.miscFlag |= DdsHeader::DDS_MISC_TEXTURECUBE;
        }
        dx10Header.arraySize = (isCubeArray) ? (imageD / RImage::CUBE_FACE_COUNT) : ((isArray) ? imageD : 1);
        ofs.write(reinterpret_cast<const char*>(&dx10Header), sizeof(dx10Header));
    }

    //-----------------------------------------------------------------------------
    // ビットマップを出力します。
    uint32_t bitmapSize = 0;
    for (uint32_t level = 0; level < mipLevel; ++level)
    {
        const uint32_t curW = RMax(imageW >> level, static_cast<uint32_t>(1));
        const uint32_t curH = RMax(imageH >> level, static_cast<uint32_t>(1));
        const uint32_t curD = (isVolume) ? RMax(imageD >> level, static_cast<uint32_t>(1)) : imageD;
        bitmapSize += curW * curH * curD * colBytes;
    }
    uint8_t* pDstBitmap = new uint8_t[bitmapSize];

    const int srcColBytes = (isHalf) ? RgbaFloat16Bytes :
        ((isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES);
    uint8_t* pDst = pDstBitmap;
    if (isVolume)
    {
        //-----------------------------------------------------------------------------
        // ボリュームテクスチャの場合、ビットマップの先頭から出力します。
        // レベル 0 のフェース 0
        // レベル 0 のフェース 1
        // ...
        // レベル 1 のフェース 0
        // レベル 1 のフェース 1
        // ...
        const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(pBitmap);
        for (uint32_t level = 0; level < mipLevel; ++level)
        {
            const uint32_t curW = RMax(imageW >> level, static_cast<uint32_t>(1));
            const uint32_t curH = RMax(imageH >> level, static_cast<uint32_t>(1));
            const uint32_t pixelCount = curW * curH;
            const uint32_t curD = (isVolume) ? RMax(imageD >> level, static_cast<uint32_t>(1)) : imageD;
            for (uint32_t iz = 0; iz < curD; ++iz)
            {
                SetDdsFaceBitmap(pDst, pSrc, colBytes, pixelCount, dxgiFormat, hasAlpha, fourCC);
                pSrc += pixelCount * srcColBytes;
                pDst += pixelCount * colBytes;
            }
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // ボリュームテクスチャ以外の場合、次の順で出力します。
        // フェース 0 のレベル 0
        // フェース 0 のレベル 1
        // ...
        // フェース 1 のレベル 0
        // フェース 1 のレベル 1
        // ...
        uint32_t srcMipOfs = 0;
        RUIntArray srcMipOfss;
        for (uint32_t level = 0; level < mipLevel; ++level)
        {
            const uint32_t curW = RMax(imageW >> level, static_cast<uint32_t>(1));
            const uint32_t curH = RMax(imageH >> level, static_cast<uint32_t>(1));
            srcMipOfss.push_back(srcMipOfs);
            srcMipOfs += curW * curH * imageD * srcColBytes;
        }
        for (uint32_t iz = 0; iz < imageD; ++iz)
        {
            for (uint32_t level = 0; level < mipLevel; ++level)
            {
                const uint32_t curW = RMax(imageW >> level, static_cast<uint32_t>(1));
                const uint32_t curH = RMax(imageH >> level, static_cast<uint32_t>(1));
                const uint32_t pixelCount = curW * curH;
                const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(pBitmap) + srcMipOfss[level] +
                    iz * pixelCount * srcColBytes;
                SetDdsFaceBitmap(pDst, pSrc, colBytes, pixelCount, dxgiFormat, hasAlpha, fourCC);
                pDst += pixelCount * colBytes;
            }
        }
    }

    ofs.write(reinterpret_cast<const char*>(pDstBitmap), bitmapSize);
    delete[] pDstBitmap;

    return status;
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief アルファ成分を持つ ctex フォーマットなら true を返します。
//!
//! @param[in] ctexFormat ctex フォーマットです。
//!
//! @return アルファ成分を持つなら true を返します。
//-----------------------------------------------------------------------------
bool CtexFormatHasAlpha(const std::string& ctexFormat)
{
    return (
        ctexFormat == "A4"      ||
        ctexFormat == "A8"      ||
        ctexFormat == "La4"     ||
        ctexFormat == "La8"     ||
        ctexFormat == "Rgb5_a1" ||
        ctexFormat == "Rgba4"   ||
        ctexFormat == "Rgba8"   ||
        ctexFormat == "Etc1_a4");
}

//-----------------------------------------------------------------------------
//! @brief 指定したフォーマットにおけるアライメントした幅（高さ）を返します。
//!
//! @param[in] orgWH 幅（高さ）です。
//! @param[in] ctexFormat ctex フォーマットです。
//!
//! @return アライメントした幅（高さ）を返します。
//-----------------------------------------------------------------------------
int GetAlignedTextureWh(const int orgWH, const std::string& ctexFormat)
{
    int alignWH = (ctexFormat == "Etc1") ? 16 : 8;
    while (orgWH > alignWH)
    {
        alignWH *= 2;
    }
    return alignWH;
}

//-----------------------------------------------------------------------------
//! @brief 4 bit のカラー値を 8 bit 値に変換します。
//!
//! @param[in] src 4 bit のカラー値です。
//!
//! @return 8 bit 値を返します。
//-----------------------------------------------------------------------------
inline uint8_t Get8BitColorFrom4Bit(const uint8_t src)
{
    return (uint8_t)((src << 4) | src);
}

//-----------------------------------------------------------------------------
//! @brief 5 bit のカラー値を 8 bit 値に変換します。
//!
//! @param[in] src 5 bit のカラー値です。
//!
//! @return 8 bit 値を返します。
//-----------------------------------------------------------------------------
inline uint8_t Get8BitColorFrom5Bit(const uint8_t src)
{
    return (uint8_t)((src << 3) | (src >> 2));
}

//-----------------------------------------------------------------------------
//! @brief 6 bit のカラー値を 8 bit 値に変換します。
//!
//! @param[in] src 6 bit のカラー値です。
//!
//! @return 8 bit 値を返します。
//-----------------------------------------------------------------------------
inline uint8_t Get8BitColorFrom6Bit(const uint8_t src)
{
    return (uint8_t)((src << 2) | (src >> 4));
}

//-----------------------------------------------------------------------------
//! @brief L4 フォーマットのテクスチャをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納します。
//! @param[in] pSrc 元データへのポインタです。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//! @param[in] alignW アライメントした画像の幅です。
//! @param[in] alignH アライメントした画像の高さです。
//!
//! @return インクリメントした元データへのポインタを返します。
//-----------------------------------------------------------------------------
const uint8_t* DecodeL4Texture(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const int alignW,
    const int alignH
)
{
    for (int iy = 0; iy < alignH; iy += CtexStandardBlockH)
    {
        for (int ix = 0; ix < alignW; ix += CtexStandardBlockW)
        {
            for (int pixelIdx = 0; pixelIdx < CtexStandardBlockPixelCount; pixelIdx += 2)
            {
                int posX = ix + CtexStandardBlockPixelPoss[pixelIdx][0];
                int posY = iy + CtexStandardBlockPixelPoss[pixelIdx][1];
                if (posX < curW && posY < curH)
                {
                    uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                    pD[0] = pD[1] = pD[2] = Get8BitColorFrom4Bit(static_cast<uint8_t>(*pSrc & 0x0f));
                    pD[3] = 0xff; // a
                }
                posX = ix + CtexStandardBlockPixelPoss[pixelIdx + 1][0];
                posY = iy + CtexStandardBlockPixelPoss[pixelIdx + 1][1];
                if (posX < curW && posY < curH)
                {
                    uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                    pD[0] = pD[1] = pD[2] = Get8BitColorFrom4Bit(static_cast<uint8_t>(*pSrc >> 4));
                    pD[3] = 0xff; // a
                }
                pSrc++;
            }
        }
    }
    return pSrc;
}

//-----------------------------------------------------------------------------
//! @brief L8 フォーマットのテクスチャをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納します。
//! @param[in] pSrc 元データへのポインタです。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//! @param[in] alignW アライメントした画像の幅です。
//! @param[in] alignH アライメントした画像の高さです。
//!
//! @return インクリメントした元データへのポインタを返します。
//-----------------------------------------------------------------------------
const uint8_t* DecodeL8Texture(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const int alignW,
    const int alignH
)
{
    for (int iy = 0; iy < alignH; iy += CtexStandardBlockH)
    {
        for (int ix = 0; ix < alignW; ix += CtexStandardBlockW)
        {
            for (int pixelIdx = 0; pixelIdx < CtexStandardBlockPixelCount; ++pixelIdx)
            {
                const int posX = ix + CtexStandardBlockPixelPoss[pixelIdx][0];
                const int posY = iy + CtexStandardBlockPixelPoss[pixelIdx][1];
                if (posX < curW && posY < curH)
                {
                    uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                    pD[0] = pD[1] = pD[2] = *pSrc;
                    pD[3] = 0xff; // a
                }
                pSrc++;
            }
        }
    }
    return pSrc;
}

//-----------------------------------------------------------------------------
//! @brief LA4 フォーマットのテクスチャをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納します。
//! @param[in] pSrc 元データへのポインタです。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//! @param[in] alignW アライメントした画像の幅です。
//! @param[in] alignH アライメントした画像の高さです。
//!
//! @return インクリメントした元データへのポインタを返します。
//-----------------------------------------------------------------------------
const uint8_t* DecodeLa4Texture(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const int alignW,
    const int alignH
)
{
    for (int iy = 0; iy < alignH; iy += CtexStandardBlockH)
    {
        for (int ix = 0; ix < alignW; ix += CtexStandardBlockW)
        {
            for (int pixelIdx = 0; pixelIdx < CtexStandardBlockPixelCount; ++pixelIdx)
            {
                const int posX = ix + CtexStandardBlockPixelPoss[pixelIdx][0];
                const int posY = iy + CtexStandardBlockPixelPoss[pixelIdx][1];
                if (posX < curW && posY < curH)
                {
                    uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                    pD[0] = pD[1] = pD[2] = Get8BitColorFrom4Bit(static_cast<uint8_t>(*pSrc >> 4));
                    pD[3] = Get8BitColorFrom4Bit(static_cast<uint8_t>(*pSrc & 0x0f)); // a
                }
                pSrc++;
            }
        }
    }
    return pSrc;
}

//-----------------------------------------------------------------------------
//! @brief LA8 フォーマットのテクスチャをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納します。
//! @param[in] pSrc 元データへのポインタです。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//! @param[in] alignW アライメントした画像の幅です。
//! @param[in] alignH アライメントした画像の高さです。
//!
//! @return インクリメントした元データへのポインタを返します。
//-----------------------------------------------------------------------------
const uint8_t* DecodeLa8Texture(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const int alignW,
    const int alignH
)
{
    for (int iy = 0; iy < alignH; iy += CtexStandardBlockH)
    {
        for (int ix = 0; ix < alignW; ix += CtexStandardBlockW)
        {
            for (int pixelIdx = 0; pixelIdx < CtexStandardBlockPixelCount; ++pixelIdx)
            {
                const int posX = ix + CtexStandardBlockPixelPoss[pixelIdx][0];
                const int posY = iy + CtexStandardBlockPixelPoss[pixelIdx][1];
                if (posX < curW && posY < curH)
                {
                    uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                    pD[0] = pD[1] = pD[2] = pSrc[1];
                    pD[3] = pSrc[0]; // a
                }
                pSrc += 2;
            }
        }
    }
    return pSrc;
}

//-----------------------------------------------------------------------------
//! @brief 法線の成分から整数型カラーの成分を取得します。
//!
//! @param[in] nn 法線の成分です。
//!
//! @return 整数型カラーの成分を返します。
//-----------------------------------------------------------------------------
inline uint8_t GetColorFromNormal(const float nn)
{
    if (nn <= -1.0f)
    {
        return 0x01;
    }
    else if (nn >= 1.0f)
    {
        return 0xff;
    }
    else
    {
        return static_cast<uint8_t>(RRound(nn * 127.0f) + 128);
    }
}

//-----------------------------------------------------------------------------
//! @brief HILO8 フォーマットのテクスチャをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納します。
//! @param[in] pSrc 元データへのポインタです。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//! @param[in] alignW アライメントした画像の幅です。
//! @param[in] alignH アライメントした画像の高さです。
//! @param[in] calcsZ 法線の Z 成分を計算して格納するなら true です。
//!
//! @return インクリメントした元データへのポインタを返します。
//-----------------------------------------------------------------------------
const uint8_t* DecodeHilo8Texture(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const int alignW,
    const int alignH,
    const bool calcsZ
)
{
    for (int iy = 0; iy < alignH; iy += CtexStandardBlockH)
    {
        for (int ix = 0; ix < alignW; ix += CtexStandardBlockW)
        {
            for (int pixelIdx = 0; pixelIdx < CtexStandardBlockPixelCount; ++pixelIdx)
            {
                const int posX = ix + CtexStandardBlockPixelPoss[pixelIdx][0];
                const int posY = iy + CtexStandardBlockPixelPoss[pixelIdx][1];
                if (posX < curW && posY < curH)
                {
                    uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                    uint8_t rgba[R_RGBA_COUNT] = { pSrc[1], pSrc[0], 0x00, 0xff };
                    if (calcsZ)
                    {
                        const float nx = (rgba[0] == 0) ? -1.0f : (rgba[0] - 128) / 127.0f;
                        const float ny = (rgba[1] == 0) ? -1.0f : (rgba[1] - 128) / 127.0f;
                        const float nz2 = 1.0f - (nx * nx + ny * ny);
                        const float nz = (nz2 >= 0.0f) ? sqrtf(nz2) : 0.0f;
                        rgba[2] = GetColorFromNormal(nz);
                    }
                    memcpy(pD, rgba, R_RGBA_BYTES);
                }
                pSrc += 2;
            }
        }
    }
    return pSrc;
}

//-----------------------------------------------------------------------------
//! @brief RGB565 フォーマットのテクスチャをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納します。
//! @param[in] pSrc 元データへのポインタです。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//! @param[in] alignW アライメントした画像の幅です。
//! @param[in] alignH アライメントした画像の高さです。
//!
//! @return インクリメントした元データへのポインタを返します。
//-----------------------------------------------------------------------------
const uint8_t* DecodeRgb565Texture(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const int alignW,
    const int alignH
)
{
    for (int iy = 0; iy < alignH; iy += CtexStandardBlockH)
    {
        for (int ix = 0; ix < alignW; ix += CtexStandardBlockW)
        {
            for (int pixelIdx = 0; pixelIdx < CtexStandardBlockPixelCount; ++pixelIdx)
            {
                const int posX = ix + CtexStandardBlockPixelPoss[pixelIdx][0];
                const int posY = iy + CtexStandardBlockPixelPoss[pixelIdx][1];
                if (posX < curW && posY < curH)
                {
                    uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                    const int rgb565 = pSrc[0] | (pSrc[1] << 8);
                    *pD++ = Get8BitColorFrom5Bit(static_cast<uint8_t>((rgb565 >> 11) & 0x1f)); // r
                    *pD++ = Get8BitColorFrom6Bit(static_cast<uint8_t>((rgb565 >>  5) & 0x3f)); // g
                    *pD++ = Get8BitColorFrom5Bit(static_cast<uint8_t>((rgb565      ) & 0x1f)); // b
                    *pD++ = 0xff; // a
                }
                pSrc += 2;
            }
        }
    }
    return pSrc;
}

//-----------------------------------------------------------------------------
//! @brief RGB8 フォーマットのテクスチャをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納します。
//! @param[in] pSrc 元データへのポインタです。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//! @param[in] alignW アライメントした画像の幅です。
//! @param[in] alignH アライメントした画像の高さです。
//!
//! @return インクリメントした元データへのポインタを返します。
//-----------------------------------------------------------------------------
const uint8_t* DecodeRgb8Texture(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const int alignW,
    const int alignH
)
{
    for (int iy = 0; iy < alignH; iy += CtexStandardBlockH)
    {
        for (int ix = 0; ix < alignW; ix += CtexStandardBlockW)
        {
            for (int pixelIdx = 0; pixelIdx < CtexStandardBlockPixelCount; ++pixelIdx)
            {
                const int posX = ix + CtexStandardBlockPixelPoss[pixelIdx][0];
                const int posY = iy + CtexStandardBlockPixelPoss[pixelIdx][1];
                if (posX < curW && posY < curH)
                {
                    uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                    *pD++ = pSrc[2]; // r
                    *pD++ = pSrc[1]; // g
                    *pD++ = pSrc[0]; // b
                    *pD++ = 0xff; // a
                }
                pSrc += 3;
            }
        }
    }
    return pSrc;
}

//-----------------------------------------------------------------------------
//! @brief RGB5_A1 フォーマットのテクスチャをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納します。
//! @param[in] pSrc 元データへのポインタです。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//! @param[in] alignW アライメントした画像の幅です。
//! @param[in] alignH アライメントした画像の高さです。
//!
//! @return インクリメントした元データへのポインタを返します。
//-----------------------------------------------------------------------------
const uint8_t* DecodeRgb5a1Texture(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const int alignW,
    const int alignH
)
{
    for (int iy = 0; iy < alignH; iy += CtexStandardBlockH)
    {
        for (int ix = 0; ix < alignW; ix += CtexStandardBlockW)
        {
            for (int pixelIdx = 0; pixelIdx < CtexStandardBlockPixelCount; ++pixelIdx)
            {
                const int posX = ix + CtexStandardBlockPixelPoss[pixelIdx][0];
                const int posY = iy + CtexStandardBlockPixelPoss[pixelIdx][1];
                if (posX < curW && posY < curH)
                {
                    uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                    const int rgb5a1 = pSrc[0] | (pSrc[1] << 8);
                    *pD++ = Get8BitColorFrom5Bit(static_cast<uint8_t>((rgb5a1 >> 11) & 0x1f)); // r
                    *pD++ = Get8BitColorFrom5Bit(static_cast<uint8_t>((rgb5a1 >>  6) & 0x1f)); // g
                    *pD++ = Get8BitColorFrom5Bit(static_cast<uint8_t>((rgb5a1 >>  1) & 0x1f)); // b
                    *pD++ = ((rgb5a1 & 0x0001) != 0) ? 0xff : 0x00; // a
                }
                pSrc += 2;
            }
        }
    }
    return pSrc;
}

//-----------------------------------------------------------------------------
//! @brief RGBA4 フォーマットのテクスチャをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納します。
//! @param[in] pSrc 元データへのポインタです。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//! @param[in] alignW アライメントした画像の幅です。
//! @param[in] alignH アライメントした画像の高さです。
//!
//! @return インクリメントした元データへのポインタを返します。
//-----------------------------------------------------------------------------
const uint8_t* DecodeRgba4Texture(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const int alignW,
    const int alignH
)
{
    for (int iy = 0; iy < alignH; iy += CtexStandardBlockH)
    {
        for (int ix = 0; ix < alignW; ix += CtexStandardBlockW)
        {
            for (int pixelIdx = 0; pixelIdx < CtexStandardBlockPixelCount; ++pixelIdx)
            {
                const int posX = ix + CtexStandardBlockPixelPoss[pixelIdx][0];
                const int posY = iy + CtexStandardBlockPixelPoss[pixelIdx][1];
                if (posX < curW && posY < curH)
                {
                    uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                    *pD++ = Get8BitColorFrom4Bit(static_cast<uint8_t>(pSrc[1] >> 4  )); // r
                    *pD++ = Get8BitColorFrom4Bit(static_cast<uint8_t>(pSrc[1] & 0x0f)); // g
                    *pD++ = Get8BitColorFrom4Bit(static_cast<uint8_t>(pSrc[0] >> 4  )); // b
                    *pD++ = Get8BitColorFrom4Bit(static_cast<uint8_t>(pSrc[0] & 0x0f)); // a
                }
                pSrc += 2;
            }
        }
    }
    return pSrc;
}

//-----------------------------------------------------------------------------
//! @brief RGBA8 フォーマットのテクスチャをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納します。
//! @param[in] pSrc 元データへのポインタです。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//! @param[in] alignW アライメントした画像の幅です。
//! @param[in] alignH アライメントした画像の高さです。
//!
//! @return インクリメントした元データへのポインタを返します。
//-----------------------------------------------------------------------------
const uint8_t* DecodeRgba8Texture(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const int alignW,
    const int alignH
)
{
    for (int iy = 0; iy < alignH; iy += CtexStandardBlockH)
    {
        for (int ix = 0; ix < alignW; ix += CtexStandardBlockW)
        {
            for (int pixelIdx = 0; pixelIdx < CtexStandardBlockPixelCount; ++pixelIdx)
            {
                const int posX = ix + CtexStandardBlockPixelPoss[pixelIdx][0];
                const int posY = iy + CtexStandardBlockPixelPoss[pixelIdx][1];
                if (posX < curW && posY < curH)
                {
                    uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                    *pD++ = pSrc[3]; // r
                    *pD++ = pSrc[2]; // g
                    *pD++ = pSrc[1]; // b
                    *pD++ = pSrc[0]; // a
                }
                pSrc += 4;
            }
        }
    }
    return pSrc;
}

//-----------------------------------------------------------------------------
//! @brief ETC1_A4 ブロックのアルファをデコードします。
//!
//! @param[out] alphas アルファを格納します。4 x 4 の配列を指定します。
//! @param[in] pSrc 元データへのポインタです。
//-----------------------------------------------------------------------------
void DecodeEtcA4BlockAlpha(uint8_t alphas[][CtexEtcBlockW], const uint8_t* pSrc)
{
    const uint8_t* pS = pSrc;
    for (int bx = 0; bx < CtexEtcBlockW; ++bx)
    {
        const uint8_t v0 = *pS++;
        const uint8_t v1 = *pS++;
        alphas[0][bx] = Get8BitColorFrom4Bit(v0 & 0x0f);
        alphas[1][bx] = Get8BitColorFrom4Bit(v0 >> 4);
        alphas[2][bx] = Get8BitColorFrom4Bit(v1 & 0x0f);
        alphas[3][bx] = Get8BitColorFrom4Bit(v1 >> 4);
    }
}

//-----------------------------------------------------------------------------
//! @brief ETC1 フォーマットのテクスチャをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納します。
//! @param[in] pSrc 元データへのポインタです。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//! @param[in] alignW アライメントした画像の幅です。
//! @param[in] alignH アライメントした画像の高さです。
//! @param[in] alphaFlag ETC1_A4 フォーマットなら true、ETC1 フォーマットなら false です。
//! @param[in] pEncoder テクスチャエンコーダへのポインタです。
//!
//! @return インクリメントした元データへのポインタを返します。
//-----------------------------------------------------------------------------
const uint8_t* DecodeEtc1Texture(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const int alignW,
    const int alignH,
    const bool alphaFlag,
    const TextureEncoder* pEncoder
)
{
    const int Etc1BlockBytes = 8;
    const int Etc1A4BlockBytes = 16;

    //-----------------------------------------------------------------------------
    // 標準的な etc1 フォーマットのデータに変換します。
    const size_t srcBlockBytes = (alphaFlag) ? Etc1A4BlockBytes : Etc1BlockBytes;
    const int srcBlockCount = alignW * alignH / CtexEtcBlockPixelCount;
    uint8_t* pEtc1 = new uint8_t[srcBlockCount * Etc1BlockBytes];
    uint8_t* pE = pEtc1;
    const uint8_t* pS = pSrc + ((alphaFlag) ? 8 : 0);
    for (int blockIdx = 0; blockIdx < srcBlockCount; ++blockIdx)
    {
        pE[0] = pS[7];
        pE[1] = pS[6];
        pE[2] = pS[5];
        pE[3] = pS[4];
        pE[4] = pS[3];
        pE[5] = pS[2];
        pE[6] = pS[1];
        pE[7] = pS[0];
        pE += Etc1BlockBytes;
        pS += srcBlockBytes;
    }

    //-----------------------------------------------------------------------------
    // テクスチャエンコーダでデコードします。
    uint8_t* pDecoded = new uint8_t[alignW * alignH * R_RGBA_COUNT];
    const bool isSucceeded = pEncoder->ConvertFormat(
        pDecoded, pEtc1, L"unorm_8_8_8_8", L"unorm_etc1", L"",
        gfx::tool::texenc::EncodeFlag_Default,
        gfx::tool::texenc::ImageDimension_2d,
        alignW, alignH, 1, 1);
    R_UNUSED_VARIABLE(isSucceeded);
    delete[] pEtc1;

    //-----------------------------------------------------------------------------
    // タイリングを解除します。
    int tiledX = 0;
    int tiledY = 0;
    pS = pSrc;
    for (int iy = 0; iy < alignH; iy += CtexEtcMetaH)
    {
        for (int ix = 0; ix < alignW; ix += CtexEtcMetaW)
        {
            for (int blockIdx = 0; blockIdx < CtexEtcMetaBlockCount; ++blockIdx)
            {
                uint8_t alphas[CtexEtcBlockH][CtexEtcBlockW];
                if (alphaFlag)
                {
                    DecodeEtcA4BlockAlpha(alphas, pS);
                }
                const int blockIx = ((blockIdx & 1) != 0) ? ix + CtexEtcBlockW : ix;
                const int blockIy = ((blockIdx & 2) != 0) ? iy + CtexEtcBlockH : iy;
                for (int by = 0; by < CtexEtcBlockH; ++by)
                {
                    const int posY = blockIy + by;
                    const uint8_t* pT = pDecoded + (tiledX + (tiledY + by) * alignW) * R_RGBA_COUNT;
                    for (int bx = 0; bx < CtexEtcBlockW; ++bx)
                    {
                        const int posX = blockIx + bx;
                        if (posX < curW && posY < curH)
                        {
                            uint8_t* pD = pDst + (posX + posY * curW) * R_RGBA_COUNT;
                            memcpy(pD, pT, R_RGBA_BYTES);
                            pD[3] = (alphaFlag) ? alphas[by][bx] : 0xff;
                        }
                        pT += R_RGBA_COUNT;
                    }
                }
                tiledX += CtexEtcBlockW;
                if (tiledX >= alignW)
                {
                    tiledX = 0;
                    tiledY += CtexEtcBlockH;
                }
                pS += srcBlockBytes;
            }
        }
    }
    delete[] pDecoded;
    return pSrc + srcBlockBytes * srcBlockCount;
}

//-----------------------------------------------------------------------------
//! @brief 画像の上下左右を反転します。
//!
//! @param[in,out] pPixels ピクセルデータです。
//!                1 ピクセルあたり 4 バイトで、RGBA の順に格納されています。
//! @param[in] curW 画像の幅です。
//! @param[in] curH 画像の高さです。
//-----------------------------------------------------------------------------
void FlipHvImage(uint8_t* pPixels, const int curW, const int curH)
{
    // uint32_t 型のポインタを使用して 4 bytes 単位で処理します。
    const int pixelCount = curW * curH;
    uint32_t* fp = reinterpret_cast<uint32_t*>(pPixels);
    uint32_t* bp = fp + pixelCount - 1;
    const int pixelCountHalf = pixelCount / 2;
    for (int pixelIdx = 0; pixelIdx < pixelCountHalf; ++pixelIdx)
    {
        const uint32_t fval = *fp;
        *fp++ = *bp;
        *bp-- = fval;
    }
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief テクスチャエンコーダを初期化します。
//-----------------------------------------------------------------------------
RStatus TextureEncoder::Initialize(const char* dllPath)
{
    //-----------------------------------------------------------------------------
    // 初期化済みなら終了処理をします。
    if (IsInitialized())
    {
        Finalize();
    }

    //-----------------------------------------------------------------------------
    // DLL ファイルのパスを設定します。
    if (dllPath != nullptr)
    {
        m_DllPath = dllPath;
    }
    //cerr << "TextureEncoder: initialize: " << m_DllPath << endl;

    //-----------------------------------------------------------------------------
    // DLL をロードします。
    // エンコーダの DLL ファイルと同じフォルダに存在する PVRTexLib.dll を
    // 優先的にロードするため LOAD_WITH_ALTERED_SEARCH_PATH を指定します。
    m_hDll = LoadLibraryEx(RGetWindowsFilePath(m_DllPath).c_str(), nullptr, LOAD_WITH_ALTERED_SEARCH_PATH);
    if (m_hDll == nullptr)
    {
        return RStatus(RStatus::FAILURE, "Cannot load dll: " + m_DllPath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // 関数アドレスを取得します。
    PROC_ADDRESS(m_hDll, ConvertFormat);

    if (ConvertFormat == nullptr)
    {
        Finalize();
        return RStatus(RStatus::FAILURE, "Cannot find function: " + m_DllPath); // RShowError
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャエンコーダを終了します。
//-----------------------------------------------------------------------------
void TextureEncoder::Finalize()
{
    //cerr << "TextureEncoder: finalize: " << m_hDll << endl;

    //-----------------------------------------------------------------------------
    // DLL を解放します。
    if (m_hDll != nullptr)
    {
        FreeLibrary(m_hDll);
        m_hDll = nullptr;
    }

    //-----------------------------------------------------------------------------
    // 関数アドレスをクリアします。
    ConvertFormat = nullptr;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルのテクスチャ要素から画像ファイルを作成します。
//!
//! @param[out] pCtexFormat ctex フォーマットを格納します。
//! @param[out] pMipLevel ミップマップのレベル数を格納します。
//! @param[in,out] pEncoder テクスチャエンコーダへのポインタです。
//! @param[in] dstPath 画像ファイル（現在は DDS のみ対応）のパスです。
//! @param[in] inputFile 入力ファイルです。
//! @param[in] texName テクスチャ名です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus CreateImageFileFromCtex(
    std::string* pCtexFormat,
    int* pMipLevel,
    TextureEncoder* pEncoder,
    const std::string& dstPath,
    const CIntermediateFile& inputFile,
    const std::string& texName
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // テクスチャ名に対応したテクスチャ要素を取得します。
    const RXMLElement* texElem = nullptr;
    const RXMLElement* texsElem = inputFile.m_GraphicsElem->FindElement("Textures");
    if (texsElem != nullptr)
    {
        for (size_t texIdx = 0; texIdx < texsElem->nodes.size(); ++texIdx)
        {
            const std::string name = texsElem->nodes[texIdx].GetAttribute("Name");
            if (name.empty() || name == texName)
            {
                texElem = &texsElem->nodes[texIdx];
                break;
            }
        }
    }
    if (texElem == nullptr)
    {
        return RStatus(RStatus::FAILURE, "Texture cannot be found: " + texName + " (" + inputFile.m_Path + ")"); // RShowError
    }

    //-----------------------------------------------------------------------------
    // テクスチャ要素の属性を取得します。
    const bool isCubeMap = (texElem->name == "CubeTextureCtr");
    const int imageW = atol(texElem->GetAttribute("Width").c_str());
    const int imageH = atol(texElem->GetAttribute("Height").c_str());
    const int imageD = (isCubeMap) ? RImage::CUBE_FACE_COUNT : 1;
    *pMipLevel = atol(texElem->GetAttribute("MipmapSize").c_str());
    *pCtexFormat = texElem->GetAttribute("Format");
    const bool hasAlpha = CtexFormatHasAlpha(*pCtexFormat);

    //-----------------------------------------------------------------------------
    // ETC フォーマットならテクスチャエンコーダを初期化します。
    if (pCtexFormat->find("Etc") == 0)
    {
        if (!pEncoder->IsInitialized())
        {
            status = pEncoder->Initialize(nullptr);
            RCheckStatus(status);
        }
    }

    //-----------------------------------------------------------------------------
    // Base64 形式のテクスチャデータをデコードします。
    uint8_t* pTextureData = nullptr;
    uint8_t* pCurTextureData = nullptr;
    for (size_t imgsIdx = 0; imgsIdx < texElem->nodes.size(); ++imgsIdx)
    {
        const RXMLElement* imgsElem = &texElem->nodes[imgsIdx];
        if (imgsElem->name.find("Images") != std::string::npos)
        {
            for (size_t imgIdx = 0; imgIdx < imgsElem->nodes.size(); ++imgIdx)
            {
                const RXMLElement* imgElem = &imgsElem->nodes[imgIdx];
                const uint32_t srcSize = static_cast<uint32_t>(imgElem->text.size());
                if (pTextureData == nullptr)
                {
                    pTextureData = new uint8_t[srcSize * imageD]; // デコード後のサイズはデコード前のサイズより絶対に小さい
                    pCurTextureData = pTextureData;
                }
                uint32_t dataBytes;
                const uint8_t* pDecodeBuf = RDecodeBase64(
                    reinterpret_cast<const uint8_t*>(imgElem->text.c_str()), srcSize, dataBytes);
                memcpy(pCurTextureData, pDecodeBuf, dataBytes);
                delete[] pDecodeBuf;
                pCurTextureData += dataBytes;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // テクスチャデータを RGBA8 形式にデコードします。
    int decodedBytes = 0;
    RIntArray levelOfss;
    for (int level = 0; level < *pMipLevel; ++level)
    {
        const int curW = RMax(imageW >> level, 1);
        const int curH = RMax(imageH >> level, 1);
        levelOfss.push_back(decodedBytes);
        decodedBytes += curW * curH * imageD * R_RGBA_BYTES;
    }
    uint8_t* pDecoded = new uint8_t[decodedBytes];

    const uint8_t* pSrc = pTextureData;
    for (int faceIdx = 0; faceIdx < imageD; ++faceIdx)
    {
        int dstFaceIdx = faceIdx;
        if (isCubeMap)
        {
            // DDS では左手座標系で +X、-X、+Y、-Y、+Z、-Z 面の順になるように格納します。
            if (dstFaceIdx == RImage::CUBE_FACE_PZ)
            {
                dstFaceIdx = RImage::CUBE_FACE_NZ;
            }
            else if (dstFaceIdx == RImage::CUBE_FACE_NZ)
            {
                dstFaceIdx = RImage::CUBE_FACE_PZ;
            }
        }

        for (int level = 0; level < *pMipLevel; ++level)
        {
            const int curW = RMax(imageW >> level, 1);
            const int curH = RMax(imageH >> level, 1);
            const int alignW = GetAlignedTextureWh(curW, *pCtexFormat);
            const int alignH = GetAlignedTextureWh(curH, *pCtexFormat);
            const int faceBytes = curW * curH * R_RGBA_BYTES;
            uint8_t* pDst = pDecoded + levelOfss[level] + dstFaceIdx * faceBytes;

            if (*pCtexFormat == "L4" || *pCtexFormat == "A4")
            {
                pSrc = DecodeL4Texture(pDst, pSrc, curW, curH, alignW, alignH);
            }
            else if (*pCtexFormat == "L8" || *pCtexFormat == "A8")
            {
                pSrc = DecodeL8Texture(pDst, pSrc, curW, curH, alignW, alignH);
            }
            else if (*pCtexFormat == "La4")
            {
                pSrc = DecodeLa4Texture(pDst, pSrc, curW, curH, alignW, alignH);
            }
            else if (*pCtexFormat == "La8")
            {
                pSrc = DecodeLa8Texture(pDst, pSrc, curW, curH, alignW, alignH);
            }
            else if (*pCtexFormat == "Hilo8")
            {
                pSrc = DecodeHilo8Texture(pDst, pSrc, curW, curH, alignW, alignH, true);
            }
            else if (*pCtexFormat == "Rgb565")
            {
                pSrc = DecodeRgb565Texture(pDst, pSrc, curW, curH, alignW, alignH);
            }
            else if (*pCtexFormat == "Rgb8")
            {
                pSrc = DecodeRgb8Texture(pDst, pSrc, curW, curH, alignW, alignH);
            }
            else if (*pCtexFormat == "Rgb5_a1")
            {
                pSrc = DecodeRgb5a1Texture(pDst, pSrc, curW, curH, alignW, alignH);
            }
            else if (*pCtexFormat == "Rgba4")
            {
                pSrc = DecodeRgba4Texture(pDst, pSrc, curW, curH, alignW, alignH);
            }
            else if (*pCtexFormat == "Rgba8")
            {
                pSrc = DecodeRgba8Texture(pDst, pSrc, curW, curH, alignW, alignH);
            }
            else if (*pCtexFormat == "Etc1")
            {
                pSrc = DecodeEtc1Texture(pDst, pSrc, curW, curH, alignW, alignH, false, pEncoder);
            }
            else if (*pCtexFormat == "Etc1_a4")
            {
                pSrc = DecodeEtc1Texture(pDst, pSrc, curW, curH, alignW, alignH, true , pEncoder);
            }

            // キューブマップの場合、+Y、-Y 面以外は上下左右反転します。
            if (isCubeMap && (dstFaceIdx != RImage::CUBE_FACE_PY && dstFaceIdx != RImage::CUBE_FACE_NY))
            {
                FlipHvImage(pDst, curW, curH);
            }
        }
    }
    delete[] pTextureData;

    //-----------------------------------------------------------------------------
    // 画像ファイルを出力します。
    status = OutputDdsFile(dstPath, pDecoded, isCubeMap, false, false,
        imageW, imageH, imageD, *pMipLevel, hasAlpha, 0);
    delete[] pDecoded;

    return status;
} // NOLINT(readability/fn_size)

//=============================================================================
// c2nn ネームスペースを終了します。
//=============================================================================
} // c2nn
} // g3dTool
} // nn

