﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// 定数
// RStatus RXMLElement RFlagArray
// RVec2D RVec3D RVec4D RMtx44D
// RIVec2D RIVec3D RIVec4D RRgbaColor ROrientedBB
// RFileBuf RFileMove RWinProcess

// RDataStream RUserData
// RExpOpt
// RScene RTransformNode RBone RSkeleton
// REnvObj RCamera RLight RFog

//=============================================================================
// include
//=============================================================================
#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>

#include <string>
#include <vector>
#include <set>
#include <map>
#include <algorithm>

#include <cassert>
#include <cfloat>
#include <climits>
#include <cmath>
#include <cstdint>
#include <ctime>

#include <winsock2.h> // force winsock2.h (u_short)
#include <windows.h>
#include <shlwapi.h>

#ifndef NN_IMPLICIT
    #define NN_IMPLICIT
#endif

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//=============================================================================
// 定数の定義です。
//=============================================================================

//-----------------------------------------------------------------------------
// color
const int R_RGB_COUNT  = 3; //!< RGB の成分数です。
const int R_RGBA_COUNT = 4; //!< RGBA の成分数です。

const int R_RGB_BYTES  = 3; //!< RGB 各 8 ビット整数のピクセルのバイト数です。
const int R_RGBA_BYTES = 4; //!< RGBA 各 8 ビット整数のピクセルのバイト数です。

const int R_RGB_FLOAT_BYTES  = 3 * sizeof(float); //!< RGB 各 32 ビット浮動小数点数のピクセルのバイト数です。
const int R_RGBA_FLOAT_BYTES = 4 * sizeof(float); //!< RGBA 各 32 ビット浮動小数点数のピクセルのバイト数です。

const int ColorSetComponentCountRa = 5; //!< RA の 2 成分を出力する場合のカラーセット名の末尾の成分数です。

//-----------------------------------------------------------------------------
// math
const double R_SAME_TOLERANCE   = 1.0e-6;  //!< double 型の値を比較する際の許容値です。
const float  R_SAME_TOLERANCE_F = 1.0e-6f; //!< float 型の値を比較する際の許容値です。

const double R_M_PI = 3.14159265358979323846;
const double R_M_RAD_TO_DEG = 180.0 / R_M_PI;
const double R_M_DEG_TO_RAD = R_M_PI / 180.0;

const int R_XYZ_COUNT  = 3; //!< XYZ の成分数です。
const int R_XYZW_COUNT = 4; //!< XYZW の成分数です。

//-----------------------------------------------------------------------------
// output
#ifndef R_ENDL
    #define R_ENDL "\r\n"
#endif

const int R_FLOAT_OUT_PRECISION = 9; //!< std::ostream::precision で指定する精度です。

const int R_BINARY_ALIGNMENT = 32; //!< バイナリ形式の中間ファイルのアライメントです。

//=============================================================================
//! @brief 未使用変数および関数の警告を抑止するためのマクロです。
//=============================================================================
#define R_UNUSED_VARIABLE(Variable) static_cast<void>(&Variable);
#define R_UNUSED_FUNCTION(Function) static_cast<void>(&Function);

//=============================================================================
//! @brief コンテナの定義です。
//=============================================================================

//! @brief char 型配列の定義です。
typedef std::vector<char> RCharArray;

//! @brief unsigned char 型配列の定義です。
typedef std::vector<unsigned char> RUCharArray;

//! @brief short 型配列の定義です。
typedef std::vector<short> RShortArray;

//! @brief unsigned short 型配列の定義です。
typedef std::vector<unsigned short> RUShortArray;

//! @brief int 型配列の定義です。
typedef std::vector<int> RIntArray;

//! @brief unsigned int 型配列の定義です。
typedef std::vector<unsigned int> RUIntArray;

//! @brief float 型配列の定義です。
typedef std::vector<float> RFloatArray;

//! @brief double 型配列の定義です。
typedef std::vector<double> RDoubleArray;

//! @brief 文字列型配列の定義です。
typedef std::vector<std::string> RStringArray;

//! @brief ユニコード文字列型配列の定義です。
typedef std::vector<std::wstring> RWStringArray;

//! @brief bool 型配列の定義です。
typedef std::vector<bool> RBoolArray;

//=============================================================================
// 一般的な数学関連の関数です。
//=============================================================================

//! @brief 絶対値を返します。
template <typename T>
inline T RAbs(const T x)
{
    return (x >= (T)0) ? x : -x;
}

//! @brief 四捨五入した値を返します。
template <typename T>
inline int RRound(const T x)
{
    return (x >= (T)0) ? static_cast<int>(x + (T)0.5) : static_cast<int>(x - (T)0.5);
}

//! @brief 小さいほうの値を返します。
template <typename T>
inline T RMin(const T a, const T b)
{
    return (a < b) ? a : b;
}

//! @brief 大きいほうの値を返します。
template <typename T>
inline T RMax(const T a, const T b)
{
    return (a > b) ? a : b;
}

//! @brief double 型の 0 に近い値を 0 にした値を返します。
inline double RSnapToZero(const double v)
{
    if (-R_SAME_TOLERANCE < v && v < R_SAME_TOLERANCE)
    {
        return 0.0;
    }
    else
    {
        return v;
    }
}

//! @brief float 型の 0 に近い値を 0 にした値を返します。
inline float RSnapToZero(const float v)
{
    if (-R_SAME_TOLERANCE_F < v && v < R_SAME_TOLERANCE_F)
    {
        return 0.0f;
    }
    else
    {
        return v;
    }
}

//! @brief 指定した範囲にクランプした値を返します。
template <typename T>
inline T RClampValue(const T valMin, const T valMax, const T val)
{
    if (val < valMin)
    {
        return valMin;
    }
    else if (val > valMax)
    {
        return valMax;
    }
    else
    {
        return val;
    }
}

//! @brief アライメントした値を返します。
template <typename T>
inline T RAlignValue(T val, const int align)
{
    return (T)((val + align - 1) & (~(align - 1)));
}

//! @brief アライメントしたポインタを返します。
template <typename T>
inline T* RAlignPointer(T* val, const int align)
{
    #ifdef _WIN64
    return (T*)(((unsigned __int64)val + align - 1) & (~(align - 1)));
    #else
    return (T*)(((unsigned __int32)val + align - 1) & (~(align - 1)));
    #endif
}

//! @brief 2 つの値を交換します。
template <typename T>
inline void RSwapValue(T& r1, T& r2)
{
    T tmpVal = r1;
    r1 = r2;
    r2 = tmpVal;
}

//! @brief [0, 360) の範囲に正規化した角度を返します。
//!
//! @param[in] ang 角度 [degree] です。
//!
//! @return [0, 360) の範囲に正規化した角度を返します。
//!
template <typename T>
inline T RNormalizeAngle(T ang)
{
    if (ang >= (T)360)
    {
        const int ishift = static_cast<int>(ang / (T)360);
        ang -= ishift * (T)360;
    }
    else if (ang < (T)0)
    {
        const int ishift = static_cast<int>(-ang / (T)360) + 1;
        ang += ishift * (T)360;
    }
    return ang;
}

//! @brief [-360, 360] の範囲にシフトした角度を返します。
//!
//! @param[in] ang 角度 [degree] です。
//!
//! @return [-360, 360] の範囲にシフトした角度を返します。
//!
template <typename T>
inline T RShiftAngle(T ang)
{
    if (ang < static_cast<T>(-360) || ang > static_cast<T>(360))
    {
        int ishift = static_cast<int>(ang / static_cast<T>(360));
        ang -= ishift * static_cast<T>(360);
    }
    return ang;
}

//! @brief 角度が (-180,180] の範囲に収まるようにシフトします。
//!
//! @param[out] pShift シフト量（360 の倍数）を格納します。nullptr なら格納しません。
//! @param[in] value 角度 [degree] です。
//!
//! @return シフトした角度を返します。
//!
inline double RShiftAngle180(double* pShift, const double value)
{
    const double shift = (value <= -180.0 || 180.0 < value) ?
        RRound(value / 360.0) * (-360.0) : 0.0;
    if (pShift != nullptr)
    {
        *pShift = shift;
    }
    return (shift != 0.0) ? value + shift : value;
}

//! @brief 角度が (-180,180] の範囲に収まるようにシフトします。
//!
//! @param[out] pShift シフト量（360 の倍数）を格納します。nullptr なら格納しません。
//! @param[in] value 角度 [degree] です。
//!
//! @return シフトした角度を返します。
//!
inline float RShiftAngle180(float* pShift, const float value)
{
    const float shift = (value <= -180.0f || 180.0f < value) ?
        RRound(value / 360.0f) * (-360.0f) : 0.0f;
    if (pShift != nullptr)
    {
        *pShift = shift;
    }
    return (shift != 0.0f) ? value + shift : value;
}

inline bool RIsSame(const double a, const double b, const double tolerance = R_SAME_TOLERANCE)
{
    if (a == b)
    {
        return true;
    }
    const double c = a - b;
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSame(const float a, const float b, const float tolerance = R_SAME_TOLERANCE_F)
{
    if (a == b)
    {
        return true;
    }
    const float c = a - b;
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSameRelative(const double a, const double b, const double tolerance = R_SAME_TOLERANCE)
{
    if (a == b)
    {
        return true;
    }
    double c = a - b;
    if (a != 0.0 && b != 0.0)
    {
        c /= a;
    }
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSameRelative(const float a, const float b, const float tolerance = R_SAME_TOLERANCE_F)
{
    if (a == b)
    {
        return true;
    }
    float c = a - b;
    if (a != 0.0f && b != 0.0f)
    {
        c /= a;
    }
    return (-tolerance < c && c < tolerance);
}

inline bool RIsSameAngle(const double a, const double b, const double tolerance = R_SAME_TOLERANCE)
{
    if (RIsSame(a, b, tolerance))
    {
        return true;
    }
    const double na = RNormalizeAngle(a);
    const double nb = RNormalizeAngle(b);
    return
        RIsSame(na, nb, tolerance) ||
        RIsSame( a, nb, tolerance) ||
        RIsSame(na,  b, tolerance);
}

inline bool RIsSameAngle(const float a, const float b, const float tolerance = R_SAME_TOLERANCE_F)
{
    if (RIsSame(a, b, tolerance))
    {
        return true;
    }
    const float na = RNormalizeAngle(a);
    const float nb = RNormalizeAngle(b);
    return
        RIsSame(na, nb, tolerance) ||
        RIsSame( a, nb, tolerance) ||
        RIsSame(na,  b, tolerance);
}

//! @brief 2 のべき乗の数なら true を返します。
//!
//! @param[in] number 判定する数です。
//! @param[in] includeOne 1 も 2 のべき乗の数（2^0）とみなすなら true を指定します。
//!
//! @return 2 のべき乗の数なら true を返します。
//!
bool RIsPower2Number(const int number, const bool includeOne = true);

//! @brief 指定した数以上で最小の 2 のべき乗の数を返します。
//!
//! @param[in] number 数です。
//! @param[in] includeOne 1 も 2 のべき乗の数（2^0）とみなすなら true を指定します。
//!
//! @return 指定した数以上で最小の 2 のべき乗の数を返します。
//!
int RGetPower2Number(const int number, const bool includeOne = true);

//=============================================================================
// 文字列関連の関数です。
//=============================================================================

//! @brief 文字が半角スペースかタブなら true を返します。
inline bool RIsSpace(const char c)
{
    return (c == ' ' || c == '\t');
}

bool RIsShiftJisFirstByte(const unsigned char val);
std::string RGetUtf8FromShiftJis(const std::string& src);
std::string RGetShiftJisFromUtf8(const std::string& src);
std::wstring RGetUnicodeFromShiftJis(const std::string& src);
std::string RGetShiftJisFromUnicode(const std::wstring& src);
bool RIsAsciiString(const std::string& str);

std::string RJoinStrings(const RStringArray& srcs, const std::string& sep);
int RSeparateString(RStringArray& dsts, const std::string& src, const std::string& sep);

//! @brief ユニコード C 配列を RStringArray に変換します。
//!
//! @param[out] dsts 変換先の文字列配列です。
//! @param[in] srcs ユニコード C 配列です。最後に NULL を格納します。
//!
//! @return 文字列数を返します。
//!
int RGetStringArray(RStringArray& dsts, const wchar_t* srcs[]);

std::string RGetLowerCaseString(const std::string& src);
std::string RGetUpperCaseString(const std::string& src);
bool RIsSameStringNoCase(const std::string& str1, const std::string& str2);
std::string RTrimLeftString(const std::string& src, const char* targets = " \t\r\n");
std::string RTrimRightString(const std::string& src, const char* targets = " \t\r\n");
std::string RTrimString(const std::string& src, const char* targets = " \t\r\n");
std::string RCutEndlString(const std::string& src);
std::string RDequoteString(const std::string& src, const char quote = '\"');
std::string RConvertToSingleQuote(const std::string& src);

//! @brief 文字列をトークンに分解します。
//!
//! @param[out] tokens トークン配列を格納します。
//! @param[in] input 文字列です。
//! @param[in] delims 区切り記号です。
//!
//! @return トークン数を返します。
//!
int RTokenizeString(RStringArray& tokens, const std::string& input, const char* delims = " \t\r\n");

//! @brief 文字列を末尾の数字の前で分割します。
//!
//! @param[out] digit 末尾の数字の文字列を格納します。
//! @param[in] src 文字列です。
//!
//! @return 末尾の数字の前の文字列を返します。
//!
std::string RSplitStringEndDigit(std::string& digit, const std::string& src);

//! @brief int 型の数値を文字列に変換します。
//!
//! @param[in] num 数値です。
//! @param[in] format フォーマット文字列です。NULL なら "%d" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetNumberString(const int num, const char* format = NULL);

//! @brief uint32_t 型の数値を文字列に変換します。
//!
//! @param[in] num 数値です。
//! @param[in] format フォーマット文字列です。NULL なら "%u" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetNumberString(const uint32_t num, const char* format = NULL);

//! @brief ポインタを文字列に変換します。
//!
//! @param[in] ptr ポインタです。
//! @param[in] format フォーマット文字列です。NULL なら "%p" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetPointerString(const void* ptr, const char* format = NULL);

//! @brief 文字列中の {n} を引数配列の n 番目の値で置換した文字列を返します。
//!
//! @param[in] src 文字列です。
//! @param[in] args 引数配列です。
//!
//! @return 置換した文字列を返します。
//!
std::string RReplaceArgumentsInString(const std::string& src, const RStringArray& args);

//! @brief Shift-JIS 文字列を XML 用にエンコードします。
std::string REncodeXmlString(const std::string& src);

//! @brief XML 用にエンコードされた Shift-JIS 文字列をデコードします。
std::string RDecodeXmlString(const std::string& src);

//! @brief Shift-JIS 文字列の制御文字をエスケープして他のコマンドに渡せる文字列を返します。
std::string REscapeString(const std::string& src);

//! @brief オプション文字列用にエンコードされたスクリプトの文字列をデコードします。
//!        シングルクォートをダブルクォート、'!' を ';'、
//!        &#x27! をシングルクォート、&#x21! を '!' に変換します。
//!
//! @param[in] src エンコードされたスクリプトの文字列です。
//!
//! @return デコードされたスクリプトの文字列を返します。
//!
std::string RDecodeScriptString(const std::string& src);

//! @brief オプション設定ファイル用の実数文字列を返します。
//!        常に小数点を含めるため、整数の場合は ".0" を付加します。
//!
//! @param[in] value 実数値です。
//!
//! @return 実数値文字列を返します。
//!
template <typename T>
std::string RGetFloatStringForSettingFile(const T value)
{
    const int iv = static_cast<int>(value);
    if (iv == value)
    {
        return RGetNumberString(iv) + ".0";
    }
    else
    {
        std::ostringstream oss;
        RInitOutStreamFormat(oss);
        oss << value;
        return oss.str();
    }
}

//=============================================================================
// ファイルパス関連の関数です。
//=============================================================================
std::string RGetWindowsFilePath(const std::string& path);
std::string RGetUnixFilePath(const std::string& path);
std::string RGetFileNameFromFilePath(const std::string& path);
std::string RGetFolderFromFilePath(const std::string& path);
std::string RGetExtensionFromFilePath(const std::string& path, const bool lower = true);
std::string RGetNoExtensionFilePath(const std::string& path);
bool RIsFullFilePath(const std::string& path);
std::string RGetFullFilePath(const std::string& path, const bool isUnix);

//! @brief 末尾にスラッシュを追加したファイルパスを返します。
std::string RGetFilePathWithEndingSlash(const std::string& path);

//! @brief 末尾にバックスラッシュを追加したファイルパスを返します。
std::string RGetFilePathWithEndingBackslash(const std::string& path);

bool RIsValidFilePathString(const std::string& path);
bool RIsValidFileNameString(const std::string& fileName);

//=============================================================================
// ファイル関連の関数です。
//=============================================================================
bool RFileExists(const std::string& path);
bool RFolderExists(const std::string& path);

//=============================================================================
// ファイル出力関連の関数です。
//=============================================================================
void RInitOutStreamFormat(std::ostream& os);
void ROutUtf8Bom(std::ostream& os);
void ROutUtf16Bom(std::ostream& os);
void ROutUtf8FromShiftJis(std::ostream& os, const std::string& sjisStr);
const char* RTab(const int count = 1);
const char* RBoolStr(const bool value);

//! @brief パディングを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] size サイズ（バイト数）です。
//! @param[in] value パディングに使用する値です。
//!
void ROutPadding(std::ostream& os, const int size, const char value = '\0');

//=============================================================================
// 配列関連の関数です。
//=============================================================================

//! @brief 配列を新たに確保したメモリにコピーします。
//!        コピー元が NULL またはサイズが 0 の場合はコピー先に NULL を設定します。
//!
//! @param[out] dst コピー先のポインタを格納します。
//! @param[in] src コピー元のポインタです。
//! @param[in] size サイズ（バイト数）です。
//!
template <typename T>
inline void RAllocAndCopyArray(T*& dst, const T* src, const int size)
{
    if (size > 0 && src != NULL)
    {
        dst = new T[size];
        memcpy(dst, src, size * sizeof(T));
    }
    else
    {
        dst = NULL;
    }
}

//! @brief std::vector 配列から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//!
template <typename T>
int RFindValueInArray(const std::vector<T>& array, const T& value, const int startIdx = 0)
{
    const int count = static_cast<int>(array.size());
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index] == value)
        {
            return index;
        }
    }
    return -1;
}

//! @brief std::vector 配列から値を検索して存在すればインデックスを返し、
//!        存在しなければ値を追加してそのインデックスを返します。
//!
template <typename T>
int RFindAppendValue(std::vector<T>& array, const T& value, const int startIdx = 0)
{
    int index = RFindValueInArray(array, value, startIdx);
    if (index == -1)
    {
        index = static_cast<int>(array.size());
        array.push_back(value);
    }
    return index;
}

//! @brief std::vector 配列に別の std::vector 配列を追加します。
//!
template <typename T>
void RAppendArrayToArray(std::vector<T>& dst, const std::vector<T>& src)
{
    const int srcCount = static_cast<int>(src.size());
    for (int isrc = 0; isrc < srcCount; ++isrc)
    {
        dst.push_back(src[isrc]);
    }
}

//! @brief std::vector 配列から指定した std::vector 配列を検索して先頭のインデックスを返します。
//!
template <typename T>
int RFindArrayInArray(const std::vector<T>& base, const std::vector<T>& comp)
{
    const int compCount = static_cast<int>(comp.size());
    const int ibaseMax = static_cast<int>(base.size()) - compCount;
    for (int ibase = 0; ibase <= ibaseMax; ++ibase)
    {
        bool sameFlag = true;
        for (int icomp = 0; icomp < compCount; ++icomp)
        {
            if (base[ibase + icomp] != comp[icomp])
            {
                sameFlag = false;
                break;
            }
        }
        if (sameFlag)
        {
            return ibase;
        }
    }
    return -1;
}

//! @brief std::vector 配列から指定した std::vector 配列を検索して
//!        存在すれば先頭のインデックスを返し、
//!        存在しなければ配列を追加してその先頭のインデックスを返します。
//!
template <typename T>
int RFindAppendArray(std::vector<T>& dst, const std::vector<T>& src)
{
    int index = RFindArrayInArray(dst, src);
    if (index == -1)
    {
        index = dst.size();
        RAppendArrayToArray(dst, src);
    }
    return index;
}

//! @brief std::vector 配列が一定値なら true を返します。
//!
template <typename T>
bool RIsConstantArray(const std::vector<T>& array)
{
    const int size = static_cast<int>(array.size());
    for (int index = 1; index < size; ++index)
    {
        if (array[index] != array[0])
        {
            return false;
        }
    }
    return true;
}

//! @brief std::vector 配列が一定値なら true を返します（許容値指定）。
//!
bool RIsConstantArray(const RFloatArray& array, const float tolerance = R_SAME_TOLERANCE_F);

//! @brief 2 つの std::vector 配列が同値なら true を返します。
//!
template <typename T>
bool RIsSameArray(const std::vector<T>& arrayA, const std::vector<T>& arrayB)
{
    const int sizeA = static_cast<int>(arrayA.size());
    if (static_cast<int>(arrayB.size()) != sizeA)
    {
        return false;
    }
    for (int index = 0; index < sizeA; ++index)
    {
        if (arrayA[index] != arrayB[index])
        {
            return false;
        }
    }
    return true;
}

//! @brief 配列の値がすべて正の値なら true を返します。
//!
template <typename T>
bool RIsArrayAllPositive(const std::vector<T>& array, const float tolerance)
{
    const int count = static_cast<int>(array.size());
    for (int index = 0; index < count; ++index)
    {
        if (!array[index].IsAllPositive(tolerance))
        {
            return false;
        }
    }
    return true;
}

bool RIsArrayAllPositive(const RFloatArray& array, const float tolerance);

//! @brief float 型配列の 0 に近い値を 0 にします。
void RSnapToZero(RFloatArray& array);

//! @brief double 型配列の 0 に近い値を 0 にします。
void RSnapToZero(RDoubleArray& array);

void RScaleArray(RFloatArray& array, const float scale);

void RScaleArray(RDoubleArray& array, const double scale);

//! @brief 3 つの float 型配列にスケールを掛けます。
//!
void RScaleArray(
    RFloatArray& xArray,
    RFloatArray& yArray,
    RFloatArray& zArray,
    const float scale
);

//! @brief 3 つの double 型配列にスケールを掛けます。
//!
void RScaleArray(
    RDoubleArray& xArray,
    RDoubleArray& yArray,
    RDoubleArray& zArray,
    const double scale
);

//! @brief std::vector 配列を出力します。
//!
template <typename T>
void RPrintArray(
    std::ostream& os,
    const std::vector<T>& array,
    const char* title = NULL,
    const bool retFlag = false
)
{
    if (title != NULL)
    {
        os << title << ": ";
        if (retFlag)
        {
            os << R_ENDL;
        }
    }
    const int count = static_cast<int>(array.size());
    for (int index = 0; index < count; ++index)
    {
        os << array[index];
        if (retFlag)
        {
            os << R_ENDL;
        }
        else if (index < count - 1)
        {
            os << ", ";
        }
    }
    if (!retFlag)
    {
        os << R_ENDL;
    }
}

//=============================================================================
// メモリ関連の関数です。
//=============================================================================

//! @brief オブジェクトへのポインタが NULL でなければ delete して NULL を代入します。
//!
//! @param[in,out] ptr オブジェクトへのポインタです。
//!
template <typename T>
inline void RFreeAndClearObject(T*& ptr)
{
    if (ptr != NULL)
    {
        delete ptr;
        ptr = NULL;
    }
}

//! @brief 配列へのポインタが NULL でなければ delete[] して NULL を代入します。
//!
//! @param[in,out] ptr 配列へのポインタです。
//!
template <typename T>
inline void RFreeAndClearArray(T*& ptr)
{
    if (ptr != NULL)
    {
        delete[] ptr;
        ptr = NULL;
    }
}

//! @brief メモリからリトルエンディアンの値を取得します。
template <typename T>
void RGetMemLittle(T& value, const void* pMemory)
{
    memcpy(&value, pMemory, sizeof(T));
    #if 0
    // swap for big endian
    uint8_t* p1 = reinterpret_cast<uint8_t*>(&value);
    uint8_t* p2 = p1 + sizeof(T) - 1;
    int byteSizeHalf = sizeof(T) / 2;
    for (int iByte = 0; iByte < byteSizeHalf; ++iByte, ++p1, --p2)
    {
        RSwapValue(*p1, *p2);
    }
    #endif
}

//! @brief メモリからビッグエンディアンの値を取得します。
template <typename T>
void RGetMemBig(T& value, const void* pMemory)
{
    memcpy(&value, pMemory, sizeof(T));
    #if 1
    // swap for little endian
    uint8_t* p1 = reinterpret_cast<uint8_t*>(&value);
    uint8_t* p2 = p1 + sizeof(T) - 1;
    int byteSizeHalf = sizeof(T) / 2;
    for (int iByte = 0; iByte < byteSizeHalf; ++iByte, ++p1, --p2)
    {
        RSwapValue(*p1, *p2);
    }
    #endif
}

//! @brief メモリからリトルエンディアンの short 値を取得します。
inline short RGetMemLittleShort(const void* pMemory)
{
    short value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリからリトルエンディアンの unsigned short 値を取得します。
inline unsigned short RGetMemLittleUShort(const void* pMemory)
{
    unsigned short value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリからリトルエンディアンの int 値を取得します。
inline int RGetMemLittleInt(const void* pMemory)
{
    int value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリからリトルエンディアンの unsigned int 値を取得します。
inline unsigned int RGetMemLittleUInt(const void* pMemory)
{
    unsigned int value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリからビッグエンディアンの short 値を取得します。
inline short RGetMemBigShort(const void* pMemory)
{
    short value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリからビッグエンディアンの unsigned short 値を取得します。
inline unsigned short RGetMemBigUShort(const void* pMemory)
{
    unsigned short value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリからビッグエンディアンの int 値を取得します。
inline int RGetMemBigInt(const void* pMemory)
{
    int value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリからビッグエンディアンの unsigned int 値を取得します。
inline unsigned int RGetMemBigUInt(const void* pMemory)
{
    unsigned int value;
    RGetMemBig(value, pMemory);
    return value;
}

//=============================================================================
// base 64 関連の関数です。
//=============================================================================
uint8_t* REncodeBase64(const uint8_t* srcBuf, const uint32_t srcSize, uint32_t& dstSize);
uint8_t* RDecodeBase64(const uint8_t* srcBuf, const uint32_t srcSize, uint32_t& dstSize);

//=============================================================================
// 一般的な Windows 関連の関数です。
//=============================================================================

//! @brief 環境変数の値を取得します。
//!
//! @param[in] name 環境変数の名前です。
//!
//! @return 環境変数の値を返します。環境変数が定義されていない場合は空文字を返します。
//!
std::string RGetEnvVar(const char* name);

//! @brief 環境変数が定義されていて値が 0 でなければ true を返します。
//!
//! @param[in] name 環境変数の名前です。
//!
//! @return 環境変数が定義されていて値が 0 でなければ true を返します。
//!
bool RIsEnvVarNotZero(const char* name);

//! @brief 呼び出し側プロセスのプロセス ID 文字列を取得します。
//!
//! @return 呼び出し側プロセスのプロセス ID 文字列を返します。
//!
std::string RGetCurrentProcessIdString();

//! @brief 一時フォルダのパスを取得します。パスの最後にはスラッシュが付きます。
//!
//! @return 一時フォルダのパスを返します。
//!
std::string RGetTempFolderPath();

//! @brief PC のユーザー名を返します。
std::string RGetUserName();
//! @brief PC のコンピュータ名を返します。
std::string RGetComputerName();

//! @brief SYSTEMTIME から time_t を取得して返します。
time_t RGetTimeT(const SYSTEMTIME& st);

//! @brief ISO 8601 に準拠した書式で現在の日時を返します。
std::string RGetDateTimeString();

//! @brief 最後に実行した Windows API のエラーメッセージを取得します。
//!
//! @return 最後に実行した Windows API のエラーメッセージを返します。
//!
std::string RGetWinErrorMessage();

//=============================================================================
//! @brief Windows プロセスのクラスです。
//=============================================================================
class RWinProcess
{
protected:
    PROCESS_INFORMATION m_Pi; //!< プロセス情報です。
    SECURITY_ATTRIBUTES m_Sa; //!< セキュリティ属性です。
    HANDLE m_hReadOut; //!< 標準出力パイプのリードハンドルです。
    HANDLE m_hWriteOut; //!< 標準出力パイプのライトハンドルです。
    HANDLE m_hReadErr; //!< 標準エラーパイプのリードハンドルです。
    HANDLE m_hWriteErr; //!< 標準エラーパイプのライトハンドルです。
    bool m_IsValid; //!< プロセスが有効なら true です。
    bool m_IsStarted; //!< プロセスが開始されていれば true です。
    bool m_IsLaunched; //!< プロセスの起動が完了していれば true です。
    int m_ExitCode; //!< 終了コードです。
    std::string m_OutMsg; //!< 出力メッセージです。
    std::string m_ErrMsg; //!< エラーメッセージです。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] getsStdOutOrErr 標準出力または標準エラーを取得するなら true です。
    //!
    explicit RWinProcess(const bool getsStdOutOrErr);

    //! @brief デストラクタです。
    virtual ~RWinProcess();

    //! @brief 有効なら true を返します。
    bool IsValid() const
    {
        return m_IsValid;
    }

    //! @brief プロセス開始済みなら true を返します。
    bool IsStarted() const
    {
        return m_IsStarted;
    }

    //! @brief プロセスの起動が完了していれば true を返します。
    bool IsLaunched() const
    {
        return m_IsLaunched;
    }

    //! @brief プロセス ID を返します。
    int GetProcessId() const
    {
        return m_Pi.dwProcessId;
    }

    //! @brief 終了コードを返します。
    int GetExitCode() const
    {
        return m_ExitCode;
    }

    //! @brief 出力メッセージを返します。
    std::string GetOutMessage() const
    {
        return m_OutMsg;
    }

    //! @brief エラーメッセージを返します。
    std::string GetErrorMessage() const
    {
        return m_ErrMsg;
    }

    //! @brief プロセスを開始します。
    //!
    //! @param[in] cmd コマンド文字列です。
    //! @param[in] showWindow ウィンドウの表示指定です。
    //!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
    //!
    //! @return 開始成功なら true を返します。
    //!
    bool Start(const char* cmd, const int showWindow);

    //! @brief プロセスが開始していれば終了まで待ちます。
    //!
    //! @param[in] checkTimeOut プロセスの終了をチェックする間隔 [ms] です。
    //!                         パイプに大量に出力する場合、間隔が長いと終了までの時間が長くなります。
    //!
    void WaitFinish(const int checkTimeOut);

    //! @brief プロセスが終了しているかチェックします。
    //!
    //! @param[in] timeOut タイムアウト時間 [ms] です。
    //!
    //! @return 終了しているかエラーが発生していれば true を返します。
    //!         タイムアウト時間が過ぎても実行中なら false を返します。
    //!
    bool CheckFinish(const int timeOut);
};

//=============================================================================
// Windows のプロセス関連の関数です。
//=============================================================================

//! @brief Windows のプロセスを実行します。
//!
//! @param[out] pStatus エラーなら 0、成功なら 0 以外を格納します。
//!                     終了コードでエラーを判別できない場合に使用します。
//!                     NULL なら何も格納しません。
//! @param[in] cmd コマンド文字列です。
//! @param[in] showWindow ウィンドウの表示指定です。
//!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
//!
//! @return コマンドの終了コードを返します。
//!
int RExecProcess(int* pStatus, const char* cmd, const int showWindow = SW_HIDE);

//! @brief Windows のプロセスを実行し、標準出力および標準エラーを取得します。
//!
//! @param[out] pStatus エラーなら 0、成功なら 0 以外を格納します。
//!                     終了コードでエラーを判別できない場合に使用します。
//!                     NULL なら何も格納しません。
//! @param[out] pOutMsg 標準出力を格納します。nullptr なら格納しません。
//! @param[out] pErrMsg 標準エラー出力を格納します。nullptr なら格納しません。
//! @param[in] cmd コマンド文字列です。
//! @param[in] showWindow ウィンドウの表示指定です。
//!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
//!
//! @return 終了コードを返します。
//!
int RExecProcessWithPipe(
    int* pStatus,
    std::string* pOutMsg,
    std::string* pErrMsg,
    const char* cmd,
    const int showWindow
);

//! @brief Windows のプロセスを並列実行し、標準出力および標準エラーを取得します。
//!
//! @param[out] pStatus エラーなら 0、成功なら 0 以外を格納します。
//!                     終了コードでエラーを判別できない場合に使用します。
//!                     nullptr なら何も格納しません。
//! @param[out] pOutMsg 標準出力を格納します。nullptr なら格納しません。
//! @param[out] pErrMsg 標準エラー出力を格納します。nullptr なら格納しません。
//! @param[in] cmds コマンド文字列配列です。
//! @param[in] showWindow ウィンドウの表示指定です。
//!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
//!
//! @return 終了コードを返します。
//!
int RExecProcessParallel(
    int* pStatus,
    std::string* pOutMsg,
    std::string* pErrMsg,
    const RStringArray& cmds,
    const int showWindow
);

//=============================================================================
// Windows のレジストリ関連の関数です。
//=============================================================================

//! @brief Windows の DWORD 型のレジストリ値を取得します。
//!
//! @param[in] hParentKey 開いている親キーのハンドルです。
//! @param[in] lpSubKey 開くべきサブキーの名前です。
//! @param[in] lpValueName 取得するレジストリエントリ名です。
//! @param[in] defaultValue 取得できなかった場合に返す値です。
//!
DWORD RGetWinRegDword(
    HKEY hParentKey,
    const char* lpSubKey,
    const char* lpValueName,
    const DWORD defaultValue
);

//! @brief Windows の文字列型のレジストリ値を取得します。
//!
//! @param[in] hParentKey 開いている親キーのハンドルです。
//! @param[in] lpSubKey 開くべきサブキーの名前です。
//! @param[in] lpValueName 取得するレジストリエントリ名です。
//! @param[in] defaultValue 取得できなかった場合に返す値です。
//!
std::string RGetWinRegString(
    HKEY hParentKey,
    const char* lpSubKey,
    const char* lpValueName,
    const char* defaultValue = ""
);

//=============================================================================
// NW 中間ファイル関連の関数です。
//=============================================================================
std::string RAdjustNameChar(const std::string& src);
bool RIsValidElementNameChar(const char c);
bool RIsValidElementNameString(const std::string& name);
std::string RAdjustElementNameString(const std::string& name);
bool RIsValidPresetNameString(const std::string& name);

//! @brief ファイル情報に出力する元ファイルのパスを取得します。
//!
//! @param[in] fullPath 元ファイルのフルパスです。
//! @param[in] projectRootPath アプリケーションのプロジェクトのルートフォルダのパスです。
//!                            空文字でなければ元ファイルのパスはルートフォルダからの相対パスとなります。
//!
//! @return 元ファイルのパスを返します。
//!
std::string RGetFileInfoSourceFilePath(
    const std::string& fullPath,
    const std::string& projectRootPath
);

//=============================================================================
// XML 関連の関数です。
//=============================================================================

//! @brief XML 要素の文字列から属性の値を取得します。
//!
//! @param[in] xmlStr XML 要素の文字列です。
//! @param[in] name 属性名です。
//! @param[in] alt 属性が見つからなかった場合に返す値です。
//!
//! @return 属性の値を返します。
//!
std::string RGetXmlAttr(
    const std::string& xmlStr,
    const std::string& name,
    const char* alt = ""
);

//=============================================================================
//! @brief 処理結果ステータスのクラスです。
//=============================================================================
class RStatus
{
public:
    //! @brief 処理結果コードを表す列挙型です。
    enum RStatusCode
    {
        SUCCESS = 0, //!< 成功です。
        FAILURE = 1  //!< 失敗です。
    };

private:
    RStatusCode m_Code; //!< 処理結果コードです。
    std::string m_MessageJp; //!< 日本語メッセージです。
    std::string m_MessageEn; //!< 英語メッセージです。

public:
    //! @brief コンストラクタです。
    NN_IMPLICIT RStatus(const RStatusCode code = SUCCESS)
    : m_Code(code)
    {
    }

    //! @brief コンストラクタです（英語メッセージのみ）。
    RStatus(const RStatusCode code, const std::string& messageEn)
    : m_Code(code),
      m_MessageEn(messageEn)
    {
    }

    //! @brief コンストラクタです（日英メッセージ）。
    RStatus(const RStatusCode code, const std::string& messageJp, const std::string& messageEn)
    : m_Code(code),
      m_MessageJp(messageJp),
      m_MessageEn(messageEn)
    {
    }

    //! @brief コンストラクタです（日英メッセージ。ワイド文字列）。
    RStatus(const RStatusCode code, const std::wstring& messageJpW, const std::string& messageEn)
    : m_Code(code),
      m_MessageJp(RGetShiftJisFromUnicode(messageJpW)),
      m_MessageEn(messageEn)
    {
    }

    //! @brief コンストラクタです（日英メッセージ。引数配列）。
    RStatus(
        const RStatusCode code,
        const std::string& messageJp,
        const std::string& messageEn,
        const RStringArray& args
    );

    //! @brief コンストラクタです（日英メッセージ。引数配列。ワイド文字列）。
    RStatus(
        const RStatusCode code,
        const std::wstring& messageJpW,
        const std::string& messageEn,
        const RStringArray& args
    );

    //! @brief コンストラクタです（日英メッセージ。引数 1 つ）。
    RStatus(
        const RStatusCode code,
        const std::string& messageJp,
        const std::string& messageEn,
        const std::string& arg0
    )
    {
        RStringArray args;
        args.push_back(arg0);
        *this = RStatus(code, messageJp, messageEn, args);
    }

    //! @brief コンストラクタです（日英メッセージ。引数 1 つ。ワイド文字列）。
    RStatus(
        const RStatusCode code,
        const std::wstring& messageJpW,
        const std::string& messageEn,
        const std::string& arg0
    )
    {
        RStringArray args;
        args.push_back(arg0);
        *this = RStatus(code, messageJpW, messageEn, args);
    }

    //! @brief コンストラクタです（日英メッセージ。引数 2 つ）。
    RStatus(
        const RStatusCode code,
        const std::string& messageJp,
        const std::string& messageEn,
        const std::string& arg0,
        const std::string& arg1
    )
    {
        RStringArray args;
        args.push_back(arg0);
        args.push_back(arg1);
        *this = RStatus(code, messageJp, messageEn, args);
    }

    //! @brief コンストラクタです（日英メッセージ。引数 2 つ。ワイド文字列）。
    RStatus(
        const RStatusCode code,
        const std::wstring& messageJpW,
        const std::string& messageEn,
        const std::string& arg0,
        const std::string& arg1
    )
    {
        RStringArray args;
        args.push_back(arg0);
        args.push_back(arg1);
        *this = RStatus(code, messageJpW, messageEn, args);
    }

    //! 成功なら true を返します。
    NN_IMPLICIT operator bool() const
    {
        return (m_Code == SUCCESS);
    }

    //! 同値であれば true を返します。
    bool operator==(const RStatus& other) const { return (m_Code == other.m_Code); }

    //! 同値でなければ true を返します。
    bool operator!=(const RStatus& other) const { return (m_Code != other.m_Code); }

    //! 同値であれば true を返します。
    bool operator==(const RStatusCode code) const { return (m_Code == code); }

    //! 同値でなければ true を返します。
    bool operator!=(const RStatusCode code) const { return (m_Code != code); }

    //! @brief 日本語メッセージを返します。
    std::string GetJapaneseMessage() const
    {
        return (!m_MessageJp.empty()) ? m_MessageJp : m_MessageEn;
    }

    //! @brief 英語メッセージを返します。
    std::string GetMessage() const
    {
        return m_MessageEn;
    }
};

//! @brief 処理結果ステータスが成功でなければ return するマクロです。
#define RCheckStatus(status)        \
{                                   \
    if (status != RStatus::SUCCESS) \
        return status;              \
}

//=============================================================================
//! @brief 時間計測のクラスです。
//=============================================================================
class RTimeMeasure
{
protected:
    std::clock_t m_StartClock; //!< 計測開始時のクロックです。

public:
    //! @brief コンストラクタです。
    RTimeMeasure()
    {
        Reset();
    }

    //! 計測開始時間を現在のクロックに設定します。
    void Reset()
    {
        m_StartClock = std::clock();
    }

    //! 計測開始時からの秒数を取得します。
    float GetSec() const
    {
        return static_cast<float>(std::clock() - m_StartClock) / CLOCKS_PER_SEC;
    }

    //! 計測開始時からのミリ秒数を取得します。
    float GetMilliSec() const
    {
        return GetSec() * 1000.0f;
    }
};

//=============================================================================
//! @brief XML 要素のクラスです。
//=============================================================================
class RXMLElement
{
public:
    typedef std::map<std::string, std::string> AttrMap;
    typedef void (*ErrorFunc)(const char* message, void* pUserData);

    std::string name;
    AttrMap attributes;
    std::string text;
    std::vector<RXMLElement> nodes;

    ErrorFunc m_ErrorFunc; //!< エラー表示関数です。
    void* m_pErrorUserData; //!< エラー表示関数に渡すユーザーデータのポインタです。

private:
    //! @brief 属性を解析してノードに記録します。
    //!
    //! @param[in] src 要素の文字列です。
    //! @param[in] end 要素の終端の文字です。
    //!
    //! @return 要素の終端までの文字数を返します。
    //!
    int LoadAttributes(const char* src, const char end);

    //! @brief エラーを表示します。
    //!
    //! @param[in] message エラーメッセージです。
    //!
    void DisplayError(const std::string& message) const;

public:
    //! コンストラクタです。
    RXMLElement(ErrorFunc errorFunc = NULL, void* pErrorUserData = NULL)
    : m_ErrorFunc(errorFunc),
      m_pErrorUserData(pErrorUserData)
    {
    }

    //! コピーコンストラクタです。
    RXMLElement(const RXMLElement &r)
    {
        operator=(r);
    }

    //! 再初期化します。
    void Clear()
    {
        name = "";
        text = "";
        attributes.clear();
        nodes.clear();
    }

    //! 代入演算子です。
    RXMLElement& operator=(const RXMLElement& r);

    //! @brief 下層ノードを新規作成します。
    //!
    //! @return 新規作成した要素のポインタを返します。
    //!
    RXMLElement* CreateElement()
    {
        nodes.push_back(RXMLElement(m_ErrorFunc, m_pErrorUserData));
        return &nodes[nodes.size() - 1];
    }

    //! @brief 指定のパスに該当する最初の要素のポインタを取得します。
    //!
    //! @param[in] path パスです。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    //! @return 要素のポインタを返します。見つからなかった場合は NULL を返します。
    //!
    const RXMLElement* FindElement(const std::string& path, const bool showError = true) const
    {
        std::vector<const RXMLElement*> list;
        FindElements(list, path, showError, 1);
        return (list.size() > 0) ? list[0] : NULL;
    }

    //! @brief 指定のパスに該当する要素を指定個数分検索します。
    //!
    //! @param[out] list 要素のポインタ配列です。
    //! @param[in] path パスです。
    //! @param[in] maxCount 取得する最大要素数です。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    void FindElements(
        std::vector<const RXMLElement*>& list,
        const std::string& path,
        const bool showError = true,
        const int maxCount = 0x7fffffff
    ) const;

    //! @brief 属性の値を取得します。
    //!
    //! @param[in] tag 属性名です。
    //! @param[in] alt 属性が見つからなかった場合に返す値です。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    //! @return 属性の値を返します。
    //!
    std::string GetAttribute(
        const std::string& tag,
        const char* alt = "",
        const bool showError = true
    ) const
    {
        AttrMap::const_iterator it = attributes.find(tag);
        if (showError && it == attributes.end())
        {
            DisplayError("Attribute is not found: " + tag + " in <" + name + ">");
        }
        return (it != attributes.end()) ? it->second : alt;
    }

    //! @brief XML 文書を解析してツリーを構築します（ASCII テキスト限定）。
    //!
    //! @param[in,out] decl 宣言要素です。宣言要素の属性を取得しないなら NULL を指定します。
    //! @param[in] xmlDoc XML 文書です。
    //!
    void LoadDocument(RXMLElement* decl, const std::string& xmlDoc);

    //! @brief XML 文書を取得します。
    //!        要素、属性、平データ、下層ノードの内容を再帰的に文字列に出力します。
    //!
    //! @param[in] tabCount 要素のインデントに必要なタブの数です。
    //! @param[in] lf 改行コードです。
    //!
    //! @return XML 文書の文字列を返します。
    //!
    std::string GetDocument(const int tabCount = 0, const char* lf = "\n") const;

    //! @brief エラー表示関数を設定します。
    //!
    //! @param[in] func エラー表示関数です。
    //! @param[in] pUserData エラー表示関数に渡すユーザーデータのポインタです。
    //!
    void SetErrorFunc(ErrorFunc func, void* pUserData)
    {
        m_ErrorFunc = func;
        m_pErrorUserData = pUserData;
    }
};

//=============================================================================
//! @brief フラグ配列のクラスです。
//!        1 要素が 1 ビットなのでメモリ使用量は 4 + 4 + (配列長 / 8) バイトです。
//=============================================================================
class RFlagArray
{
private:
    int m_Count; //!< フラグ数です。
    int m_ArraySize; //!< 配列のサイズです。
    uint8_t* m_Array; //!< 配列です。

public:
    //! @brief コンストラクタです。
    RFlagArray(const int count = 0, const bool initialValue = FALSE)
    : m_Count(count),
      m_ArraySize(0),
      m_Array(NULL)
    {
        if (count > 0)
        {
            Resize(count, initialValue);
        }
    }

    //! デストラクタです。
    virtual ~RFlagArray() { FreeMemory(); }

    //! フラグ数を変更し、値を初期化します。
    void Resize(const int count, const bool initialValue = FALSE)
    {
        FreeMemory();

        m_Count = count;
        m_ArraySize = (m_Count + 7) >> 3;
        if (m_Count > 0)
        {
            m_Array = new uint8_t[m_ArraySize];
            SetAll(initialValue);
        }
    }

    //! すべてのフラグの値を設定します。
    void SetAll(const bool value)
    {
        if (m_ArraySize > 0)
        {
            uint8_t fill = (value) ? 0xff : 0x00;
            memset(m_Array, fill, m_ArraySize);
        }
    }

    //! 指定したフラグを true にします。
    void SetTrue(const int index)
    {
        m_Array[index >> 3] |= 1 << (index & 0x07);
    }

    //! 指定したフラグを false にします。
    void SetFalse(const int index)
    {
        m_Array[index >> 3] &= ~(1 << (index & 0x07));
    }

    //! 指定したフラグを取得します。
    bool Get(const int index) const
    {
        return ((m_Array[index >> 3] & (1 << (index & 0x07))) != 0);
    }

    //! フラグ数を取得します。
    int GetCount() const { return m_Count; };

    //! メモリを解放します。
    void FreeMemory()
    {
        RFreeAndClearArray(m_Array);
    }
};

//=============================================================================
//! @brief float 型 2 次元ベクトルのクラスです。
//=============================================================================
class RVec3;
class RVec4;

class RVec2 // RVec2D
{
public:
    float x; //!< X 成分です。
    float y; //!< Y 成分です。

public:
    //! コンストラクタです（引数なし）。ゼロベクトルで初期化されます。
    RVec2()
    : x(0.0f),
      y(0.0f)
    {
    }

    //! コンストラクタです（XY 値指定）。
    RVec2(float inX, float inY)
    : x(inX),
      y(inY)
    {
    }

    //! コンストラクタです（3 次元ベクトルから生成）。
    explicit RVec2(const RVec3& v3);

    //! コンストラクタです（4 次元ベクトルから生成）。
    explicit RVec2(const RVec4& v4);

    bool IsInt() const { return false; }

    bool IsFloat() const { return true; }

    int ComponentCount() const { return 2; }

    float& operator[](int iXy)
    {
        if (iXy == 0) return x;
        else          return y;
    }

    float operator[](int iXy) const
    {
        if (iXy == 0) return x;
        else          return y;
    }

    float operator*(const RVec2& right) const
    {
        return x * right.x + y * right.y;
    }

    float operator^(const RVec2& right) const
    {
        return x * right.y - y * right.x;
    }

    RVec2 operator+(const RVec2& other) const
    {
        return RVec2(x + other.x, y + other.y);
    }

    RVec2& operator+=(const RVec2& other)
    {
        x += other.x;
        y += other.y;
        return *this;
    }

    RVec2 operator-() const
    {
        return RVec2(-x, -y);
    }

    RVec2 operator-(const RVec2& other) const
    {
        return RVec2(x - other.x, y - other.y);
    }

    RVec2& operator-=(const RVec2& other)
    {
        x -= other.x;
        y -= other.y;
        return *this;
    }

    RVec2 operator*(float scalar) const
    {
        return RVec2(x * scalar, y * scalar);
    }

    friend RVec2 operator*(float scalar, const RVec2& right)
    {
        return RVec2(right.x * scalar, right.y * scalar);
    }

    RVec2& operator*=(float scalar)
    {
        x *= scalar;
        y *= scalar;
        return *this;
    }

    RVec2 operator/(float scalar) const
    {
        if (scalar != 0.0f)
        {
            float mul = 1.0f / scalar;
            return RVec2(x * mul, y * mul);
        }
        else
        {
            return *this;
        }
    }

    RVec2& operator/=(float scalar)
    {
        if (scalar != 0.0f)
        {
            float mul = 1.0f / scalar;
            x *= mul;
            y *= mul;
        }
        return *this;
    }

    float Length() const
    {
        return static_cast<float>(sqrt(x * x + y * y));
    }

    float LengthSquare() const
    {
        return x * x + y * y;
    }

    RVec2 Normal() const
    {
        const float len = Length();
        return (len != 0.0f) ? *this / len : *this;
    }

    void Normalize()
    {
        const float len = Length();
        if (len != 0.0f)
        {
            *this /= len;
        }
    }

    bool IsAllPositive(const float tolerance) const
    {
        return (x >= -tolerance && y >= -tolerance);
    }

    float MaxAbsValue() const
    {
        const float ax = RAbs(x);
        const float ay = RAbs(y);
        return (ax > ay) ? ax : ay;
    }

    RVec2& SnapToZero()
    {
        x = RSnapToZero(x);
        y = RSnapToZero(y);
        return *this;
    }

    bool IsEquivalent(const RVec2& other, const float tolerance = R_SAME_TOLERANCE_F) const
    {
        return (
            RIsSame(x, other.x, tolerance) &&
            RIsSame(y, other.y, tolerance));
    }

    //! XY 成分がすべて有限かつ NaN でなければ true を返します。
    bool IsFinite() const
    {
        return (_finite(x) && _finite(y));
    }

    //! 同値であれば true を返します。
    bool operator==(const RVec2& rhs) const
    {
        return (x == rhs.x && y == rhs.y);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RVec2& rhs) const
    {
        return (x != rhs.x || y != rhs.y);
    }

    friend std::ostream& operator<<(std::ostream& os, const RVec2& v)
    {
        os << v.x << ' ' << v.y;
        return os;
    }

    void OutSome(std::ostream& os, const int compCount) const
    {
        if (compCount == 2)
        {
            os << *this;
        }
        else
        {
            os << x;
        }
    }

    static const RVec2 kZero;
    static const RVec2 kOne;
};

//! @brief float 型 2 次元ベクトル配列の定義です。
typedef std::vector<RVec2> RVec2Array;

//! @brief float 型 2 次元ベクトルのポインタ配列の定義です。
typedef std::vector<RVec2*> RVec2PtrArray;

//=============================================================================
//! @brief float 型 3 次元ベクトルのクラスです。
//=============================================================================
class RMtx44;

class RVec3 // RVec3D
{
public:
    float x; //!< X 成分です。
    float y; //!< Y 成分です。
    float z; //!< Z 成分です。

public:
    //! コンストラクタです（引数なし）。ゼロベクトルで初期化されます。
    RVec3()
    : x(0.0f),
      y(0.0f),
      z(0.0f)
    {
    }

    //! コンストラクタです（XYZ 値指定）。
    RVec3(const float inX, const float inY, const float inZ)
    : x(inX),
      y(inY),
      z(inZ)
    {
    }

    //! コンストラクタです（2 次元ベクトルから生成）。
    RVec3(const RVec2& v2, const float inZ = 0.0f)
    : x(v2.x),
      y(v2.y),
      z(inZ)
    {
    }

    //! コンストラクタです（4 次元ベクトルから生成）。
    explicit RVec3(const RVec4& v4);

    bool IsInt() const { return false; }

    bool IsFloat() const { return true; }

    int ComponentCount() const { return 3; }

    float& operator[](int iXyz)
    {
        if      (iXyz == 0) return x;
        else if (iXyz == 1) return y;
        else                return z;
    }

    float operator[](int iXyz) const
    {
        if      (iXyz == 0) return x;
        else if (iXyz == 1) return y;
        else                return z;
    }

    float operator*(const RVec3& right) const
    {
        return x * right.x + y * right.y + z * right.z;
    }

    RVec3 operator^(const RVec3& right) const
    {
        return RVec3(
            y * right.z - z * right.y,
            z * right.x - x * right.z,
            x * right.y - y * right.x);
    }

    RVec3 operator+(const RVec3& other) const
    {
        return RVec3(x + other.x, y + other.y, z + other.z);
    }

    RVec3& operator+=(const RVec3& other)
    {
        x += other.x;
        y += other.y;
        z += other.z;
        return *this;
    }

    RVec3 operator-() const
    {
        return RVec3(-x, -y, -z);
    }

    RVec3 operator-(const RVec3& other) const
    {
        return RVec3(x - other.x, y - other.y, z - other.z);
    }

    RVec3& operator-=(const RVec3& other)
    {
        x -= other.x;
        y -= other.y;
        z -= other.z;
        return *this;
    }

    RVec3 operator*(float scalar) const
    {
        return RVec3(x * scalar, y * scalar, z * scalar);
    }

    friend RVec3 operator*(float scalar, const RVec3& right)
    {
        return RVec3(right.x * scalar, right.y * scalar, right.z * scalar);
    }

    RVec3& operator*=(float scalar)
    {
        x *= scalar;
        y *= scalar;
        z *= scalar;
        return *this;
    }

    RVec3 operator/(float scalar) const
    {
        if (scalar != 0.0f)
        {
            float mul = 1.0f / scalar;
            return RVec3(x * mul, y * mul, z * mul);
        }
        else
        {
            return *this;
        }
    }

    RVec3& operator/=(float scalar)
    {
        if (scalar != 0.0f)
        {
            float mul = 1.0f / scalar;
            x *= mul;
            y *= mul;
            z *= mul;
        }
        return *this;
    }

    //! 4 x 4 行列を乗算したベクトルを返します。行列の移動成分は適用しません。
    RVec3 operator*(const RMtx44& m) const;

    //! 4 x 4 行列を乗算します。行列の移動成分は適用しません。
    RVec3& operator*=(const RMtx44& m);

    float Length() const
    {
        return static_cast<float>(sqrt(x * x + y * y + z * z));
    }

    float LengthSquare() const
    {
        return x * x + y * y + z * z;
    }

    RVec3 Normal() const
    {
        const float len = Length();
        return (len != 0.0f) ? *this / len : *this;
    }

    void Normalize()
    {
        const float len = Length();
        if (len != 0.0f)
        {
            *this /= len;
        }
    }

    bool IsAllPositive(const float tolerance) const
    {
        return (x >= -tolerance && y >= -tolerance && z >= -tolerance);
    }

    float MaxAbsValue() const
    {
        const float ax = RAbs(x);
        const float ay = RAbs(y);
        const float az = RAbs(z);
        if (ax > ay)
        {
            return (ax > az) ? ax : az;
        }
        else
        {
            return (ay > az) ? ay : az;
        }
    }

    RVec3& SnapToZero()
    {
        x = RSnapToZero(x);
        y = RSnapToZero(y);
        z = RSnapToZero(z);
        return *this;
    }

    RVec3& ShiftAngle()
    {
        x = RShiftAngle(x);
        y = RShiftAngle(y);
        z = RShiftAngle(z);
        return *this;
    }

    //! @brief 指定した値に近くて回転行列が同じになるオイラー回転を取得します。
    //!        回転順序が XYZ の場合に対応しています。角度の単位は degree です。
    //!
    //! @param[in] dst 近づけるオイラー回転です。
    //!
    //! @return 指定した値に近いオイラー回転を返します。
    //!
    RVec3 GetClosestRotationXyz(const RVec3& dst) const;

    //! @brief 指定した値に近くて回転行列が同じになるオイラー回転を取得します。
    //!        回転順序が ZXY の場合に対応しています。角度の単位は degree です。
    //!
    //! @param[in] dst 近づけるオイラー回転です。
    //!
    //! @return 指定した値に近いオイラー回転を返します。
    //!
    RVec3 GetClosestRotationZxy(const RVec3& dst) const;

    bool IsEquivalent(const RVec3& other, const float tolerance = R_SAME_TOLERANCE_F) const
    {
        return (
            RIsSame(x, other.x, tolerance) &&
            RIsSame(y, other.y, tolerance) &&
            RIsSame(z, other.z, tolerance));
    }

    //! XYZ 成分がすべて有限かつ NaN でなければ true を返します。
    bool IsFinite() const
    {
        return (_finite(x) && _finite(y) && _finite(z));
    }

    //! 同値であれば true を返します。
    bool operator==(const RVec3& rhs) const
    {
        return (x == rhs.x && y == rhs.y && z == rhs.z);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RVec3& rhs) const
    {
        return (x != rhs.x || y != rhs.y || z != rhs.z);
    }

    friend std::ostream& operator<<(std::ostream& os, const RVec3& v)
    {
        os << v.x << ' ' << v.y << ' ' << v.z;
        return os;
    }

    void OutSome(std::ostream& os, const int compCount) const
    {
        if (compCount == 3)
        {
            os << *this;
        }
        else if (compCount == 2)
        {
            os << x << ' ' << y;
        }
        else
        {
            os << x;
        }
    }

    static const RVec3 kZero;
    static const RVec3 kOne;
    static const RVec3 kXAxis;
    static const RVec3 kYAxis;
    static const RVec3 kZAxis;
    static const RVec3 kXNegAxis;
    static const RVec3 kYNegAxis;
    static const RVec3 kZNegAxis;
};

//! @brief float 型 3 次元ベクトル配列の定義です。
typedef std::vector<RVec3> RVec3Array;

//! @brief float 型 3 次元ベクトルのポインタ配列の定義です。
typedef std::vector<RVec3*> RVec3PtrArray;

//=============================================================================
//! @brief float 型 4 次元ベクトルのクラスです。
//=============================================================================
class RVec4 // RVec4D
{
public:
    float x; //!< X 成分です。
    float y; //!< Y 成分です。
    float z; //!< Z 成分です。
    float w; //!< W 成分です。

public:
    //! コンストラクタです（引数なし）。(0, 0, 0, 1) で初期化されます。
    RVec4()
    : x(0.0f),
      y(0.0f),
      z(0.0f),
      w(1.0f)
    {
    }

    //! コンストラクタです（XYZW 値指定）。
    RVec4(const float inX, const float inY, const float inZ, const float inW)
    : x(inX),
      y(inY),
      z(inZ),
      w(inW)
    {
    }

    //! コンストラクタです（2 次元ベクトルから生成）。
    RVec4(const RVec2& v2, const float inZ = 0.0f, const float inW = 1.0f)
    : x(v2.x),
      y(v2.y),
      z(inZ),
      w(inW)
    {
    }

    //! コンストラクタです（3 次元ベクトルから生成）。
    RVec4(const RVec3& v3, const float inW = 1.0f)
    : x(v3.x),
      y(v3.y),
      z(v3.z),
      w(inW)
    {
    }

    bool IsInt() const { return false; }

    bool IsFloat() const { return true; }

    int ComponentCount() const { return 4; }

    float& operator[](int iXyzw)
    {
        if      (iXyzw == 0) return x;
        else if (iXyzw == 1) return y;
        else if (iXyzw == 2) return z;
        else                 return w;
    }

    float operator[](int iXyzw) const
    {
        if      (iXyzw == 0) return x;
        else if (iXyzw == 1) return y;
        else if (iXyzw == 2) return z;
        else                 return w;
    }

    float operator*(const RVec4& right) const
    {
        return x * right.x + y * right.y + z * right.z + w * right.w;
    }

    RVec4 operator+(const RVec4& other) const
    {
        return RVec4(x + other.x, y + other.y, z + other.z, w + other.w);
    }

    RVec4& operator+=(const RVec4& other)
    {
        x += other.x;
        y += other.y;
        z += other.z;
        w += other.w;
        return *this;
    }

    RVec4 operator-() const
    {
        return RVec4(-x, -y, -z, -w);
    }

    RVec4 operator-(const RVec4& other) const
    {
        return RVec4(x - other.x, y - other.y, z - other.z, w - other.w);
    }

    RVec4& operator-=(const RVec4& other)
    {
        x -= other.x;
        y -= other.y;
        z -= other.z;
        w -= other.w;
        return *this;
    }

    RVec4 operator*(float scalar) const
    {
        return RVec4(x * scalar, y * scalar, z * scalar, w * scalar);
    }

    friend RVec4 operator*(float scalar, const RVec4& right)
    {
        return RVec4(right.x * scalar, right.y * scalar, right.z * scalar, right.w * scalar);
    }

    RVec4& operator*=(float scalar)
    {
        x *= scalar;
        y *= scalar;
        z *= scalar;
        w *= scalar;
        return *this;
    }

    RVec4 operator/(float scalar) const
    {
        if (scalar != 0.0f)
        {
            float mul = 1.0f / scalar;
            return RVec4(x * mul, y * mul, z * mul, w * mul);
        }
        else
        {
            return *this;
        }
    }

    RVec4& operator/=(float scalar)
    {
        if (scalar != 0.0f)
        {
            float mul = 1.0f / scalar;
            x *= mul;
            y *= mul;
            z *= mul;
            w *= mul;
        }
        return *this;
    }

    //! 4 x 4 行列を乗算したベクトルを返します。
    RVec4 operator*(const RMtx44& m) const;

    //! 4 x 4 行列を乗算します。
    RVec4& operator*=(const RMtx44& m);

    float Length() const
    {
        return static_cast<float>(sqrt(x * x + y * y + z * z + w * w));
    }

    float LengthSquare() const
    {
        return x * x + y * y + z * z + w * w;
    }

    RVec4 Normal() const
    {
        const float len = Length();
        return (len != 0.0f) ? *this / len : *this;
    }

    void Normalize()
    {
        const float len = Length();
        if (len != 0.0f)
        {
            *this /= len;
        }
    }

    bool IsAllPositive(const float tolerance) const
    {
        return (x >= -tolerance && y >= -tolerance &&
                z >= -tolerance && w >= -tolerance);
    }

    float MaxAbsValue() const
    {
        const float ax = RAbs(x);
        const float ay = RAbs(y);
        const float az = RAbs(z);
        const float aw = RAbs(w);
        float maxVal = (ax > ay) ? ax : ay;
        maxVal = (maxVal > az) ? maxVal : az;
        return (maxVal > aw) ? maxVal : aw;
    }

    RVec4& SnapToZero()
    {
        x = RSnapToZero(x);
        y = RSnapToZero(y);
        z = RSnapToZero(z);
        w = RSnapToZero(w);
        return *this;
    }

    bool IsEquivalent(const RVec4& other, const float tolerance = R_SAME_TOLERANCE_F) const
    {
        return (
            RIsSame(x, other.x, tolerance) &&
            RIsSame(y, other.y, tolerance) &&
            RIsSame(z, other.z, tolerance) &&
            RIsSame(w, other.w, tolerance));
    }

    //! XYZW 成分がすべて有限かつ NaN でなければ true を返します。
    bool IsFinite() const
    {
        return (_finite(x) && _finite(y) && _finite(z) && _finite(w));
    }

    //! 同値であれば true を返します。
    bool operator==(const RVec4& rhs) const
    {
        return (x == rhs.x && y == rhs.y && z == rhs.z && w == rhs.w);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RVec4& rhs) const
    {
        return (x != rhs.x || y != rhs.y || z != rhs.z || w != rhs.w);
    }

    friend std::ostream& operator<<(std::ostream& os, const RVec4& v)
    {
        os << v.x << ' ' << v.y << ' ' << v.z << ' ' << v.w;
        return os;
    }

    void OutSome(std::ostream& os, const int compCount) const
    {
        if (compCount == 4)
        {
            os << *this;
        }
        else if (compCount == 3)
        {
            os << x << ' ' << y << ' ' << z;
        }
        else if (compCount == 2)
        {
            os << x << ' ' << y;
        }
        else
        {
            os << x;
        }
    }

    static const RVec4 kZero;
    static const RVec4 kOne;
    static const RVec4 kBlack;
    static const RVec4 kWhite;
};

//! @brief float 型 4 次元ベクトル配列の定義です。
typedef std::vector<RVec4> RVec4Array;

//! @brief float 型 4 次元ベクトルのポインタ配列の定義です。
typedef std::vector<RVec4*> RVec4PtrArray;

//=============================================================================
//! @brief float 型 4 x 4 行列のクラスです。
//=============================================================================
class RMtx44 // RMtx44D
{
public:
    float m_Value[4][4]; //!< 行列の各成分です。

public:
    //! コンストラクタです。単位行列を作成します。
    RMtx44() { Ident(); }

    //! @brief コンストラクタです。
    //!
    //! @param[in] src 行列の各成分です。
    //!
    explicit RMtx44(const float src[4][4])
    {
        for (int row = 0; row < 4; ++row)
        {
            for (int col = 0; col < 4; ++col)
            {
                m_Value[row][col] = src[row][col];
            }
        }
    }

    //! 行を float 型の配列として返します。
    float* operator[](int row)
    {
        return m_Value[row];
    }

    //! 行を float 型の配列として返します。
    const float* operator[](int row) const
    {
        return m_Value[row];
    }

    //! @brief 他の行列を乗算した行列を返します。
    //!
    //! @param[in] rhs 他の行列です。
    //!
    //! @return 他の行列を乗算した行列を返します。
    //!
    RMtx44 operator*(const RMtx44& rhs) const;

    //! 単位行列にします。
    void Ident()
    {
        for (int row = 0; row < 4; ++row)
        {
            for (int col = 0; col < 4; ++col)
            {
                if (row == col)
                {
                    m_Value[row][col] = 1.0f;
                }
                else
                {
                    m_Value[row][col] = 0.0f;
                }
            }
        }
    }

    //! @brief 逆行列を返します。
    //!
    //! @return 逆行列を返します。
    //!
    RMtx44 Inverse() const;

    //! @brief 0 に近い成分の値を 0 にします。
    //!
    //! @return 自身の参照を返します。
    //!
    RMtx44& SnapToZero()
    {
        for (int row = 0; row < 4; ++row)
        {
            for (int col = 0; col < 4; ++col)
            {
                m_Value[row][col] = RSnapToZero(m_Value[row][col]);
            }
        }
        return *this;
    }

    //! @brief 変換行列（座標を左から掛けるタイプ）を設定します。
    //!
    //! @param[in] scale スケールです。
    //! @param[in] rotate 回転です。単位は度数です。回転順序は XYZ で固定です。
    //! @param[in] translate 移動です。
    //!
    //! @return 自身の参照を返します。
    //!
    RMtx44& SetTransform(
        const RVec3& scale,
        const RVec3& rotate,
        const RVec3& translate
    );

    //! @brief 変換行列（座標を左から掛けるタイプ）を設定します。
    //!
    //! @param[in] scale スケールです。
    //! @param[in] rotate 回転（クォータニオン）です。
    //! @param[in] translate 移動です。
    //!
    //! @return 自身の参照を返します。
    //!
    RMtx44& SetTransformQuat(
        const RVec3& scale,
        const RVec4& rotate,
        const RVec3& translate
    );

    //! @brief 回転を取得します。行列は座標を左から掛けるタイプで、正規化されている必要があります。
    //!
    //! @return 回転を返します。単位は度数です。回転順序は XYZ で固定です。
    //!
    RVec3 GetRotate() const;

    //! @brief 変換を取得します。行列は座標を左から掛けるタイプになっている必要があります。
    //!
    //! @param[out] scale スケールを格納します。
    //! @param[out] rotate 回転を格納します。単位は度数です。回転順序は XYZ で固定です。
    //! @param[out] translate 移動を格納します。
    //!
    void GetTransform(RVec3& scale, RVec3& rotate, RVec3& translate) const;

    //! @brief 他の行列と各成分の値が同等なら true を返します。
    //!
    //! @param[in] other 他の行列です。
    //! @param[in] tolerance 誤差の許容値です。
    //!
    //! @return 他の行列と各成分の値が同等なら true を返します。
    //!
    bool IsEquivalent(const RMtx44& other, const float tolerance = R_SAME_TOLERANCE_F) const
    {
        // compare float value only
        for (int row = 0; row < 4; ++row)
        {
            for (int col = 0; col < 4; ++col)
            {
                if (!RIsSame(m_Value[row][col], other.m_Value[row][col], tolerance))
                {
                    return false;
                }
            }
        }
        return true;
    }

    //! @brief 3 x 4 成分を出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount インデントに必要なタブの数です。
    //!
    void Out34(std::ostream& os, const int tabCount) const;

    //! 同値であれば true を返します。
    bool operator==(const RMtx44& rhs) const
    {
        for (int row = 0; row < 4; ++row)
        {
            for (int col = 0; col < 4; ++col)
            {
                if (m_Value[row][col] != rhs.m_Value[row][col])
                {
                    return false;
                }
            }
        }
        return true;
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RMtx44& rhs) const
    {
        return !(*this == rhs);
    }

    //! @brief 出力ストリームに各成分の値を出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] m 行列です。
    //!
    //! @return 出力ストリームを返します。
    //!
    friend std::ostream& operator<<(std::ostream& os, const RMtx44& m)
    {
        for (int row = 0; row < 4; ++row)
        {
            os << m[row][0] << ' ' << m[row][1] << ' ' << m[row][2] << ' ' << m[row][3] << R_ENDL;
        }
        return os;
    }
};

//=============================================================================
//! @brief int 型 2 次元ベクトルのクラスです。
//=============================================================================
class RIVec3;
class RIVec4;

class RIVec2 // RIVec2D
{
public:
    int x; //!< X 成分です。
    int y; //!< Y 成分です。

public:
    //! コンストラクタです（引数なし）。ゼロベクトルで初期化されます。
    RIVec2()
    : x(0),
      y(0)
    {
    }

    //! コンストラクタです（XY 値指定）。
    RIVec2(int inX, int inY)
    : x(inX),
      y(inY)
    {
    }

    //! コンストラクタです（3 次元ベクトルから生成）。
    explicit RIVec2(const RIVec3& v3);

    //! コンストラクタです（4 次元ベクトルから生成）。
    explicit RIVec2(const RIVec4& v4);

    bool IsInt() const { return true; }

    bool IsFloat() const { return false; }

    int ComponentCount() const { return 2; }

    int& operator[](int iXy)
    {
        if (iXy == 0) return x;
        else          return y;
    }

    int operator[](int iXy) const
    {
        if (iXy == 0) return x;
        else          return y;
    }

    int operator*(const RIVec2& right) const
    {
        return x * right.x + y * right.y;
    }

    int operator^(const RIVec2& right) const
    {
        return x * right.y - y * right.x;
    }

    RIVec2 operator+(const RIVec2& other) const
    {
        return RIVec2(x + other.x, y + other.y);
    }

    RIVec2& operator+=(const RIVec2& other)
    {
        x += other.x;
        y += other.y;
        return *this;
    }

    RIVec2 operator-() const
    {
        return RIVec2(-x, -y);
    }

    RIVec2 operator-(const RIVec2& other) const
    {
        return RIVec2(x - other.x, y - other.y);
    }

    RIVec2& operator-=(const RIVec2& other)
    {
        x -= other.x;
        y -= other.y;
        return *this;
    }

    RIVec2 operator*(int scalar) const
    {
        return RIVec2(x * scalar, y * scalar);
    }

    friend RIVec2 operator*(int scalar, const RIVec2& right)
    {
        return RIVec2(right.x * scalar, right.y * scalar);
    }

    RIVec2& operator*=(int scalar)
    {
        x *= scalar;
        y *= scalar;
        return *this;
    }

    RIVec2 operator/(int scalar) const
    {
        if (scalar != 0)
        {
            return RIVec2(x / scalar, y / scalar);
        }
        else
        {
            return *this;
        }
    }

    RIVec2& operator/=(int scalar)
    {
        if (scalar != 0)
        {
            x /= scalar;
            y /= scalar;
        }
        return *this;
    }

    int Length() const
    {
        return static_cast<int>(sqrt(static_cast<float>(x * x + y * y)));
    }

    int LengthSquare() const
    {
        return x * x + y * y;
    }

    //! 同値であれば true を返します。
    bool operator==(const RIVec2& rhs) const
    {
        return (x == rhs.x && y == rhs.y);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RIVec2& rhs) const
    {
        return (x != rhs.x || y != rhs.y);
    }

    friend std::ostream& operator<<(std::ostream& os, const RIVec2& v)
    {
        os << v.x << ' ' << v.y;
        return os;
    }

    void OutSome(std::ostream& os, const int compCount) const
    {
        if (compCount == 2)
        {
            os << *this;
        }
        else
        {
            os << x;
        }
    }
};

//! @brief int 型 2 次元ベクトル配列の定義です。
typedef std::vector<RIVec2> RIVec2Array;

//! @brief int 型 2 次元ベクトルのポインタ配列の定義です。
typedef std::vector<RIVec2*> RIVec2PtrArray;

//=============================================================================
//! @brief int 型 3 次元ベクトルのクラスです。
//=============================================================================
class RIVec3 // RIVec3D
{
public:
    int x; //!< X 成分です。
    int y; //!< Y 成分です。
    int z; //!< Z 成分です。

public:
    //! コンストラクタです（引数なし）。ゼロベクトルで初期化されます。
    RIVec3()
    : x(0),
      y(0),
      z(0)
    {
    }

    //! コンストラクタです（XYZ 値指定）。
    RIVec3(const int inX, const int inY, const int inZ)
    : x(inX),
      y(inY),
      z(inZ)
    {
    }

    //! コンストラクタです（2 次元ベクトルから生成）。
    RIVec3(const RIVec2& v2, const int inZ = 0)
    : x(v2.x),
      y(v2.y),
      z(inZ)
    {
    }

    //! コンストラクタです（4 次元ベクトルから生成）。
    explicit RIVec3(const RIVec4& v4);

    bool IsInt() const { return true; }

    bool IsFloat() const { return false; }

    int ComponentCount() const { return 3; }

    int& operator[](int iXyz)
    {
        if      (iXyz == 0) return x;
        else if (iXyz == 1) return y;
        else                return z;
    }

    int operator[](int iXyz) const
    {
        if      (iXyz == 0) return x;
        else if (iXyz == 1) return y;
        else                return z;
    }

    int operator*(const RIVec3& right) const
    {
        return x * right.x + y * right.y + z * right.z;
    }

    RIVec3 operator^(const RIVec3& right) const
    {
        return RIVec3(
            y * right.z - z * right.y,
            z * right.x - x * right.z,
            x * right.y - y * right.x);
    }

    RIVec3 operator+(const RIVec3& other) const
    {
        return RIVec3(x + other.x, y + other.y, z + other.z);
    }

    RIVec3& operator+=(const RIVec3& other)
    {
        x += other.x;
        y += other.y;
        z += other.z;
        return *this;
    }

    RIVec3 operator-() const
    {
        return RIVec3(-x, -y, -z);
    }

    RIVec3 operator-(const RIVec3& other) const
    {
        return RIVec3(x - other.x, y - other.y, z - other.z);
    }

    RIVec3& operator-=(const RIVec3& other)
    {
        x -= other.x;
        y -= other.y;
        z -= other.z;
        return *this;
    }

    RIVec3 operator*(int scalar) const
    {
        return RIVec3(x * scalar, y * scalar, z * scalar);
    }

    friend RIVec3 operator*(int scalar, const RIVec3& right)
    {
        return RIVec3(right.x * scalar, right.y * scalar, right.z * scalar);
    }

    RIVec3& operator*=(int scalar)
    {
        x *= scalar;
        y *= scalar;
        z *= scalar;
        return *this;
    }

    RIVec3 operator/(int scalar) const
    {
        if (scalar != 0)
        {
            return RIVec3(x / scalar, y / scalar, z / scalar);
        }
        else
        {
            return *this;
        }
    }

    RIVec3& operator/=(int scalar)
    {
        if (scalar != 0)
        {
            x /= scalar;
            y /= scalar;
            z /= scalar;
        }
        return *this;
    }

    int Length() const
    {
        return static_cast<int>(sqrt(static_cast<float>(x * x + y * y + z * z)));
    }

    int LengthSquare() const
    {
        return x * x + y * y + z * z;
    }

    //! 同値であれば true を返します。
    bool operator==(const RIVec3& rhs) const
    {
        return (x == rhs.x && y == rhs.y && z == rhs.z);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RIVec3& rhs) const
    {
        return (x != rhs.x || y != rhs.y || z != rhs.z);
    }

    friend std::ostream& operator<<(std::ostream& os, const RIVec3& v)
    {
        os << v.x << ' ' << v.y << ' ' << v.z;
        return os;
    }

    void OutSome(std::ostream& os, const int compCount) const
    {
        if (compCount == 3)
        {
            os << *this;
        }
        else if (compCount == 2)
        {
            os << x << ' ' << y;
        }
        else
        {
            os << x;
        }
    }
};

//! @brief int 型 3 次元ベクトル配列の定義です。
typedef std::vector<RIVec3> RIVec3Array;

//! @brief int 型 3 次元ベクトルのポインタ配列の定義です。
typedef std::vector<RIVec3*> RIVec3PtrArray;

//=============================================================================
//! @brief int 型 4 次元ベクトルのクラスです。
//=============================================================================
class RIVec4 // RIVec4D
{
public:
    int x; //!< X 成分です。
    int y; //!< Y 成分です。
    int z; //!< Z 成分です。
    int w; //!< W 成分です。

public:
    //! コンストラクタです（引数なし）。(0, 0, 0, 1) で初期化されます。
    RIVec4()
    : x(0),
      y(0),
      z(0),
      w(1)
    {
    }

    //! コンストラクタです（XYZW 値指定）。
    RIVec4(const int inX, const int inY, const int inZ, const int inW)
    : x(inX),
      y(inY),
      z(inZ),
      w(inW)
    {
    }

    //! コンストラクタです（2 次元ベクトルから生成）。
    RIVec4(const RIVec2& v2, const int inZ = 0, const int inW = 1)
    : x(v2.x),
      y(v2.y),
      z(inZ),
      w(inW)
    {
    }

    //! コンストラクタです（3 次元ベクトルから生成）。
    RIVec4(const RIVec3& v3, const int inW = 1)
    : x(v3.x),
      y(v3.y),
      z(v3.z),
      w(inW)
    {
    }

    bool IsInt() const { return true; }

    bool IsFloat() const { return false; }

    int ComponentCount() const { return 4; }

    int& operator[](int iXyzw)
    {
        if      (iXyzw == 0) return x;
        else if (iXyzw == 1) return y;
        else if (iXyzw == 2) return z;
        else                 return w;
    }

    int operator[](int iXyzw) const
    {
        if      (iXyzw == 0) return x;
        else if (iXyzw == 1) return y;
        else if (iXyzw == 2) return z;
        else                 return w;
    }

    int operator*(const RIVec4& right) const
    {
        return x * right.x + y * right.y + z * right.z + w * right.w;
    }

    RIVec4 operator+(const RIVec4& other) const
    {
        return RIVec4(x + other.x, y + other.y, z + other.z, w + other.w);
    }

    RIVec4& operator+=(const RIVec4& other)
    {
        x += other.x;
        y += other.y;
        z += other.z;
        w += other.w;
        return *this;
    }

    RIVec4 operator-() const
    {
        return RIVec4(-x, -y, -z, -w);
    }

    RIVec4 operator-(const RIVec4& other) const
    {
        return RIVec4(x - other.x, y - other.y, z - other.z, w - other.w);
    }

    RIVec4& operator-=(const RIVec4& other)
    {
        x -= other.x;
        y -= other.y;
        z -= other.z;
        w -= other.w;
        return *this;
    }

    RIVec4 operator*(int scalar) const
    {
        return RIVec4(x * scalar, y * scalar, z * scalar, w * scalar);
    }

    friend RIVec4 operator*(int scalar, const RIVec4& right)
    {
        return RIVec4(right.x * scalar, right.y * scalar, right.z * scalar, right.w * scalar);
    }

    RIVec4& operator*=(int scalar)
    {
        x *= scalar;
        y *= scalar;
        z *= scalar;
        w *= scalar;
        return *this;
    }

    RIVec4 operator/(int scalar) const
    {
        if (scalar != 0)
        {
            return RIVec4(x / scalar, y / scalar, z / scalar, w / scalar);
        }
        else
        {
            return *this;
        }
    }

    RIVec4& operator/=(int scalar)
    {
        if (scalar != 0)
        {
            x /= scalar;
            y /= scalar;
            z /= scalar;
            w /= scalar;
        }
        return *this;
    }

    int Length() const
    {
        return static_cast<int>(sqrt(static_cast<float>(x * x + y * y + z * z + w * w)));
    }

    int LengthSquare() const
    {
        return x * x + y * y + z * z + w * w;
    }

    //! 同値であれば true を返します。
    bool operator==(const RIVec4& rhs) const
    {
        return (x == rhs.x && y == rhs.y && z == rhs.z && w == rhs.w);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RIVec4& rhs) const
    {
        return (x != rhs.x || y != rhs.y || z != rhs.z || w != rhs.w);
    }

    friend std::ostream& operator<<(std::ostream& os, const RIVec4& v)
    {
        os << v.x << ' ' << v.y << ' ' << v.z << ' ' << v.w;
        return os;
    }

    void OutSome(std::ostream& os, const int compCount) const
    {
        if (compCount == 4)
        {
            os << *this;
        }
        else if (compCount == 3)
        {
            os << x << ' ' << y << ' ' << z;
        }
        else if (compCount == 2)
        {
            os << x << ' ' << y;
        }
        else
        {
            os << x;
        }
    }
};

//! @brief int 型 4 次元ベクトル配列の定義です。
typedef std::vector<RIVec4> RIVec4Array;

//! @brief int 型 4 次元ベクトルのポインタ配列の定義です。
typedef std::vector<RIVec4*> RIVec4PtrArray;

//=============================================================================
//! @brief int 型 RGBA カラーのクラスです。
//=============================================================================
class RRgbaColor
{
public:
    int r; //!< R 成分です。
    int g; //!< G 成分です。
    int b; //!< B 成分です。
    int a; //!< A 成分です。

public:
    //! 成分の最大値です。
    enum { COL_MAX = 255 };

    RRgbaColor()
    : r(0),
      g(0),
      b(0),
      a(COL_MAX)
    {
    }

    RRgbaColor(int ar, int ag, int ab, int aa = COL_MAX)
    : r(ar),
      g(ag),
      b(ab),
      a(aa)
    {
    }

    int& operator[](const int irgba)
    {
        if (irgba == 0)
        {
            return r;
        }
        else if (irgba == 1)
        {
            return g;
        }
        else if (irgba == 2)
        {
            return b;
        }
        else
        {
            return a;
        }
    }
    int operator[](const int irgba) const
    {
        if (irgba == 0)
        {
            return r;
        }
        else if (irgba == 1)
        {
            return g;
        }
        else if (irgba == 2)
        {
            return b;
        }
        else
        {
            return a;
        }
    }
    void OutRgb(std::ostream& os) const
    {
        os << r << ' ' << g << ' ' << b;
    }
    void OutRgba(std::ostream& os) const
    {
        os << r << ' ' << g << ' ' << b << ' ' << a;
    }

    //! 同値であれば true を返します。
    bool operator==(const RRgbaColor& rhs) const
    {
        return (r == rhs.r && g == rhs.g && b == rhs.b && a == rhs.a);
    }

    //! 同値でなければ true を返します。
    bool operator!=(const RRgbaColor& rhs) const
    {
        return (r != rhs.r || g != rhs.g || b != rhs.b || a != rhs.a);
    }
    friend std::ostream& operator<<(std::ostream& os, const RRgbaColor& col)
    {
        os << col.r << ' ' << col.g << ' ' << col.b << ' ' << col.a;
        return os;
    }

    static const RRgbaColor kBlack;
    static const RRgbaColor kWhite;
};

//! @brief int 型 RGBA カラー配列の定義です。
typedef std::vector<RRgbaColor> RRgbaColorArray;

//! @brief int 型 RGBA カラーのポインタ配列の定義です。
typedef std::vector<RRgbaColor*> RRgbaColorPtrArray;

//=============================================================================
//! @brief 有向バウンディングボックス（OBB）のクラスです。
//=============================================================================
class ROrientedBB
{
public:
    RVec3 m_CenterPos;
    RVec3 m_AxisVecs[3];
    float m_Sizes[3];

public:
    ROrientedBB() { Init(); }

    void Init()
    {
        m_CenterPos = RVec3::kZero;
        m_AxisVecs[0] = RVec3::kXAxis;
        m_AxisVecs[1] = RVec3::kYAxis;
        m_AxisVecs[2] = RVec3::kZAxis;
        m_Sizes[0] = m_Sizes[1] = m_Sizes[2] = 0.0f;
    }

    void CalculateByPca(const RVec3Array& poss, const bool roundToZero = false);

    void Out(std::ostream& os, const int tabCount) const;
};

//=============================================================================
// ベクトル型の数学関連の関数です。
//=============================================================================

//! @brief 座標の重心を返します。
//!
//! @param[in] poss 座標の配列です。
//!
//! @return 座標の重心を返します。
//!
RVec3 RGetCenterOfGravity(const RVec3Array& poss);

//! @brief 頂点座標と UV から接線と従法線を計算します。
//!        法線がゼロベクトルでなければ、接線と法線および従法線と法線が
//!        それぞれ直交するように補正します（接線と従法線は直交するとは限りません）。
//!
//! @param[out] tangent 接線を格納します。
//! @param[out] binormal 従法線を格納します。
//! @param[in] pos0 対象の頂点の座標です。
//! @param[in] pos1 対象の頂点に隣接する頂点 1 の座標です。
//! @param[in] pos2 対象の頂点に隣接する頂点 2 の座標です。
//! @param[in] uv0 対象の頂点の UV です。
//! @param[in] uv1 対象の頂点に隣接する頂点 1 の UV です。
//! @param[in] uv2 対象の頂点に隣接する頂点 2 の UV です。
//! @param[in] normal 対象の頂点の法線です。
//!
void RCalcTangent(
    RVec3& tangent,
    RVec3& binormal,
    const RVec3& pos0,
    const RVec3& pos1,
    const RVec3& pos2,
    const RVec2& uv0,
    const RVec2& uv1,
    const RVec2& uv2,
    const RVec3& normal
);

//! @brief 接線、従法線、法線が右手座標系（UV のワインド順序が時計回り、前向き）
//!        なら true を返します。
//!
//! @param[in] tangent 接線です。
//! @param[in] binormal 従法線です。
//! @param[in] normal 法線です。
//!
//! @return 接線、従法線、法線が右手座標系なら true を返します。
//!
bool RIsRightHandTangent(const RVec3& tangent, const RVec3& binormal, const RVec3& normal);

//=============================================================================
// ベクトル型の配列関連の関数です。
//=============================================================================

int RFindValueInArray(const RFloatArray& array, const float& value, const float tolerance, const int startIdx = 0);
int RFindValueInArray(const RVec2Array& array, const RVec2& value, const float tolerance, const int startIdx = 0);
int RFindValueInArray(const RVec3Array& array, const RVec3& value, const float tolerance, const int startIdx = 0);
int RFindValueInArray(const RVec4Array& array, const RVec4& value, const float tolerance, const int startIdx = 0);

void RGetArrayRange(float& valueMin, float& valueMax, const RFloatArray& array);
void RGetArrayRange(RVec2& valueMin, RVec2& valueMax, const RVec2Array& array);
void RGetArrayRange(RVec3& valueMin, RVec3& valueMax, const RVec3Array& array);
void RGetArrayRange(RVec4& valueMin, RVec4& valueMax, const RVec4Array& array);

bool RIsAllZeroZ(const RVec3Array& array, const float tolerance = R_SAME_TOLERANCE_F);

bool RIsAllAlphaMax(const RRgbaColorArray& array);

//! @brief 配列の値の絶対値の最大値を返します。
template <typename T>
float RGetArrayMaxAbsValue(const std::vector<T>& array)
{
    T valMin, valMax;
    RGetArrayRange(valMin, valMax, array);
    const float abs0 = valMin.MaxAbsValue();
    const float abs1 = valMax.MaxAbsValue();
    return (abs0 > abs1) ? abs0 : abs1;
}

float RGetArrayMaxAbsValue(const RFloatArray& array);

//=============================================================================
//! @brief ファイルバッファのクラスです。
//=============================================================================
class RFileBuf
{
protected:
    std::string m_FilePath; //!< ファイルのパスです。
    uint8_t* m_FileBuf; //!< ファイルバッファです。
    size_t m_FileSize; //!< ファイルサイズです。

public:
    //! @biref コンストラクタです。
    //!        ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    explicit RFileBuf(const std::string& path)
    : m_FileBuf(NULL)
    {
        Read(path);
    }

    //! @biref コンストラクタです。
    //!        ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    explicit RFileBuf(const std::wstring& path)
    : m_FileBuf(NULL)
    {
        Read(RGetShiftJisFromUnicode(path));
    }

    //! デストラクタです。
    virtual ~RFileBuf()
    {
        //std::cerr << "~RFileBuf(): " << m_FilePath << std::endl;
        RFreeAndClearArray(m_FileBuf);
    }

    //! メモリを解放します。
    void Clear()
    {
        RFreeAndClearArray(m_FileBuf);
        m_FilePath.clear();
        m_FileSize = 0;
    }

    //! @biref ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    //! @return リード成功なら true を返します。
    //!
    bool Read(const std::string& path);

    //! リード成功なら true を返します。
    NN_IMPLICIT operator bool() const
    {
        return (m_FileBuf != NULL);
    }

    //! ファイルのパスを返します。
    std::string GetPath() const
    {
        return m_FilePath;
    };

    //! ファイルバッファを返します。
    const uint8_t* GetBuf() const
    {
        return m_FileBuf;
    };

    //! ファイルバッファを返します（非 const 版）。
    uint8_t* GetBuf()
    {
        return m_FileBuf;
    };

    //! ファイルサイズを返します。
    size_t GetSize() const
    {
        return m_FileSize;
    };

    //! 文字コードが UTF-8 なら true を返します。
    bool IsUtf8() const
    {
        return (m_FileSize >= 3 &&
            m_FileBuf[0] == 0xef &&
            m_FileBuf[1] == 0xbb &&
            m_FileBuf[2] == 0xbf);
    }

    //! @brief ファイルバッファの要素の参照を返す添え字演算子です。
    //!
    //! @param[in] i 要素のインデックスです。
    //!
    //! @return 要素の参照を返します。
    //!
    const uint8_t& operator[](size_t i) const
    {
        return m_FileBuf[i];
    }

    //! @brief ファイルバッファの要素の参照を返す添え字演算子です（非 const 版）。
    //!
    //! @param[in] i 要素のインデックスです。
    //!
    //! @return 要素の参照を返します。
    //!
    uint8_t& operator[](size_t i)
    {
        return m_FileBuf[i];
    }

private:
    RFileBuf(const RFileBuf& other);
    RFileBuf& operator=(const RFileBuf& other);
};

//=============================================================================
//! @brief ファイル移動情報のクラスです。
//!        テンポラリフォルダに出力した中間ファイルを出力フォルダに移動するために使用します。
//=============================================================================
class RFileMove
{
public:
    std::string m_Src; //!< 移動元のファイルパスです。
    std::string m_Dst; //!< 移動先のファイルパスです。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] src 移動元のファイルパスです。
    //! @param[in] dst 移動先のファイルパスです。
    //!
    RFileMove(const std::string& src, const std::string& dst)
    : m_Src(src),
      m_Dst(dst)
    {
    }

    //! @brief ファイルを移動します。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus Move() const;
};

//! @brief ファイル移動情報配列の定義です。
typedef std::vector<RFileMove> RFileMoveArray;

//=============================================================================
// @brief データ列のクラスです。
//=============================================================================
class RDataStream
{
public:
    //! 値の型を表す列挙型です。
    enum Type { FLOAT, INT, BYTE, STRING, WSTRING };

    //! バイナリ形式のヘッダサイズです。
    enum { HEADER_SIZE = 0x20 };

    Type m_Type; //!< 値の型です。
    int m_Column; //!< アスキー出力時の列数です。
    RFloatArray m_FloatValues; //!< 浮動小数点数値の配列です。
    RIntArray m_IntValues; //!< 整数値の配列です。
    RUCharArray m_ByteValues; //!< バイト値の配列です。
    std::string m_StringValue; //!< 文字列値（Shift-JIS）です。

public:
    //! @brief 浮動小数点数値用のコンストラクタです。
    //!
    //! @param[in] array 浮動小数点数値の配列です。
    //! @param[in] column アスキー出力時の列数です。
    //!
    RDataStream(const RFloatArray& array, const int column = 4)
    : m_Type(FLOAT),
      m_Column(column)
    {
        m_FloatValues = array;
    }

    //! @brief 整数値用のコンストラクタです。
    //!
    //! @param[in] array 整数値の配列です。
    //! @param[in] column アスキー出力時の列数です。
    //!
    RDataStream(const RIntArray& array, const int column = 4)
    : m_Type(INT),
      m_Column(column)
    {
        m_IntValues = array;
    }

    //! @brief バイト値用のコンストラクタです。
    //!
    //! @param[in] array 配列の先頭の値へのポインタです。
    //! @param[in] bytes 配列のバイト数です。
    //! @param[in] column アスキー出力時の列数です。
    //!
    RDataStream(const uint8_t* array, const int bytes, const int column = 4);

    //! @brief 文字列値のコンストラクタです。
    //!
    //! @param[in] str 文字列値（Shift-JIS）です。
    //! @param[in] isWstring ASCII 文字列として出力するなら false、
    //!                      Unicode 文字列として出力するなら true を指定します。
    //!
    RDataStream(const std::string& str, const bool isWstring = false)
    : m_Column(0)
    {
        m_Type = (isWstring) ? WSTRING : STRING;
        m_StringValue = str;
    }

    //! @brief Unicode 文字列値のコンストラクタです。
    //!
    //! @param[in] wstr Unicode 文字列値です。
    //!
    explicit RDataStream(const std::wstring& wstr)
    : m_Type(WSTRING),
      m_Column(0)
    {
        m_StringValue = RGetShiftJisFromUnicode(wstr);
    }

    //! @brief ベクトル値用のコンストラクタです。テンプレートです。
    //!
    //! @param[in] array ベクトル値の配列です。RVec2/3/4、RIVec2/3/4 型の配列を指定できます。
    //! @param[in] column アスキー出力時の列数です。
    //!                   省略した場合はベクトル値の成分数が設定されます。
    //!
    template <typename T>
    RDataStream(const std::vector<T>& array, const int column = -1)
    {
        const int elemCount = static_cast<int>(array.size());
        if (elemCount > 0)
        {
            m_Type = array[0].IsInt() ? INT : FLOAT;
            const int compCount = array[0].ComponentCount();
            m_Column = (column != -1) ? column : compCount;
            if (m_Type == FLOAT)
            {
                for (int ielem = 0; ielem < elemCount; ++ielem)
                {
                    const T& elem = array[ielem];
                    for (int ic = 0; ic < compCount; ++ic)
                    {
                        m_FloatValues.push_back(static_cast<float>(elem[ic]));
                    }
                }
            }
            else
            {
                for (int ielem = 0; ielem < elemCount; ++ielem)
                {
                    const T& elem = array[ielem];
                    for (int ic = 0; ic < compCount; ++ic)
                    {
                        m_IntValues.push_back(static_cast<int>(elem[ic]));
                    }
                }
            }
        }
        else
        {
            m_Type = INT;
            m_Column = (column != -1) ? column : 4;
        }
    }

    //! バイト値を追加します。
    //!
    //! @param[in] array 配列の先頭の値へのポインタです。
    //! @param[in] bytes 配列のバイト数です。
    //!
    void Append(const uint8_t* array, const int bytes);

    //! バイト値のパディングを追加します。
    //!
    //! @param[in] size パディングのバイト数です。
    //! @param[in] value パディングに使用する値です。
    //!
    void AppendPadding(const int bytes, const char value = '\0')
    {
        if (bytes > 0)
        {
            uint8_t* buf = new uint8_t[bytes];
            memset(buf, value, bytes);
            Append(buf, bytes);
            delete[] buf;
        }
    }

    //! 値の個数を取得します。
    const int GetCount() const
    {
        if      (m_Type == FLOAT) return static_cast<int>(m_FloatValues.size());
        else if (m_Type == INT  ) return static_cast<int>(m_IntValues.size()  );
        else if (m_Type == BYTE ) return static_cast<int>(m_ByteValues.size() );
        else                      return 1; // STRING, WSTRING
    }

    //! バイナリ形式の実データサイズ（バイト数）を取得します。
    const int GetSize() const
    {
        if      (m_Type == FLOAT ) return static_cast<int>(m_FloatValues.size()) * sizeof(float);
        else if (m_Type == INT   ) return static_cast<int>(m_IntValues.size()  ) * sizeof(int);
        else if (m_Type == BYTE  ) return static_cast<int>(m_ByteValues.size() );
        else if (m_Type == STRING) return static_cast<int>(m_StringValue.size()) + 1; // 終端文字を含むので +1 します。
        else // WSTRING
        {
            // 終端文字を含むので +1 します。
            return (static_cast<int>(RGetUnicodeFromShiftJis(m_StringValue).size()) + 1) * sizeof(wchar_t);
        }
    }

    //! ヘッダを含むバイナリ形式のアライメントしたデータサイズ（バイト数）を取得します。
    const int GetBinarySize() const
    {
        return RAlignValue(HEADER_SIZE + GetSize(), R_BINARY_ALIGNMENT);
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <stream> 要素のインデントに必要なタブの数です。
    //! @param[in] index <stream_array> 内でのインデックスです。
    //!
    void Out(std::ostream& os, const int tabCount, const int index) const;

    //! @brief バイナリ形式で出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //!
    void OutBinary(std::ostream& os) const;
};

//! @brief データ列配列の定義です。
typedef std::vector<RDataStream> RDataStreamArray;

//! @brief データ列のポインタ配列の定義です。
typedef std::vector<RDataStream*> RDataStreamPtrArray;

//! @brief データ列配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] dataStreams データ列配列です。
//!
void ROutDataStreams(std::ostream& os, const RDataStreamArray& array);

//! @brief データ列配列をバイナリ形式で出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] dataStreams データ列配列です。
//! @param[in] outZeroPadding バイナリ形式の後半部の前のゼロパディングを出力するなら true を指定します。
//!
void ROutBinaryDataStreams(
    std::ostream& os,
    const RDataStreamArray& dataStreams,
    const bool outZeroPadding
);

//! @brief 中間ファイルの先頭からバイナリデータまでのオフセットを取得します。
//!
//! @param[in] fileBuf 中間ファイルをリードしたメモリのポインタです。
//! @param[in] fileSize 中間ファイルのサイズ（バイト数）です。
//!
//! @return バイナリデータまでのオフセットを返します。
//!         バイナリデータがない場合は fileSize を返します。
//!
int RGetBinaryOffset(const void* fileBuf, const int fileSize);

//=============================================================================
//! @brief ユーザーデータのクラスです。
//=============================================================================
class RUserData
{
public:
    //! 値の型を表す列挙型です。
    enum Type
    {
        INT,        //!< 整数型です。
        FLOAT,      //!< 浮動小数点数型です。
        STRING,     //!< ASCII 文字列型です。
        WSTRING     //!< Unicode 文字列型です。
    };

    std::string m_Name; //!< ユーザーデータ名です。
    Type m_Type; //!< 値の型です。
    RIntArray m_IntValues; //!< 整数値の配列です。
    RFloatArray m_FloatValues; //!< 浮動小数点数値の配列です。
    RStringArray m_StringValues; //!< 文字列値（Shift-JIS）の配列です。

public:
    //! @brief 値の数を返します。
    //!
    int GetValueCount() const
    {
        if (m_Type == INT)
        {
            return static_cast<int>(m_IntValues.size());
        }
        else if (m_Type == FLOAT)
        {
            return static_cast<int>(m_FloatValues.size());
        }
        else // STRING, WSTRING
        {
            return static_cast<int>(m_StringValues.size());
        }
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <user_data> 要素のインデントに必要なタブの数です。
    //! @param[in] dataIdx <user_data_array> 内でのインデックスです。
    //!
    void Out(std::ostream& os, const int tabCount, const int dataIdx) const;

    //! @brief 有効な文字列型ユーザーデータの値なら true を返します。
    //!
    static bool IsValidString(const std::string& str);
};

//! @brief ユーザーデータ配列の定義です。
typedef std::vector<RUserData> RUserDataArray;

//=============================================================================
//! @brief エクスポートオプションのクラスです。
//=============================================================================
class RExpOpt
{
public:
    //! @brief Export 対象を表す列挙型です。
    enum ExportTarget
    {
        EXPORT_TARGET_ALL,      //!< シーン中の全ノードが対象です。
        EXPORT_TARGET_SELECTION //!< 選択したノードが対象です。
    };

    //! @brief フレーム範囲を表す列挙型です。
    enum FrameRange
    {
        FRAME_RANGE_ALL,      //!< アニメーションの全範囲です。
        FRAME_RANGE_PLAYBACK, //!< アニメーションの再生範囲です。
        FRAME_RANGE_SPECIFY   //!< 指定した範囲です。
    };

    //! @brief テクスチャ SRT の計算方式を表す列挙型です。
    enum TexSrtMode
    {
        TEX_SRT_MODE_MAYA,      //!< Maya の計算方式です。
        TEX_SRT_MODE_3DSMAX,    //!< 3dsMax の計算方式です。
        TEX_SRT_MODE_SOFTIMAGE, //!< Softimage の計算方式です。
    };

    //! @brief 中間ファイルタイプを表す列挙型です。
    enum FileType
    {
        FMD, //!< モデルです。
        FSK, //!< スケルタルアニメーションです。
        FVB, //!< ボーンビジビリティアニメーションです。
        FCL, //!< カラーアニメーションです。
        FTS, //!< テクスチャ SRT アニメーションです。
        FTP, //!< テクスチャパターンアニメーションです。
        FSH, //!< シェイプアニメーションです。

        FSN, //!< シーンアニメーションです。

        FTX, //!< テクスチャです。

        FILE_TYPE_COUNT = FTX //!< 中間ファイルのタイプ数です（テクスチャ以外）。
    };

    //! @brief ボーン圧縮モードを表す列挙型です。
    enum CompressBone
    {
        COMPRESS_BONE_NONE,     //!< ボーン圧縮なしです。
        COMPRESS_BONE_CULL,     //!< 階層構造の末端の不要なボーンを削除します。
        COMPRESS_BONE_MERGE,    //!< Cull の処理に加えて、行列を合成できるボーンを合成します。
        COMPRESS_BONE_UNITE,    //!< 圧縮不可でないボーンを１つにまとめます。
        COMPRESS_BONE_UNITE_ALL //!< すべてのボーンを１つにまとめます。
    };

    //! @brief プリミティブ最適化モードを表す列挙型です。
    enum OptimizePrimitive
    {
        OPTIMIZE_PRIMITIVE_NONE,        //!< プリミティブ最適化なしです。
        OPTIMIZE_PRIMITIVE_DEFAULT,     //!< 通常の最適化をします。
        OPTIMIZE_PRIMITIVE_FORCE,       //!< 時間を掛けて最適化をします。
        OPTIMIZE_PRIMITIVE_BRUTE_FORCE, //!< より長い時間を掛けて最適化をします。
        OPTIMIZE_PRIMITIVE_FORSYTH,     //!< Forsyth 式の最適化をします。
        OPTIMIZE_PRIMITIVE_INVALID      //!< 無効なプリミティブ最適化モードです。
    };

    //! @brief 三角形分割最適化モードを表す列挙型です。
    enum OptimizeTriangulation
    {
        OPTIMIZE_TRIANGULATION_NONE,    //!< 三角形分割最適化なしです。
        OPTIMIZE_TRIANGULATION_ANGLE,   //!< できるだけ鋭角にならないように最適化をします。
    };

    static const int VERTEX_SKINNING_COUNT_MAX = 8; //!< スキニングのウェイト値を分配可能な最大ノード数の最大値です。

    //-----------------------------------------------------------------------------
    // preset
    std::string m_PresetName; //!< プリセット名です。プリセットなしなら空文字です。
    std::string m_PresetFilePath; //!< プリセットファイルのパスです。プリセットなしなら空文字です。

    //-----------------------------------------------------------------------------
    // output
    bool m_AnimRangeFlag; //!< アニメーションレンジ出力モードなら true です。
    ExportTarget m_Target; //!< Export 対象です。
    bool m_ExportsLod; //!< LOD 出力モードなら true です。
    std::string m_ModelName; //!< モデル名です。シングル出力モードでは m_OutFileName と同じです。
    std::string m_OutFileName; //!< 出力する中間ファイル名です。拡張子は含みません。
    std::string m_OutFolderPath; //!< 中間ファイルを出力するフォルダのパスです。
    std::string m_OrgOutFileName; //!< 出力する中間ファイル名（空欄、@node 展開前）です。拡張子は含みません。
    std::string m_OrgOutFolderPath; //!< 中間ファイルを出力するフォルダのパス（環境変数展開前）です。
    std::string m_SpecialTmpFolderPath; //!< 中間ファイルを一時的に出力するテンポラリフォルダを指定する場合のパスです。

    //-----------------------------------------------------------------------------
    // merge
    bool m_MergeFmdFlag; //!< fmd ファイルをマージするなら true です。
    std::string m_MergeFmdPath; //!< マージする fmd ファイルのパスです。
    bool m_MergesOutputFmd; //!< 出力先の既存の fmd ファイルをマージするなら true です。
    bool m_MergeFtxFlag; //!< ftx ファイルをマージするなら true です。
    bool m_MergeAnimFlag; //!< アニメーション中間ファイルをマージするなら true です。
    std::string m_MergeAnimFolder; //!< マージするアニメーション中間ファイルのフォルダのパスです。
    std::string m_MergeAnimName; //!< マージするアニメーション中間ファイルのファイル名です。拡張子は含みません。
    bool m_MergesOutputAnim; //!< 出力先フォルダにあるアニメーション中間ファイルをマージするなら true です。
    std::string m_OrgMergeFmdPath; //!< マージする fmd ファイルのパス（環境変数展開前）です。
    std::string m_OrgMergeAnimFolder; //!< マージするアニメーション中間ファイルのフォルダのパス（環境変数展開前）です。
    std::string m_OrgMergeAnimName; //!< マージするアニメーション中間ファイルのファイル名（空欄展開前）です。拡張子は含みません。

    //-----------------------------------------------------------------------------
    // general
    double m_Magnify; //!< 移動値や頂点座標に掛ける倍率です。
    TexSrtMode m_TexSrtMode; //!< テクスチャ SRT の計算方式です。
    bool m_RemoveNamespace; //!< ボーン名、マテリアル名などのネームスペース部分を削除して出力するなら true です。
    std::string m_CommentText; //!< 編集用コメント文です。

    //-----------------------------------------------------------------------------
    // animation
    FrameRange m_FrameRange; //!< アニメーションを出力するフレーム範囲です。
    int m_StartFrame; //!< アニメーションを出力する開始フレームです。
    int m_EndFrame; //!< アニメーションを出力する終了フレームです。
    bool m_LoopAnim; //!< ループ再生するデータとして出力するなら true です。

    //! @brief 全アニメーションについてベイクしたデータから
    //!        アニメーションカーブデータを自動作成するなら true です。
    //!        可能な限り DCC ツール上のアニメーションカーブをそのまま出力するなら fale です。
    bool m_BakeAllAnim;

    //! @brief ベイクしたデータからアニメーションカーブデータを
    //!        自動作成する際の、フレームの精度です。
    //!        1 フレームをこの値で分割したサブフレーム単位でベイクします。
    //!           1 = 整数フレーム単位でベイク
    //!           2 = 0.5 フレーム単位でベイク
    //!           5 = 0.2 フレーム単位でベイク
    //!          10 = 0.1 フレーム単位でベイク
    int m_FramePrecision;

    //-----------------------------------------------------------------------------
    // output file selection
    bool m_IsBinaryFormat; //!< バイナリ形式の中間ファイルを出力するなら true です。
    bool m_OutFileFlag[FILE_TYPE_COUNT]; //!< ftx 以外の中間ファイルタイプごとの出力フラグです。
    bool m_OutFtxFlag; //!< ftx ファイルを出力するなら true です。

    //-----------------------------------------------------------------------------
    // optimization
    CompressBone m_CompressBone; //!< ボーン圧縮モードです。
    bool m_UniteChild; //!< 階層構造の末端のボーンを 1 つにまとめるなら true です。
    bool m_CompressMaterial; //!< マテリアルを圧縮するなら true です。
    bool m_CompressShape; //!< シェイプを圧縮するなら true です。
    bool m_DivideSubmesh; //!< サブメッシュ分割するなら true です。
    std::string m_DivideSubmeshMode; //!< サブメッシュ分割モードです。
    bool m_PolyReduction; //!< ポリゴンリダクションするなら true です。
    std::string m_PolyReductionMode; //!< ポリゴンリダクションモードです。
    std::string m_OptimizerExtraOptionFmd; //!< fmd ファイルに対する中間ファイルオプティマイザーの追加オプションです。

    //-----------------------------------------------------------------------------
    // model

    //-----------------------------------------------------------------------------
    // tolerance
    float m_TolS; //!< fsk ファイルのスケールについての誤差の許容値です。
    float m_TolR; //!< fsk ファイルの回転についての誤差の許容値です。
    float m_TolT; //!< fsk ファイルの移動についての誤差の許容値です。
    float m_TolC; //!< カラーアニメーションのカラーについての誤差の許容値です。
    float m_TolTexS; //!< fts ファイルのスケールについての誤差の許容値です。
    float m_TolTexR; //!< fts ファイルの回転についての誤差の許容値です。
    float m_TolTexT; //!< fts ファイルの移動についての誤差の許容値です。

    float m_QuantTolS; //!< fsk ファイルを量子化する際のスケールについての誤差の許容値です。
    float m_QuantTolR; //!< fsk ファイルを量子化する際の回転についての誤差の許容値です。
    float m_QuantTolT; //!< fsk ファイルを量子化する際の移動についての誤差の許容値です。
    float m_QuantTolTexS; //!< fts ファイルを量子化する際のスケールについての誤差の許容値です。
    float m_QuantTolTexR; //!< fts ファイルを量子化する際の回転についての誤差の許容値です。
    float m_QuantTolTexT; //!< fts ファイルを量子化する際の移動についての誤差の許容値です。

    //-----------------------------------------------------------------------------
    // script
    std::string m_PreExpScript; //!< Export 前に実行するスクリプトです。
    std::string m_PostExpScript; //!< Export 後に実行するスクリプトです。

    //-----------------------------------------------------------------------------
    // コンフィグファイルで指定するオプション
    bool m_DisplaysProfile; //!< 各処理の処理時間を表示するなら true です。
    std::string m_ProjectRootPath; //!< アプリケーションのプロジェクトのルートフォルダのパスです。
    int m_MaxVertexSkinningCount; //!< スキニングのウェイト値を分配可能な最大ノード数です。
    bool m_AdjustsSmoothSkinning; //!< DCC ツール上のノード単位でスムーススキニングか判定するなら true、シェイプ単位で判定するなら false です。
    bool m_CompressesIgnoringVertexSkinningCount; //!< 1 頂点に影響するスキニングのノード数が異なるシェイプを圧縮するなら true です。
    OptimizePrimitive m_OptimizePrimitive; //!< プリミティブ最適化モードです。
    OptimizeTriangulation m_OptimizeTriangulation; //!< 三角形分割最適化モードです（ユーザーには非公開）。
    bool m_QuantizationAnalysis; //!< 量子化分析するなら true です。
    std::string m_DeleteNearVertex; //!< 近似頂点削除の条件です。空文字なら削除しません。

    //! @brief テクスチャカスタマイズの構造体です。
    class RTexCustom
    {
    public:
        std::string m_Hint; //!< ヒント情報です。
        int m_LinearFlag; //!< リニア変換フラグです。
    public:
        //! コンストラクタです。
        RTexCustom(const std::string& hint, const int linearFlag)
        : m_Hint(hint),
          m_LinearFlag(linearFlag)
        {
        }
    };
    std::vector<RTexCustom> m_TexCustoms; //!< テクスチャカスタマイズ配列です。

    bool m_UsesSrgbFetch; //!< sRGB フォーマットを使用するなら true です。
    int m_NormalTextureFormat; //!< 法線マップ用テクスチャのフォーマット（RImage::Format）です。
    std::string m_NormalTextureCompSel; //!< 法線マップ用テクスチャの成分選択です。空文字ならフォーマットに応じた成分選択を適用します。
    bool m_EnablesWeightedCompress; //!< テクスチャの重み付け圧縮を有効にするなら true です。
    bool m_IsFilterMipLinear; //!< ミップマップフィルタが linear なら true、point なら false です。
    bool m_SamplerMergePriority; //!< DCC ツール側のサンプラ情報を優先してマージするなら true です。
    bool m_UsesFclFtsFtp; //!< fcl / fts / ftp ファイルを使用するなら true、fma ファイルを使用するなら false です。
    bool m_SeparatesFma; //!< 各種類のマテリアルアニメーションを個別の fma ファイルに出力するなら true です。
    bool m_BoneVisMergePriority; //!< DCC ツール側のボーンビジビリティを優先してマージするなら true です。
    bool m_ExportsBoneVisAll; //!< すべてのノードのボーンビジビリティアニメーションをバイナリファイルに出力するなら true、アニメーションが設定されているノードのみ出力するなら false です。
    bool m_ChecksPresetNoneError; //!< プリセットが None の場合にエラーを発生するなら true です。
    bool m_ChecksWrongFileNameError; //!< 出力する中間ファイル名にサポート外の文字が含まれる場合にエラーを発生するなら true です。
    bool m_ChecksNoTextureFileErrorAlways; //!< ftx ファイルを出力しない場合でもテクスチャファイルがなければエラーを発生するなら true です。
    bool m_WarnsNodeNameChanged; //!< ノード名を変更して出力した場合の警告を表示するなら true です。
    bool m_UsesTmpFolderForScript; //!< テンポラリフォルダに出力した状態で Post-Export Script を実行するなら true です。
    std::string m_UserCustomUiScript; //!< エクスポートプラグインにユーザーカスタム UI を追加するスクリプトです。
    std::string m_MultiExportUserCustomUiScript; //!< マルチエクスポートプラグインにユーザーカスタム UI を追加するスクリプトです。

    //-----------------------------------------------------------------------------
    // internal
    bool m_IsMultiExport; //!< マルチエクスポートプラグインで出力中なら true です。
    bool m_IsCancelledByScript; //!< スクリプトから出力をキャンセルされたなら true です。
    bool m_IsScriptOnly; //!< スクリプトの実行のみで出力はしないなら true です。
    std::string m_TexFolderPath; //!< ftx ファイルを出力するフォルダのパスです。
    std::string m_MergeFtxFolderPath; //!< マージする ftx ファイルの存在するフォルダのパスです。

    //! @brief アニメーションのフレーム数です。
    //!        m_EndFrame - m_StartFrame + 1 が格納されます。
    int m_FrameCount;

    //! @brief 出力用のアニメーションのフレーム数です。
    //!        m_FrameCount - 1 が格納されます。
    int m_OutFrameCount; //!< 出力用のアニメーションの整数フレーム数です。

    //! @brief アニメーションを出力するサブフレーム数です。
    //!        m_FrameCount * m_FramePrecision - (m_FramePrecision - 1) が格納されます。
    int m_SubFrameCount;

    std::string m_G3dToolPath; //!< G3dTool の exe ファイルが存在するフォルダのパスです。

    //! @brief 各中間ファイルタイプのアニメーションが存在しない場合の
    //!        警告を表示済みなら true です。
    bool m_NoAnimWarnedFlag[FILE_TYPE_COUNT];

    bool m_Arranges3dIfFormat; //!< 中間ファイルフォーマッターで中間ファイルの書式を調整するなら true です。

    //-----------------------------------------------------------------------------
    // static member variable

    //! @brief 選択されたノードを中間ファイル名する場合の特別な中間ファイル名です。
    static const char* SpecialOutFileNameForNode;

    static const char* TexFolderName; //!< テクスチャを出力するフォルダの名前です。

public:
    //! 全中間ファイルタイプの出力フラグを false にします。
    void ClearOutFileFlag()
    {
        for (int fileTypeIdx = 0; fileTypeIdx < FILE_TYPE_COUNT; ++fileTypeIdx)
        {
            m_OutFileFlag[fileTypeIdx] = false;
        }
        m_OutFtxFlag = false;
    }

    //! 最適化オプションを初期化します。
    void InitOptimization()
    {
        m_CompressBone            = COMPRESS_BONE_NONE;
        m_UniteChild              = false;
        m_CompressMaterial        = false;
        m_CompressShape           = false;
        m_DivideSubmesh           = false;
        m_DivideSubmeshMode       = "";
        m_PolyReduction           = false;
        m_PolyReductionMode       = "";
        m_OptimizerExtraOptionFmd = "";

        m_OptimizePrimitive       = OPTIMIZE_PRIMITIVE_FORSYTH;
        m_OptimizeTriangulation   = OPTIMIZE_TRIANGULATION_ANGLE;
        m_QuantizationAnalysis    = true;
        m_DeleteNearVertex        = "";
    }

    //! @brief LOD モデルのいずれかのレベルを出力中なら true を返します。
    bool ExportsLodLevel() const
    {
        // LOD 出力で初回は Pre-Export Script の実行と情報取得のためだけに
        // エクスポートコマンドを実行するので false を返します。
        return (m_ExportsLod && !m_IsScriptOnly);
    }

    //! @brief LOD モデルのベースレベルを出力中なら true を返します。
    bool ExportsLodBaseLevel() const
    {
        return (m_ExportsLod && m_PostExpScript == "nnLodExport_BasePostExport");
    }

    //! モデル中間ファイル（fmd ファイル）を出力するなら true を返します。
    bool ExportsModelFile() const { return m_OutFileFlag[FMD]; }

    //! テクスチャ中間ファイル（ftx ファイル）を出力するなら true を返します。
    bool ExportsTexture() const { return m_OutFtxFlag; }

    //! @brief カラーアニメーションを出力するなら true を返します。
    bool ExportsColorAnim() const
    {
        return m_OutFileFlag[FCL];
    }

    //! @brief テクスチャ SRT アニメーションを出力するなら true を返します。
    bool ExportsTexSrtAnim() const
    {
        return m_OutFileFlag[FTS];
    }

    //! @brief テクスチャパターンアニメーションを出力するなら true を返します。
    bool ExportsTexPatAnim() const
    {
        return m_OutFileFlag[FTP];
    }

    //! @brief マテリアル関連のアニメーション中間ファイル（fma、fcl、fts、ftp ファイル）を出力するなら true を返します。
    bool ExportsMaterialAnim() const
    {
        return (
            ExportsColorAnim()  ||
            ExportsTexSrtAnim() ||
            ExportsTexPatAnim());
    }

    //! @brief 中間ファイルタイプが fcl / fts / ftp のいずれかなら true を返します。
    static bool IsFclFtsFtp(const FileType fileType)
    {
        return (fileType == FCL || fileType == FTS || fileType == FTP);
    }

    //! @brief 全種類のマテリアルアニメーションを 1 つの fma ファイルに格納するなら true を返します。
    bool UsesSingleFma() const
    {
        return (!m_UsesFclFtsFtp && !m_SeparatesFma);
    }

    //! @brief マテリアルアニメーションの種類ごとに個別の fma ファイルに格納するなら true を返します。
    bool UsesSeparatedFma() const
    {
        return (!m_UsesFclFtsFtp && m_SeparatesFma);
    }

    //! @brief fma ファイル（全種類のマテリアルアニメーションを 1 ファイルに格納）を出力するなら true を返します。
    bool ExportsSingleFma() const
    {
        return (UsesSingleFma() && ExportsMaterialAnim());
    }

    //! @brief fma ファイル（マテリアルアニメーションの種類ごとに個別のファイルに格納）を出力するなら true を返します。
    bool ExportsSeparatedFma() const
    {
        return (UsesSeparatedFma() && ExportsMaterialAnim());
    }

    //! シェイプアニメーション中間ファイル（fsh ファイル）を出力するなら true を返します。
    bool ExportsShapeAnim() const { return m_OutFileFlag[FSH]; }

    //! モデル関連のアニメーション中間ファイル（fsk、fvb、fma、fcl、fts、ftp、fsh ファイル）を出力するなら true を返します。
    bool ExportsModelAnim() const
    {
        return (
            m_OutFileFlag[FSK]    ||
            m_OutFileFlag[FVB]    ||
            ExportsMaterialAnim() ||
            ExportsShapeAnim());
    }

    //! シーンアニメーション中間ファイル（fsn ファイル）を出力するなら true を返します。
    bool ExportsSceneAnim() const { return m_OutFileFlag[FSN]; }

    //! モデル関連またはシーン関連のアニメーション中間ファイルを出力するなら true を返します。
    bool ExportsAnim() const { return (ExportsModelAnim() || ExportsSceneAnim()); }

    //! @brief アニメーションのフレーム数を更新します。
    //!        m_FrameCount、m_SubFrameCount、m_OutFrameCount の値を設定します。
    //!        呼ぶ前に m_StartFrame、m_EndFrame、m_FramePrecision をセットしておく必要があります。
    void UpdateFrameCount();

    //! アニメーションが存在しない場合の警告表示済みフラグをクリアします。
    void ClearNoAnimWarnedFlag()
    {
        for (int fileTypeIdx = 0; fileTypeIdx < FILE_TYPE_COUNT; ++fileTypeIdx)
        {
            m_NoAnimWarnedFlag[fileTypeIdx] = false;
        }
    }

    //! 階層構造の末端のボーンを 1 つにまとめるなら true を返します。
    bool UnitesChildBone() const
    {
        return m_UniteChild                              &&
               m_CompressBone != COMPRESS_BONE_UNITE     &&
               m_CompressBone != COMPRESS_BONE_UNITE_ALL;
    }

    //! ボーンを最適化するなら true を返します。
    bool OptimizesBone() const
    {
        return (m_CompressBone != COMPRESS_BONE_NONE || UnitesChildBone());
    }

    //! fmd ファイルを最適化するなら true を返します。
    bool OptimizesFmd() const
    {
        return (ExportsModelFile() &&
            (OptimizesBone()    ||
             m_CompressMaterial ||
             m_CompressShape    ||
             (m_DivideSubmesh && !m_DivideSubmeshMode.empty()) ||
             (m_PolyReduction && !m_PolyReductionMode.empty()) ||
             !m_OptimizerExtraOptionFmd.empty() ||
             (!m_ExportsLod && m_OptimizePrimitive != OPTIMIZE_PRIMITIVE_NONE) ||
             (!m_ExportsLod && m_QuantizationAnalysis                        ) ||
             !m_DeleteNearVertex.empty()));
    }

    //! fsk ファイルを最適化するなら true を返します。
    bool OptimizesFsk() const
    {
        return (m_OutFileFlag[FSK] && (OptimizesBone() || m_QuantizationAnalysis));
    }

    //! fvb ファイルを最適化するなら true を返します。
    bool OptimizesFvb() const
    {
        return (m_OutFileFlag[FVB] && (OptimizesBone() || m_QuantizationAnalysis));
    }

    //! fma ファイルを最適化するなら true を返します。
    bool OptimizesFma() const
    {
        return (ExportsMaterialAnim() && (m_QuantizationAnalysis));
    }

    //! fcl ファイルを最適化するなら true を返します。
    bool OptimizesFcl() const
    {
        return (ExportsColorAnim() && (m_QuantizationAnalysis));
    }

    //! fts ファイルを最適化するなら true を返します。
    bool OptimizesFts() const
    {
        return (ExportsTexSrtAnim() && (m_QuantizationAnalysis));
    }

    //! ftp ファイルを最適化するなら true を返します。
    bool OptimizesFtp() const
    {
        return (ExportsTexPatAnim() && (m_QuantizationAnalysis));
    }

    //! fsh ファイルを最適化するなら true を返します。
    bool OptimizesFsh() const
    {
        return (m_OutFileFlag[FSH] && (m_QuantizationAnalysis));
    }

    //! fsn ファイルを最適化するなら true を返します。
    bool OptimizesFsn() const
    {
        return (m_OutFileFlag[FSN] && (m_QuantizationAnalysis));
    }

    //! いずれかのアニメーション中間ファイルを最適化するなら true を返します。
    bool OptimizesSomeAnim() const
    {
        return (
            OptimizesFsk() ||
            OptimizesFvb() ||
            OptimizesFcl() ||
            OptimizesFts() ||
            OptimizesFtp() ||
            OptimizesFsh() ||
            OptimizesFsn());
    }

    //! いずれかの中間ファイルを最適化するなら true を返します。
    bool OptimizesSome() const
    {
        return (OptimizesFmd() || OptimizesSomeAnim());
    }

    //! fmd ファイルをマージするなら true を返します。
    bool MergesFmd() const { return (ExportsModelFile() && m_MergeFmdFlag); }

    //! ftx ファイルをマージするなら true を返します。
    bool MergesFtx() const { return (ExportsTexture() && m_MergeFtxFlag); }

    //! いずれかのアニメーション中間ファイルをマージするなら true を返します。
    bool MergesAnim() const { return (ExportsAnim() && m_MergeAnimFlag); }

    //! fsk ファイルをマージするなら true を返します。
    bool MergesFsk() const { return (m_OutFileFlag[FSK] && m_MergeAnimFlag); }

    //! fvb ファイルをマージするなら true を返します。
    bool MergesFvb() const { return (m_OutFileFlag[FVB] && m_MergeAnimFlag); }

    //! fma ファイルをマージするなら true を返します。
    bool MergesFma() const { return (ExportsMaterialAnim() && m_MergeAnimFlag); }

    //! fcl ファイルをマージするなら true を返します。
    bool MergesFcl() const { return (ExportsColorAnim() && m_MergeAnimFlag); }

    //! fts ファイルをマージするなら true を返します。
    bool MergesFts() const { return (ExportsTexSrtAnim() && m_MergeAnimFlag); }

    //! ftp ファイルをマージするなら true を返します。
    bool MergesFtp() const { return (ExportsTexPatAnim() && m_MergeAnimFlag); }

    //! fsh ファイルをマージするなら true を返します。
    bool MergesFsh() const { return (m_OutFileFlag[FSH] && m_MergeAnimFlag); }

    //! fsn ファイルをマージするなら true を返します。
    bool MergesFsn() const { return (m_OutFileFlag[FSN] && m_MergeAnimFlag); }

    //! 中間ファイルオプティマイザーを使用するなら true を返します。
    bool Uses3dOptimizer() const
    {
        return (OptimizesSome() || MergesFmd() || MergesFtx() || MergesAnim());
    }

    //! 中間ファイルフォーマッターを使用するなら true を返します。
    bool Uses3dFormatter() const
    {
        return (m_Arranges3dIfFormat && ExportsModelFile() && !m_ExportsLod && !OptimizesFmd() && !MergesFmd());
    }

    //! @brief 中間ファイルがバイナリ形式なら true を返します。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return 中間ファイルがバイナリ形式なら true を返します。
    //!
    bool IsBinaryFormat(const FileType fileType) const;

    //! @brief 中間ファイル名に追加するサフィックスを返します。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return 中間ファイル追加するサフィックスを返します。
    //!
    std::string GetSuffix(const FileType fileType) const;

    //! @brief 中間ファイルの拡張子を返します。
    //!        コンフィグファイルの設定に従って fma と fcl / fts / ftp を切り替えます。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return 中間ファイルの拡張子を返します。
    //!
    std::string GetExtension(const FileType fileType) const;

    //! @brief コンフィグファイルの設定に関係なくマテリアルアニメーションに関しては
    //!        fcl / fts / ftp の形式の中間ファイルの拡張子を返します。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return fcl / fts / ftp の形式の中間ファイルの拡張子を返します。
    //!
    std::string GetNoFmaExtension(const RExpOpt::FileType fileType) const;

    //! @brief 中間ファイル名（フォルダパスなし、拡張子あり）を返します。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return 中間ファイル名を返します。
    //!
    std::string GetFileName(const FileType fileType) const
    {
        return m_OutFileName + GetSuffix(fileType) + "." + GetExtension(fileType);
    }

    //! @brief 中間ファイルのパスを返します。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return 中間ファイルのパスを返します。
    //!
    std::string GetFilePath(const FileType fileType) const
    {
        return m_OutFolderPath + GetFileName(fileType);
    }

    //! @brief マージするアニメーション中間ファイル名（フォルダパスなし、拡張子あり）を返します。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return マージするアニメーション中間ファイル名を返します。
    //!
    std::string GetMergeAnimFileName(const FileType fileType) const
    {
        return m_MergeAnimName + GetSuffix(fileType) + "." + GetExtension(fileType);
    }

    //! @brief マージするアニメーション中間ファイルのパスを返します。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return マージするアニメーション中間ファイルのパスを返します。
    //!
    std::string GetMergeAnimFilePath(const FileType fileType) const
    {
        return m_MergeAnimFolder + GetMergeAnimFileName(fileType);
    }

    //! @brief 中間ファイルのルート要素のバージョンを取得します。
    //!
    //! @return ルート要素のバージョンを返します。
    //!
    static std::string GetRootElementVersion();

    //! @brief 各中間ファイルのタイプ要素の名前を取得します。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return タイプ要素の名前を返します。
    //!
    std::string GetTypeElementName(const FileType fileType) const;

    //! @brief 各中間ファイルのタイプ要素のバージョンを取得します。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return タイプ要素のバージョンを返します。
    //!
    std::string GetTypeElementVersion(const FileType fileType) const;

    //! @brief コンフィグファイルを解析して一部のオプションを設定します。
    //!
    //! @param[in] configPath コンフィグファイルのパスです。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus ParseConfigFile(const std::string& configPath);

    //! @brief テクスチャのリニア変換フラグを取得します。
    //!
    //! @param[in] hint ヒント情報です。
    //!
    //! @return リニア変換フラグを返します。
    //!
    int GetTexLinearFlag(const std::string& hint) const;

protected:
    //! @brief テクスチャカスタマイズをヒント情報から検索します（const 版）。
    //!        見つからなければ NULL を返します。
    const RTexCustom* FindTexCustom(const std::string& hint) const;

    //! @brief テクスチャカスタマイズをヒント情報から検索します。
    //!        見つからなければ NULL を返します。
    RTexCustom* FindTexCustom(const std::string& hint);
};

//=============================================================================
//! @brief シーンデータのクラスです。
//=============================================================================
class RScene
{
protected:
    std::ostream* m_pErr; //!< エラー出力ストリームのポインタです。
    int m_ErrorCount; //!< エラー数です。
    int m_WarningCount; //!< 警告数です。
    bool m_IsJapaneseUi; //!< UI を日本語で表示するなら true です。
    bool m_IsEvaluating; //!< シーン評価中なら true、通常のエクスポート中なら false です。

public:
    //! コンストラクタです。
    RScene()
    : m_pErr(NULL),
      m_ErrorCount(0),
      m_WarningCount(0),
      m_IsJapaneseUi(false),
      m_IsEvaluating(false)
    {
    }

    //! エラー出力ストリームの参照を返します。
    std::ostream& Err() const { return *m_pErr; }

    //! エラー出力ストリームのポインタを返します。
    std::ostream* ErrPtr() const { return m_pErr; }

    //! エラー出力ストリームのポインタを設定します。
    void SetErrPtr(std::ostream* pErr) { m_pErr = pErr; }

    //! エラー数を返します。
    int GetErrorCount() const { return m_ErrorCount; }

    //! エラー数を設定します。
    void SetErrorCount(const int count) { m_ErrorCount = count; }

    //! エラー数に加算します。
    void AddErrorCount(const int add = 1) { m_ErrorCount += add; }

    //! 警告数を返します。
    int GetWarningCount() const { return m_WarningCount; }

    //! 警告数を設定します。
    void SetWarningCount(const int count) { m_WarningCount = count; }

    //! 警告数に加算します。
    void AddWarningCount(const int add = 1) { m_WarningCount += add; }

    //! UI を日本語で表示するなら true を返します。
    bool IsJapaneseUi() const
    {
        return m_IsJapaneseUi;
    }

    //! UI を日本語で表示するかどうかを設定します。
    void SetJapaneseUi(const bool isJapaneseUi)
    {
        m_IsJapaneseUi = isJapaneseUi;
    }

    //! シーン評価中なら true、通常のエクスポート中なら false を返します。
    bool IsEvaluating() const { return m_IsEvaluating; }

    //! シーン評価中かどうかを設定します。
    void SetEvaluating(const bool isEvaluating) { m_IsEvaluating = isEvaluating; }
};

//=============================================================================
//! @brief トランスフォームノードのクラスです。
//=============================================================================
class RTransformNode
{
public:
    //! @brief 名前です。
    //!        DCC ツール上のノード名から使用できない文字やネームスペースを除き、
    //!        他の同種のノードと名前が重複しないように調整した名前です。
    std::string m_Name;

    RVec3 m_Scale; //!< スケールです。
    RVec3 m_Rotate; //!< 回転です。
    RVec3 m_Translate; //!< 移動です。

    bool m_BinarizesScale; //!< スケールアニメーションをバイナリ出力するなら true です。
    bool m_BinarizesRotate; //!< 回転アニメーションをバイナリ出力するなら true です。
    bool m_BinarizesTranslate; //!< 移動アニメーションをバイナリ出力するなら true です。

    RUserDataArray m_UserDatas; //!< ユーザーデータ配列です。

public:
    //! コンストラクタです。
    RTransformNode()
    : m_Scale(RVec3::kOne),
      m_Rotate(RVec3::kZero),
      m_Translate(RVec3::kZero),
      m_BinarizesScale(true),
      m_BinarizesRotate(true),
      m_BinarizesTranslate(true)
    {
    }

    //! @brief SRT を出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount 属性のインデントに必要なタブの数です。
    //!
    void OutSrt(std::ostream& os, const int tabCount) const;
};

//=============================================================================
//! @brief ボーンのクラスです。
//=============================================================================
class RBone : public RTransformNode
{
public:
    //! @brief ビルボードモードを表す列挙型です。
    enum Billboard
    {
        BILLBOARD_NONE,     //!< ビルボードなしです。
        WORLD_VIEWVECTOR,   //!< Z 軸がカメラの視線と並行になります。
        WORLD_VIEWPOINT,    //!< Z 軸がカメラの方向を向きます。
        SCREEN_VIEWVECTOR,  //!< Y 軸がカメラの上方向を向き、Z 軸がカメラの視線と並行になります。
        SCREEN_VIEWPOINT,   //!< Y 軸がカメラの上方向を向き、Z 軸がカメラの方向を向きます。
        YAXIS_VIEWVECTOR,   //!< Y 軸のみ回転する状態で、Z 軸がカメラの視線と並行になります。
        YAXIS_VIEWPOINT,    //!< Y 軸のみ回転する状態で、Z 軸がカメラの方向を向きます。
        BILLBOARD_COUNT     //!< ビルボードモードの総数です。
    };

    //! @brief ラベルのサイドを表す列挙型です。
    enum Side
    {
        SIDE_NONE,      //!< なしです。
        SIDE_CENTER,    //!< 中央です。
        SIDE_LEFT,      //!< 左側です。
        SIDE_RIGHT,     //!< 右側です。
        SIDE_COUNT      //!< ラベルのサイドの総数です。
    };

    //! @brief スケルタルアニメーションするパラメータを表す列挙型です。
    enum Param
    {
        SCALE_X,        //!< スケールの X 成分です。
        SCALE_Y,        //!< スケールの Y 成分です。
        SCALE_Z,        //!< スケールの Z 成分です。
        ROTATE_X,       //!< 回転の X 成分です。
        ROTATE_Y,       //!< 回転の Y 成分です。
        ROTATE_Z,       //!< 回転の Z 成分です。
        TRANSLATE_X,    //!< 移動の X 成分です。
        TRANSLATE_Y,    //!< 移動の Y 成分です。
        TRANSLATE_Z,    //!< 移動の Z 成分です。
        PARAM_COUNT     //!< パラメータの総数です。
    };

    // attr
    bool m_RigidBody; //!< 行列がリジッドボディの描画に使用されるなら true です。
    int m_SmoothMtxIdx; //!< スムーススキンの描画に使用する行列の行列パレットインデックスです。
    int m_RigidMtxIdx; //!< リジッドスキンの描画に使用する行列の行列パレットインデックスです。
    bool m_ScaleCompensate; //!< セグメントスケール補正が有効なら true です。
    Billboard m_Billboard; //!< ビルボード設定です。
    bool m_Visibility; //!< 可視性です。
    bool m_CompressEnable; //!< 最適化で圧縮可能なら true です。
    Side m_Side; //!< ラベルのサイドです。
    std::string m_Type; //!< ラベルのタイプです。
    RMtx44 m_InvModelMtx; //!< バインドポーズの逆行列です。

    // static member variable

    //! @brief Maya 上でルートボーンが複数ある場合にグループ化するルートボーンの名前です。
    static const char* DefaultRootName;

public:
    //! コンストラクタです。
    RBone()
    :
      // attr
      m_RigidBody(false),
      m_SmoothMtxIdx(-1),
      m_RigidMtxIdx(-1),
      m_ScaleCompensate(false),
      m_Billboard(BILLBOARD_NONE),
      m_Visibility(true),
      m_CompressEnable(true),
      m_Side(SIDE_NONE)
    {
    }

    //! @brief 行列関連の属性を出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount 属性のインデントに必要なタブの数です。
    //!
    void OutMtxAttrib(std::ostream& os, const int tabCount) const;

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <bone> 要素のインデントに必要なタブの数です。
    //! @param[in] boneIdx <bone_array> 内でのインデックスです。
    //! @param[in] parentName 親のボーン名です。親がなければ空文字を指定します。
    //!
    void Out(
        std::ostream& os,
        const int tabCount,
        const int boneIdx,
        const std::string& parentName
    ) const;

    //! @brief スケールのパラメータなら true を返します。
    static bool IsScale(const int paramIdx)
    {
        return (SCALE_X <= paramIdx && paramIdx <= SCALE_Z);
    }

    //! @brief 回転のパラメータなら true を返します。
    static bool IsRotate(const int paramIdx)
    {
        return (ROTATE_X <= paramIdx && paramIdx <= ROTATE_Z);
    }

    //! @brief 移動のパラメータなら true を返します。
    static bool IsTranslate(const int paramIdx)
    {
        return (TRANSLATE_X <= paramIdx && paramIdx <= TRANSLATE_Z);
    }

    //! @brief スケルタルアニメーションアニメーションするパラメータの名前を返します。
    static const char* GetParamName(const int paramIdx);
};

//! @brief ボーン配列の定義です。
typedef std::vector<RBone> RBoneArray;

//=============================================================================
//! @brief スケルトンのクラスです。
//=============================================================================
class RSkeleton
{
public:
    //! @brief スケール計算モードを表す列挙型です。
    enum ScaleMode
    {
        STANDARD,   //!< 通常の方式でスケールを計算します。
        MAYA,       //!< Maya のセグメントスケール補正を考慮して、スケールを計算します。
        SOFTIMAGE   //!< Softimage のスケール計算方式を考慮して、スケールを計算します。
    };

    //! @brief 回転計算モードを表す列挙型です。
    enum RotateMode
    {
        EULER_XYZ,  //!< オイラー角の回転を XYZ の順に計算します。回転値は度数で指定します。
        QUATERNION  //!< クォータニオンで計算します。
    };

    bool m_ScaleEnable; //!< ボーン行列を計算する際にスケールを計算するなら true です。
    ScaleMode m_ScaleMode; //!< ボーン行列のスケール計算方法です。
    RotateMode m_RotateMode; //!< ボーン行列の回転計算方法です。
    bool m_MotionMirroringEnable; //!< モーションのミラーリングをするなら true です。

public:
    //! コンストラクタです。
    RSkeleton()
    : m_ScaleEnable(true),
      m_ScaleMode(STANDARD),
      m_RotateMode(EULER_XYZ),
      m_MotionMirroringEnable(false)
    {
    }

    //! @brief スケールと回転の計算方法を出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount 属性のインデントに必要なタブの数です。
    //!
    void OutScaleRotateMode(std::ostream& os, const int tabCount) const;

    //! @brief スケルトン情報を出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <skeleton_info> 要素のインデントに必要なタブの数です。
    //!
    void OutInfo(std::ostream& os, const int tabCount) const;
};

//=============================================================================
//! @brief 環境オブジェクトのクラスです。
//=============================================================================
class REnvObj : public RTransformNode
{
public:
    int m_FrameCount; //!< アニメーションのフレーム数です。
    bool m_LoopAnim; //!< アニメーションをループ再生するなら true です。
};

//=============================================================================
//! @brief カメラのクラスです。
//=============================================================================
class RCamera : public REnvObj
{
public:
    //! @brief 回転モードを表す列挙型です。
    enum RotateMode
    {
        ROTATE_AIM,         //!< 注視点と捻りで回転します。
        ROTATE_EULER_ZXY    //!< ZXY 軸回転で回転します。
    };

    //! @brief 投影モードを表す列挙型です。
    enum ProjectionMode
    {
        ORTHO,  //!< 正射影です。
        PERSP   //!< 透視射影です。
    };

    //! @brief アニメーションするパラメータを表す列挙型です。
    enum Param
    {
        POSITION_X,     //!< 位置の X 成分です。
        POSITION_Y,     //!< 位置の Y 成分です。
        POSITION_Z,     //!< 位置の Z 成分です。
        AIM_X,          //!< 注視点の位置の X 成分です。
        AIM_Y,          //!< 注視点の位置の Y 成分です。
        AIM_Z,          //!< 注視点の位置の Z 成分です。
        TWIST,          //!< 捻りです。
        ROTATE_X,       //!< 回転の X 成分です。
        ROTATE_Y,       //!< 回転の Y 成分です。
        ROTATE_Z,       //!< 回転の Z 成分です。
        ASPECT,         //!< 射影のアスペクト比（横 / 縦）です。
        NEAR_CLIP,      //!< ニアクリップ距離です。
        FAR_CLIP,       //!< ファークリップ距離です。
        ORTHO_HEIGHT,   //!< 正射影の高さです。
        PERSP_FOVY,     //!< 透視射影の垂直方向の画角です。
        PARAM_COUNT     //!< パラメータの総数です。
    };

    // attr
    RotateMode m_RotateMode; //!< 回転モードです。
    ProjectionMode m_ProjectionMode; //!< 投影モードです。

    RVec3 m_Aim; //!< 注視点の位置です。
    float m_Twist; //!< 捻りです。
    float m_Aspect; //!< 射影のアスペクト比（幅 / 高さ）です。
    float m_NearClip; //!< ニアクリップ距離です。
    float m_FarClip; //!< ファークリップ距離です。
    float m_OrthoHeight; //!< 正射影の高さです。
    float m_PerspFovy; //!< 透視射影の垂直方向の画角です。

public:
    //! @brief 回転のパラメータなら true を返します。
    static bool IsRotate(const int paramIdx)
    {
        return (
            (ROTATE_X <= paramIdx && paramIdx <= ROTATE_Z) ||
            paramIdx == TWIST ||
            paramIdx == PERSP_FOVY);
    }

    //! @brief 移動のパラメータなら true を返します。
    static bool IsTranslate(const int paramIdx)
    {
        return (
            (POSITION_X <= paramIdx && paramIdx <= POSITION_Z) ||
            (AIM_X      <= paramIdx && paramIdx <= AIM_Z     ) ||
            paramIdx == NEAR_CLIP ||
            paramIdx == FAR_CLIP  ||
            paramIdx == ORTHO_HEIGHT);
    }

    //! @brief アニメーションするパラメータの名前を返します。
    static const char* GetParamName(const int paramIdx);
};

//=============================================================================
//! @brief ライトのクラスです。
//=============================================================================
class RLight : public REnvObj
{
public:
    //! @brief タイプを表す列挙型です。
    enum Type
    {
        AMBIENT,        //!< アンビエントライトです。
        DIRECTIONAL,    //!< ディレクショナルライトです。
        POINT,          //!< ポイントライトです。
        SPOT            //!< スポットライトです。
    };

    //! @brief アニメーションするパラメータを表す列挙型です。
    enum Param
    {
        ENABLE,             //!< 有効状態です。
        POSITION_X,         //!< 位置の X 成分です。
        POSITION_Y,         //!< 位置の Y 成分です。
        POSITION_Z,         //!< 位置の Z 成分です。
        DIRECTION_X,        //!< 方向の X 成分です。
        DIRECTION_Y,        //!< 方向の Y 成分です。
        DIRECTION_Z,        //!< 方向の Z 成分です。
        AIM_X,              //!< 目標の位置の X 成分です。
        AIM_Y,              //!< 目標の位置の Y 成分です。
        AIM_Z,              //!< 目標の位置の Z 成分です。
        DIST_ATTN_START,    //!< 距離減衰の開始距離です。
        DIST_ATTN_END,      //!< 距離減衰の終了距離です。
        ANGLE_ATTN_START,   //!< 角度減衰の開始角度です。
        ANGLE_ATTN_END,     //!< 角度減衰の終了角度です。
        COLOR0_R,           //!< カラー 0 の R 成分です。
        COLOR0_G,           //!< カラー 0 の G 成分です。
        COLOR0_B,           //!< カラー 0 の B 成分です。
        COLOR1_R,           //!< カラー 1 の R 成分です。
        COLOR1_G,           //!< カラー 1 の G 成分です。
        COLOR1_B,           //!< カラー 1 の B 成分です。
        PARAM_COUNT         //!< パラメータの総数です。
    };

    // attr
    Type m_Type; //!< タイプです。
    std::string m_DistAttnFunc; //!< 距離減衰関数です。
    std::string m_AngleAttnFunc; //!< 角度減衰関数です。

    bool m_Enable; //!< 有効状態です。
    RVec3 m_Direction; //!< 方向です。
    RVec3 m_Aim; //!< 目標の位置です。
    bool m_UsesDistAttn; //!< 距離減衰を使用するなら true です。
    float m_DistAttnStart; //!< 距離減衰の開始距離です。
    float m_DistAttnEnd; //!< 距離減衰の終了距離です。
    float m_AngleAttnStart; //!< 角度減衰の開始角度です。
    float m_AngleAttnEnd; //!< 角度減衰の終了角度です。
    RVec3 m_Color0; //!< カラー 0 です。
    RVec3 m_Color1; //!< カラー 1 です。

public:
    //! @brief 回転のパラメータなら true を返します。
    static bool IsRotate(const int paramIdx)
    {
        return (
            paramIdx == ANGLE_ATTN_START ||
            paramIdx == ANGLE_ATTN_END);
    }

    //! @brief 移動のパラメータなら true を返します。
    static bool IsTranslate(const int paramIdx)
    {
        return (
            (POSITION_X <= paramIdx && paramIdx <= POSITION_Z) ||
            (AIM_X      <= paramIdx && paramIdx <= AIM_Z     ) ||
            paramIdx == DIST_ATTN_START ||
            paramIdx == DIST_ATTN_END);
    }

    //! @brief ベクトルのパラメータなら true を返します。
    static bool IsVector(const int paramIdx)
    {
        return (DIRECTION_X <= paramIdx && paramIdx <= DIRECTION_Z);
    }

    //! @brief カラーのパラメータなら true を返します。
    static bool IsColor(const int paramIdx)
    {
        return (COLOR0_R <= paramIdx && paramIdx <= COLOR1_B);
    }

    //! @brief アニメーションするパラメータの名前を返します。
    static const char* GetParamName(const int paramIdx);
};

//=============================================================================
//! @brief フォグのクラスです。
//=============================================================================
class RFog : public REnvObj
{
public:
    //! @brief アニメーションするパラメータを表す列挙型です。
    enum Param
    {
        DIST_ATTN_START,    //!< 距離減衰の開始距離です。
        DIST_ATTN_END,      //!< 距離減衰の終了距離です。
        COLOR_R,            //!< カラーの R 成分です。
        COLOR_G,            //!< カラーの G 成分です。
        COLOR_B,            //!< カラーの B 成分です。
        PARAM_COUNT
    };

    // attr
    std::string m_DistAttnFunc; //!< 距離減衰関数です。

    float m_DistAttnStart; //!< 距離減衰の開始距離です。
    float m_DistAttnEnd; //!< 距離減衰の終了距離です。
    RVec3 m_Color; //!< カラーです。

public:
    //! @brief 移動のパラメータなら true を返します。
    static bool IsTranslate(const int paramIdx)
    {
        return (
            paramIdx == DIST_ATTN_START ||
            paramIdx == DIST_ATTN_END);
    }

    //! @brief カラーのパラメータなら true を返します。
    static bool IsColor(const int paramIdx)
    {
        return (COLOR_R <= paramIdx && paramIdx <= COLOR_B);
    }

    //! @brief アニメーションするパラメータの名前を返します。
    static const char* GetParamName(const int paramIdx);
};

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

