﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinShaderParamAnim.h>
#include <BinMaterial.h>
#include <util/UtilMath.h>
#include <algorithm>


namespace nn {
namespace g3dTool {
namespace {
NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    shader_param, type, ctype,
    BinAnimCurve::BOOL, BinAnimCurve::BOOL, BinAnimCurve::BOOL, BinAnimCurve::BOOL,
    BinAnimCurve::INT, BinAnimCurve::INT, BinAnimCurve::INT, BinAnimCurve::INT,
    BinAnimCurve::INT, BinAnimCurve::INT, BinAnimCurve::INT, BinAnimCurve::INT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT, BinAnimCurve::FLOAT, BinAnimCurve::FLOAT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT, BinAnimCurve::FLOAT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT, BinAnimCurve::FLOAT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT, BinAnimCurve::FLOAT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT
    );
}

void BinShaderParamMatAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_shader_param_mat_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    std::vector<const nw::g3d::tool::g3dif::elem_anim_curve*> curveArray;
    for (auto iter = elem.param_anim_array.cbegin();
        iter != elem.param_anim_array.cend(); ++iter)
    {
        for (auto iterTarget = iter->param_anim_target_array.cbegin();
            iterTarget != iter->param_anim_target_array.cend();
            ++iterTarget)
        {
            if (iterTarget->curve)
            {
                curveArray.push_back(&iterTarget->curve.Get());
            }
        }
    }
    m_CurveArray.resize(curveArray.size());
    SetParentBlockArray(m_CurveArray, this);
    BuildArray(pCtx, m_CurveArray, curveArray);

    int curveIndex = 0;
    int numConstant = 0;
    for (auto iter = elem.param_anim_array.cbegin();
        iter != elem.param_anim_array.cend(); ++iter)
    {
        for (auto iterTarget = iter->param_anim_target_array.cbegin();
            iterTarget != iter->param_anim_target_array.cend();
            ++iterTarget)
        {
            if (iterTarget->curve)
            {
                BinAnimCurve& curve = m_CurveArray[curveIndex];
                curve.SetTargetOffset(sizeof(nn::Bit32) * iterTarget->component_index.value);
                curve.SetType(static_cast<BinAnimCurve::Type>(ToEnum_ctype(iter->type.value)));
                if (iter->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_texsrt
                    && iterTarget->component_index.value == 0)
                {
                    curve.SetType(BinAnimCurve::INT);
                }

                // rotate の場合は無条件で Degree であると仮定する。
                // Quaternion の場合は SetRotateMode で false に設定する。
                if ((iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_2D_COMPONENT_INDEX) &&
                    (iter->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_srt2d))
                {
                    curve.SetDegreeValue(true);
                }
                else if (
                    ((iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_X) ||
                    (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_Y) ||
                    (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_Z)) &&
                    iter->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_srt3d)
                {
                    curve.SetDegreeValue(true);
                }
                else if ((iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_2D_COMPONENT_INDEX_SRT) &&
                         (iter->type.value >= nw::g3d::tool::g3dif::elem_shader_param::type_texsrt))
                {
                    // texsrt と texsrt_ex
                    curve.SetDegreeValue(true);
                }

                ++curveIndex;
            }
            else
            {
                ++numConstant;
            }
        }
    }
    m_NumConstant = numConstant;

    // 文字列の登録。
    pCtx->SetStr( elem.mat_name.value.c_str() );
    for (auto iter = elem.param_anim_array.cbegin(); iter != elem.param_anim_array.cend(); ++iter)
    {
        pCtx->SetStr( iter->id.value.c_str() );
    }
}

void BinShaderParamMatAnim::CalculateSize()
{
    m_Chunk[PARAM_INFO].size = sizeof( nn::g3d::ResShaderParamAnimInfo ) * m_pElem->param_anim_array.size();
    m_Chunk[CONSTANT].size = sizeof(nn::g3d::ResAnimConstantData) * m_NumConstant;
    m_Chunk[ChunkType_Curve].size = sizeof(nn::g3d::ResAnimCurveData) * m_CurveArray.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinShaderParamMatAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Curve].offset;
    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResAnimCurveData);
    }
}

void BinShaderParamMatAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResPerMaterialAnimData& matAnim = *GetPtr<nn::g3d::ResPerMaterialAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));
    matAnim.shaderParamAnimCount = static_cast<uint16_t>(m_pElem->param_anim_array.size());
    matAnim.curveCount = static_cast<uint16_t>(m_CurveArray.size());
    matAnim.constantCount = static_cast<uint16_t>(m_NumConstant);
    matAnim.beginShaderParamCurveIndex	= static_cast<uint16_t>( m_BeginCurve );
    matAnim.beginTexturePatternCurveIndex	= nn::g3d::AnimFlag_NotBound;
    matAnim.visibilityConstantIndex	= nn::g3d::AnimFlag_NotBound;
    matAnim.visibilityCurveIndex	= nn::g3d::AnimFlag_NotBound;

    pCtx->LinkStr( &matAnim.pName, m_pElem->mat_name.value.c_str() );
    pCtx->LinkPtr( &matAnim.pCurveArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Curve].offset));

    nn::g3d::ResShaderParamAnimInfo* pParamAnimInfo = GetPtr<nn::g3d::ResShaderParamAnimInfo>(pCtx, Context::MemBlockType_Main, m_Chunk[PARAM_INFO].offset);
    int paramInfoIndex = 0;
    int curveIndex = 0;
    int constantIndex = 0;
    for (auto iter = m_pElem->param_anim_array.cbegin(); iter != m_pElem->param_anim_array.cend(); ++iter, ++paramInfoIndex)
    {
        nn::g3d::ResShaderParamAnimInfo& paramAnimInfo = pParamAnimInfo[paramInfoIndex];
        pCtx->LinkStr( &paramAnimInfo.pName, iter->id.value.c_str() );
        paramAnimInfo.beginConstant = static_cast<uint16_t>(constantIndex);
        paramAnimInfo.beginCurve =static_cast<uint16_t>(curveIndex);
        paramAnimInfo.subbindIndex = nn::g3d::AnimFlag_NotBound;

        int numFloatCurve = 0;
        int numIntCurve = 0;
        int numConstant = 0;
        for (auto iterTarget = iter->param_anim_target_array.cbegin(); iterTarget != iter->param_anim_target_array.cend(); ++ iterTarget)
        {
            if (iterTarget->curve)
            {
                BinAnimCurve& curve = m_CurveArray[curveIndex];
                if (curve.GetType() == BinAnimCurve::FLOAT)
                {
                    ++numFloatCurve;
                }
                else
                {
                    ++numIntCurve;
                }

                ++curveIndex;
            }
            else
            {
                ++numConstant;
                ++constantIndex;
            }
        }

        paramAnimInfo.floatCurveCount = static_cast<uint16_t>( numFloatCurve );
        paramAnimInfo.intCurveCount = static_cast<uint16_t>( numIntCurve );
        paramAnimInfo.constantCount = static_cast<uint16_t>( numConstant );
    }

    pCtx->LinkPtr( &matAnim.pShaderParamAnimInfoArray,
        GetPtr<nn::g3d::ResShaderParamAnimInfo>(pCtx, Context::MemBlockType_Main, m_Chunk[PARAM_INFO].offset) );

    nn::g3d::ResAnimConstantData* pConstant = GetPtr<nn::g3d::ResAnimConstantData>(pCtx, Context::MemBlockType_Main, m_Chunk[CONSTANT].offset);
    int targetIndex = 0;
    for (auto iter = m_pElem->param_anim_array.cbegin();
        iter != m_pElem->param_anim_array.cend(); ++iter)
    {
        for (auto iterTarget = iter->param_anim_target_array.cbegin();
            iterTarget != iter->param_anim_target_array.cend();
            ++iterTarget)
        {
            if (!iterTarget->curve)
            {
                // 型を判定して CONSTANT に値を入れる。
                if ((iter->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_srt2d) &&
                    (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_2D_COMPONENT_INDEX))
                {
                    // テクスチャ srt の Rotate の場合は Degree を Radian にして格納する。
                    pConstant[targetIndex].fValue = nw::g3d::tool::util::DegToRad(iterTarget->base_value.value);
                }
                else if ((iter->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_srt3d) &&
                    ((iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_X) ||
                    (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_Y) ||
                    (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_Z)))
                {
                    pConstant[targetIndex].fValue = nw::g3d::tool::util::DegToRad(iterTarget->base_value.value);
                }
                else if (iter->type.value >= nw::g3d::tool::g3dif::elem_shader_param::type_texsrt)
                {
                    if (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_2D_COMPONENT_INDEX_SRT)
                    {
                        // テクスチャ srt の Rotate の場合は Degree を Radian にして格納する。
                        pConstant[targetIndex].fValue = nw::g3d::tool::util::DegToRad(iterTarget->base_value.value);
                    }
                    else if (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::MODE_COMPONENT_INDEX)
                    {
                        // テクスチャ srt の Mode の場合は uint32_t に変換して格納する。
                        pConstant[targetIndex].iValue = static_cast<uint32_t>(iterTarget->base_value.value);
                    }
                    else
                    {
                        pConstant[targetIndex].fValue = iterTarget->base_value.value;
                    }
                }
                else if (iter->type.value >= nw::g3d::tool::g3dif::elem_shader_param::type_float)
                {
                    pConstant[targetIndex].fValue = iterTarget->base_value.value;
                }
                else
                {
                    pConstant[targetIndex].iValue = static_cast<int>(iterTarget->base_value.value);
                }

                pConstant[targetIndex].targetOffset = iterTarget->component_index.value * sizeof(Bit32);
                ++targetIndex;
            }
        }
    }
    pCtx->LinkPtr( &matAnim.pConstantArray, pConstant );
}

void BinShaderParamAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_shader_param_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    m_ShaderParamMatAnimArray.resize(elem.shader_param_mat_anim_array.size());
    SetParentBlockArray(m_ShaderParamMatAnimArray, this);
    BuildArray(pCtx, m_ShaderParamMatAnimArray, elem.shader_param_mat_anim_array);

    int32_t numParam = 0;
    for (auto iter = m_ShaderParamMatAnimArray.begin(); iter != m_ShaderParamMatAnimArray.end(); ++iter)
    {
        iter->SetBeginParam(numParam);
        numParam += iter->GetParamCount();
    }
    m_ParamCount = numParam;

    int32_t numCurve = 0;
    for (auto iter = m_ShaderParamMatAnimArray.begin(); iter != m_ShaderParamMatAnimArray.end(); ++iter)
    {
        iter->SetBeginCurve(numCurve);
        numCurve += iter->GetCurveCount();
    }
    m_CurveCount = numCurve;

    m_DicUserData.Build(pCtx, elem.user_data_array.size());
    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr(elem.path.c_str());
    pCtx->SetStr(elem.name.c_str());

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinShaderParamAnim::CalculateSize()
{
    auto numAnim =  m_pElem->shader_param_mat_anim_array.size();
    //m_Chunk[SHADER_PARAM_ANIM].size = sizeof(nn::g3d::ResMaterialAnimData);		// 親から割り当てます。
    m_Chunk[ChunkType_BindIndex].size = nw::g3d::tool::util::Align(sizeof(uint16_t) * numAnim, ALIGNMENT_DEFAULT);
    m_Chunk[ChunkType_ShaderParamMaterialAnim].size = sizeof(nn::g3d::ResPerMaterialAnimData) * numAnim;
    m_Chunk[ChunkType_UserDataData].size = sizeof( nn::gfx::ResUserDataData ) * m_UserDataArray.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinShaderParamAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_ShaderParamMaterialAnim].offset;
    for (auto iter = m_ShaderParamMatAnimArray.begin(); iter != m_ShaderParamMatAnimArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResPerMaterialAnimData);
    }

    offset = GetBlockOffset( Context::MemBlockType_Main ) + m_Chunk[ ChunkType_UserDataData ].offset;
    for (auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::gfx::ResUserDataData);
    }
}

void BinShaderParamAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResMaterialAnimData& shaderParamAnim = *GetPtr<nn::g3d::ResMaterialAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));

    shaderParamAnim.blockHeader.signature.SetPacked( nn::g3d::ResMaterialAnim::Signature );
    pCtx->LinkStr( &shaderParamAnim.pName, m_pElem->name.c_str() );
    pCtx->LinkStr( &shaderParamAnim.pPath, m_pElem->path.c_str() );

    const nw::g3d::tool::g3dif::elem_shader_param_anim_info& info = m_pElem->shader_param_anim_info;
    uint16_t flag = 0;
    if (info.loop.value)
    {
        flag |= nn::g3d::AnimFlag_PlayPolicyLoop;
    }

    shaderParamAnim.flag = flag;
    shaderParamAnim.frameCount = m_pElem->shader_param_anim_info.frame_count.value;
    shaderParamAnim.shaderParamAnimCount = static_cast<uint16_t>( m_ParamCount );
    shaderParamAnim.perMaterialAnimCount = static_cast<uint16_t>(m_ShaderParamMatAnimArray.size());
    shaderParamAnim.curveCount = static_cast<uint16_t>( m_CurveCount );
    shaderParamAnim.bakedSize = 0;

    shaderParamAnim.pBindModel.Set( nullptr );
    uint16_t* pBindIndexArray = GetPtr<uint16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BindIndex].offset);
    pCtx->LinkPtr( &shaderParamAnim.pBindIndexArray, pBindIndexArray);
    int numAnim = static_cast<int>(m_ShaderParamMatAnimArray.size());
    for (int idxAnim = 0; idxAnim < numAnim; ++idxAnim)
    {
        pBindIndexArray[idxAnim] = nn::g3d::AnimFlag_NotBound;
    }
    if (numAnim & 0x1)
    {
        // 奇数個の場合は最後にパディングを埋める。
        pBindIndexArray[numAnim] = 0;
    }

    pCtx->LinkPtr(&shaderParamAnim.pPerMaterialAnimArray,
        GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_ShaderParamMaterialAnim].offset));

    // UserData
    shaderParamAnim.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    m_DicUserData.ConvertData(pCtx, shaderParamAnim.pUserDataDic, m_UserDataArray);

    pCtx->LinkPtr( &shaderParamAnim.pUserDataArray, GetPtr( pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_UserDataData].offset ) );
}

void BinShaderParamAnim::Adjust( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResMaterialAnim* shaderParamAnim = GetPtr<nn::g3d::ResMaterialAnim>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));
    nn::g3d::ResMaterialAnimData& shaderParamAnimData = shaderParamAnim->ToData();
    shaderParamAnimData.pPerMaterialAnimArray.Relocate( pCtx->GetBasePtr() );

    size_t size = 0;
    for (int idxMatAnim = 0, numMatAnim = shaderParamAnim->GetPerMaterialAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        nn::g3d::ResPerMaterialAnim* pMatAnim = shaderParamAnim->GetPerMaterialAnim(idxMatAnim);
        nn::g3d::ResPerMaterialAnimData& resPerMaterialAnimData = pMatAnim->ToData();
        resPerMaterialAnimData.pCurveArray.Relocate( pCtx->GetBasePtr() );

        for (int idxCurve = 0, numCurve = pMatAnim->GetCurveCount();
            idxCurve < numCurve; ++idxCurve)
        {
            nn::g3d::ResAnimCurve* curve = pMatAnim->GetCurve(idxCurve);
            size += curve->CalculateBakedSize();
        }

        resPerMaterialAnimData.pCurveArray.Unrelocate( pCtx->GetBasePtr() );
    }
    shaderParamAnimData.bakedSize = static_cast<uint32_t>(size);
    shaderParamAnimData.pPerMaterialAnimArray.Unrelocate( pCtx->GetBasePtr() );
}

}
}
