﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinModel.h>
#include <util/UtilError.h>

namespace nn {
namespace g3dTool {


class RenderSetShapeSort
{
public:
    bool operator()(const nw::g3d::tool::g3dif::elem_shape* rLeft, const nw::g3d::tool::g3dif::elem_shape* rRight) const
    {
        return rLeft->name.value < rRight->name.value;
    }
};

class RenderSetMatSort
{
public:
    bool operator()(const nw::g3d::tool::g3dif::elem_shape* rLeft, const nw::g3d::tool::g3dif::elem_shape* rRight) const
    {
        return rLeft->shape_info.mat_name.value < rRight->shape_info.mat_name.value;
    }
};

void BinModel::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_model& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    // バイナリ化対象のファイルを収集する。
    const nw::g3d::tool::g3dif::elem_skeleton* skeleton = NULL;
    std::vector<const nw::g3d::tool::g3dif::elem_shape*> shapeArray;
    std::vector<const nw::g3d::tool::g3dif::elem_vertex*> vertexArray;
    std::vector<const nw::g3d::tool::g3dif::elem_material*> materialArray;
    std::vector<const nw::g3d::tool::g3dif::elem_user_data*> userDataArray;

    auto numShape = elem.shape_array.size();
    shapeArray.reserve(numShape);

    auto numVertex = elem.vertex_array.size();
    vertexArray.reserve(numVertex);

    auto numMaterial = elem.material_array.size();
    materialArray.reserve(numMaterial);

    auto numUserData = elem.user_data_array.size();
    userDataArray.reserve(numUserData);


    if (elem.skeleton)
    {
        skeleton = &elem.skeleton.Get();
    }

    for (auto iter = elem.shape_array.cbegin(); iter != elem.shape_array.cend(); ++iter)
    {
        shapeArray.push_back(&*iter);
    }

    for (auto iter = elem.vertex_array.cbegin(); iter != elem.vertex_array.cend(); ++iter)
    {
        vertexArray.push_back(&*iter);
    }

    for (auto iter = elem.material_array.cbegin(); iter != elem.material_array.cend(); ++iter)
    {
        materialArray.push_back(&*iter);
    }

    for (auto iter = elem.user_data_array.cbegin(); iter != elem.user_data_array.cend(); ++iter)
    {
        userDataArray.push_back(&*iter);
    }
    if (skeleton)
    {
        m_Skeleton.Get().SetParentBlock(this);
        m_Skeleton.Get().Build(pCtx, *skeleton);
        m_Skeleton.Validate();
    }

    // 可変長要素の辞書をコンバート対象に追加する。
    m_DicShape.Build(pCtx, numShape);
    m_DicMaterial.Build(pCtx, numMaterial);
    m_DicUserData.Build(pCtx, numUserData);

    m_ShapeArray.resize(shapeArray.size());
    m_VertexArray.resize(vertexArray.size());
    m_UserDataArray.resize(userDataArray.size());
    m_MaterialArray.resize(materialArray.size());

    SetParentBlockArray(m_ShapeArray, this);
    SetParentBlockArray(m_VertexArray, this);
    SetParentBlockArray(m_UserDataArray, this);
    SetParentBlockArray(m_MaterialArray, this);

    BuildArray(pCtx, m_ShapeArray, shapeArray);
    BuildArray(pCtx, m_VertexArray, vertexArray);
    BuildArray(pCtx, m_UserDataArray, userDataArray);
    BuildArray(pCtx, m_MaterialArray, materialArray);

    // 文字列の登録。
    pCtx->SetStr(elem.name.c_str());
    if( elem.path.length() != 0 )	// 空文字の登録を行うとライブラリが持つ辞書のノードがリークする。
    {
        pCtx->SetStr(elem.path.c_str());
    }

    // shape と vertex の相互参照の解決
    int shapeIndex = 0;
    for (auto iter = shapeArray.cbegin(); iter != shapeArray.cend(); ++iter, ++shapeIndex)
    {
        int vertexIndex = (*iter)->shape_info.vertex_index.value;

        if (vertexIndex < 0 || vertexIndex >= static_cast<int>(m_VertexArray.size()))
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_SHAPE_INVALID_VERTEX_INDEX, "Identifier_InvalidVertexIndex", vertexIndex);
        }

        // 複数の shape が同じ vertex を参照している場合は、１つの shape が所有権を持つ
        if (!m_VertexArray[vertexIndex].HasOwner())
        {
            m_ShapeArray[shapeIndex].SetOwnership(true);
            m_VertexArray[vertexIndex].SetOwner(true);
        }

        m_ShapeArray[shapeIndex].SetIndex(shapeIndex);
        m_ShapeArray[shapeIndex].SetVertex(&m_VertexArray[vertexIndex]);
        m_VertexArray[vertexIndex].SetSkinningCount((*iter)->shape_info.vertex_skinning_count.value);
        if (m_Skeleton)
        {
            m_ShapeArray[shapeIndex].SetSkeleton(&(m_Skeleton.Get()));
        }

        m_DicShape.SetName(shapeIndex, (*iter)->name.value);
    }

    int materialIndex = 0;
    for (auto iter = materialArray.cbegin(); iter != materialArray.cend(); ++iter, ++materialIndex)
    {
        m_DicMaterial.SetName(materialIndex, (*iter)->name.value);
    }

    int userDataIndex = 0;
    for (auto iter = m_pElem->user_data_array.cbegin();
        iter != m_pElem->user_data_array.cend(); ++iter, ++userDataIndex)
    {
        m_DicUserData.SetName(userDataIndex, iter->name.value);
    }

    int materialArrayIndex = 0;
    for (auto iter = m_MaterialArray.begin(); iter != m_MaterialArray.end(); ++iter, ++materialArrayIndex)
    {
        iter->SetIndex(materialArrayIndex);
    }
}

void BinModel::CalculateSize()
{
    m_Chunk[ChunkType_Vtx].size = sizeof(nn::g3d::ResVertexData) * m_pElem->vertex_array.size();
    m_Chunk[ ChunkType_MatArray ].size = sizeof( nn::g3d::ResMaterialData ) * m_MaterialArray.size();
    m_Chunk[ ChunkType_ShapeArray ].size = sizeof( nn::g3d::ResShapeData ) * m_ShapeArray.size();
    m_Chunk[ ChunkType_UserDataArray ].size = sizeof( nn::gfx::ResUserDataData ) * m_UserDataArray.size();

    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinModel::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);
    ptrdiff_t vertexOffset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Vtx].offset;
    for (auto iter = m_VertexArray.begin(); iter != m_VertexArray.end(); ++iter)
    {
        iter->SetStructOffset(vertexOffset);
        vertexOffset += sizeof(nn::g3d::ResVertexData);
    }

    // Material Array のオフセット計算
    ptrdiff_t ofsMaterialArray = GetBlockOffset( Context::MemBlockType_Main ) + m_Chunk[ ChunkType_MatArray ].offset;
    for ( auto iter = m_MaterialArray.begin(); iter != m_MaterialArray.end(); ++iter )
    {
        iter->SetStructOffset( ofsMaterialArray );
        ofsMaterialArray += sizeof(nn::g3d::ResMaterialData);
    }

    // Shape Array のオフセット計算
    ptrdiff_t ofsShapeArray = GetBlockOffset( Context::MemBlockType_Main ) + m_Chunk[ ChunkType_ShapeArray ].offset;
    for ( auto iter = m_ShapeArray.begin(); iter != m_ShapeArray.end(); ++iter )
    {
        iter->SetStructOffset( ofsShapeArray );
        ofsShapeArray += sizeof(nn::g3d::ResShapeData);
    }

    // UserData Array のオフセット計算
    ptrdiff_t ofsUserDataArray = GetBlockOffset( Context::MemBlockType_Main ) + m_Chunk[ ChunkType_UserDataArray ].offset;
    for ( auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter )
    {
        iter->SetStructOffset( ofsUserDataArray );
        ofsUserDataArray += sizeof(nn::gfx::ResUserDataData);
    }
}

void BinModel::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResModelData& model = *GetPtr<nn::g3d::ResModelData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    pCtx->AddBinBlockHeader( &model.blockHeader );

    model.blockHeader.signature.SetPacked( nn::g3d::ResModel::Signature );
    pCtx->LinkStr( &model.pName, nn::util::string_view( m_pElem->name.c_str() ) );
    pCtx->LinkStr( &model.pPath, nn::util::string_view( m_pElem->path.c_str() ));
    model.pUserPtr.Set(nullptr);

    // Skeleton
    if (m_pElem->skeleton)
    {
        nn::g3d::ResSkeletonData& skeleton
            = *( m_Skeleton.Get().GetPtr<nn::g3d::ResSkeletonData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) ) );
        pCtx->LinkPtr( &model.pSkeleton, &skeleton );
    }
    else
    {
        model.pSkeleton.Set(nullptr);
    }

    // Vertex
    ConvertVertex(model, pCtx);

    // Shape
    model.shapeCount = static_cast<uint16_t>(m_ShapeArray.size());
    pCtx->LinkPtr( &model.pShapeArray, GetPtr< void >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_ShapeArray ].offset ) );
    m_DicShape.ConvertData(pCtx, model.pShapeDic, m_ShapeArray);

    // Material
    model.materialCount = static_cast< uint16_t >(m_MaterialArray.size());
    void* pMaterialArray = GetPtr< void >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_MatArray ].offset );
    pCtx->LinkPtr( &model.pMaterialArray, pMaterialArray );
    m_DicMaterial.ConvertData(pCtx, model.pMaterialDic, m_MaterialArray);

    // UserData
    model.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    void* pUserDataArray = GetPtr< void >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_UserDataArray ].offset );
    pCtx->LinkPtr( &model.pUserDataArray, pUserDataArray );
    m_DicUserData.ConvertData(pCtx, model.pUserDataDic, m_UserDataArray);
}

void BinModel::ConvertVertex( nn::g3d::ResModelData &model, std::shared_ptr<Context> pCtx )
{
    model.vertexCount = static_cast<uint16_t>(m_VertexArray.size());
    nn::g3d::ResVertexData* pVertex =
        GetPtr<nn::g3d::ResVertexData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Vtx].offset);
    pCtx->LinkPtr( &model.pVertexArray, pVertex );

    int index = 0;
    for (auto iter = m_pElem->vertex_array.cbegin();
        iter != m_pElem->vertex_array.cend(); ++iter, ++pVertex, ++index)
    {
        // index 以外の要素は BinVertex で代入する。
        pVertex->index = static_cast<uint16_t>(index);
    }
}

}
}
