﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinMaterialVisibilityAnim.h>

namespace nn {
namespace g3dTool {

void BinMatVisibilityAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_mat_visibility_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    std::vector<const nw::g3d::tool::g3dif::elem_anim_curve*> curveArray;
    for (auto iter = m_pElem->mat_vis_mat_anim_array.cbegin(); iter != m_pElem->mat_vis_mat_anim_array.cend(); ++iter)
    {
        pCtx->SetStr( iter->mat_name.value.c_str() );

        if (iter->curve)
        {
            curveArray.push_back(&iter->curve.Get());
        }
    }
    m_CurveArray.resize(curveArray.size());
    SetParentBlockArray(m_CurveArray, this);
    BuildArray(pCtx, m_CurveArray, curveArray);

    int animIndex = 0;
    int curveIndex = 0;
    for (auto iter = m_pElem->mat_vis_mat_anim_array.cbegin(); iter != m_pElem->mat_vis_mat_anim_array.cend(); ++iter)
    {
        if (iter->curve)
        {
            BinAnimCurve& curve = m_CurveArray[curveIndex];
            curve.SetTargetOffset(animIndex);
            curve.SetType(BinAnimCurve::BOOL);
            ++curveIndex;
        }
        ++animIndex;
    }

    m_DicUserData.Build(pCtx, elem.user_data_array.size());

    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr( elem.path.c_str() );
    pCtx->SetStr( elem.name.c_str() );
    for (auto iter = m_pElem->mat_vis_mat_anim_array.cbegin(); iter != m_pElem->mat_vis_mat_anim_array.cend(); ++iter )
    {
        pCtx->SetStr( iter->mat_name.value.c_str() );
    }

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinMatVisibilityAnim::CalculateSize()
{
    auto numAnim =  m_pElem->mat_vis_mat_anim_array.size();

    m_Chunk[ChunkType_PerMaterialAnimDataArray].size = sizeof( nn::g3d::ResPerMaterialAnimData ) * numAnim;
    m_Chunk[ChunkType_BindIndex].size = nw::g3d::tool::util::Align(sizeof(uint16_t) * numAnim, ALIGNMENT_DEFAULT);
    m_Chunk[ChunkType_Curve].size = sizeof(nn::g3d::ResAnimCurveData) * m_CurveArray.size();
    m_Chunk[ChunkType_BaseValue].size = numAnim * sizeof( nn::g3d::ResAnimConstantData );
    m_Chunk[ChunkType_UserDataData].size = sizeof( nn::gfx::ResUserDataData ) * m_UserDataArray.size();

    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinMatVisibilityAnim::CalculateOffset(std::shared_ptr<Context> pCtx)
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Curve].offset;
    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResAnimCurveData);
    }

    offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ ChunkType_UserDataData ].offset;
    for( auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter )
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::gfx::ResUserDataData);
    }
}

void BinMatVisibilityAnim::Convert(std::shared_ptr<Context> pCtx)
{
    nn::g3d::ResMaterialAnimData& visibilityAnim = *GetPtr<nn::g3d::ResMaterialAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));

    visibilityAnim.blockHeader.signature.SetPacked( nn::g3d::ResMaterialAnim::Signature );
    pCtx->LinkStr( &visibilityAnim.pName, m_pElem->name.c_str() );
    pCtx->LinkStr( &visibilityAnim.pPath, m_pElem->path.c_str() );

    const nw::g3d::tool::g3dif::elem_mat_visibility_anim_info& info = m_pElem->mat_visibility_anim_info;
    uint16_t flag = 0;
    if (info.loop.value)
    {
        flag |= nn::g3d::AnimFlag_PlayPolicyLoop;
    }

    flag |= nn::g3d::ResMaterial::Flag_Visibility;

    visibilityAnim.flag = flag;
    visibilityAnim.frameCount = info.frame_count.value;
    visibilityAnim.perMaterialAnimCount = static_cast<uint16_t>(m_pElem->mat_vis_mat_anim_array.size());
    visibilityAnim.visibilityAnimCount = static_cast<uint16_t>(m_pElem->mat_vis_mat_anim_array.size());
    visibilityAnim.curveCount = static_cast<uint16_t>(m_CurveArray.size());
    visibilityAnim.bakedSize = 0;
    visibilityAnim.textureCount = 0;

    visibilityAnim.pBindModel.Set(nullptr);
    int numAnim = static_cast<int>(m_pElem->mat_vis_mat_anim_array.size());
    uint16_t* pBindIndexArray = GetPtr<uint16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BindIndex].offset);
    for (int idxAnim = 0; idxAnim < numAnim; ++idxAnim)
    {
        pBindIndexArray[idxAnim] = nn::g3d::AnimFlag::AnimFlag_NotBound;
    }
    if (numAnim & 0x1)
    {
        // 奇数個の場合は最後にパディングを埋める。
        pBindIndexArray[numAnim] = 0;
    }
    pCtx->LinkPtr( &visibilityAnim.pBindIndexArray, GetPtr<uint16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BindIndex].offset) );


    // 名前の配列とベースバリュー
    nn::g3d::ResAnimConstantData* pBaseValue = GetPtr<nn::g3d::ResAnimConstantData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BaseValue].offset);
    memset(pBaseValue, 0, m_Chunk[ChunkType_BaseValue].size);

    nn::g3d::ResPerMaterialAnimData* pResPerMaterialAnimData =
        GetPtr<nn::g3d::ResPerMaterialAnimData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_PerMaterialAnimDataArray].offset);

    uint16_t idxAnim = 0;
    uint16_t idxVisibilityCurve = 0;
    nn::g3d::ResAnimCurveData*	pResAnimCurveData =
        GetPtr<nn::g3d::ResAnimCurveData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Curve].offset);
    for (auto iter = m_pElem->mat_vis_mat_anim_array.cbegin();
        iter != m_pElem->mat_vis_mat_anim_array.cend();
        ++iter, ++idxAnim, ++pResPerMaterialAnimData )
    {
        pResPerMaterialAnimData->curveCount		= 0;
        pResPerMaterialAnimData->constantCount	= 0;
        if( iter->curve )
        {
            pResPerMaterialAnimData->curveCount		= 1;
            pResPerMaterialAnimData->beginVisibilityCurveIndex	= idxVisibilityCurve;
            pResPerMaterialAnimData->visibilityCurveIndex		= 0;
            pResPerMaterialAnimData->visibilityConstantIndex	= nn::g3d::AnimFlag_NotBound;
            pCtx->LinkPtr( &pResPerMaterialAnimData->pCurveArray, pResAnimCurveData );
            ++idxVisibilityCurve;
            ++pResAnimCurveData;
        }
        else    // アニメカーブを持たない場合は定数アニメと判定する。
        {
            pResPerMaterialAnimData->constantCount	= 1;
            pResPerMaterialAnimData->beginVisibilityCurveIndex	= nn::g3d::AnimFlag_NotBound;
            pResPerMaterialAnimData->visibilityCurveIndex		= nn::g3d::AnimFlag_NotBound;
            pResPerMaterialAnimData->visibilityConstantIndex	= 0;
        }

        pResPerMaterialAnimData->shaderParamAnimCount = 0;
        pResPerMaterialAnimData->beginShaderParamCurveIndex		= nn::g3d::AnimFlag_NotBound;
        pResPerMaterialAnimData->beginTexturePatternCurveIndex	= nn::g3d::AnimFlag_NotBound;

        pCtx->LinkStr( &pResPerMaterialAnimData->pName, iter->mat_name.value.c_str() );

        pBaseValue[idxAnim].iValue = static_cast<int32_t>( iter->base_value.value );
        pCtx->LinkPtr( &pResPerMaterialAnimData->pConstantArray, &pBaseValue[idxAnim] );
    }

    pCtx->LinkPtr( &visibilityAnim.pPerMaterialAnimArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_PerMaterialAnimDataArray].offset) );

    // UserData
    visibilityAnim.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    m_DicUserData.ConvertData(pCtx, visibilityAnim.pUserDataDic, m_UserDataArray);
    pCtx->LinkPtr( &visibilityAnim.pUserDataArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_UserDataData].offset) );

}

void BinMatVisibilityAnim::Adjust(std::shared_ptr<Context> pCtx)
{
    nn::g3d::ResMaterialAnim* materialVisibilityAnim = GetPtr<nn::g3d::ResMaterialAnim>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));
    nn::g3d::ResMaterialAnimData& materialVisibilityAnimData = materialVisibilityAnim->ToData();
    materialVisibilityAnimData.pPerMaterialAnimArray.Relocate( pCtx->GetBasePtr() );

    size_t size = 0;
    for (int idxMatAnim = 0, numMatAnim = materialVisibilityAnim->GetPerMaterialAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        nn::g3d::ResPerMaterialAnim* pMatAnim = materialVisibilityAnim->GetPerMaterialAnim(idxMatAnim);
        nn::g3d::ResPerMaterialAnimData& resPerMaterialAnimData = pMatAnim->ToData();
        resPerMaterialAnimData.pCurveArray.Relocate( pCtx->GetBasePtr() );

        for (int idxCurve = 0, numCurve = pMatAnim->GetCurveCount();
            idxCurve < numCurve; ++idxCurve)
        {
            nn::g3d::ResAnimCurve* curve = pMatAnim->GetCurve(idxCurve);
            size += curve->CalculateBakedSize();
        }

        resPerMaterialAnimData.pCurveArray.Unrelocate( pCtx->GetBasePtr() );
    }
    materialVisibilityAnimData.bakedSize = static_cast<uint32_t>(size);
    materialVisibilityAnimData.pPerMaterialAnimArray.Unrelocate( pCtx->GetBasePtr() );
}


} // namespace cvtr
} // namespace tool
