﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <util/UtilMath.h>
#include <BinMaterialAnim.h>
#include <BinMaterial.h>
#include <util/UtilError.h>


namespace nn {
namespace g3dTool {

void BinMaterialAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_nn_material_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    m_PerMaterialAnimArray.resize(elem.per_material_anim_array.size());
    SetParentBlockArray(m_PerMaterialAnimArray, this);
    BuildArray(pCtx, m_PerMaterialAnimArray, elem.per_material_anim_array);

    uint16_t numParam = 0;
    for (auto iter = m_PerMaterialAnimArray.begin(); iter != m_PerMaterialAnimArray.end(); ++iter)
    {
        iter->SetBeginParam(numParam);
        numParam += iter->GetParamCount();
    }
    m_ParamCount = numParam;

    uint16_t numCurve = 0;
    for (auto iter = m_PerMaterialAnimArray.begin(); iter != m_PerMaterialAnimArray.end(); ++iter)
    {
        iter->SetBeginCurve(numCurve);
        numCurve += iter->GetCurveCount();
    }
    m_CurveCount = numCurve;

    m_DicUserData.Build(pCtx, elem.user_data_array.size());

    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr(elem.path.c_str());
    pCtx->SetStr(elem.name.c_str());

    // texture 名の文字列プールへの登録。
    for (auto iter = m_pElem->tex_pattern_array.cbegin(); iter != m_pElem->tex_pattern_array.cend(); ++iter)
    {
        pCtx->SetStr(iter->tex_name.value.c_str());
    }

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinMaterialAnim::CalculateSize()
{
    int matCount =  static_cast<int>( m_pElem->per_material_anim_array.size() );
    m_Chunk[ChunkType_BindIndex].size = nw::g3d::tool::util::Align(sizeof(uint16_t) * matCount, ALIGNMENT_DEFAULT);
    m_Chunk[ChunkType_PerMaterialAnim].size = sizeof(nn::g3d::ResPerMaterialAnimData) * matCount;
    m_Chunk[ChunkType_UserDataData].size = sizeof( nn::gfx::ResUserDataData ) * m_UserDataArray.size();
    m_Chunk[ChunkType_TextureDescriptorSlotArray].size = sizeof( nn::gfx::DescriptorSlotData ) * m_pElem->tex_pattern_array.size();
    m_Chunk[ChunkType_GfxTextureViewPtrArray].size = sizeof( nn::util::BinTPtr<const nn::gfx::TextureView> ) * m_pElem->tex_pattern_array.size();
    m_Chunk[ChunkType_TextureNamePtrArray].size = sizeof( nn::util::BinPtrToString ) * m_pElem->tex_pattern_array.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinMaterialAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    // PerMaterialAnim のコンバートクラスにヒープを渡す。
    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_PerMaterialAnim].offset;
    for (auto iter = m_PerMaterialAnimArray.begin(); iter != m_PerMaterialAnimArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResPerMaterialAnimData);
    }

    offset = GetBlockOffset( Context::MemBlockType_Main ) + m_Chunk[ ChunkType_UserDataData ].offset;
    for (auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::gfx::ResUserDataData);
    }
}

void BinMaterialAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResMaterialAnimData& materialAnim = *GetPtr<nn::g3d::ResMaterialAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));
    materialAnim.blockHeader.signature.SetPacked( nn::g3d::ResMaterialAnim::Signature );
    pCtx->LinkStr( &materialAnim.pName, m_pElem->name.c_str() );
    pCtx->LinkStr( &materialAnim.pPath, m_pElem->path.c_str() );
    materialAnim.pBindModel.Set( nullptr );

    const nw::g3d::tool::g3dif::elem_nn_material_anim_info& info = m_pElem->material_anim_info;
    uint16_t flag = 0;
    if (info.loop.value)
    {
        flag |= nn::g3d::AnimFlag_PlayPolicyLoop;
    }
    materialAnim.flag = flag;
    materialAnim.frameCount = m_pElem->material_anim_info.frame_count.value;
    materialAnim.shaderParamAnimCount = m_ParamCount;
    materialAnim.perMaterialAnimCount = static_cast<uint16_t>( m_pElem->per_material_anim_array.size() );
    materialAnim.curveCount = m_CurveCount;
    materialAnim.bakedSize = 0;

    uint16_t* pBindIndexArray = GetPtr<uint16_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_BindIndex].offset);
    pCtx->LinkPtr( &materialAnim.pBindIndexArray, pBindIndexArray);

    const int matCount = static_cast<int>( m_pElem->per_material_anim_array.size() );
    for (int idxAnim = 0; idxAnim < matCount; ++idxAnim)
    {
        pBindIndexArray[idxAnim] = nn::g3d::AnimFlag_NotBound;
    }
    if (matCount & 0x1)
    {
        // 奇数個の場合は最後にパディングを埋める。
        pBindIndexArray[matCount] = 0;
    }

    pCtx->LinkPtr(&materialAnim.pPerMaterialAnimArray,
        GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_PerMaterialAnim].offset));

    // TexturePattern
    {
        materialAnim.textureCount = static_cast<uint16_t>(m_pElem->tex_pattern_array.size());
        pCtx->LinkPtr( &materialAnim.pTextureArray, GetPtr( pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_GfxTextureViewPtrArray].offset ) );
        uint16_t texPatternAnimCount = 0;
        for( auto iterPerMatAnim = m_pElem->per_material_anim_array.begin(); iterPerMatAnim < m_pElem->per_material_anim_array.end(); ++iterPerMatAnim)
        {
            if( iterPerMatAnim->tex_pattern_anim_array )
            {
                texPatternAnimCount += static_cast<uint16_t>(iterPerMatAnim->tex_pattern_anim_array.Get().size());	//!< 配下にある、全 sampler の合計数
            }
        }
        materialAnim.texturePatternAnimCount = texPatternAnimCount;

        if( m_pElem->tex_pattern_array.size() != 0 )
        {
            nn::util::BytePtr offsetToTextureDescriptorSlotArray( GetPtr<void>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_TextureDescriptorSlotArray ].offset ) );
            if( !nn::util::is_aligned( reinterpret_cast<size_t>( offsetToTextureDescriptorSlotArray.Get() ), 8 ) )
            {
                THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_INTERNAL, "Identifier_InvalidAddressAlignment", "ResMaterialAnimData pTextureSlotArray", offsetToTextureDescriptorSlotArray.Get() );
            }
            // DescriptorSlotを初期化しておく。無効値で初期化される。
            nn::gfx::DescriptorSlotData* pDescriptorSlotDataArray = offsetToTextureDescriptorSlotArray.Get<nn::gfx::DescriptorSlotData>();
            for (unsigned int idxTex = 0; idxTex < m_pElem->tex_pattern_array.size(); ++idxTex)
            {
                new(&pDescriptorSlotDataArray[idxTex]) nn::gfx::DescriptorSlot();
            }
            pCtx->LinkPtr( &materialAnim.pTextureSlotArray,
                GetPtr<int32_t>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_TextureDescriptorSlotArray ].offset ) );

            nn::util::BinPtrToString* pTextureNameArray =
                GetPtr<nn::util::BinPtrToString>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_TextureNamePtrArray ].offset );
            pCtx->LinkPtr( &materialAnim.pTextureNameArray, pTextureNameArray );

            for( auto iter = m_pElem->tex_pattern_array.cbegin(); iter != m_pElem->tex_pattern_array.cend(); ++iter, ++pTextureNameArray )
            {
                pCtx->LinkStr( pTextureNameArray, nn::util::string_view( iter->tex_name.value.data() ) );
            }
        }
    }

    // UserData
    materialAnim.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    m_DicUserData.ConvertData(pCtx, materialAnim.pUserDataDic, m_UserDataArray);
}

void BinMaterialAnim::Adjust( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResMaterialAnim* materialAnim = GetPtr<nn::g3d::ResMaterialAnim>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));
    nn::g3d::ResMaterialAnimData& shaderParamAnimData = materialAnim->ToData();
    shaderParamAnimData.pPerMaterialAnimArray.Relocate( pCtx->GetBasePtr() );

    size_t size = 0;
    for (int idxMatAnim = 0, numMatAnim = materialAnim->GetPerMaterialAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        nn::g3d::ResPerMaterialAnim* pMatAnim = materialAnim->GetPerMaterialAnim(idxMatAnim);
        nn::g3d::ResPerMaterialAnimData& resPerMaterialAnimData = pMatAnim->ToData();
        resPerMaterialAnimData.pCurveArray.Relocate( pCtx->GetBasePtr() );

        for (int idxCurve = 0, numCurve = pMatAnim->GetCurveCount();
            idxCurve < numCurve; ++idxCurve)
        {
            nn::g3d::ResAnimCurve* curve = pMatAnim->GetCurve(idxCurve);
            size += curve->CalculateBakedSize();
        }

        resPerMaterialAnimData.pCurveArray.Unrelocate( pCtx->GetBasePtr() );
    }
    shaderParamAnimData.bakedSize = static_cast<uint32_t>(size);
    shaderParamAnimData.pPerMaterialAnimArray.Unrelocate( pCtx->GetBasePtr() );
}

}
}
