﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <algorithm>
#include <regex>
#include <binlib.h>
#include <util/UtilPrint.h>
#include <util/UtilString.h>
#include <util/UtilXMLParser.h>
#include <util/UtilFlag.h>
#include <util/UtilFile.h>
#include <g3dif/MaterialAnim.h>
#include <BinConverter.h>

#include <util/UtilError.h>

#define OPTION_EQUALS(name, option) \
    (wcscmp((name), (option)) == 0 || wcscmp((name), (option##_FULL)) == 0)

#define OPTION_EQUALS_FULL(name, option) \
    (wcscmp((name), (option)) == 0 || wcscmp((name), (option)) == 0)

namespace nn {
namespace g3dTool {
namespace {

std::vector<std::string> g_MaterialAnimFileNames;
bool g_HasNwMaterialAnim;
bool g_HasNnMaterialAnim;
inline void CheckAnimationFileNameDupplication( const std::string& key, const wchar_t* ext )
{
    std::wstring targetExtName = std::wstring( ext );
    targetExtName.erase( targetExtName.begin() );		// . の削除
    targetExtName.pop_back();							// 末尾 'a' 'b' の削除 c.f. fma|a fma|b

    const wchar_t* NwMatAnimExtNameArray[] =
    {
        L"fsp",
        L"fcl",
        L"fts",
        L"ftp",
        L"fvm",
    };
    auto findNwMatExtName = [&](const wchar_t* name)->const wchar_t*
    {
        for( int idxNwMatAnimExtNameArray = 0; idxNwMatAnimExtNameArray < sizeof(NwMatAnimExtNameArray) / sizeof( const wchar_t* ); ++idxNwMatAnimExtNameArray )
        {
            if( std::wstring( name ) == NwMatAnimExtNameArray[idxNwMatAnimExtNameArray] )
            {
                return NwMatAnimExtNameArray[idxNwMatAnimExtNameArray];
            }
        }
        return nullptr;
    };
    if( findNwMatExtName( targetExtName.c_str() ) != nullptr )
    {
        g_HasNwMaterialAnim = true;
    }

    const wchar_t* NnMaterialAnimExtName = L"fma";
    if( std::wstring( NnMaterialAnimExtName ) == targetExtName )
    {
        g_HasNnMaterialAnim = true;
    }

    if( g_HasNwMaterialAnim && g_HasNnMaterialAnim )
    {
        THROW_TRANSLATED_ERROR(ERRCODE_DUPLICAED_MATERIAL_ANIM_FILE_TYPE, "Identifier_DuplicatedMaterialAnimFile");
    }

    std::vector<std::string>::iterator fileNameIter =
        std::find( g_MaterialAnimFileNames.begin(), g_MaterialAnimFileNames.end(), key );

    if( std::find( g_MaterialAnimFileNames.begin(), g_MaterialAnimFileNames.end(), key.c_str() ) != g_MaterialAnimFileNames.end() )
    {
        THROW_TRANSLATED_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "Identifier_DuplicatedMaterialAnimName", key.c_str());
    }
    g_MaterialAnimFileNames.push_back( key );
}

const char* const FILE_TYPE_ARRAY[] =
{
    ".fmda", ".fmdb",
    ".ftxa", ".ftxb",
    ".fska", ".fskb",
    ".fspa", ".fspb",
    ".fcla", ".fclb",
    ".ftsa", ".ftsb",
    ".ftpa", ".ftpb",
    ".fvba", ".fvbb",
    ".fvma", ".fvmb",
    ".fsha", ".fshb",
    ".fsna", ".fsnb",
    ".fmaa", ".fmab",
};

const int FILE_TYPE_COUNT = sizeof(FILE_TYPE_ARRAY) / sizeof(char*);

const wchar_t* const s_NameRegexString = L"[0-9A-Za-z\\-\\._]+";

} // anonymous namespace

bool Converter::Initialize()
{
    try
    {
        return Clear();
    }
    CATCH_ERROR_ALL(false);
}

bool Converter::Clear()
{
    try
    {
        m_Ctx.reset(new Context());	// Context 破壊前に File を破壊するとコンテキストが保持する辞書が破壊できなくなりリークする。
        m_File.reset(new BinFile());
        m_RootPathUtf.clear();
        m_CheckFlag = CHECK_BINARY_DEFAULT;
        m_UseAbsolutePath = false;
        g_MaterialAnimFileNames.resize(0);
        g_HasNnMaterialAnim = false;
        g_HasNwMaterialAnim = false;
        return true;
    }
    CATCH_ERROR_ALL(false);
}

bool Converter::Shutdown()
{
    return true;
}

bool Converter::SetOptions(const wchar_t* options[])
{
    try
    {
        // ログのサイレント設定を先に行っておく
        {
            int index = 0;
            while (options[index])
            {
                const wchar_t* name = options[index++];
                const wchar_t* value = options[index++];
                NN_UNUSED(value);
                if (OPTION_EQUALS(name, OPT_SILENT))
                {
                    nw::g3d::tool::LogConfig::s_Level = nw::g3d::tool::LogConfig::Silent;
                }
            }
        }

        int index = 0;
        while (options[index])
        {
            const wchar_t* name = options[index++];
            const wchar_t* value = options[index++];
            if (OPTION_EQUALS(name, OPT_OUTPUT_FILE))
            {
                std::wstring fileName;
                std::wstring ext;
                wchar_t nameBuf[_MAX_FNAME];
                wchar_t extBuf[_MAX_EXT];
                // 拡張子抽出
                if (_wsplitpath_s(value, nullptr, 0, nullptr, 0, nameBuf, _MAX_FNAME, extBuf, _MAX_EXT) == 0)
                {
                    ext.assign(extBuf);
                }

                fileName.assign(value);
                if (ext == OUTPUT_FILE_EXT)
                {
                    std::wstring::size_type pos = fileName.rfind(OUTPUT_FILE_EXT);
                    // 拡張子削除
                    if( pos != std::wstring::npos )
                    {
                        fileName.resize(pos);
                    }
                }

                // セパレータ変換
                std::replace(fileName.begin(), fileName.end(), L'\\', L'/');
                // ベース名取得
                fileName = nw::g3d::tool::util::GetBaseName(fileName, L'/');

                // 名前の命名規則に合っていなかった場合例外
                if (!std::regex_match(fileName, std::wregex(s_NameRegexString)))
                {
                    THROW_TRANSLATED_ERROR(ERRCODE_INVALID_FILENAME, "Identifier_InvalidOutputFileName", nw::g3d::tool::util::ToAscii(fileName.c_str()).c_str());
                }

                // wstring から string に変換する
                std::string cStr = nw::g3d::tool::util::ToAscii(&fileName[0]);

                this->m_File->SetName(cStr);

                PRINT_LOG("bfres name: %hs", cStr.c_str());
            }
            else if (OPTION_EQUALS(name, OPT_ROOT_PATH))
            {
                wchar_t fullpath[_MAX_PATH];
                _wfullpath(fullpath, value, _MAX_PATH);
                std::wstring rootPath;
                rootPath.assign(fullpath);
                if (L'\\' != rootPath.back())
                {
                    // フォルダ名が部分一致しないように確実に終端する。
                    rootPath.push_back(L'\\');
                }

                std::replace(rootPath.begin(), rootPath.end(), L'\\', L'/');
                m_RootPathUtf = nw::g3d::tool::util::ToUTF8(rootPath.c_str());

                PRINT_LOG("root path: %hs", m_RootPathUtf.c_str());
            }
            else if (OPTION_EQUALS(name, OPT_IGNORE_ASSIGN))
            {
                // シェーダアサインを無視してコンバートを行います。
                m_CheckFlag = nw::g3d::tool::util::DisableFlag(m_CheckFlag, CHECK_SHADER_ASSIGN);
            }
            else if (OPTION_EQUALS(name, OPT_IGNORE_VERSION))
            {
                // バージョン番号を無視してコンバートを行います。
                m_CheckFlag = nw::g3d::tool::util::DisableFlag(m_CheckFlag, CHECK_VERSION);
            }
            else if (0 == wcscmp(name, OPT_VIEWER_FULL))
            {
                // マテリアルエディタ用フラグ処理
                // ビューアーではシェーダー関連チェックを行いません。
                m_CheckFlag = nw::g3d::tool::util::DisableFlag(m_CheckFlag, CHECK_PARAM_ANIM_ASSIGN);
                m_CheckFlag = nw::g3d::tool::util::DisableFlag(m_CheckFlag, CHECK_SHADER_ASSIGN);

                // ビューアーではテクスチャ名のチェックを行いません。
                m_CheckFlag = nw::g3d::tool::util::DisableFlag(m_CheckFlag, CHECK_TEXTURE_NAME);

                m_UseAbsolutePath = true;
            }
            else if (0 == wcscmp(name, OPT_EDITOR_FULL))
            {
                // 3Dエディタ用フラグ処理
                m_UseAbsolutePath = true;
            }
            else if (OPTION_EQUALS(name, OPT_SILENT) || OPTION_EQUALS(name, OPT_VERIFY) || OPTION_EQUALS(name, OPT_VERSION))
            {
                // do nothing.
            }
            else if( OPTION_EQUALS_FULL( name, OPT_DISABLE_ATTRIBUTE_ALIGNMENT_SORT ) )
            {
                m_Ctx->SetDisableAttributeAlignmentSort( true );
            }
            else
            {
                THROW_TRANSLATED_ERROR(ERRCODE_UNKNOWN_OPTION, "Identifier_UnknownOption", nw::g3d::tool::util::ToAscii(name).c_str(), nw::g3d::tool::util::ToAscii(value).c_str());
            }
        }

        return true;
    }
    CATCH_ERROR_ALL(false);
}

// 各中間ファイルバージョンをチェックする
template<typename T>
void Converter::CheckVersion(T pElem, int major, int minor, int micro, const char* elementName)
{
    if (nw::g3d::tool::util::CheckFlag(m_CheckFlag, CHECK_VERSION))
    {
        if (pElem->version.value.major != major ||
            pElem->version.value.minor != minor ||
            pElem->version.value.micro != micro)
        {
            THROW_ERROR(
                ERRCODE_XML_VERSION_MISMATCH,
                "Xml version mismatch.\n%hs supports <%hs> version: %d.%d.%d.\nBut input <%hs> version: %d.%d.%d.",
                nw::g3d::tool::util::GetModuleName(nullptr).c_str(),
                elementName,
                major,
                minor,
                micro,
                elementName,
                pElem->version.value.major,
                pElem->version.value.minor,
                pElem->version.value.micro);
        }
    }
}

bool Converter::AddFile(void* pData, size_t dataSize, const wchar_t* path[3], const wchar_t* options[])
{
    PRINT_LOG("Path: %ls", path[0]);
    PRINT_LOG("Filename: %ls%ls", path[1], path[2]);

    try
    {
        enum
        {
            PATH_FULLPATH,
            PATH_FNAME,
            PATH_EXT
        };
        const wchar_t* fname = path[PATH_FNAME];
        const wchar_t* ext = path[PATH_EXT];

        // ファイル名を char に変換する。
        std::string fnameAscii = nw::g3d::tool::util::ToAscii(&fname[0]);
        std::string extAscii = nw::g3d::tool::util::ToAscii(&ext[0]);

        if (!std::regex_match(fname, std::wregex(s_NameRegexString)))
        {
            THROW_TRANSLATED_ERROR(ERRCODE_INVALID_FILENAME, "Identifier_InvalidInputFileName", fnameAscii.c_str());
        }

        if (extAscii.empty())
        {
            THROW_TRANSLATED_ERROR(ERRCODE_INVALID_FILENAME, "Identifier_InvalidInputFileExtension", fnameAscii.c_str());
        }

        std::wstring wpath(path[PATH_FULLPATH]);
        // ワイド文字でセパレータを置換してから utf-8 に変換
        std::replace(wpath.begin(), wpath.end(), L'\\', L'/');
        std::string fullpathUtf = nw::g3d::tool::util::ToUTF8(wpath.c_str());

        // root_path が設定されていない場合は path を入れない。
        std::string trimString = nw::g3d::tool::util::TrimString(fullpathUtf, m_RootPathUtf);
        if (!m_UseAbsolutePath && m_RootPathUtf.empty())
        {
            trimString = "";
        }

        bool isExternalFile = false;
        size_t externalFileAlignment = ALIGNMENT_EXTERNAL_FILE;

        // ローカルオプションを処理する
        int index = 0;
        while (options[index])
        {
            const wchar_t* name = options[index++];
            const wchar_t* value = options[index++];

            if (0 == wcscmp(name, OPT_EXTERNAL_FILE_FULL))
            {
                isExternalFile = true;
            }
            else if (0 == wcscmp(name, OPT_EXTERNAL_FILE_ALIGNMENT_FULL))
            {
                int alignment = 0;
                if (nw::g3d::tool::util::TryParse(alignment, value))
                {
                    externalFileAlignment = alignment;
                }
            }
            else
            {
                THROW_TRANSLATED_ERROR(ERRCODE_UNKNOWN_OPTION, "Identifier_UnknownOption", nw::g3d::tool::util::ToAscii(name).c_str(), nw::g3d::tool::util::ToAscii(value).c_str());
            }
        }

        std::string extLower;
        std::transform(extAscii.begin(), extAscii.end(), std::back_inserter(extLower), ::tolower);
        bool isSupportType = false;
        for (int i = 0; i < FILE_TYPE_COUNT; ++i)
        {
            const char* filetype = FILE_TYPE_ARRAY[i];
            if (extLower == filetype)
            {
                isSupportType = true;
                break;
            }
        }

        if (!isExternalFile && !isSupportType)
        {
            THROW_TRANSLATED_ERROR(ERRCODE_UNSUPPORTED_FILETYPE, "Identifier_UnsupportedFileType", extAscii.c_str());
        }

        if (isExternalFile)
        {
            BinExternalFile::ExternalFileElem elem;
            elem.data.reset(malloc(dataSize));
            memcpy(elem.data.get(), pData, dataSize);
            std::stringstream basename;
            basename << fnameAscii << extAscii;
            elem.name = basename.str();
            elem.size = dataSize;
            elem.alignment = externalFileAlignment;
            m_File->Add(elem);
        }
        else
        {
            nw::g3d::tool::util::XMLParser parser;
            // バイナリ中間ファイルのために NULL 終端を含んだサイズを計算する。
            size_t textSize = strnlen_s(static_cast<const char*>(pData), dataSize) + sizeof(char);
            bool success = parser.ParseTree(static_cast<char*>(pData), textSize);
            if (!success)
            {
                THROW_TRANSLATED_ERROR(ERRCODE_XML_CONVERT_PARSE_FAILURE, "Identifier_FailXmlParse");
            }
            const nw::g3d::tool::util::XMLElement* pRoot = parser.Root();

            std::wstring ifExt(ext);
            // データサイズ＋１とテキスト部分のサイズが同一の場合、テキスト中間ファイルとして扱う。
            if ((dataSize + 1) == textSize)
            {
                ifExt.pop_back();
                ifExt.push_back(L'a');
            }
            std::transform(ifExt.begin(), ifExt.end(), ifExt.begin(), ::tolower);

            std::shared_ptr<nw::g3d::tool::g3dif::elem_nw4f_3dif> nw4f_3dif(new nw::g3d::tool::g3dif::elem_nw4f_3dif());
            *nw4f_3dif << pRoot;

            // 全中間ファイル拡張子で統一されたコンテナバージョンをチェックします。
            CheckVersion(nw4f_3dif, NN_G3D_INTERMEDIATE_VERSION_MAJOR, NN_G3D_INTERMEDIATE_VERSION_MINOR, NN_G3D_INTERMEDIATE_VERSION_MICRO, "nw4f_3dif");

            // モデル
            if (0 == wcscmp(L".fmda", ifExt.c_str()))
            {
                std::shared_ptr<nw::g3d::tool::g3dif::elem_model> elem =
                    nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_model>(pRoot, m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FMD_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FMD_VERSION_MINOR, NN_G3D_INTERMEDIATE_FMD_VERSION_MICRO, "model");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            else if (0 == wcscmp(L".fmdb", ifExt.c_str()))
            {
                std::shared_ptr<nw::g3d::tool::g3dif::elem_model> elem =
                    nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_model>(pRoot, nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))), m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FMD_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FMD_VERSION_MINOR, NN_G3D_INTERMEDIATE_FMD_VERSION_MICRO, "model");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            // スケルタルアニメ
            else if (0 == wcscmp(L".fska", ifExt.c_str()))
            {
                 std::shared_ptr<nw::g3d::tool::g3dif::elem_skeletal_anim> elem =
                     nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_skeletal_anim>(pRoot, m_CheckFlag);
                 CheckVersion(elem, NN_G3D_INTERMEDIATE_FSK_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSK_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSK_VERSION_MICRO, "skeletal_anim");
                 nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                 m_File->Add(elem);
            }
            else if (0 == wcscmp(L".fskb", ifExt.c_str()))
            {
                std::shared_ptr<nw::g3d::tool::g3dif::elem_skeletal_anim> elem =
                    nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_skeletal_anim>(pRoot, nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))), m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FSK_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSK_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSK_VERSION_MICRO, "skeletal_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            // シェーダパラメータアニメ
            else if (0 == wcscmp(L".fspa", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_param_anim> elem =
                    nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_shader_param_anim>(pRoot, m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FSP_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MICRO, "shader_param_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->AddFsp(elem);
            }
            else if (0 == wcscmp(L".fcla", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext  );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_param_anim> elem =
                    nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_shader_param_anim>(pRoot, m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FSP_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MICRO, "shader_param_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->AddFcl(elem);
            }
            else if (0 == wcscmp(L".ftsa", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_param_anim> elem =
                    nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_shader_param_anim>(pRoot, m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FSP_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MICRO, "shader_param_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->AddFts(elem);
            }
            else if (0 == wcscmp(L".fspb", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_param_anim> elem =
                    nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_shader_param_anim>(pRoot, nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))), m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FSP_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MICRO, "shader_param_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->AddFsp(elem);
            }
            else if (0 == wcscmp(L".fclb", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_param_anim> elem =
                    nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_shader_param_anim>(pRoot, nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))), m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FSP_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MICRO, "shader_param_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->AddFcl(elem);
            }
            else if (0 == wcscmp(L".ftsb", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_param_anim> elem =
                    nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_shader_param_anim>(pRoot, nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))), m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FSP_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSP_VERSION_MICRO, "shader_param_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->AddFts(elem);
            }
            // テクスチャパターンアニメ
            else if (0 == wcscmp(L".ftpa", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_tex_pattern_anim> elem =
                    nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_tex_pattern_anim>(pRoot, m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FTP_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FTP_VERSION_MINOR, NN_G3D_INTERMEDIATE_FTP_VERSION_MICRO, "tex_pattern_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            else if (0 == wcscmp(L".ftpb", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_tex_pattern_anim> elem =
                    nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_tex_pattern_anim>(pRoot, nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))), m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FTP_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FTP_VERSION_MINOR, NN_G3D_INTERMEDIATE_FTP_VERSION_MICRO, "tex_pattern_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            // ボーンビジビリティアニメ
            else if (0 == wcscmp(L".fvba", ifExt.c_str()))
            {
                std::shared_ptr<nw::g3d::tool::g3dif::elem_bone_visibility_anim> elem =
                    nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_bone_visibility_anim>(pRoot, m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FVB_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FVB_VERSION_MINOR, NN_G3D_INTERMEDIATE_FVB_VERSION_MICRO, "bone_visibility_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            else if (0 == wcscmp(L".fvbb", ifExt.c_str()))
            {
                std::shared_ptr<nw::g3d::tool::g3dif::elem_bone_visibility_anim> elem =
                    nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_bone_visibility_anim>(pRoot, nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))), m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FVB_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FVB_VERSION_MINOR, NN_G3D_INTERMEDIATE_FVB_VERSION_MICRO, "bone_visibility_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            // マテリアルビジビリティアニメ
            else if (0 == wcscmp(L".fvma", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_mat_visibility_anim> elem =
                    nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_mat_visibility_anim>(pRoot, m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FVM_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FVM_VERSION_MINOR, NN_G3D_INTERMEDIATE_FVM_VERSION_MICRO, "mat_visibility_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            else if (0 == wcscmp(L".fvmb", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_mat_visibility_anim> elem =
                    nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_mat_visibility_anim>(pRoot, nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))), m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FVM_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FVM_VERSION_MINOR, NN_G3D_INTERMEDIATE_FVM_VERSION_MICRO, "mat_visibility_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            // シーンアニメーション
            else if (0 == wcscmp(L".fsna", ifExt.c_str()))
            {
                std::shared_ptr<nw::g3d::tool::g3dif::elem_scene_anim> elem =
                    nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_scene_anim>(pRoot, m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FSN_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSN_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSN_VERSION_MICRO, "scene_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            else if (0 == wcscmp(L".fsnb", ifExt.c_str()))
            {
                std::shared_ptr<nw::g3d::tool::g3dif::elem_scene_anim> elem =
                    nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_scene_anim>(pRoot, nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))), m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FSN_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSN_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSN_VERSION_MICRO, "scene_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            // シェイプアニメーション
            else if (0 == wcscmp(L".fsha", ifExt.c_str()))
            {
                std::shared_ptr<nw::g3d::tool::g3dif::elem_shape_anim> elem =
                    nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_shape_anim>(pRoot, m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FSH_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSH_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSH_VERSION_MICRO, "shape_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            else if (0 == wcscmp(L".fshb", ifExt.c_str()))
            {
                std::shared_ptr<nw::g3d::tool::g3dif::elem_shape_anim> elem =
                    nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_shape_anim>(pRoot, nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))), m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FSH_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FSH_VERSION_MINOR, NN_G3D_INTERMEDIATE_FSH_VERSION_MICRO, "shape_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            else if(0 == wcscmp(L".fmaa", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_nn_material_anim> elem =
                    nw::g3d::tool::util::AddElem<nw::g3d::tool::g3dif::elem_nn_material_anim>(pRoot, m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FMA_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FMA_VERSION_MINOR, NN_G3D_INTERMEDIATE_FMA_VERSION_MICRO, "material_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            else if(0 == wcscmp(L".fmab", ifExt.c_str()))
            {
                CheckAnimationFileNameDupplication( fnameAscii, ext );
                std::shared_ptr<nw::g3d::tool::g3dif::elem_nn_material_anim> elem =
                    nw::g3d::tool::util::AddBinaryElem<nw::g3d::tool::g3dif::elem_nn_material_anim>(
                                pRoot,
                                nw::g3d::tool::util::AddOffset(pData, nw::g3d::tool::util::Align(textSize, static_cast<int>(nw::g3d::tool::g3dif::STREAM_BINARY_ALIGNMENT))),
                                m_CheckFlag);
                CheckVersion(elem, NN_G3D_INTERMEDIATE_FMA_VERSION_MAJOR, NN_G3D_INTERMEDIATE_FMA_VERSION_MINOR, NN_G3D_INTERMEDIATE_FMA_VERSION_MICRO, "material_anim");
                nw::g3d::tool::util::AddPath(elem, fnameAscii.c_str(), trimString.c_str());
                m_File->Add(elem);
            }
            else
            {
                THROW_TRANSLATED_ERROR(ERRCODE_UNSUPPORTED_FILETYPE, "Identifier_UnsupportedFileType");
            }
        }

        return true;
    }
    CATCH_ERROR_ALL(false)
}

size_t Converter::CalculateSize()
{
    try
    {
        // 文字列の登録やサイズ計算の下準備
        m_File->Build( m_Ctx );

        // 各 Bin* クラスのサイズを計算します。
        auto execCalculateSize = [&](BinaryBlock* pBlock)
        {
            pBlock->CalculateSize();
        };
        std::for_each(m_Ctx->blocks.begin(), m_Ctx->blocks.end(), execCalculateSize);

        // 並列処理をする場合はこちらを用います。
        //parallel_for_each(blocks.begin(), blocks.end(), execCalculateSize);

        // 各 Bin* クラスが使用するメモリブロックでの offset を計算します。
        auto execCalcOffset = [&](BinaryBlock* pBlock)
        {
            pBlock->CalculateOffset( m_Ctx );
        };
        std::for_each(m_Ctx->blocks.begin(), m_Ctx->blocks.end(), execCalcOffset); // 並列化不可。

        // MemoryPool ブロックのサイズとアライメントの設定
        if ( ( m_Ctx->GetMemBlockSize( Context::MemBlockType_IdxStream ) != 0 ) ||	// 安全のためにどちらか片方の存在でも 2 つ分の MemoryPool を用意する。
            ( m_Ctx->GetMemBlockSize( Context::MemBlockType_VtxStream ) != 0 ) )
        {
            size_t memPoolBlocksize = GetGfxMaxImplDataSize<nn::gfx::MemoryPoolImplData>();
            m_Ctx->SetMemBlockSize( Context::MemBlockType_GfxMemPool, memPoolBlocksize );
            m_Ctx->SetAlignment( Context::MemBlockType_GfxMemPool, 8 );
        }

        // 各メモリブロックのアライメントを計算し、セットします。
        {
            int maxAlignment = 0;
            if ( m_Ctx->GetMemBlockSize( Context::MemBlockType_IdxStream ) )
            {
                size_t size = m_Ctx->GetMemBlockSize( Context::MemBlockType_IdxStream );
                size = nw::g3d::tool::util::Align(size, ALIGNMENT_VERTEX_STREAM );
                m_Ctx->SetMemBlockSize( Context::MemBlockType_IdxStream, size );

                // メモリプールの要求するアドレスアライメントにする
                m_Ctx->SetAlignment( Context::MemBlockType_IdxStream, ALIGNMENT_MEMORY_POOL_ADDR );
                maxAlignment = std::max(maxAlignment, static_cast<int>(ALIGNMENT_MEMORY_POOL_ADDR));
            }

            if ( m_Ctx->GetMemBlockSize( Context::MemBlockType_VtxStream ) )
            {
                // 頂点バッファとインデックスバッファの合計サイズをメモリプールの要求するサイズアライメントにする
                size_t size = m_Ctx->GetMemBlockSize( Context::MemBlockType_VtxStream ) +
                              m_Ctx->GetMemBlockSize( Context::MemBlockType_IdxStream );
                size_t padding = nw::g3d::tool::util::Align(size, ALIGNMENT_MEMORY_POOL_SIZE ) - size;

                size = m_Ctx->GetMemBlockSize( Context::MemBlockType_VtxStream ) + padding;
                m_Ctx->SetMemBlockSize( Context::MemBlockType_VtxStream, size );
                m_Ctx->SetAlignment( Context::MemBlockType_VtxStream, ALIGNMENT_VERTEX_STREAM );
                maxAlignment = std::max(maxAlignment, static_cast<int>(ALIGNMENT_VERTEX_STREAM));
            }

            if ( m_Ctx->GetMemBlockSize( Context::MemBlockType_ExternalFile ) )
            {
                // ExternalFile に含まれる最大アライメント
                int externalFileAlignment = m_File->GetExternalFileAlignment();
                m_Ctx->SetAlignment( Context::MemBlockType_ExternalFile, externalFileAlignment );
                maxAlignment = std::max(maxAlignment, externalFileAlignment);
            }

            maxAlignment = std::max( maxAlignment, static_cast<int>(ALIGNMENT_FILE ) );
            m_File->SetMaxAlignment( maxAlignment );
        }

        // シリアライズコンテキストを初期化します。
        m_Ctx->Build( nn::util::string_view( m_File->GetName().c_str() ) );

        return m_Ctx->GetBaseSize();
    }
    CATCH_ERROR_ALL(0)
}

size_t Converter::GetAlignmentSize()
{
    try
    {
        return m_File->GetMaxAlignment();
    }
    CATCH_ERROR_ALL(0)
}

bool Converter::Convert(void* pBuffer, size_t size)
{
    NN_UNUSED( pBuffer );
    NN_UNUSED( size );
    try
    {
        if ( m_Ctx->GetBasePtr() == nullptr )
        {
            THROW_ERROR_INTERNAL( ERRCODE_INTERNAL, "Buffer is NULL\n" );
        }

        auto execConvert = [&](BinaryBlock* pBlock)
        {
            pBlock->Convert( m_Ctx );
        };
        std::for_each(m_Ctx->blocks.begin(), m_Ctx->blocks.end(), execConvert);

        // コンバート後に補正を行います。
        auto execAdjust = [&](BinaryBlock* pBlock)
        {
            pBlock->Adjust( m_Ctx );
        };
        std::for_each(m_Ctx->blocks.begin(), m_Ctx->blocks.end(), execAdjust);

        // 並列処理をする場合はこちらを用います。
        //parallel_for_each(blocks.begin(), blocks.end(), execConvert);

        // relocation table のサイズ計算と作成
        m_Ctx->Convert();

        return true;
    }
    CATCH_ERROR_ALL(false)
}

bool Converter::Bind(void* pBuffer, size_t size)
{
    NN_UNUSED( size );

    try
    {
        nn::g3d::ResFileData* pFile	= static_cast<nn::g3d::ResFileData*>( pBuffer );
        pFile->pModelArray.Relocate( pBuffer );
        nn::g3d::ResFile* pResFile	= reinterpret_cast<nn::g3d::ResFile*>( pFile );

        // ビルボードをしている子ノード以下にフラグを立てる。
        for (int i = 0; i < pResFile->GetModelCount(); ++i)
        {
            nn::g3d::ResModel* pResModel = pResFile->GetModel( i );
            nn::g3d::ResModelData* pResModelData = &pResModel->ToData();
            pResModelData->pSkeleton.Relocate( pBuffer );

            nn::g3d::ResSkeleton* pResSkeleton = pResModelData->pSkeleton.Get();
            nn::g3d::ResSkeletonData* pResSkeletonData = &pResSkeleton->ToData();
            pResSkeletonData->pBoneArray.Relocate( pBuffer );
            pResSkeleton->UpdateBillboardMode();

            pResSkeletonData->pBoneArray.Unrelocate( pBuffer );
            pResModelData->pSkeleton.Unrelocate( pBuffer );
        }

        pFile->pModelArray.Unrelocate( pBuffer );

        return true;
    }
    CATCH_ERROR_ALL(false)
}

bool Converter::SwapEndian(void* pBuffer, size_t /*size*/)
{
    NN_UNUSED( pBuffer );

    try
    {
#if !defined( NN_GFX_TOOL_ENDIAN_SWAP_IS_NOT_IMPLEMENTED )
        void* pBuffer = GetBasePtr();
        nn::g3d::ResFileData* pFile = static_cast<nn::g3d::ResFileData*>(pBuffer);
        nn::util::BinaryFileHeader* pBinaryFileHeader = &pFile->fileHeader;

        if( pBinaryFileHeader->IsEndianReverse() )	// ビッグエンディアンの時はアーリーアウト
        {
            PRINT_LOG( "SwapEndian: Endian has already been big endian\n" );
            return true;
        }

        // エンディアン反転
        nn::g3d::Endian::SwapEndian( pFile );
        pBinaryFileHeader->SetByteOrderMark( nn::util::ByteOrderMark_Reverse );
#endif
        return true;
    }
    CATCH_ERROR_ALL(false)
}

}
}
