﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <g3dif/Model.h>
#include <nn/g3d/g3d_ResShape.h>

namespace nn {
namespace g3dTool {

typedef struct HalfEdgeRec
{
    uint32_t Vert;				// Vertex index at the end of this half-edge
    struct HalfEdgeRec* Twin;	// Oppositely oriented adjacent half-edge
    struct HalfEdgeRec* Next;	// Next half-edge around the face
} HalfEdge;

typedef struct BoundingVolumeRec
{
    nn::util::Float3 volumeMax;
    nn::util::Float3 volumeMin;
} BoundingVolume;

class LessIndex
{
public:
    bool operator()(const nw::g3d::tool::util::RefBoneWeight& rLeft, const nw::g3d::tool::util::RefBoneWeight& rRight) const
    {
        return rLeft.index < rRight.index;
    }
};

template<typename Source, typename Dest>
void ComputeAdjacency(void* dest, const void* source, int faceCount, int /*vertCount*/)
{
    HalfEdge* edges = (HalfEdge*)calloc(faceCount * 3, sizeof(HalfEdge));
    memset(edges, 0, faceCount * 3 * sizeof(HalfEdge));

    std::unordered_map<Dest, HalfEdge*> edgeTable;

    // Plow through faces and fill all half-edge info except twin pointers:
    HalfEdge* pEdge = edges;
    const Source* pSrc = static_cast<const Source*>(source);
    const int INDEX_SIZE = sizeof(Dest) * 8 / 2;
    const uint64_t BIT_MASK = (0x1u << (INDEX_SIZE - 1)) - 1 | (0x1u << (INDEX_SIZE - 1));
    for (int faceIndex = 0; faceIndex < faceCount; ++faceIndex)
    {
        Dest A = *pSrc++;
        Dest B = *pSrc++;
        Dest C = *pSrc++;

        edgeTable.insert(std::make_pair(C | (A << INDEX_SIZE), pEdge));
        pEdge->Vert = static_cast<uint32_t>(A);
        pEdge->Next = pEdge + 1;
        ++pEdge;

        edgeTable.insert(std::make_pair(A | (B << INDEX_SIZE), pEdge));
        pEdge->Vert = static_cast<uint32_t>(B);
        pEdge->Next = pEdge + 1;
        ++pEdge;

        edgeTable.insert(std::make_pair(B | (C << INDEX_SIZE), pEdge));
        pEdge->Vert = static_cast<uint32_t>(C);
        pEdge->Next = pEdge - 2;
        ++pEdge;
    }

    unsigned long numEntries = static_cast<unsigned long>(edgeTable.size());
    if (numEntries != static_cast<unsigned long>(faceCount * 3))
    {
        THROW_ERROR(ERRCODE_SHAPE_INVALID_MESH, "Invalid mesh. Detected duplicated edges or inconsistent winding.");
    }

    for (auto iterEdge = edgeTable.begin(); iterEdge != edgeTable.end(); ++iterEdge)
    {
        HalfEdge* edge = iterEdge->second;
        Dest edgeIndex = iterEdge->first;
        Dest twinIndex = ((edgeIndex & BIT_MASK) << INDEX_SIZE) | (edgeIndex >> INDEX_SIZE);
        auto comp = edgeTable.find(twinIndex);

        if (comp != edgeTable.end())
        {
            HalfEdge* pTwinEdge = edgeTable.find(twinIndex)->second;
            edge->Twin = pTwinEdge;
            pTwinEdge->Twin = edge;
        }
    }

    Source* pDest = static_cast<Source*>(dest);
    pEdge = edges;
    for (int faceIndex = 0; faceIndex < faceCount; ++faceIndex, pEdge += 3, pDest += 6)
    {
        pDest[0] = pEdge[0].Vert;
        pDest[1] = pEdge[1].Twin ? (pEdge[1].Twin->Next->Vert) : pDest[0];
        pDest[2] = pEdge[1].Vert;
        pDest[3] = pEdge[2].Twin ? (pEdge[2].Twin->Next->Vert) : pDest[2];
        pDest[4] = pEdge[2].Vert;
        pDest[5] = pEdge[0].Twin ? (pEdge[0].Twin->Next->Vert) : pDest[4];
    }

    free(edges);
}

// 座標データからバウンディングを構築する。
BoundingVolume ComputeBoundingVolume(const std::vector<nw::g3d::tool::util::Vec3_t>& positions,
                                const int* indices, int startIndex, int count, int skinningCount, int lodOffset );

// バウンディングを拡張する。
BoundingVolume MergeBoundingVolume(const BoundingVolume& lhs, const BoundingVolume& rhs);


// 中心点からバウンディングスフィアを計算する。
float ComputeBoundingSphere(const std::vector<nw::g3d::tool::util::Vec3_t>& positions, const nn::util::Float3& center,
                            const int* indices, int startIndex, int count, int skinningCount, int lodOffset );

// min, max から center, extent に変換する。
nn::g3d::Bounding BoundingVolumeToBounding(const BoundingVolume& BoundingVolume);

BoundingVolume BoundingToBoundingVolume(const nn::g3d::Bounding& bounding);

void CalculateBounding(nn::g3d::Bounding* pBoundingArray,
    float* pRadiusArray,
    const nw::g3d::tool::g3dif::elem_shape& elemShape,
    const std::vector<nw::g3d::tool::util::Vec3_t>* pLocalPositions,
    const std::vector<int>* pLodOffsets);

float GetAllowableError(float value, int significantDigits);

bool IsFloatEqual(float value1, float value2, int significantDigits);

// 各サブメッシュを LOD 毎に計算して誤差を比較する。T は nw::g3d::Bounding か nn::g3d::Bounding
bool HasBoundingPerLod( const nn::g3d::Bounding* pBoundingArray, const nw::g3d::tool::g3dif::elem_shape& elemShape );

}
}
