﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <g3dif/Shape.h>
#include <bindefs.h>

#if defined( near ) && defined( far )	// windows.h で両方 define される
#undef near
#undef far
#include <nn/g3d/g3d_ResShape.h>	// nn:util:math で near, far を使用している
#define near
#define far
#else
#include <nn/g3d/g3d_ResShape.h>
#endif

#include <BinaryBlock.h>
#include <BinDictionary.h>

namespace nn {
namespace g3dTool {

class BinVertex;
class BinSkeleton;

//! @brief ポリゴンを定義する頂点インデックスのバイナリ化を行うクラスです。
class BinMesh : public BinaryBlock
{
public:
    BinMesh()
        : BinaryBlock()
        , m_pElem(nullptr)
        , m_IsBinarizeAdjacency(false)
        , m_VertexCount(0)
        , m_AdjList()
        , m_QuantizeType(nw::g3d::tool::g3dif::elem_mesh::none)
        , m_IndicesCount(0)
        , m_AdjIndices(nullptr)
    {
    }

    void Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_mesh& elem);
    virtual void CalculateSize();
    virtual void Convert( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("Mesh(index) : %d", m_pElem->index);
    }

    void BinarizeAdjacency()
    {
        m_IsBinarizeAdjacency = true;
    }

    void SetVertexCount(uint32_t vertexCount)
    {
        m_VertexCount = vertexCount;
    }
private:
    const nw::g3d::tool::g3dif::elem_mesh* m_pElem;

    bool m_IsBinarizeAdjacency;
    uint32_t m_VertexCount;
    std::vector<std::vector<int>> m_AdjList;
    nw::g3d::tool::g3dif::elem_mesh::enum_quantize_type m_QuantizeType;
    uint32_t m_IndicesCount;
    std::shared_ptr<void> m_AdjIndices;

    enum ChunkType
    {
        ChunkType_Submesh,
        ChunkType_GfxBuff,
        ChunkType_GfxBuffInfoData,
        ChunkType_Count
    };

    Chunk m_Chunk[ChunkType_Count];
};

//--------------------------------------------------------------------------------------------------

//! @brief 形状データのバイナリ化を行うクラスです。
class BinShape : public BinaryBlock
{
public:
    BinShape()
        : BinaryBlock()
        , m_pElem(nullptr)
        , m_Meshes()
        , m_KeyShapeDic()
        , m_HasOwnership(false)
        , m_Vertex(nullptr)
        , m_Skeleton(nullptr)
        , m_Index(0)
        , m_Chunk()
    {
    }

    void Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_shape& elem);
    virtual void CalculateSize();
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );
    virtual void Convert( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("Shape : %hs", m_pElem->name.value.c_str());
    }

    void SetOwnership(bool hasOwnership) { this->m_HasOwnership = hasOwnership; }
    void SetIndex(int index) { m_Index = index; }
    void SetVertex(BinVertex* vertex) { m_Vertex = vertex; }
    void SetSkeleton(BinSkeleton* skeleton) { m_Skeleton = skeleton; }

private:
    const nw::g3d::tool::g3dif::elem_shape* m_pElem;
    BinVertex* m_Vertex;
    BinSkeleton* m_Skeleton;

    std::vector<BinMesh> m_Meshes;
    BinDictionary m_KeyShapeDic;

    bool m_HasOwnership;
    int m_Index;

    enum ChunkType
    {
        ChunkType_Mesh,
        ChunkType_SkinBoneIdx,
        ChunkType_KeyShape,
        ChunkType_SubmeshBounding,
        ChunkType_BoundingSpherePerLod,		//!< shape の lod 毎のバウンディング球
        ChunkType_Count
    };

    Chunk m_Chunk[ChunkType_Count];
};

}
}
