﻿namespace G3dCore.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Entities;
    using Opal.ViewModels;

    /// <summary>
    /// G3d ドキュメントビューモデルの抽象クラスです。
    /// </summary>
    public abstract class G3dDocumentViewModel : DocumentViewModel
    {
        private G3dFileViewModel fileViewModel = null;

        /// <summary>
        /// ファイルビューモデルを取得します。
        /// </summary>
        public G3dFileViewModel FileViewModel
        {
            get
            {
                return this.fileViewModel;
            }
        }

        /// <summary>
        /// G3d 中間ファイルを取得します。
        /// </summary>
        protected internal G3dFile File
        {
            get
            {
                return this.fileViewModel.File;
            }
        }

        /// <summary>
        /// 設定処理を実行します。
        /// </summary>
        /// <param name="parameter">設定用のパラメーターです。</param>
        public override sealed void Setup(object parameter)
        {
            Debug.Assert(parameter != null);
            Debug.Assert(parameter is G3dFileViewModel);

            this.fileViewModel = (G3dFileViewModel)parameter;
            this.UpdateTitle();

            // TODO: 今は強参照なのでWeakReferenceで勝手に不要になったら消えてくれるようにする
            this.File.PropertyChanged += this.OnFilePropertyChanged;

            this.SetupInternal();
        }

        /// <summary>
        /// 内部用の設定処理を実行します。
        /// </summary>
        protected abstract void SetupInternal();

        /// <summary>
        /// 閉じるときに実行される処理です。
        /// </summary>
        /// <returns>閉じることに成功したかどうかを返します。</returns>
        protected override bool TryClose()
        {
            this.File.PropertyChanged -= this.OnFilePropertyChanged;
            return base.TryClose();
        }

        private void OnFilePropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            // ファイル名の変更かDirtyフラグの変更があればタブのタイトル名を変更する
            if (e.PropertyName.Equals("FilePath") || e.PropertyName.Equals("IsDirty"))
            {
                this.UpdateTitle();
            }
        }

        private void UpdateTitle()
        {
            string displayFileName = Path.GetFileName(this.File.FilePath);
            if (this.File.IsDirty)
            {
                displayFileName = "*" + displayFileName;
            }

            this.Title = displayFileName;
        }
    }
}
