﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using ShaderAssistAddons.Resources;

    /// <summary>
    /// シェーダ設定を行うビューモデルです。
    /// </summary>
    public class ShaderSettingViewModel : WizardPageViewModel
    {
        private const string DefaultShaderName = "NewShader";
        private readonly ShaderSourceSettingViewModel vertexShaderSourceSettingViewModel;
        private readonly ShaderSourceSettingViewModel geometryShaderSourceSettingViewModel;
        private readonly ShaderSourceSettingViewModel fragmentShaderSourceSettingViewModel;
        private readonly ShaderSourceSettingViewModel computeShaderSourceSettingViewModel;
        private readonly ErrorMessageTextBlockViewModel errorNameViewModel = new ErrorMessageTextBlockViewModel();

        private WizardShaderViewModel wizardShaderViewModel;
        private string name = string.Empty;
        private bool isMaterialShader = true;

        /// <summary>
        /// シェーダ設定名を取得設定します。
        /// </summary>
        public string Name
        {
            get
            {
                return name;
            }

            set
            {
                if (value != name)
                {
                    this.CheckName(value);
                    name = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// マテリアルシェーダかどうかを取得設定します。
        /// </summary>
        public bool IsMaterialShader
        {
            get
            {
                return this.isMaterialShader;
            }

            set
            {
                this.SetProperty(ref this.isMaterialShader, value);
            }
        }

        /// <summary>
        /// シェーダ設定ファイルパスを取得します。
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.wizardShaderViewModel.FilePath;
            }
        }

        /// <summary>
        /// 頂点シェーダソースファイル設定ビューモデルを取得します。
        /// </summary>
        public ShaderSourceSettingViewModel VertexShaderSourceSettingViewModel
        {
            get
            {
                return this.vertexShaderSourceSettingViewModel;
            }
        }

        /// <summary>
        /// ジオメトリシェーダソースファイル設定ビューモデルを取得します。
        /// </summary>
        public ShaderSourceSettingViewModel GeometryShaderSourceSettingViewModel
        {
            get
            {
                return this.geometryShaderSourceSettingViewModel;
            }
        }

        /// <summary>
        /// フラグメントシェーダソースファイル設定ビューモデルを取得します。
        /// </summary>
        public ShaderSourceSettingViewModel FragmentShaderSourceSettingViewModel
        {
            get
            {
                return this.fragmentShaderSourceSettingViewModel;
            }
        }

        /// <summary>
        /// 演算シェーダソースファイル設定ビューモデルを取得します。
        /// </summary>
        public ShaderSourceSettingViewModel ComputeShaderSourceSettingViewModel
        {
            get
            {
                return this.computeShaderSourceSettingViewModel;
            }
        }

        /// <summary>
        /// シェーダ設定名のエラー表示ビューモデルを取得します。
        /// </summary>
        public ErrorMessageTextBlockViewModel ErrorNameViewModel
        {
            get
            {
                return this.errorNameViewModel;
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="wizardViewModel">ウィザードビューモデルです。</param>
        /// <param name="isAddMode">追加モードかどうかです。</param>
        public ShaderSettingViewModel(WizardViewModel wizardViewModel, bool isAddMode)
            : base(wizardViewModel)
        {
            this.wizardShaderViewModel = wizardViewModel as WizardShaderViewModel;

            if (isAddMode)
            {
                this.Description = ShaderConfigMessage.WizardDescriptionAddShader;
            }
            else
            {
                this.Description = ShaderConfigMessage.WizardDescriptionEditShader;
            }

            this.vertexShaderSourceSettingViewModel =
                new ShaderSourceSettingViewModel(
                    ShaderSourceSettingViewModel.EnumShaderType.Vertex,
                    this.wizardShaderViewModel.FilePath,
                    this.wizardShaderViewModel.IncludeFullPaths);
            this.geometryShaderSourceSettingViewModel =
                new ShaderSourceSettingViewModel(
                    ShaderSourceSettingViewModel.EnumShaderType.Geometry,
                    this.wizardShaderViewModel.FilePath,
                    this.wizardShaderViewModel.IncludeFullPaths);
            this.fragmentShaderSourceSettingViewModel =
                new ShaderSourceSettingViewModel(
                    ShaderSourceSettingViewModel.EnumShaderType.Fragment,
                    this.wizardShaderViewModel.FilePath,
                    this.wizardShaderViewModel.IncludeFullPaths);
            this.computeShaderSourceSettingViewModel =
                new ShaderSourceSettingViewModel(
                    ShaderSourceSettingViewModel.EnumShaderType.Compute,
                    this.wizardShaderViewModel.FilePath,
                    this.wizardShaderViewModel.IncludeFullPaths);

            // 既存の名前と同じ名前にならないように初期名をつける。
            this.name = DefaultShaderName;
            int index = 1;
            while (this.wizardShaderViewModel.ShaderConfigViewModel.ShaderViewModels.Any(x => x.Name == this.name))
            {
                this.name = DefaultShaderName + index;
                ++index;
            }
        }

        /// <summary>
        /// 次ページに進めるかを取得します。
        /// </summary>
        /// <returns>進める場合 true を返します。</returns>
        public override bool CanGoNext()
        {
            return (this.Name != string.Empty && !this.ErrorNameViewModel.IsShown) &&
                (
                ((this.VertexShaderSourceSettingViewModel.ShaderSourceFullPath == string.Empty) &&
                (this.GeometryShaderSourceSettingViewModel.ShaderSourceFullPath == string.Empty) &&
                (this.FragmentShaderSourceSettingViewModel.ShaderSourceFullPath == string.Empty) &&
                (this.ComputeShaderSourceSettingViewModel.ShaderSourceFullPath != string.Empty))
                ||
                ((this.VertexShaderSourceSettingViewModel.ShaderSourceFullPath != string.Empty) &&
                (this.ComputeShaderSourceSettingViewModel.ShaderSourceFullPath == string.Empty))
                );
        }

        /// <summary>
        /// 前ページに進めるかを取得します。
        /// </summary>
        /// <returns>進める場合 true を返します。</returns>
        public override bool CanGoBack()
        {
            return true;
        }

        private bool CheckName(string name)
        {
            bool hasSameName = this.wizardShaderViewModel.ShaderConfigViewModel.ShaderViewModels.Any(x => x.Name == name);
            if (this.wizardShaderViewModel.ShaderViewModel != null &&
                this.wizardShaderViewModel.ShaderViewModel.Name == name)
            {
                hasSameName = false;
            }

            bool hasNoName = string.IsNullOrEmpty(name);

            if (hasSameName)
            {
                ErrorNameViewModel.Message = ShaderConfigMessage.ErrorHasSameShaderSettingName;
                ErrorNameViewModel.IsShown = true;
            }
            else if (hasNoName)
            {
                ErrorNameViewModel.Message = ShaderConfigMessage.ErrorNoShaderSettingName;
                ErrorNameViewModel.IsShown = true;
            }
            else
            {
                ErrorNameViewModel.IsShown = false;
            }

            return !hasSameName && !hasNoName;
        }
    }
}
