﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Collections.Specialized;
    using System.ComponentModel;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows;
    using Opal.ViewModels;
    using ShaderAssistAddons.Modules.ShaderConfig.Commands;

    /// <summary>
    /// 追加・削除ができるリストボックスビューモデルです。
    /// </summary>
    public class EditableListBoxViewModel : ViewModel
    {
        private readonly ObservableCollection<EditableListBoxItemViewModel> listBoxItems = new ObservableCollection<EditableListBoxItemViewModel>();
        private EditableListBoxItemViewModel selectedItem;
        private bool isEditable;
        private bool isEditing;

        /// <summary>
        /// リストボックスが更新された時のデリゲートです。
        /// </summary>
        public delegate void ListCollectionChangedDelegate();

        /// <summary>
        /// アイテム追加時のデリゲートです。
        /// </summary>
        /// <param name="items">追加されるアイテムです。</param>
        public delegate void AddingDelegate(EditableListBoxItemViewModel[] items);

        /// <summary>
        /// リストボックスアイテムを取得します。
        /// </summary>
        public ObservableCollection<EditableListBoxItemViewModel> ListBoxItems
        {
            get
            {
                return this.listBoxItems;
            }
        }

        /// <summary>
        /// 選択アイテムを取得します。
        /// </summary>
        public EditableListBoxItemViewModel SelectedItem
        {
            get
            {
                return this.selectedItem;
            }

            protected set
            {
                if (value != this.selectedItem)
                {
                    this.selectedItem = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// 編集中かどうかを取得します。
        /// </summary>
        public bool IsEditing
        {
            get
            {
                return this.isEditing;
            }

            protected set
            {
                if (value != this.isEditing)
                {
                    this.isEditing = value;
                    this.RaisePropertyChanged();
                    //CanExecuteChangedInvoker.InvalidateRequerySuggested();
                }
            }
        }

        /// <summary>
        /// リストボックスが更新された時のデリゲートを取得設定します。
        /// </summary>
        public ListCollectionChangedDelegate OnListCollectionChanged
        {
            get;
            set;
        }

        /// <summary>
        /// アイテム追加時デリゲートを取得設定します。
        /// </summary>
        public AddingDelegate OnAdding
        {
            get;
            set;
        }

        /// <summary>
        /// 追加実行時アクションを取得設定します。
        /// </summary>
        public Action AddExecute
        {
            get;
            set;
        }

        /// <summary>
        /// アイテム編集前デリゲートを取得設定します。
        /// </summary>
        public EditableListBoxItemViewModel.BeforeEditDelegate ItemBeforeEdit
        {
            get;
            set;
        }

        /// <summary>
        /// アイテム編集後デリゲートを取得設定します。
        /// </summary>
        public EditableListBoxItemViewModel.AfterEditDelegate ItemAfterEdit
        {
            get;
            set;
        }

        /// <summary>
        /// 追加可能かを取得設定します。
        /// </summary>
        public bool CanAdd
        {
            get;
            set;
        }

        /// <summary>
        /// 追加コマンドを取得します。
        /// </summary>
        public DelegateCommand AddCommand
        {
            get
            {
                return new DelegateCommand(this.AddCommandExecute, () => this.CanAdd);
            }
        }

        /// <summary>
        /// 削除コマンドを取得します。
        /// </summary>
        public DelegateCommand RemoveCommand
        {
            get
            {
                return new DelegateCommand(this.RemoveCommandExecute, this.CanRemove);
            }
        }

        /// <summary>
        /// 上へコマンドを取得します。
        /// </summary>
        public DelegateCommand UpCommand
        {
            get
            {
                return new DelegateCommand(this.UpCommandExecute, this.CanUp);
            }
        }

        /// <summary>
        /// 下へコマンドを取得します。
        /// </summary>
        public DelegateCommand DownCommand
        {
            get
            {
                return new DelegateCommand(this.DownCommandExecute, this.CanDown);
            }
        }

        /// <summary>
        /// マウス Up 時コマンドを取得します。
        /// </summary>
        public DelegateCommand MouseUpCommand
        {
            get
            {
                return new DelegateCommand(this.MouseUpExecute);
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public EditableListBoxViewModel(bool isEditable)
        {
            this.isEditable = isEditable;
            this.listBoxItems.CollectionChanged += this.ListCollectionChangedInternal;
            this.CanAdd = true;
        }

        /// <summary>
        /// リストに追加します。
        /// </summary>
        /// <param name="item">追加するアイテムです。</param>
        public void AddToList(params string[] item)
        {
            if (item == null || item.Length == 0)
            {
                return;
            }

            var itemViewModels = new EditableListBoxItemViewModel[item.Length];
            for (int i = 0; i < itemViewModels.Length; ++i)
            {
                itemViewModels[i] = new EditableListBoxItemViewModel(this.isEditable)
                {
                    Label = item[i],
                    BeforeEdit = this.ItemBeforeEdit,
                    AfterEdit = this.ItemAfterEdit
                };
            }

            if (this.OnAdding != null)
            {
                this.OnAdding.Invoke(itemViewModels);
            }

            foreach (EditableListBoxItemViewModel itemViewModel in itemViewModels)
            {
                if (!string.IsNullOrEmpty(itemViewModel.Label))
                {
                    this.ListBoxItems.Add(itemViewModel);
                }
            }
        }

        /// <summary>
        /// リストをクリアします。
        /// </summary>
        public void ClearItems()
        {
            this.ListBoxItems.Clear();
        }

        /*public void CheckList()
        {
            if (OnAdding != null)
            {
                OnAdding.Invoke(ListBoxItems.ToArray());
            }
        }*/

        private void AddCommandExecute()
        {
            if (this.AddExecute != null)
            {
                this.AddExecute.Invoke();
            }
        }

        private void RemoveCommandExecute()
        {
            this.ListBoxItems.Remove(SelectedItem);
            this.SelectedItem = null;
        }

        private bool CanRemove()
        {
            return this.SelectedItem != null;
        }

        private void UpCommandExecute()
        {
            EditableListBoxItemViewModel swap = this.SelectedItem;
            int index = this.ListBoxItems.IndexOf(swap);
            this.ListBoxItems.Remove(swap);
            this.ListBoxItems.Insert(index - 1, swap);
            this.SelectedItem = swap;
        }

        private bool CanUp()
        {
            return this.SelectedItem != null &&
                   this.ListBoxItems.IndexOf(this.SelectedItem) > 0;
        }

        private void DownCommandExecute()
        {
            EditableListBoxItemViewModel swap = this.SelectedItem;
            int index = this.ListBoxItems.IndexOf(swap);
            this.ListBoxItems.Remove(swap);
            this.ListBoxItems.Insert(index + 1, swap);
            this.SelectedItem = swap;
        }

        private bool CanDown()
        {
            return this.SelectedItem != null &&
                   this.ListBoxItems.IndexOf(this.SelectedItem) < this.ListBoxItems.Count - 1;
        }

        private void MouseUpExecute()
        {
            // ListBox の空白をクリックすると選択を解除する。
            bool reset = !this.ListBoxItems.Any(x => x.IsMouseDownProcessed);
            foreach (EditableListBoxItemViewModel item in this.ListBoxItems)
            {
                item.IsMouseDownProcessed = false;
                if (reset)
                {
                    item.IsSelected = false;
                }
            }

            this.IsEditing = this.ListBoxItems.Any(x => x.TextBoxVisibility == Visibility.Visible);
        }

        private void ListCollectionChangedInternal(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (this.OnListCollectionChanged != null)
            {
                this.OnListCollectionChanged.Invoke();
            }

            //CanExecuteChangedInvoker.InvalidateRequerySuggested();
        }
    }
}
