﻿namespace ShaderAssistAddons.Modules.ShaderConfig.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Entities;
    using Opal.ViewModels;

    /// <summary>
    /// シェーダバリエーションのビューモデルクラスです。
    /// </summary>
    public class VariationViewModel : ViewModel
    {
        private WeakReference<Variation> variationData;
        private OptionVar optionVar;
        private bool isSelected = false;
        private string id = string.Empty;
        private string shaderConfigChoice = string.Empty;
        private string shaderConfigDefault = string.Empty;
        private bool branch = false;
        private bool isOutputable = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="dataModel">データモデルです。</param>
        /// <param name="optionVar">バリエーションの元になるオプション変数です。</param>
        public VariationViewModel(Variation dataModel, OptionVar optionVar)
        {
            Debug.Assert(dataModel != null);
            Debug.Assert(optionVar != null);

            this.variationData = new WeakReference<Variation>(dataModel);
            dataModel.PropertyChanged += this.VariationPropertyChanged;

            this.optionVar = optionVar;
            this.SetVariationData(dataModel);
        }

        /// <summary>
        /// バリエーションのデータを取得します。
        /// </summary>
        public Variation VariationData
        {
            get
            {
                Variation data;
                if (this.variationData != null && this.variationData.TryGetTarget(out data))
                {
                    return data;
                }

                return null;
            }
        }

        /// <summary>
        /// このバリエーションが選択されているかを取得設定します。
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (value != this.isSelected)
                {
                    this.isSelected = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// 識別子を取得設定します。
        /// </summary>
        public string Id
        {
            get
            {
                return this.id;
            }

            set
            {
                if (value != this.id)
                {
                    this.id = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// シェーダ設定の choice を取得設定します。
        /// </summary>
        public string ShaderConfigChoice
        {
            get
            {
                return this.shaderConfigChoice;
            }

            set
            {
                if (value != this.shaderConfigChoice)
                {
                    this.shaderConfigChoice = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// シェーダ定義の choice を取得します。
        /// </summary>
        public string ShaderDefinitionChoice
        {
            get
            {
                return this.optionVar.Choice;
            }
        }

        /// <summary>
        /// シェーダ設定の default を取得設定します。
        /// </summary>
        public string ShaderConfigDefault
        {
            get
            {
                return this.shaderConfigDefault;
            }

            set
            {
                if (value != this.shaderConfigDefault)
                {
                    this.shaderConfigDefault = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// シェーダ定義の default を取得します。
        /// </summary>
        public string ShaderDefinitionDefault
        {
            get
            {
                return this.optionVar.Default;
            }
        }

        /// <summary>
        /// 静的分岐かどうかを取得設定します。
        /// </summary>
        public bool Branch
        {
            get
            {
                return this.branch;
            }

            set
            {
                if (value != this.branch)
                {
                    this.branch = value;
                    this.RaisePropertyChanged();
                }
            }
        }

        /// <summary>
        /// 中間ファイルに出力するかどうかを取得設定します。
        /// </summary>
        public bool IsOutputable
        {
            get
            {
                return this.isOutputable;
            }

            set
            {
                if (value != this.isOutputable)
                {
                    this.isOutputable = value;
                    this.IsOutputableChanged = true;
                    this.RaisePropertyChanged();
                    this.RaisePropertyChanged(() => this.IsOutputableChanged);
                }
            }
        }

        /// <summary>
        /// 出力が変化したかを取得します。
        /// </summary>
        public bool IsOutputableChanged
        {
            get;
            private set;
        }

        /// <summary>
        /// 出力が変化したかどうかをリセットします。
        /// </summary>
        public void ResetIsOutputableChanged()
        {
            this.IsOutputableChanged = false;
        }

        private void SetVariationData(Variation variation)
        {
            if (variation != null)
            {
                this.Id = variation.Id;
                this.ShaderConfigChoice = variation.Choice;
                this.ShaderConfigDefault = variation.Default;
                this.Branch = variation.Branch;
                this.IsOutputable = variation.IsOutputable;
            }

            this.ResetIsOutputableChanged();
        }

        private void VariationPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            Variation variation = this.VariationData;
            if (variation != null)
            {
                this.SetVariationData(variation);
            }
        }
    }
}
