﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Windows.Forms;
using System.Diagnostics;
using App;
using App.Controls;

namespace Viewer
{
    public sealed partial class LoadProgressDialog : ModalDialog
    {
        private Action	CancelButtonClick{ get; set; }
        private bool	IsHideOnCancelButtonClick{ get; set; }

        /// <summary>
        /// この変数は TheApp.MainThread のみでアクセス可能。
        /// それ以外のスレッドからのアクセスを想定した実装は行っていない。
        /// </summary>
        private static App.Utility.WaitCursor cursor;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public LoadProgressDialog()
        {
            InitializeComponent();
        }

        /// <summary>
        /// ダイアログが表示中か？
        /// </summary>
        public static bool IsDialogVisible
        {
            get
            {
                Debug.Assert(System.Threading.Thread.CurrentThread == TheApp.MainThread);
                return dialog_ != null && dialog_.Visible;
            }
        }

        private void btnCancel_Click(object sender, EventArgs e)
        {
            if (CancelButtonClick != null)
            {
                CancelButtonClick();
            }
        }

        /// <summary>
        /// この変数は TheApp.MainThread のみでアクセス可能。
        /// それ以外のスレッドからのアクセスを想定した実装は行っていない。
        /// </summary>
        private static LoadProgressDialog dialog_ = null;

        public class Arg
        {
            public string	Message				{ get; set; }
            public int		ParentLeft			{ get; set; }
            public int		ParentTop			{ get; set; }
            public int		ParentWidth			{ get; set; }
            public int		ParentHeight		{ get; set; }
            public Action	CancelButtonClick	{ get; set; }
            public bool		IsHideOnCancelButtonClick	{ get; set; }
        }

        private static void Show(string message, Action cancelButtonClick, bool isHideOnCancelButtonClick)
        {
            Message = message;
            Action showAction = () =>
            {
                if (dialog_ != null)
                {
                    return;
                }

                // dialog_ の Dispose() はここで行う。
                // LoadProgressDialog.Hide() では Close() のみを行う。
                // 詳細は LoadProgressDialog.Hide() を参照。
                using (var dialog = dialog_ = new LoadProgressDialog()
                {
                    CancelButtonClick = cancelButtonClick,
                    IsHideOnCancelButtonClick = isHideOnCancelButtonClick,
                })
                {
                    RemoveCloseButton(dialog_);
                    dialog_.StartPosition = FormStartPosition.CenterParent;
                    dialog_.lblMessage.Text = Message;
                    dialog_.timerConnectionCheck.Start();
                    if (cursor != null)
                    {
                        cursor.Dispose();
                        cursor = null;
                    }
                    dialog_.ShowDialog(TheApp.MainFrame);
                }
            };


            Action action = () =>
                {
                    cursor = new App.Utility.WaitCursor();
                    DebugConsole.WriteLine("Create WaitCursor");
                    TheApp.MainFrame.LoadProgreessTimerAction = showAction;

                    // 表示待ちのタイマーを開始
                    // 注意: 他のモーダルダイアログが出ている間はダイアログは表示されない。
                    TheApp.MainFrame.LoadProgressTimer.Start();
                };
            if (TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
            {
                TheApp.MainFrame.BeginInvoke(action,null);
            }
        }

        private static string message;
        private static readonly object messageLock = new object();

        /// <summary>
        /// dialog_ に設定する文字列を取得または設定する。
        /// dialog_ への設定は TheApp.MainThread で非同期実行されるので、ここで取得した値が dialog_ に設定済みになっている保障はない。
        /// また、設定した値が dialog_ に即時設定される保証もない。
        /// </summary>
        private static string Message
        {
            get
            {
                // message に非同期アクセスするための排他制御。
                lock (messageLock)
                {
                    return message;
                }
            }
            set
            {
                // message に非同期アクセスするための排他制御。
                // set() 全体を lock しない方がよい。
                // 現在の実装においては全体を lock しても lock 期間が僅かに延びるだけで実害はないが、
                // BeginInvoke().AsyncWaitHandle.WaitOne() のような同期処理に変更してしまうと、
                // setMessage() での Message 取得でデッドロックする可能性があるため。
                lock (messageLock)
                {
                    message = value;
                }

                // 非同期で setMessage を呼び出すことで、実行時の最新 Message が dialog_ に反映される。
                Action setMessage = () =>
                {
                    if (dialog_ != null)
                    {
                        dialog_.lblMessage.Text = Message;
                    }
                };
                if (TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
                {
                    TheApp.MainFrame.BeginInvoke(setMessage, null);
                }
            }
        }

        public static new void Hide()
        {
            Action action = () =>
                {
                    TheApp.MainFrame.LoadProgressTimer.Stop();
                    if (dialog_ != null)
                    {
                        // ここで dialog_ を Dispose() してはならない。
                        // LoadProgressDialog.Show() 内で dialog_ を ShowDialog() しており、
                        // ここで Dispose() すると Dispose 後のハンドルに ShowDialog() 側からアクセスされることがあるため。
                        // Dispose() は LoadProgressDialog.Show() の ShowDialog() 後に行うこと。
                        dialog_.Close();
                        dialog_ = null;
                    }
                    if (cursor != null)
                    {
                        cursor.Dispose();
                        cursor = null;
                    }
                    DebugConsole.WriteLine("Dispose WaitCursor");
                };
            if (TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
            {
                TheApp.MainFrame.BeginInvoke(action, null);
            }
        }

        // http://www.developmentnow.com/g/29_2003_12_0_0_118087/WinForm-Question.htm
        private static void RemoveCloseButton(Form frmForm)
        {
            var hMenu = App.Win32.NativeMethods.GetSystemMenu(frmForm.Handle, 0);
            if (hMenu != IntPtr.Zero)
            {
                var n = App.Win32.NativeMethods.GetMenuItemCount(hMenu);
                if (n > 0)
                {
                    App.Win32.NativeMethods.RemoveMenu(hMenu, n - 1, App.Win32.NativeMethods.MF_BYPOSITION);
                    App.Win32.NativeMethods.RemoveMenu(hMenu, n - 2, App.Win32.NativeMethods.MF_BYPOSITION);
                }

                App.Win32.NativeMethods.DrawMenuBar(frmForm.Handle);
            }
        }

        public class DialogBlock : IDisposable
        {
            private static int count = 0;
            private static readonly object syncObject = new object();

            public DialogBlock(string message, Action cancelButtonClick, bool isHideOnCancelButtonClick)
            {
                // DialogBlock.count 操作と LoadProgressDialog.Show() を同時実行するための排他制御。
                lock (syncObject)
                {
                    count++;
                    if (count > 1)
                    {
                        return;
                    }

                    Action onCancel = () =>
                    {
                        cancelButtonClick();
                    };

                    LoadProgressDialog.Show(message, onCancel, isHideOnCancelButtonClick);
                }
            }

            public string Message
            {
                get
                {
                    // LoadProgressDialog.Message で排他制御されているため、ここでの排他制御は不要。
                    return LoadProgressDialog.Message;
                }
                set
                {
                    // LoadProgressDialog.Message で排他制御されているため、ここでの排他制御は不要。
                    LoadProgressDialog.Message = value;
                }
            }

            public void Dispose()
            {
                // DialogBlock.count 操作と LoadProgressDialog.Hide() を同時実行するための排他制御。
                lock (syncObject)
                {
                    count--;
                    if (count == 0)
                    {
                        LoadProgressDialog.Hide();
                    }
                }
            }
        }

        private void timerConnectionCheck_Tick(object sender, EventArgs e)
        {
            if (!Viewer.Manager.Instance.IsConnected)
            {
                timerConnectionCheck.Stop();
                App.TheApp.MainFrame.ConnectToHio(false);
            }
        }
    }
}
