﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Utility
{
    public static class ShaderTypeUtility
    {
        public static bool IsMaterialShader(this shading_modelType shadingModel)
        {
            Debug.Assert(shadingModel != null);
            return shadingModel.material_shader;
        }

        public static IEnumerable<attrib_varType> Attributes(this shading_modelType definition)
        {
            if (definition == null || definition.attrib_var_array == null)
            {
                return Enumerable.Empty<attrib_varType>();
            }

            return definition.attrib_var_array.attrib_var;
        }

        public static IEnumerable<option_varType> Options(this shading_modelType definition)
        {
            if (definition == null || definition.option_var_array == null)
            {
                return Enumerable.Empty<option_varType>();
            }

            return definition.option_var_array.option_var;
        }

        public static IEnumerable<interleaveType> Interleave(this shading_modelType definition)
        {
            if (definition == null || definition.interleave_array == null)
            {
                return Enumerable.Empty<interleaveType>();
            }

            return definition.interleave_array.interleave;
        }

        public static IEnumerable<uniform_varType> MaterialUniforms(this shading_modelType definition)
        {
            if (definition == null || definition.block_var_array == null)
            {
                return Enumerable.Empty<uniform_varType>();
            }

            return definition.block_var_array.block_var.Where(x => x.type == block_var_typeType.material && x.uniform_var_array != null).
                SelectMany(x => x.uniform_var_array.uniform_var);
        }

        public static IEnumerable<uniform_varType> Uniforms(this shading_modelType definition)
        {
            if (definition == null || definition.block_var_array == null)
            {
                return Enumerable.Empty<uniform_varType>();
            }

            return definition.block_var_array.block_var.Where(x => x.uniform_var_array != null).
                SelectMany(x => x.uniform_var_array.uniform_var);
        }

        public static IEnumerable<block_varType> UniformBlocks(this shading_modelType definition)
        {
            if (definition == null || definition.block_var_array == null)
            {
                return Enumerable.Empty<block_varType>();
            }

            return definition.block_var_array.block_var;
        }

        public static IEnumerable<uniform_varType> UniformsInBlock(this block_varType block)
        {
            if (block.uniform_var_array == null)
            {
                return Enumerable.Empty<uniform_varType>();
            }

            return block.uniform_var_array.uniform_var;
        }

        public static IEnumerable<buffer_varType> Buffers(this shading_modelType definition)
        {
            if (definition == null || definition.shader_storage_block_var_array == null)
            {
                return Enumerable.Empty<buffer_varType>();
            }

            return definition.shader_storage_block_var_array.shader_storage_block_var.Where(x => x.buffer_var_array != null).
                SelectMany(x => x.buffer_var_array.buffer_var);
        }

        public static IEnumerable<shader_storage_block_varType> BufferBlocks(this shading_modelType definition)
        {
            if (definition == null || definition.shader_storage_block_var_array == null)
            {
                return Enumerable.Empty<shader_storage_block_varType>();
            }

            return definition.shader_storage_block_var_array.shader_storage_block_var;
        }

        public static IEnumerable<buffer_varType> BuffersInBlock(this shader_storage_block_varType block)
        {
            if (block.buffer_var_array == null)
            {
                return Enumerable.Empty<buffer_varType>();
            }

            return block.buffer_var_array.buffer_var;
        }

        public static IEnumerable<sampler_varType> Samplers(this shading_modelType definition)
        {
            if (definition == null || definition.sampler_var_array == null)
            {
                return Enumerable.Empty<sampler_varType>();
            }

            return definition.sampler_var_array.sampler_var;
        }

        public static IEnumerable<render_info_slotType> RenderInfoSlots(this shading_modelType definition)
        {
            if (definition == null || definition.render_info_slot_array == null)
            {
                return Enumerable.Empty<render_info_slotType>();
            }

            return definition.render_info_slot_array.render_info_slot;
        }

        public static string Label(this sampler_varType sampler)
        {
            return sampler.ui_label == null ? null : sampler.ui_label.value;
        }

        public static string Label(this uniform_varType uniform)
        {
            return uniform.ui_label == null ? null : uniform.ui_label.value;
        }

        public static string Label(this option_varType option)
        {
            return option.ui_label == null ? null : option.ui_label.value;
        }

        public static string Label(this attrib_varType attrib)
        {
            return attrib.ui_label == null ? null : attrib.ui_label.value;
        }

        public static string Label(this render_info_slotType ris)
        {
            return ris.ui_label == null ? null : ris.ui_label.value;
        }

        public static string Label(this groupType group)
        {
            return group.ui_label == null ? null : group.ui_label.value;
        }

        public static string Label(this pageType page)
        {
            return page.ui_label == null ? null : page.ui_label.value;
        }

        public static string Comment(this sampler_varType sampler)
        {
            return sampler.ui_comment == null ? string.Empty : sampler.ui_comment.value;
        }

        public static string Comment(this uniform_varType uniform)
        {
            return uniform.ui_comment == null ? string.Empty : uniform.ui_comment.value;
        }

        public static string Comment(this option_varType option)
        {
            return option.ui_comment == null ? string.Empty : option.ui_comment.value;
        }

        public static string Comment(this attrib_varType attrib)
        {
            return attrib.ui_comment == null ? string.Empty : attrib.ui_comment.value;
        }

        public static string Comment(this render_info_slotType ris)
        {
            return ris.ui_comment == null ? string.Empty : ris.ui_comment.value;
        }

        public static string Comment(this groupType group)
        {
            return group.ui_comment == null ? string.Empty : group.ui_comment.value;
        }

        public static string Comment(this pageType page)
        {
            return page.ui_comment == null ? string.Empty : page.ui_comment.value;
        }

        public static bool Editable(this attrib_varType attrib)
        {
            return attrib.ui_editable == null ? true : attrib.ui_editable.value;
        }

        public static string Group(this sampler_varType uniform)
        {
            return uniform.ui_group == null ? string.Empty : uniform.ui_group.group_name;
        }

        public static string Group(this uniform_varType uniform)
        {
            return uniform.ui_group == null ? string.Empty : uniform.ui_group.group_name;
        }

        public static string Group(this option_varType option)
        {
            return option.ui_group == null ? string.Empty : option.ui_group.group_name;
        }

        public static string Group(this render_info_slotType slot)
        {
            return slot.ui_group == null ? string.Empty : slot.ui_group.group_name;
        }

        public static string Group(this attrib_varType attrib)
        {
            return attrib.ui_group == null ? DefaultAttribGroup.name : attrib.ui_group.group_name;
        }

        public static string Group(this groupType group)
        {
            return group.ui_group == null ? string.Empty : group.ui_group.group_name;
        }

        private static groupType DefaultAttribGroup = new groupType()
        {
            name = ShaderTypeUtility.AttribAssignGroupName(),
            ui_label = new ui_labelType() { value = ShaderTypeUtility.AttribAssignGroupLabel() },
            ui_order = new ui_orderType() { value = ShaderTypeUtility.AttribAssignGroupOrder() },
            page_name = "",
        };

        public static string AttribAssignGroupName()
        {
            // グループ名
            // 空白が入っているので他と重複しない。
            return "attrib_assign group";
        }

        public static string AttribAssignGroupLabel()
        {
            return App.res.Strings.AttribAssignGroup;
        }

        public static int AttribAssignGroupOrder()
        {
            return int.MinValue;
        }

        public static string RemainsGroupLabel()
        {
            return App.res.Strings.RemainsGroupLabel;
        }

        /// <summary>
        /// グループに属さないものの順番
        /// </summary>
        public static int RemainsGroupOrder()
        {
            return int.MaxValue;
        }

        public static int Order(this attrib_varType attrib_assign)
        {
            return attrib_assign.ui_order == null ? 0 : attrib_assign.ui_order.value;
        }

        public static int Order(this sampler_varType sampler)
        {
            return sampler.ui_order == null ? 0 : sampler.ui_order.value;
        }

        public static int Order(this uniform_varType uniform)
        {
            return uniform.ui_order == null ? 0 : uniform.ui_order.value;
        }

        public static int Order(this option_varType option)
        {
            return option.ui_order == null ? 0 : option.ui_order.value;
        }

        public static int Order(this render_info_slotType slot)
        {
            return slot.ui_order == null ? 0 : slot.ui_order.value;
        }

        public static int Order(this groupType group)
        {
            return group.ui_order == null ? 0 : group.ui_order.value;
        }

        public static int Order(this pageType page)
        {
            return page.ui_order == null ? 0 : page.ui_order.value;
        }

        public static ui_item_valueType Item(this uniform_varType uniform)
        {
            return uniform.ui_item == null ? ui_item_valueType.auto : uniform.ui_item.value;
        }

        public static ui_item_valueType Item(this render_info_slotType slot)
        {
            return slot.ui_item == null ? ui_item_valueType.auto : slot.ui_item.value;
        }

        public static ui_visible_valueType Visible(this uniform_varType uniform)
        {
            return uniform.ui_visible == null ? ui_visible_valueType.@true : uniform.ui_visible.value;
        }

        public static ui_visible_valueType Visible(this sampler_varType sampler)
        {
            return sampler.ui_visible == null ? ui_visible_valueType.@true : sampler.ui_visible.value;
        }

        public static ui_visible_valueType Visible(this option_varType option)
        {
            return option.ui_visible == null ? ui_visible_valueType.@true : option.ui_visible.value;
        }

        public static ui_visible_valueType Visible(this render_info_slotType slot)
        {
            return slot.ui_visible == null ? ui_visible_valueType.@true : slot.ui_visible.value;
        }

        private static bool IsUiVisible(ui_visibleType uiVisible)
        {
            if (uiVisible != null)
            {
                switch (uiVisible.value)
                {
                    case ui_visible_valueType.@true:
                    case ui_visible_valueType.param_only:
                        return true;
                    case ui_visible_valueType.@false:
                    case ui_visible_valueType.anim_only:
                        return false;
                }
            }

            return true;
        }

        public static bool SamplerVisible(this sampler_varType sampler)
        {
            return IsUiVisible(sampler.ui_visible);
        }

        public static bool OptionVisible(this option_varType option)
        {
            return IsUiVisible(option.ui_visible);
        }

        public static bool RenderInfoVisible(this render_info_slotType slot)
        {
            return IsUiVisible(slot.ui_visible);
        }

        /// <summary>
        /// パラメーターを表示するかどうか
        /// </summary>
        public static bool IsParamVisible(this uniform_varType uniform)
        {
            return IsUiVisible(uniform.ui_visible);
        }

        /// <summary>
        /// アニメーションを表示するかどうか
        /// </summary>
        public static bool IsAnimVisible(this uniform_varType uniform)
        {
            if (uniform.ui_visible != null)
            {
                switch (uniform.ui_visible.value)
                {
                    case ui_visible_valueType.@true:
                    case ui_visible_valueType.anim_only:
                        return true;
                    case ui_visible_valueType.@false:
                    case ui_visible_valueType.param_only:
                        return false;

                }
            }

            return true;
        }



        public static IEnumerable<interleaveType> Interleaves(this shading_modelType shadingModel)
        {
            return shadingModel.interleave_array == null ? Enumerable.Empty<interleaveType>() : shadingModel.interleave_array.interleave;
        }

        public static IEnumerable<groupType> Groups(this shading_modelType shadingModel)
        {
            return Enumerable.Repeat(DefaultAttribGroup, 1).Concat(GroupsExceptDefaultAttribGroup(shadingModel));
        }

        public static IEnumerable<groupType> GroupsExceptDefaultAttribGroup(this shading_modelType shadingModel)
        {
            return shadingModel.group_array == null ? Enumerable.Empty<groupType>() : shadingModel.group_array.group;
        }

        public static IEnumerable<pageType> Pages(this shading_modelType shadingModel)
        {
            return shadingModel.page_array == null ? Enumerable.Empty<pageType>() : shadingModel.page_array.page;
        }

        public static string Default(this uniform_varType uniform)
        {
            // 3DEditor で生成した場合はnull もあり
            int count = Dimension(uniform.type);
            int count1 = 0;
            if (uniform.Item() == ui_item_valueType.color)
            {
                count1 = 4;
            }
            else if (IsSrt2d(uniform.type))
            {
                count1 = 2;
            }
            else if (uniform.type == shader_param_typeType.srt3d)
            {
                count1 = 3;
            }

            var array = new float[count];
            for (int i = 0; i != count; i++)
            {
                if (uniform.@default != null && i < uniform.@default.Length)
                {
                    array[i] = uniform.@default[i];
                }
                else if (i < count1)
                {
                    array[i] = 1;
                }
                else
                {
                    array[i] = 0;
                }
            }

            return IfUtility.MakeArrayString(array);
        }

        public static string Default(this render_info_slotType slot)
        {
            return slot.@default;
        }
        /// <summary>
        /// 最小値
        /// </summary>
        public static float[] Min(this uniform_varType uniform)
        {
            if (uniform.ui_min != null)
            {
                return Enumerable.Repeat(uniform.ui_min.value, Dimension(uniform.type)).ToArray();
            }

            // 設定されていない場合はnull
            return null;
        }

        /// <summary>
        /// 最小値
        /// </summary>
        public static float[] Max(this uniform_varType uniform)
        {
            if (uniform.ui_max != null)
            {
                return Enumerable.Repeat(uniform.ui_max.value, Dimension(uniform.type)).ToArray();
            }

            // 設定されていない場合はnull
            return null;
        }

        /// <summary>
        /// 次元
        /// </summary>
        public static int Dimension(shader_param_typeType type)
        {
            int count = 0;
            switch (type)
            {
                case shader_param_typeType.@bool: count = 1; break;
                case shader_param_typeType.bool2: count = 2; break;
                case shader_param_typeType.bool3: count = 3; break;
                case shader_param_typeType.bool4: count = 4; break;
                case shader_param_typeType.@int: count = 1; break;
                case shader_param_typeType.int2: count = 2; break;
                case shader_param_typeType.int3: count = 3; break;
                case shader_param_typeType.int4: count = 4; break;
                case shader_param_typeType.@uint: count = 1; break;
                case shader_param_typeType.uint2: count = 2; break;
                case shader_param_typeType.uint3: count = 3; break;
                case shader_param_typeType.uint4: count = 4; break;
                case shader_param_typeType.@float: count = 1; break;
                case shader_param_typeType.float2: count = 2; break;
                case shader_param_typeType.float3: count = 3; break;
                case shader_param_typeType.float4: count = 4; break;
                case shader_param_typeType.float2x2: count = 2 * 2; break;
                case shader_param_typeType.float2x3: count = 2 * 3; break;
                case shader_param_typeType.float2x4: count = 2 * 4; break;
                case shader_param_typeType.float3x2: count = 3 * 2; break;
                case shader_param_typeType.float3x3: count = 3 * 3; break;
                case shader_param_typeType.float3x4: count = 3 * 4; break;
                case shader_param_typeType.float4x2: count = 4 * 2; break;
                case shader_param_typeType.float4x3: count = 4 * 3; break;
                case shader_param_typeType.float4x4: count = 4 * 4; break;
                case shader_param_typeType.srt2d:
                    count = 5;
                    break;
                case shader_param_typeType.texsrt:
                    count = 6;
                    break;
                case shader_param_typeType.srt3d:
                    count = 9;
                    break;
                default: Debug.Assert(false); break;
            }
            return count;
        }

        /// <summary>
        /// 2次元テクスチャ座標かどうか
        /// </summary>
        public static bool IsSrt2d(shader_param_typeType type)
        {
            switch (type)
            {
                case shader_param_typeType.srt2d:
                case shader_param_typeType.texsrt:
                    return true;
            }
            return false;
        }

        public enum ShaderParamTypeKind
        {
            Scalar,
            Vector,
            Matrix,
            TextureSrt
        }

        public static ShaderParamTypeKind ShaderParamTypeKindFromType(shader_param_typeType type)
        {
            switch (type)
            {
                case shader_param_typeType.@bool:
                case shader_param_typeType.@int:
                case shader_param_typeType.@uint:
                case shader_param_typeType.@float:
                    return ShaderParamTypeKind.Scalar;

                case shader_param_typeType.bool2:
                case shader_param_typeType.bool3:
                case shader_param_typeType.bool4:
                case shader_param_typeType.int2:
                case shader_param_typeType.int3:
                case shader_param_typeType.int4:
                case shader_param_typeType.uint2:
                case shader_param_typeType.uint3:
                case shader_param_typeType.uint4:
                case shader_param_typeType.float2:
                case shader_param_typeType.float3:
                case shader_param_typeType.float4:
                    return ShaderParamTypeKind.Vector;

                case shader_param_typeType.float2x2:
                case shader_param_typeType.float2x3:
                case shader_param_typeType.float2x4:
                case shader_param_typeType.float3x2:
                case shader_param_typeType.float3x3:
                case shader_param_typeType.float3x4:
                case shader_param_typeType.float4x2:
                case shader_param_typeType.float4x3:
                case shader_param_typeType.float4x4:
                    return ShaderParamTypeKind.Matrix;

                case shader_param_typeType.srt2d:
                case shader_param_typeType.texsrt:
                case shader_param_typeType.srt3d:
                    return ShaderParamTypeKind.TextureSrt;

                default:
                    Debug.Assert(false);
                    return ShaderParamTypeKind.Vector;
            }
        }

        public enum ParamPrimitiveTypeKind
        {
            @bool,
            @int,
            @uint,
            @float,
            //
            Unknown,
        }

        public static ParamPrimitiveTypeKind ParamPrimitiveTypeFromType(shader_param_typeType type)
        {
                switch (type)
                {
                    case shader_param_typeType.@bool:
                    case shader_param_typeType.bool2:
                    case shader_param_typeType.bool3:
                    case shader_param_typeType.bool4:
                        return ParamPrimitiveTypeKind.@bool;

                    case shader_param_typeType.@int:
                    case shader_param_typeType.int2:
                    case shader_param_typeType.int3:
                    case shader_param_typeType.int4:
                        return ParamPrimitiveTypeKind.@int;

                    case shader_param_typeType.@uint:
                    case shader_param_typeType.uint2:
                    case shader_param_typeType.uint3:
                    case shader_param_typeType.uint4:
                        return ParamPrimitiveTypeKind.@uint;

                    case shader_param_typeType.@float:
                    case shader_param_typeType.float2:
                    case shader_param_typeType.float3:
                    case shader_param_typeType.float4:
                    case shader_param_typeType.float2x2:
                    case shader_param_typeType.float2x3:
                    case shader_param_typeType.float2x4:
                    case shader_param_typeType.float3x2:
                    case shader_param_typeType.float3x3:
                    case shader_param_typeType.float3x4:
                    case shader_param_typeType.float4x2:
                    case shader_param_typeType.float4x3:
                    case shader_param_typeType.float4x4:
                    case shader_param_typeType.srt2d:
                    case shader_param_typeType.srt3d:
                    case shader_param_typeType.texsrt:
                        return ParamPrimitiveTypeKind.@float;

                    default:
                        Debug.Assert(false);
                        return ParamPrimitiveTypeKind.Unknown;
                }
        }

        /// <summary>
        /// sampler_varType の比較
        /// </summary>
        public class SamplerVarComparer : IEqualityComparer<sampler_varType>
        {
            public static bool SamplerVarEquals(sampler_varType x, sampler_varType y)
            {
                return x.id == y.id &&
                x.hint == y.hint && // 必要?
                x.type == y.type &&
                (x.ui_label == y.ui_label || (x.ui_label != null && y.ui_label != null && x.ui_label.value == y.ui_label.value)) &&
                (x.ui_comment == y.ui_comment || (x.ui_comment != null && y.ui_comment != null && x.ui_comment.value == y.ui_comment.value)) &&
                (x.ui_order == y.ui_order || (x.ui_order != null && y.ui_order != null && x.ui_order.value == y.ui_order.value)) &&
                (x.ui_visible == y.ui_visible || (x.ui_visible != null && y.ui_visible != null && x.ui_visible.value == y.ui_visible.value)) &&
                (x.ui_group == y.ui_group || (x.ui_group != null && y.ui_group != null && x.ui_group.group_name == y.ui_group.group_name));
            }

            bool IEqualityComparer<sampler_varType>.Equals(sampler_varType x, sampler_varType y)
            {
                return SamplerVarEquals(x, y);
            }

            int IEqualityComparer<sampler_varType>.GetHashCode(sampler_varType x)
            {
                int code = 0;
                code = MixHash(code, x.id);
                code = MixHash(code, x.hint);
                code = MixValueHash(code, x.type);
                code = MixHash(code, x.ui_label == null ? null : x.ui_label.value);
                code = MixHash(code, x.ui_comment == null ? null : x.ui_comment.value);
                code = MixValueHash(code, x.ui_order == null ? -1 : x.ui_order.value);
                code = MixValueHash(code, x.ui_visible == null ? -1 : Convert.ToInt32(x.ui_visible.value));
                code = MixHash(code, x.ui_group == null ? null : x.ui_group.group_name);
                return code;
            }

            public static SamplerVarComparer TheComparer = new SamplerVarComparer();
        }

        /// <summary>
        /// sampler_varType の比較
        /// </summary>
        public class SamplerVarIdTypeComparer : IEqualityComparer<sampler_varType>
        {
            public static bool SamplerVarEquals(sampler_varType x, sampler_varType y)
            {
                return x.id == y.id &&
                x.type == y.type;
            }

            bool IEqualityComparer<sampler_varType>.Equals(sampler_varType x, sampler_varType y)
            {
                return SamplerVarEquals(x, y);
            }

            int IEqualityComparer<sampler_varType>.GetHashCode(sampler_varType x)
            {
                int code = 0;
                code = MixHash(code, x.id);
                code = MixValueHash(code, x.type);
                return code;
            }

            public static SamplerVarIdTypeComparer TheComparer = new SamplerVarIdTypeComparer();
        }

        public class UniformUI
        {
            public uniform_varType uniform { get; set; }
            public bool colorControl { get; set; }
        }

        public class UniformUIComparer : IEqualityComparer<UniformUI>
        {
            public static bool UniformUIEquals(UniformUI x, UniformUI y)
            {
                return UniformComparer.UniformEquals(x.uniform, y.uniform) &&
                    x.colorControl == y.colorControl;
            }

            public bool Equals(UniformUI x, UniformUI y)
            {
                return UniformUIEquals(x, y);
            }

            public int GetHashCode(UniformUI x)
            {
                int code = 0;
                code = MixValueHash(code, (new UniformComparer()).GetHashCode(x.uniform));
                code = MixValueHash(code, x.colorControl);
                return code;
            }
        }

        /// <summary>
        /// uniform_varType の比較
        /// </summary>
        public class UniformComparer : IEqualityComparer<uniform_varType>
        {
            public static bool UniformEquals(uniform_varType x, uniform_varType y)
            {
                return x.id == y.id &&
                    x.hint == y.hint &&
                    x.type == y.type &&
                    x.@default == y.@default &&
                    (x.ui_label == y.ui_label || (x.ui_label != null && y.ui_label != null && x.ui_label.value == y.ui_label.value)) &&
                    (x.ui_comment == y.ui_comment || (x.ui_comment != null && y.ui_comment != null && x.ui_comment.value == y.ui_comment.value)) &&
                    (x.ui_group == y.ui_group || (x.ui_group != null && y.ui_group != null && x.ui_group.group_name == y.ui_group.group_name)) &&
                    (x.ui_order == y.ui_order || (x.ui_order != null && y.ui_order != null && x.ui_order.value == y.ui_order.value)) &&
                    (x.ui_item == y.ui_item || (x.ui_item != null && y.ui_item != null && x.ui_item.value == y.ui_item.value)) &&
                    (x.ui_min == y.ui_min || (x.ui_min != null && y.ui_min != null && x.ui_min.value == y.ui_min.value)) &&
                    (x.ui_max == y.ui_max || (x.ui_max != null && y.ui_max != null && x.ui_max.value == y.ui_max.value)) &&
                    (x.ui_default_min == y.ui_default_min || (x.ui_default_min != null && y.ui_default_min != null && x.ui_default_min.value == y.ui_default_min.value)) &&
                    (x.ui_default_max == y.ui_default_max || (x.ui_default_max != null && y.ui_default_max != null && x.ui_default_max.value == y.ui_default_max.value)) &&
                    (x.ui_visible == y.ui_visible || (x.ui_visible != null && y.ui_visible != null && x.ui_visible.value == y.ui_visible.value));
            }

            public bool Equals(uniform_varType x, uniform_varType y)
            {
                return UniformEquals(x, y);
            }

            public int GetHashCode(uniform_varType x)
            {
                int code = 0;
                code = MixHash(code, x.id);
                code = MixHash(code, x.hint);
                code = MixValueHash(code, x.type);
                code = MixHash(code, x.ui_label == null ? null : x.ui_label.value);
                code = MixHash(code, x.ui_comment == null ? null : x.ui_comment.value);
                code = MixHash(code, x.ui_group == null ? null : x.ui_group.group_name);
                code = MixValueHash(code, x.ui_order == null ? -1 : x.ui_order.value);
                code = MixValueHash(code, x.ui_item == null ? -1 : (int)x.ui_item.value);
                code = MixValueHash(code, x.ui_min == null ? float.NaN : x.ui_min.value);
                code = MixValueHash(code, x.ui_max == null ? float.NaN : x.ui_max.value);
                code = MixValueHash(code, x.ui_default_min == null ? float.NaN : x.ui_default_min.value);
                code = MixValueHash(code, x.ui_default_max == null ? float.NaN : x.ui_default_max.value);
                code = MixValueHash(code, x.ui_visible == null ? -1 : Convert.ToInt32(x.ui_visible.value));
                return code;
            }

            public static UniformComparer TheComparer = new UniformComparer();
        }

        /// <summary>
        /// uniform_varType の比較
        /// </summary>
        public class UniformIdTypeComparer : IEqualityComparer<uniform_varType>
        {
            public static bool UniformEquals(uniform_varType x, uniform_varType y)
            {
                return x.id == y.id &&
                    x.type == y.type;
            }

            public bool Equals(uniform_varType x, uniform_varType y)
            {
                return UniformEquals(x, y);
            }

            public int GetHashCode(uniform_varType x)
            {
                int code = 0;
                code = MixHash(code, x.id);
                code = MixValueHash(code, x.type);
                return code;
            }

            public static UniformIdTypeComparer TheComparer = new UniformIdTypeComparer();
        }

        public class OptionVarComparer : IEqualityComparer<option_varType>
        {
            public static bool OptionEquals(option_varType x, option_varType y)
            {
                return x.id == y.id &&
                    x.choice == y.choice &&
                    x.@default == y.@default &&
                    (x.ui_label == y.ui_label || (x.ui_label != null && y.ui_label != null && x.ui_label.value == y.ui_label.value)) &&
                    (x.ui_comment == y.ui_comment || (x.ui_comment != null && y.ui_comment != null && x.ui_comment.value == y.ui_comment.value)) &&
                    (x.ui_group == y.ui_group || (x.ui_group != null && y.ui_group != null && x.ui_group.group_name == y.ui_group.group_name)) &&
                    (x.ui_visible == y.ui_visible || (x.ui_visible != null && y.ui_visible != null && x.ui_visible.value == y.ui_visible.value)) &&
                    (x.ui_order == y.ui_order || (x.ui_order != null && y.ui_order != null && x.ui_order.value == y.ui_order.value));

            }

            public bool Equals(option_varType x, option_varType y)
            {
                return OptionEquals(x, y);
            }

            public int GetHashCode(option_varType x)
            {
                int code = 0;
                code = MixHash(code, x.id);
                code = MixHash(code, x.choice);
                code = MixHash(code, x.@default);
                code = MixHash(code, x.ui_label == null ? null : x.ui_label.value);
                code = MixHash(code, x.ui_comment == null ? null : x.ui_comment.value);
                code = MixHash(code, x.ui_group == null ? null : x.ui_group.group_name);
                code = MixValueHash(code, x.ui_item == null ? -1 : Convert.ToInt32(x.ui_item.value));
                code = MixValueHash(code, x.ui_order == null ? -1 : x.ui_order.value);
                code = MixValueHash(code, x.ui_visible == null ? -1 : Convert.ToInt32(x.ui_visible.value));
                return code;
            }

            public static OptionVarComparer TheComparer = new OptionVarComparer();
        }

        public class OptionIdChoiceComparer : IEqualityComparer<option_varType>
        {
            public static bool OptionEquals(option_varType x, option_varType y)
            {
                return x.id == y.id &&
                    IfShaderOptionUtility.ChoiceEquivalent(x.choice, y.choice);
            }

            public bool Equals(option_varType x, option_varType y)
            {
                return OptionEquals(x, y);
            }

            public int GetHashCode(option_varType x)
            {
                int code = 0;
                code = MixHash(code, x.id);
                // choice の部分は未実装。動作はする。
                return code;
            }

            public static OptionIdChoiceComparer TheComparer = new OptionIdChoiceComparer();
        }

        public class AttribComparer : IEqualityComparer<attrib_varType>
        {
            public static bool AttribEquals(attrib_varType x, attrib_varType y)
            {
                return x.id == y.id &&
                    x.hint == y.hint &&
                    x.type == y.type &&
                    (x.ui_label == y.ui_label || (x.ui_label != null && y.ui_label != null && x.ui_label.value == y.ui_label.value)) &&
                    (x.ui_comment == y.ui_comment || (x.ui_comment != null && y.ui_comment != null && x.ui_comment.value == y.ui_comment.value)) &&
                    (x.ui_editable == y.ui_editable || (x.ui_editable != null && y.ui_editable != null && x.ui_editable == y.ui_editable)) &&
                    (x.ui_group == y.ui_group || (x.ui_group != null && y.ui_group != null && x.ui_group.group_name == y.ui_group.group_name));
            }

            public bool Equals(attrib_varType x, attrib_varType y)
            {
                return AttribEquals(x, y);
            }

            public int GetHashCode(attrib_varType x)
            {
                int code = 0;
                code = MixHash(code, x.id);
                code = MixHash(code, x.hint);
                code = MixValueHash(code, x.type);
                code = MixHash(code, x.ui_label == null ? null : x.ui_label.value);
                code = MixHash(code, x.ui_comment == null ? null : x.ui_comment.value);
                code = MixValueHash(code, x.ui_editable == null ? -1 : x.ui_editable.value ? 1 : 0);
                code = MixHash(code, x.ui_group == null ? null : x.ui_group.group_name);
                return code;
            }

            public static AttribComparer TheComparer = new AttribComparer();
        }

        public class AttribIdTypeComparer : IEqualityComparer<attrib_varType>
        {
            public static bool AttribEquals(attrib_varType x, attrib_varType y)
            {
                return x.id == y.id &&
                    x.type == y.type;
            }

            public bool Equals(attrib_varType x, attrib_varType y)
            {
                return AttribEquals(x, y);
            }

            public int GetHashCode(attrib_varType x)
            {
                int code = 0;
                code = MixHash(code, x.id);
                code = MixValueHash(code, x.type);
                return code;
            }

            public static AttribIdTypeComparer TheComparer = new AttribIdTypeComparer();
        }

        public class RenderInfoSlotComparer : IEqualityComparer<render_info_slotType>
        {
            public static bool RenderInfoSlotEquals(render_info_slotType x, render_info_slotType y)
            {
                return x.name == y.name &&
                    x.type == y.type &&
                    x.optional == y.optional &&
                    x.count == y.count &&
                    x.choice == y.choice &&
                    x.@default == y.@default &&
                    (x.ui_label == y.ui_label || (x.ui_label != null && y.ui_label != null && x.ui_label.value == y.ui_label.value)) &&
                    (x.ui_comment == y.ui_comment || (x.ui_comment != null && y.ui_comment != null && x.ui_comment.value == y.ui_comment.value)) &&
                    (x.ui_item == y.ui_item || (x.ui_item != null && y.ui_item != null && x.ui_item.value == y.ui_item.value)) &&
                    (x.ui_group == y.ui_group || (x.ui_group != null && y.ui_group != null && x.ui_group.group_name == y.ui_group.group_name));
            }

            public bool Equals(render_info_slotType x, render_info_slotType y)
            {
                return RenderInfoSlotEquals(x, y);
            }

            public int GetHashCode(render_info_slotType x)
            {
                int code = 0;
                code = MixHash(code, x.name);
                code = MixValueHash(code, x.type);
                code = MixValueHash(code, x.optional);
                code = MixValueHash(code, x.count);
                code = MixHash(code, x.choice);
                code = MixHash(code, x.@default);
                code = MixHash(code, x.ui_label == null ? null : x.ui_label.value);
                code = MixHash(code, x.ui_comment == null ? null : x.ui_comment.value);
                code = MixValueHash(code, x.ui_item == null ? -1 : (int)x.ui_item.value);
                code = MixHash(code, x.ui_group == null ? null : x.ui_group.group_name);
                return code;
            }

            public static RenderInfoSlotComparer TheComparer = new RenderInfoSlotComparer();
        }

        public class RenderInfoSlotNameTypeComparer : IEqualityComparer<render_info_slotType>
        {
            public static bool RenderInfoSlotEquals(render_info_slotType x, render_info_slotType y)
            {
                return x.name == y.name &&
                    x.type == y.type;
            }

            public bool Equals(render_info_slotType x, render_info_slotType y)
            {
                return RenderInfoSlotEquals(x, y);
            }

            public int GetHashCode(render_info_slotType x)
            {
                int code = 0;
                code = MixHash(code, x.name);
                code = MixValueHash(code, x.type);
                return code;
            }

            public static RenderInfoSlotNameTypeComparer TheComparer = new RenderInfoSlotNameTypeComparer();
        }

        public class OptionComparer : IEqualityComparer<optionType>
        {
            public bool Equals(optionType x, optionType y)
            {
                return x.id == y.id && x.choice == y.choice;
            }

            public int GetHashCode(optionType x)
            {
                int code = 0;
                code = MixHash(code, x.id);
                code = MixHash(code, x.choice);
                return code;
            }

            public static OptionComparer TheComparer = new OptionComparer();
        }

#if false
        public class GroupComparer : IEqualityComparer<groupType>
        {
            bool IEqualityComparer<groupType>.Equals(groupType x, groupType y)
            {
                return x.name == y.name &&
                    x.condition == y.condition &&
                    x.page_name == y.page_name &&
                    (x.ui_label == y.ui_label || (x.ui_label != null && y.ui_label != null && x.ui_label.value == y.ui_label.value)) &&
                    (x.ui_comment == y.ui_comment || (x.ui_comment != null && y.ui_comment != null && x.ui_comment.value == y.ui_comment.value));
            }

            int IEqualityComparer<groupType>.GetHashCode(groupType x)
            {
                int code = 0;
                code = MixHash(code, x.name);
                code = MixHash(code, x.condition);
                code = MixHash(code, x.page_name);
                code = MixHash(code, x.ui_label == null ? null : x.ui_label.value);
                code = MixHash(
                return code;
            }

            public static OptionComparer TheComparer = new OptionComparer();
        }
#endif
        public class MacroComparer : IEqualityComparer<macroType>
        {
            public bool Equals(macroType x, macroType y)
            {
                return x.name == y.name && x.value == y.value;
            }

            public int GetHashCode(macroType x)
            {
                int code = 0;
                code = MixHash(code, x.name);
                code = MixHash(code, x.value);
                return code;
            }

            public static MacroComparer TheComparer = new MacroComparer();
        }
        public class ShaderProgramComparer : IEqualityComparer<shader_programType>
        {
            public bool Equals(shader_programType x, shader_programType y)
            {
                return x.option_array.GetItems().SequenceEqual(y.option_array.GetItems(), OptionComparer.TheComparer);
            }

            public int GetHashCode(shader_programType x)
            {
                var optionHash = x.option_array != null ? ArrayHash(x.option_array.Items, OptionComparer.TheComparer) : 0;
                unchecked
                {
                    return 31 * optionHash;
                }
            }

            public static ShaderProgramComparer TheComparer = new ShaderProgramComparer();
        }

        public static int ArrayHash<T>(T[] a, IEqualityComparer<T> comparer)
        {
            int x = a.Length;
            foreach (var item in a)
            {
                unchecked
                {
                    x *= 31;
                }

                x ^= comparer.GetHashCode(item);
            }

            return x;
        }

        /// <summary>
        /// ハッシュ値を合成します。
        /// </summary>
        public static int MixHash<T>(int x, T obj) where T: class
        {
            unchecked
            {
                x *= 31;
            }

            return x ^ (obj == null ? 0: obj.GetHashCode());
        }

        /// <summary>
        /// ハッシュ値を合成します。
        /// </summary>
        public static int MixValueHash<T>(int x, T value) where T : struct
        {
            unchecked
            {
                x *= 31;
            }

            return x ^ value.GetHashCode();
        }

        public static render_info_typeType Type(this render_info_slotType slot)
        {
            switch (slot.type)
            {
                case render_info_slot_typeType.@float:
                    return render_info_typeType.@float;
                case render_info_slot_typeType.@int:
                    return render_info_typeType.@int;
                case render_info_slot_typeType.@string:
                    return render_info_typeType.@string;
            }

            Debug.Assert(false);
            throw new NotImplementedException();
        }

        public static IEnumerable<shader_paramType> ShaderParams(this shader_assignType shader_assign)
        {
            return shader_assign != null && shader_assign.shader_param_array != null ?
                shader_assign.shader_param_array.shader_param :
                Enumerable.Empty<shader_paramType>();
        }

        public static IEnumerable<shader_optionType> ShaderOptions(this shader_assignType shader_assign)
        {
            return shader_assign != null && shader_assign.shader_option_array != null ?
                shader_assign.shader_option_array.shader_option :
                Enumerable.Empty<shader_optionType>();
        }

        public static IEnumerable<sampler_assignType> SamplerAssigns(this shader_assignType shader_assign)
        {
            return shader_assign != null && shader_assign.sampler_assign_array != null ?
                shader_assign.sampler_assign_array.sampler_assign :
                Enumerable.Empty<sampler_assignType>();
        }

        public static IEnumerable<attrib_assignType> AttribAssigns(this shader_assignType shader_assign)
        {
            return shader_assign != null && shader_assign.attrib_assign_array != null ?
                shader_assign.attrib_assign_array.attrib_assign :
                Enumerable.Empty<attrib_assignType>();
        }

        public static IEnumerable<render_infoType> RenderInfos(this shader_assignType shader_assign)
        {
            return shader_assign != null && shader_assign.render_info_array != null ?
                shader_assign.render_info_array.render_info :
                Enumerable.Empty<render_infoType>();
        }

        public static IEnumerable<samplerType> Samplers(this materialType material)
        {
            return material.sampler_array != null ? material.sampler_array.sampler : Enumerable.Empty<samplerType>();
        }
    }
}
