﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using App.ConfigData;
using App.res;
using App.Controls;
using App.Data;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Utility
{
    public static class PrePostIO
    {
        private static bool Execute(string exeFilePath, string[] paths, StringBuilder errorBuilder, out Exception exception)
        {
            exception = null;
            string error = null;

            if (paths.Length == 0)
            {
                return true;
            }

            if (String.IsNullOrEmpty(exeFilePath))
            {
                return true;
            }

            if (exeFilePath.Contains("%"))
            {
                exeFilePath = Environment.ExpandEnvironmentVariables(exeFilePath);
            }

            if (File.Exists(exeFilePath))
            {
                using (var fileName = TemporaryFileUtility.MakeDisposableFileName(".txt"))
                {
                    var writer = new StreamWriter(fileName.Path, false, new UTF8Encoding(false));
                    foreach (var path in paths)
                    {
                        writer.WriteLine(Path.GetFullPath(path));
                    }
                    writer.Close();

                    var info = new ProcessStartInfo(exeFilePath, fileName.Path)
                    {
                        CreateNoWindow = true,
                        // リダイレクトするときは必要
                        UseShellExecute = false,
                        RedirectStandardError = true,
                        // RedirectStandardError だけ true にするとエラー出力がされる場合がある。
                        RedirectStandardOutput = true
                    };

                    Process process = null;
                    try
                    {
                        process = Process.Start(info);

                        // 受け取らなくてもよい
                        /*
                        var outputBuilder = new StringBuilder();
                        process.OutputDataReceived += (s, a) =>
                        {
                            if (!string.IsNullOrEmpty(a.Data))
                            {
                                outputBuilder.AppendLine(a.Data);
                            }
                        };
                        */
                        // これを行わないとプロセスが終了しないことがある。
                        process.BeginOutputReadLine();

                        process.ErrorDataReceived += (s, a) =>
                        {
                            if (!String.IsNullOrEmpty(a.Data))
                            {
                                errorBuilder.AppendLine(a.Data);
                            }
                        };
                        process.BeginErrorReadLine();

                        //process.WaitForExit(); だと Abort されない。詳しい理由は不明
                        while (!process.WaitForExit(10000))
                        {
                        }

                        // 問題なければ終わり
                        if (process.ExitCode == 0)
                        {
                            return true;
                        }
                    }
                    catch (Exception e)
                    {
                        if (process != null)
                        {
                            try
                            {
                                // プロセスが走っているかもしれないので止める
                                process.Kill();
                            }
                            catch
                            {
                                // 何もしない
                                DebugConsole.WriteLine("Failed to Kill process");
                            }
                        }

                        exception = e;
                    }
                }
            }
            else
            {
                errorBuilder.AppendLine(Strings.IO_PrePostSaveCommandFileNotFound);
                errorBuilder.AppendLine(exeFilePath);
            }

            error = errorBuilder.ToString();
            return false;
        }

        public static bool ExecutePreSave(string[] paths)
        {
            Exception exception;
            var errorBuilder = new StringBuilder();
            if (Execute(ApplicationConfig.FileIo.PreSaveCommand, paths, errorBuilder, out exception))
            {
                return true;
            }

            var labelText = exception == null ? Strings.IO_PreSaveCommandFailed : Strings.IO_PreSaveCommandFailedException;
            if (exception != null) errorBuilder.AppendLine(exception.ToString());

            App.AppContext.NotificationHandler.OkListBoxDialog(
                MessageContext.IO_PreSaveCommand,
                Strings.IO_PreSaveCommand,
                labelText,
                new[] { errorBuilder.ToString() }
                );
            return false;
        }

        public static bool ExecutePostSave(string[] paths)
        {
            Exception exception;
            var errorBuilder = new StringBuilder();
            if (Execute(ApplicationConfig.FileIo.PostSaveCommand, paths, errorBuilder, out exception))
            {
                return true;
            }

            var labelText = exception == null ? Strings.IO_PostSaveCommandFailed : Strings.IO_PostSaveCommandFailedException;
            if (exception != null) errorBuilder.AppendLine(exception.ToString());

            App.AppContext.NotificationHandler.OkListBoxDialog(
                MessageContext.IO_PostSaveCommand,
                Strings.IO_PostSaveCommand,
                labelText,
                new[] { errorBuilder.ToString() }
                );

            return false;
        }

        public static bool ExecutePreOpen(string[] paths, out Action showDialog)
        {
            Exception exception;
            var errorBuilder = new StringBuilder();
            if (Execute(ApplicationConfig.FileIo.PreOpenCommand, paths, errorBuilder, out exception))
            {
                showDialog = null;
                return true;
            }

            showDialog = () =>
            {
                var labelText = exception == null
                    ? Strings.IO_PreOpenCommandFailed + Strings.IO_PreOpenCommandFailed_Stop
                    : Strings.IO_PreOpenCommandFailedException + Strings.IO_PreOpenCommandFailed_Stop;
                    if (exception != null) errorBuilder.AppendLine(exception.ToString());

                    App.AppContext.NotificationHandler.OkListBoxDialog(
                        MessageContext.IO_PreOpenCommand,
                        Strings.IO_PreOpenCommand,
                        labelText,
                        new[] { errorBuilder.ToString() }
                        );
                };

            return false;
        }

        public static bool ExecutePreOpen(string[] paths)
        {
            Action showDialog;
            var result = ExecutePreOpen(paths, out showDialog);

            if (showDialog != null)
            {
                showDialog();
            }

            return result;
        }

        private static bool ExecutePreOpenUndoRedo(string[] paths, bool undo)
        {
            Exception exception;
            var errorBuilder = new StringBuilder();
            if (Execute(ApplicationConfig.FileIo.PreOpenCommand, paths, errorBuilder, out exception))
            {
                return true;
            }

            string append = undo ? Strings.OkCancelUndo : Strings.OkCancelRedo;
            var labelText = exception == null
                ? Strings.IO_PreOpenCommandFailed + append
                : Strings.IO_PreOpenCommandFailedException + append;
            if (exception != null) errorBuilder.AppendLine(exception.ToString());

            return App.AppContext.NotificationHandler.OkCancelTextBoxDialog(
                MessageContext.IO_PreOpenCommand,
                Strings.IO_PreOpenCommand,
                labelText,
                new[] { errorBuilder.ToString() }
                );
        }

        /// <summary>
        /// docs の各 Document.FilePath で PreOpenCommand を実行
        /// コマンド実行中に FilePath のファイルで変更があった場合は Document.NeedsReload に true が設定される。
        /// ただし Document が同ファイルを既に監視中であれば設定されない。
        /// </summary>
        public static bool ExecutePreOpenUndoRedo(IEnumerable<Document> docs, bool undo)
        {
            return ExecutePreOpenUndoRedo(docs.Select(x => Tuple.Create(x, x.FilePath)), undo);
        }

        /// <summary>
        /// docs の各 Item2 のファイルパスで PreOpenCommand を実行
        /// コマンド実行中に Item2 のファイルで変更があった場合は Item1.NeedsReload に true が設定される。
        /// ただし Item1 が同ファイルを既に監視中であれば設定されない。
        /// </summary>
        public static bool ExecutePreOpenUndoRedo(IEnumerable<Tuple<Document, string>> docs, bool undo)
        {
            bool ret;

            var groups = docs
                .Select((x, i) => new
                {
                    Index = i,
                    Document = x.Item1,
                    FileLocation = !string.IsNullOrEmpty(x.Item2) ? Path.GetDirectoryName(x.Item2) : string.Empty,
                    FileName = !string.IsNullOrEmpty(x.Item2) ? Path.GetFileName(x.Item2) : x.Item2,
                    Watcher = new FileSystemWatcher(),
                    NeedsWatching = !(x.Item1.IsWatched && string.Equals(x.Item1.FilePath, x.Item2, StringComparison.OrdinalIgnoreCase)) && !string.IsNullOrEmpty(x.Item2)
                }).ToArray();

            var needsReload = Enumerable.Repeat<long>(0, groups.Length).ToArray();
            using (var dc = new DisposableCollection<FileSystemWatcher>(groups.Select(x => x.Watcher)))
            {
                dc.ItemDisposing += (ss, ee) =>
                {
                    ee.Item.EnableRaisingEvents = false;
                };

                foreach (var group in groups.Where(x => x.NeedsWatching && Directory.Exists(Path.GetDirectoryName(x.FileLocation))))
                {
                    var doc = group.Document;
                    var watcher = group.Watcher;
                    watcher.NotifyFilter = NotifyFilters.LastWrite | NotifyFilters.CreationTime | NotifyFilters.FileName;
                    watcher.SynchronizingObject = null;
                    watcher.Path = group.FileLocation;
                    watcher.Filter = group.FileName;
                    watcher.Changed += (ss, ee) => { Interlocked.Exchange(ref needsReload[group.Index], 1); };
                    watcher.Renamed += (ss, ee) => { Interlocked.Exchange(ref needsReload[group.Index], 1); };
                    watcher.Created += (ss, ee) => { Interlocked.Exchange(ref needsReload[group.Index], 1); };
                    watcher.EnableRaisingEvents = true;
                }

                ret = ExecutePreOpenUndoRedo(docs.Select(x => x.Item2).ToArray(), undo);
            }

            // DisposableCollection.Dispose() でのウォッチャー破棄後に Document.NeedsReload に値を設定する。
            foreach (var doc in Enumerable.Range(0, needsReload.Length)
                .Select(x => new { Value = Interlocked.Read(ref needsReload[x]), Index = x })
                .Where(x => x.Value != 0)
                .Select(x => groups[x.Index].Document))
            {
                doc.NeedsReload = true;
            }

            return ret;
        }

        public static List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> ExecutePreBinarize(List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> list)
        {
            if (list == null || !list.Any() || !ApplicationConfig.FileIo.PreBinarizeCommand.HasCommand)
            {
                return list;
            }

            var idlist = ApplicationConfig.FileIo.PreBinarizeCommand.FileIDs;
            if (idlist == null || idlist.Count == 0)
            {
                idlist = new List<GuiObjectID>();
                foreach (var ext in ApplicationConfig.FileIo.PreBinarizeCommand.FilterExts)
                {
                    var id = ObjectIDUtility.ExtToId(ext);
                    if (id != null)
                    {
                        idlist.Add(id.GetValueOrDefault());
                    }
                }
                ApplicationConfig.FileIo.PreBinarizeCommand.FileIDs = idlist;
            }

            var newlist = new List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>>();

            // ドキュメントをファイルに出力してからコマンドを実行する
            using (var tempDir = TemporaryFileUtility.MakeDisposableDirectoryName())
            {
                var disposableFiles = new List<DisposableFileName>();

                foreach (var tuple in list)
                {
                    // Filterが指定されていない時はすべての種類でコマンドを実行
                    if (idlist.Any() && !idlist.Contains(tuple.Item1.ObjectID))
                    {
                        newlist.Add(tuple);
                        disposableFiles.Add(null);
                        continue;
                    }

                    var filepath = Path.Combine(tempDir.Path,
                        tuple.Item4 + "." + ObjectIDUtility.IdToExt(tuple.Item1.ObjectID).First());
                    if (G3dPath.IsTextPath(filepath))
                    {
                        filepath = G3dPath.ToBinaryPath(filepath);
                    }

                    disposableFiles.Add(new DisposableFileName(filepath));

                    // fileinfoのcreateがnullの時はfileinfo自体をnullにする(IfTextFormaterで落ちるため)
                    if (tuple.Item2.file_info != null && tuple.Item2.file_info.create == null)
                    {
                        tuple.Item2.file_info = null;
                    }

                    IfWriteUtility.Write(tuple.Item2, tuple.Item3, filepath);
                    var newtuple =
                        new Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>(
                            (IntermediateFileDocument) tuple.Item1,
                            null,
                            null,
                            tuple.Item4
                            );

                    newlist.Add(newtuple);
                }

                Debug.Assert(disposableFiles.Count == newlist.Count);

                // call command
                Exception exception;
                var files = disposableFiles.Where(x => x != null).Select(x => x.Path).ToArray();
                var errorBuilder = new StringBuilder();
                if (!Execute(ApplicationConfig.FileIo.PreBinarizeCommand.FullPath, files, errorBuilder,out exception))
                {
                    var labelText = exception == null
                        ? Strings.PrePostIO_ExecutePreBinarizeCommandFailed
                        : Strings.PrePostIO_ExecutePreBinarizeCommandFailedException;
                    if (exception != null) errorBuilder.AppendLine(exception.ToString());

                    App.AppContext.NotificationHandler.OkListBoxDialog(
                        MessageContext.IO_PreBinarizeCommand,
                        Strings.IO_PreBinarizeCommand,
                        labelText,
                        new[] {errorBuilder.ToString()}
                        );

                    return null;
                }

                for (var i = 0; i < newlist.Count; i++)
                {
                    var tuple = newlist[i];
                    var file = disposableFiles[i];
                    if (file != null)
                    {
                        var newStreams = new List<G3dStream>();
                        var newIfType = IfReadUtility.Read(newStreams, file.Path, null);
                        var newtuple =
                            new Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>(
                                (IntermediateFileDocument)tuple.Item1,
                                newIfType,
                                newStreams,
                                tuple.Item4
                                );
                        newlist[i] = newtuple;
                    }
                }
            }

            return newlist;
        }

        public static bool ExecutePostClose(string[] paths, out Action showDialog)
        {
            Exception exception;
            var errorBuilder = new StringBuilder();
            if (Execute(ApplicationConfig.FileIo.PostCloseCommand, paths, errorBuilder, out exception))
            {
                showDialog = null;
                return true;
            }

            showDialog = () =>
            {
                var labelText = exception == null
                    ? Strings.IO_PostCloseCommandFailed + Strings.IO_PostCloseCommandFailed_Stop
                    : Strings.IO_PostCloseCommandFailedException + Strings.IO_PostCloseCommandFailed_Stop;
                if (exception != null) errorBuilder.AppendLine(exception.ToString());

                App.AppContext.NotificationHandler.OkListBoxDialog(
                    MessageContext.IO_PostCloseCommand,
                    Strings.IO_PostCloseCommand,
                    labelText,
                    new[] { errorBuilder.ToString() }
                    );
            };

            return false;
        }

        public static bool ExecutePostClose(string[] paths)
        {
            Action showDialog;
            var result = ExecutePostClose(paths, out showDialog);

            if (showDialog != null)
            {
                showDialog();
            }

            return result;
        }
    }
}
