﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using App.ConfigData;
using nw.g3d.nw4f_3dif;
using System.IO;
using System.Linq;

namespace App.Utility
{
    static public class G3dHioLibProxy
    {
        public static NintendoWare.G3d.Edit.HIO Hio{ get; private set; }

        static G3dHioLibProxy()
        {
        }

        public static void Initialize()
        {
            NintendoWare.G3d.Edit.HIO.InitializeHIO();
            Hio = NintendoWare.G3d.Edit.HIO.GetInstance();

            OneShotIdleProcess.Execute(() =>
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=893
                ChangeCommDevice(ApplicationConfig.Setting.Preview.Target, App.AppContext.SelectedPlatformPreset)
            );
        }

        public static void Destroy()
        {
            NintendoWare.G3d.Edit.HIO.FinalizeHIO();
        }

        public static void ChangeCommDevice(ConfigCommon.HioTarget target, TeamConfig.PlatformPreset platform)
        {
            bool useHtc = platform.Name != "Cafe";
            platform.ActiveDeviceOption = platform.DeviceOptions.FirstOrDefault(x => (x.DeviceType == "Pc") == (target == ConfigCommon.HioTarget.Pc));

            string peerType = platform.ActiveDeviceOption.PeerType;
            NintendoWare.G3d.Edit.HIOBase.TargetType targetType;
            if (useHtc)
            {
                targetType = NintendoWare.G3d.Edit.HIOBase.TargetType.Htc;
            }
            else
            {
                if (target == ConfigCommon.HioTarget.Device)
                {
                    targetType = NintendoWare.G3d.Edit.HIOBase.TargetType.Cafe;
                }
                else
                {
                    targetType = NintendoWare.G3d.Edit.HIOBase.TargetType.OldPc;
                }
            }

            if (Viewer.Manager.Instance.PollPingLockObject == null)
            {
                Hio.ChangeCommDevice(targetType, peerType);
            }
            else
            {
                lock(Viewer.Manager.Instance.PollPingLockObject)
                {
                    Hio.ChangeCommDevice(targetType, peerType);
                }
            }
        }

        private static UInt16?	codePage_;
        public static UInt16?	CodePage
        {
            get{	return Hio.IsConnected ? codePage_ : null;	}
            set{	codePage_ = value;						}
        }

        public static object CreateVector(float[] v)
        {
            switch (v.Length)
            {
                case 2:	return new NintendoWare.G3d.Edit.Math.Vector2(){ X = v[0], Y = v[1]						};
                case 3:	return new NintendoWare.G3d.Edit.Math.Vector3(){ X = v[0], Y = v[1], Z = v[2]			};
                case 4:	return new NintendoWare.G3d.Edit.Math.Vector4(){ X = v[0], Y = v[1], Z = v[2], W = v[3]	};
            }

            Debug.Assert(false);
            return null;
        }

        public static object CreateVectorI(int[] v)
        {
            switch (v.Length)
            {
                case 2:	return new NintendoWare.G3d.Edit.Math.Vector2i(){ X = v[0], Y = v[1]						};
                case 3:	return new NintendoWare.G3d.Edit.Math.Vector3i(){ X = v[0], Y = v[1], Z = v[2]				};
                case 4:	return new NintendoWare.G3d.Edit.Math.Vector4i(){ X = v[0], Y = v[1], Z = v[2], W = v[3]	};
            }

            Debug.Assert(false);
            return null;
        }

        public static object CreateVectorU(uint[] v)
        {
            switch (v.Length)
            {
                case 2:	return new NintendoWare.G3d.Edit.Math.Vector2u(){ X = v[0], Y = v[1]						};
                case 3:	return new NintendoWare.G3d.Edit.Math.Vector3u(){ X = v[0], Y = v[1], Z = v[2]				};
                case 4:	return new NintendoWare.G3d.Edit.Math.Vector4u(){ X = v[0], Y = v[1], Z = v[2], W = v[3]	};
            }

            Debug.Assert(false);
            return null;
        }

        public static object CreateVectorB(bool[] v)
        {
            switch (v.Length)
            {
                case 2:	return new NintendoWare.G3d.Edit.Math.Vector2b(){ X = v[0], Y = v[1]						};
                case 3:	return new NintendoWare.G3d.Edit.Math.Vector3b(){ X = v[0], Y = v[1], Z = v[2]				};
                case 4:	return new NintendoWare.G3d.Edit.Math.Vector4b(){ X = v[0], Y = v[1], Z = v[2], W = v[3]	};
            }

            Debug.Assert(false);
            return null;
        }

        public static object CreateMatrix(shader_param_typeType type, float[] v)
        {
            Debug.Assert(v != null);
            Debug.Assert(ShaderTypeUtility.ShaderParamTypeKindFromType(type) == ShaderTypeUtility.ShaderParamTypeKind.Matrix);
            Debug.Assert(ShaderTypeUtility.Dimension(type) == v.Length);

            switch(type)
            {
                case shader_param_typeType.float2x2:
                    return
                        new NintendoWare.G3d.Edit.Math.Matrix22()
                        {
                            M00 = v[ 0], M01 = v[ 1],
                            M10 = v[ 2], M11 = v[ 3]
                        };

                case shader_param_typeType.float2x3:
                    return
                        new NintendoWare.G3d.Edit.Math.Matrix23()
                        {
                            M00 = v[ 0], M01 = v[ 1], M02 = v[ 2],
                            M10 = v[ 3], M11 = v[ 4], M12 = v[ 5]
                        };

                case shader_param_typeType.float2x4:
                    return
                        new NintendoWare.G3d.Edit.Math.Matrix24()
                        {
                            M00 = v[ 0], M01 = v[ 1], M02 = v[ 2], M03 = v[ 3],
                            M10 = v[ 4], M11 = v[ 5], M12 = v[ 6], M13 = v[ 7]
                        };

                case shader_param_typeType.float3x2:
                    return
                        new NintendoWare.G3d.Edit.Math.Matrix32()
                        {
                            M00 = v[ 0], M01 = v[ 1],
                            M10 = v[ 2], M11 = v[ 3],
                            M20 = v[ 4], M21 = v[ 5]
                        };

                case shader_param_typeType.float3x3:
                    return
                        new NintendoWare.G3d.Edit.Math.Matrix33()
                        {
                            M00 = v[ 0], M01 = v[ 1], M02 = v[ 2],
                            M10 = v[ 3], M11 = v[ 4], M12 = v[ 5],
                            M20 = v[ 6], M21 = v[ 7], M22 = v[ 8]
                        };

                case shader_param_typeType.float3x4:
                    return
                        new NintendoWare.G3d.Edit.Math.Matrix34()
                        {
                            M00 = v[ 0], M01 = v[ 1], M02 = v[ 2], M03 = v[ 3],
                            M10 = v[ 4], M11 = v[ 5], M12 = v[ 6], M13 = v[ 7],
                            M20 = v[ 8], M21 = v[ 9], M22 = v[10], M23 = v[11]
                        };

                case shader_param_typeType.float4x2:
                    return
                        new NintendoWare.G3d.Edit.Math.Matrix42()
                        {
                            M00 = v[ 0], M01 = v[ 1],
                            M10 = v[ 2], M11 = v[ 3],
                            M20 = v[ 4], M21 = v[ 5],
                            M30 = v[ 6], M31 = v[ 7]
                        };

                case shader_param_typeType.float4x3:
                    return
                        new NintendoWare.G3d.Edit.Math.Matrix43()
                        {
                            M00 = v[ 0], M01 = v[ 1], M02 = v[ 2],
                            M10 = v[ 3], M11 = v[ 4], M12 = v[ 5],
                            M20 = v[ 6], M21 = v[ 7], M22 = v[ 8],
                            M30 = v[ 9], M31 = v[10], M32 = v[11]
                        };

                case shader_param_typeType.float4x4:
                    return
                        new NintendoWare.G3d.Edit.Math.Matrix44()
                        {
                            M00 = v[ 0], M01 = v[ 1], M02 = v[ 2], M03 = v[ 3],
                            M10 = v[ 4], M11 = v[ 5], M12 = v[ 6], M13 = v[ 7],
                            M20 = v[ 8], M21 = v[ 9], M22 = v[10], M23 = v[11],
                            M30 = v[12], M31 = v[13], M32 = v[14], M33 = v[15]
                        };
            }

            Debug.Assert(false);
            return null;
        }

        public static object CreateTextureSrt(shader_param_typeType type, float[] v)
        {
            Debug.Assert(v != null);
            Debug.Assert(ShaderTypeUtility.ShaderParamTypeKindFromType(type) == ShaderTypeUtility.ShaderParamTypeKind.TextureSrt);
            Debug.Assert(ShaderTypeUtility.Dimension(type) == v.Length);

            switch(type)
            {
                case shader_param_typeType.srt2d:
                {
                    var srt = new NintendoWare.G3d.Edit.Math.TextureSRT2D();
                    {
                        srt.Scale.X = v[0];
                        srt.Scale.Y = v[1];
                        srt.Rotate = MathUtility.ToRadian(v[2]);
                        srt.Trans.X = v[3];
                        srt.Trans.Y = v[4];
                        srt.Kind = NintendoWare.G3d.Edit.Math.TextureSRT2D.SRTKind.SRTNormal;
                    }
                    return srt;
                }
                case shader_param_typeType.texsrt:
                {
                    var srt = new NintendoWare.G3d.Edit.Math.TextureSRT2D();
                    {
                        srt.Scale.X = v[1];
                        srt.Scale.Y = v[2];
                        srt.Rotate  = MathUtility.ToRadian(v[3]);
                        srt.Trans.X = v[4];
                        srt.Trans.Y = v[5];
                    }

                    switch ((int)v[0])
                    {
                        case 0:
                            srt.Kind = NintendoWare.G3d.Edit.Math.TextureSRT2D.SRTKind.SRTMaya;
                            break;
                        case 1:
                            srt.Kind = NintendoWare.G3d.Edit.Math.TextureSRT2D.SRTKind.SRT3DSMax;
                            break;
                        case 2:
                            srt.Kind = NintendoWare.G3d.Edit.Math.TextureSRT2D.SRTKind.SRTSoftimage;
                            break;
                        default:
                            Debug.Assert(false);
                            break;
                    }

                    return srt;
                }

                case shader_param_typeType.srt3d:
                {
                    var srt = new NintendoWare.G3d.Edit.Math.TextureSRT3D();
                    {
                        srt.Scale.X  = v[0];
                        srt.Scale.Y  = v[1];
                        srt.Scale.Z  = v[2];
                        srt.Rotate.X = MathUtility.ToRadian(v[3]);
                        srt.Rotate.Y = MathUtility.ToRadian(v[4]);
                        srt.Rotate.Z = MathUtility.ToRadian(v[5]);
                        srt.Trans.X  = v[6];
                        srt.Trans.Y  = v[7];
                        srt.Trans.Z  = v[8];
                    }
                    return srt;
                }
            }

            Debug.Assert(false);
            return null;
        }

        public static NintendoWare.G3d.Edit.Math.VectorChannelFlag VectorChannelFlag(uint elementBits)
        {
            NintendoWare.G3d.Edit.Math.VectorChannelFlag f = 0;
            {
                if((elementBits & (1<<0)) != 0){	f |= NintendoWare.G3d.Edit.Math.VectorChannelFlag.XAxis;	}
                if((elementBits & (1<<1)) != 0){	f |= NintendoWare.G3d.Edit.Math.VectorChannelFlag.YAxis;	}
                if((elementBits & (1<<2)) != 0){	f |= NintendoWare.G3d.Edit.Math.VectorChannelFlag.ZAxis;	}
                if((elementBits & (1<<3)) != 0){	f |= NintendoWare.G3d.Edit.Math.VectorChannelFlag.WAxis;	}
            }
            return f;
        }

        public static NintendoWare.G3d.Edit.Drawing.ColorChannelFlag ColorChannelFlag(uint elementBits)
        {
            NintendoWare.G3d.Edit.Drawing.ColorChannelFlag f = 0;
            {
                if((elementBits & (1<<0)) != 0){	f |= NintendoWare.G3d.Edit.Drawing.ColorChannelFlag.Red;	}
                if((elementBits & (1<<1)) != 0){	f |= NintendoWare.G3d.Edit.Drawing.ColorChannelFlag.Green;	}
                if((elementBits & (1<<2)) != 0){	f |= NintendoWare.G3d.Edit.Drawing.ColorChannelFlag.Blue;	}
                if((elementBits & (1<<3)) != 0){	f |= NintendoWare.G3d.Edit.Drawing.ColorChannelFlag.Alpha;	}
            }
            return f;
        }

        public static bool Connect()
        {
            NintendoWare.G3d.Edit.HtcResult result = G3dHioLibProxy.Hio.Connect();
            DebugConsole.WriteLine(
                string.Format(
                "Connect Result: ResultCode = {0}, ErrorDetail = {1}, ErrorDetailRawValue = {2}",
                result.ResultCode, result.ErrorDetailCode, result.ErrorDetailRawValue));
            return result.IsSuccess;
        }

        public static bool CancelAttachShaderArchive(uint shaderArchiveKey)
        {
            DebugConsole.WriteLine(string.Format("CancelAttachShaderArchive {0}", shaderArchiveKey));
            return G3dHioLibProxy.Hio.CancelAttachShaderArchive(shaderArchiveKey);
        }

        public static bool CancelAttachModel(uint modelObjKey)
        {
            DebugConsole.WriteLine(string.Format("CancelAttachModel {0}", modelObjKey));
            return G3dHioLibProxy.Hio.CancelAttachModel(modelObjKey);
        }
    }
}
