﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.IO;
using System.Linq;
using System.Security.AccessControl;
using System.Security.Principal;

namespace App.Utility
{
    public static class FileUtility
    {
        // http://stackoverflow.com/questions/1281620/checking-for-directory-and-file-write-permissions-in-net/1281638
        // http://intre.net/item_6569.html
        // ここを参考に

        // path にはファイルパス、ディレクトリパスの両方を指定出来ます。
        // 絶対パスのみに対して有効
        public static bool HasWritePermission(string path)
        {
            FileSystemSecurity accessControlList = null;
            var isDir = false;
            {
                if(Directory.Exists(path))
                {
                    accessControlList = Directory.GetAccessControl(path);
                    isDir = true;
                }
                else
                if (File.Exists(path))
                {
                    accessControlList = File.GetAccessControl(path);
                    isDir = false;
                }
            }

            if (accessControlList != null)
            {
                // ファイル上書き時、もしくはディレクトリ
                var writeAllow = false;
                var writeDeny = false;
                {
                    var accessRules = accessControlList.GetAccessRules(true, true, typeof(SecurityIdentifier)).OfType<FileSystemAccessRule>();

                    var currentIdentity = WindowsIdentity.GetCurrent();
                    var sids = new[] { currentIdentity.User }.Concat(currentIdentity.Groups);

                    foreach (var rule in accessRules)
                    {
                        if ((rule.FileSystemRights & FileSystemRights.Write) != FileSystemRights.Write)
                        {
                            continue;
                        }

                        if (sids.Any(sid => sid == rule.IdentityReference) == false)
                        {
                            continue;
                        }

                        if (rule.AccessControlType == AccessControlType.Allow)
                        {
                            writeAllow = true;
                        }
                        else
                        if (rule.AccessControlType == AccessControlType.Deny)
                        {
                            writeDeny = true;
                        }
                    }
                }

                if (isDir)
                {
                    return writeAllow && (writeDeny == false);
                }
                else
                {
                    return writeAllow && (writeDeny == false) && HasWritePermission(Path.GetDirectoryName(path));
                }
            }
            else
            {
                // 新規書き込みファイル時は、その出力先のディレクトリをチェックする
                return HasWritePermission(Path.GetDirectoryName(path));
            }
        }
    }
}
